(* TBTree16             Copyright (c)  1988,1989       Dean H. Farwell II    *)

unit ByteData;

(*****************************************************************************)
(*                                                                           *)
(*                   B Y T E   D A T A   A R R A Y   R O U T I N E S         *)
(*                                                                           *)
(*****************************************************************************)

(* This unit implements a new data type, the byte array, which is an array of
   bytes similar to a string.  It is similar in that it can hold between 0 and
   255 bytes of data.  It can have a maximum length of 256 bytes. The first
   byte in the array (byte 0) contains the number of data bytes in the array
   (0 - 255).

   One big difference between a byte array and a string is that you can
   declare a string variable to be whatever size you want.  You can not do
   this with other types, including this one.  However, you can declare your
   own type to be an array of bytes the length you want.  These routines use
   untyped parameters so that they will handle an array of between 0 and 255
   bytes of data (length of 1 to 256 bytes including the 0th byte, the length
   byte).

   The only other real difference between a string and a byte array is that
   the byte array contains bytes which are just raw data and do not represent
   characters.

   You can use this unit in two ways.  You can declare variables of the type
   ByteData which will take 256 bytes of memory and you can use the routines
   as desired.  Or, you can declare a  variable to be an array of bytes the
   length you need (plus one byte for the length).  You can still use the
   routines.  You must ensure that you use variables that are big enough.  If
   you use a routine that produces a byte array of length 10 (eleven bytes
   required) you better have a variable big enough.  If not, this will write
   over some other variable in memory and really bad things are bound to
   happen.  If you do not understand how this works, do not get fancy.

   This unit can be used to implement concatenated indexes, but this is not
   obvious.  You do this by creating the index with type ByteArrayValue and
   using these routines to concatenate the values together.  It really only
   makes sense to do retrievals on equality if this method is used.  Someday,
   I'll get around to developing an example of how to use this.  This was
   developed for use with TBASE.

   Note - To use the 8087 types you must compile this unit using {$N+}       *)

(*\*)
(* Version Information

   Version 1.1 - Unit did not exist.

   Version 1.2 - Unit did not exist.

   Version 1.3 - Unit did not exist.

   Version 1.4 - Unit created with this version.

   Version 1.5 - Changed code internally to use newly added FastMove unit

   Version 1.6 - No Changes                                                  *)


(*////////////////////////// I N T E R F A C E //////////////////////////////*)

interface

uses
    FastMove,
    Numbers,
    Strings;

type
    ByteArrayRange = 0 .. MAXBYTE;

    ByteArray = Array [ByteArrayRange] of Byte;    (* This handy type is used
                                                      to store a from 1 to
                                                      255 bytes.  It is much
                                                      a string in that the
                                                      first element is the
                                                      number of bytes in the
                                                      array. All bytes after
                                                      the significant number
                                                      of bytes are not
                                                      significant.  This is
                                                      used for concatenated
                                                      indexes                *)

(*\*)
(* This routine will take an input string and create a byte array.  The number
of bytes held in the byte array is equal to the maximum number of characters
which the string will handle.  For example if a string is passed in which was
declared as a type String[20] then the new byte array will contain 20 bytes of
information (plus the 0th byte which contains the length).  The maximum length
is passed in using the maxLength parameter.  If the string passed in is
shorter than the maximum allowed, the end of the string is filled with as many
0's as required.                                                             *)

procedure ConvertStringToByteArray(str : String;
                                   maxLength : StringLengthRange;
                                   var resultArray);


(* This routine will take two values and will create a new byte array which
is a concatenation of the two input values.  The input values are of type
ValueType.  All the ValueType(s)s are supported.  The size of the resulting
byte array will be equal to the sum of the sizes of the two input values.
However, if the resulting size will exceed 255 byte, the resulting array will
be truncated and only the first 255 bytes will be returned.

Note: Be sure that your variable which is used for the returned array is big
enough!!!                                                                    *)

procedure ConcatenateTwoValues(var val1;
                               vType1 : ValueType;
                               var val2;
                               vType2 : ValueType;
                               var resultArray);


(* This routine will take three values and will create a new byte array which
is a concatenation of the three input values.  The input values are of type
ValueType.  All the ValueType(s)s are supported.  The size of the resulting
byte array will be equal to the sum of the sizes of the three input values.
However, if the resulting size will exceed 255 byte, the resulting array will
be truncated and only the first 255 bytes will be returned.

Note: Be sure that your variable which is used for the returned array is big
enough!!!                                                                    *)

procedure ConcatenateThreeValues(var val1;
                                 vType1 : ValueType;
                                 var val2;
                                 vType2 : ValueType;
                                 var val3;
                                 vType3 : ValueType;
                                 var resultArray);

(*!*)
(*\*)
(*///////////////////// I M P L E M E N T A T I O N /////////////////////////*)

implementation

(* This routine will take an input string and create a byte array.  The number
of bytes held in the byte array is equal to the maximum number of characters
whcih the string will handle.  For example if a string is passed in which was
declared as a type String[20] then the new byte array will contain 20 bytes of
information (plus the 0th byte which contains the length).  The maximum length
is passed in using the maxLength parameter.  If the string passed in is
shorter than the maximum allowed, the end of the string is filled with as many
0's as required.                                                             *)

procedure ConvertStringToByteArray(str : String;
                                   maxLength : StringLengthRange;
                                   var resultArray);

var
    resultByteArray : ByteArray absolute resultArray;

    begin
    FastMover(str,resultByteArray,Length(str) + 1);
    FillChar(resultByteArray[resultByteArray[0] + 1],
             maxLength - resultByteArray[0],0);
    resultByteArray[0] := maxLength;
    end;                                   (* end of NormalizeString routine *)

(*\*)
(* This routine will take two values and will create a new byte array which
is a concatenation of the two input values.  The input values are of type
ValueType.  All the ValueType(s)s are supported.  The size of the resulting
byte array will be equal to the sum of the sizes of the two input values.
However, if the resulting size will exceed 255 byte, the resulting array will
be truncated and only the first 255 bytes will be returned.

Note: Be sure that your variable which is used for the returned array is big
enough!!!                                                                    *)

procedure ConcatenateTwoValues(var val1;
                               vType1 : ValueType;
                               var val2;
                               vType2 : ValueType;
                               var resultArray);

var
    byteArray1 : ByteArray absolute val1;
    byteArray2 : ByteArray absolute val2;
    resultBytearray : ByteArray absolute resultArray;
    size1,
    size2,
    size3 : ByteArrayRange;   (* always equal to the number of bytes of data
                                 not counting the byte required to keep the
                                 length (BYTEARRAYVALUE and STRINGVALUE)     *)

    begin
    case vType1 of
        BYTEARRAYVALUE : size1 := byteArray1[0];
        BYTEVALUE : size1 := BYTESIZE;
        SHORTINTVALUE : size1 := SHORTINTSIZE;
        INTEGERVALUE : size1 := INTEGERSIZE;
        LONGINTVALUE : size1 := LONGINTSIZE;
        WORDVALUE : size1 := WORDSIZE;
        STRINGVALUE : size1 := byteArray1[0];
        REALVALUE : size1 := REALSIZE;
(*   The following types are only for 8087 - and are compiled only if the unit
     is compiled using {$N+}                                                 *)

{$IFOPT N+}
        SINGLEVALUE : size1 := SINGLESIZE;
        DOUBLEVALUE : size1 := DOUBLESIZE;
        EXTENDEDVALUE : size1 := EXTENDEDSIZE;
        COMPVALUE : size1 := COMPSIZE;
{$ENDIF}
        end;                                       (* end of case statement *)
    if vType1 in [BYTEARRAYVALUE,STRINGVALUE] then
        begin
        FastMover(byteArray1[1],resultByteArray[1],size1);
        end
    else
        begin
        FastMover(val1,resultByteArray[1],size1);
        end;
    case vType2 of
        BYTEARRAYVALUE : size2 := byteArray2[0];
        BYTEVALUE : size2 := BYTESIZE;
        SHORTINTVALUE : size2 := SHORTINTSIZE;
        INTEGERVALUE : size2 := INTEGERSIZE;
        LONGINTVALUE : size2 := LONGINTSIZE;
        WORDVALUE : size2 := WORDSIZE;
        STRINGVALUE : size2 := byteArray2[0];
        REALVALUE : size2 := REALSIZE;
(*   The following types are only for 8087 - and are compiled only if the unit
     is compiled using {$N+}                                                 *)

{$IFOPT N+}
        SINGLEVALUE : size2 := SINGLESIZE;
        DOUBLEVALUE : size2 := DOUBLESIZE;
        EXTENDEDVALUE : size2 := EXTENDEDSIZE;
        COMPVALUE : size2 := COMPSIZE;
{$ENDIF}

        end;                                       (* end of case statement *)
    if (size1 + size2) > MAXBYTE then
        begin
        size2 := MAXBYTE - size1;
        end;
    if vType2 in [BYTEARRAYVALUE,STRINGVALUE] then
        begin
        FastMover(byteArray2[1],resultByteArray[size1 + 1],size2);
        end
    else
        begin
        FastMover(val2,resultByteArray[size1 + 1],size2);
        end;
    resultByteArray[0] := size1 + size2;
    end;                              (* end of ConcatenateTwoValues routine *)

(*\*)
(* This routine will take three values and will create a new byte array which
is a concatenation of the three input values.  The input values are of type
ValueType.  All the ValueType(s)s are supported.  The size of the resulting
byte array will be equal to the sum of the sizes of the three input values.
However, if the resulting size will exceed 255 byte, the resulting array will
be truncated and only the first 255 bytes will be returned.

Note: Be sure that your variable which is used for the returned array is big
enough!!!                                                                    *)

procedure ConcatenateThreeValues(var val1;
                                 vType1 : ValueType;
                                 var val2;
                                 vType2 : ValueType;
                                 var val3;
                                 vType3 : ValueType;
                                 var resultArray);

var
    byteArray1 : ByteArray;
    resultBytearray : ByteArray absolute resultArray;

    begin
    ConcatenateTwoValues(val1,vType1,val2,vType2,byteArray1);
    ConcatenateTwoValues(byteArray1,BYTEARRAYVALUE,
                         val3,vType3,
                         resultByteArray);
    end;                            (* end of ConcatenateThreeValues routine *)


end.                                                 (* end of ByteData unit *)
