//*************************************************************
//  File name: ttdmp.c
//
//  Description:
//
//      WinMain and window procedure routines.
//
// This sample demonstrates the usage of new TrueType API available
// in Windows 3.10.  The sample demonstrates the TrueType functions
// GetOutlineTextMetrics() and GetRasterizerCaps() functions.  The
// sample has the ability to dump the OUTLINETEXTMETRIC structure,
// the NEWTEXTMETRIC structure, the PANOSE structure and the ENUMLOGFONT
// structure for a TrueType font.  Also, the sample demonstrates the
// use of the EnumFontFamilies() which is new to Win 3.10.  This
// function can retrieve the new ENUMLOGFONT and NEWTEXTMETRIC 
// information.
//
//
// The sample also demonstrates how to use the ChooseFont and Print
// common dialogs.  The sample allows the user to pick a font and
// display a line of text using that font to the screen.  Also, the
// user can print out the same line of text to a printer selected
// through the Print common dialog.
//
// Development Team:
//
//      Don Miller
//
//
// Written by Microsoft Product Support Services, Windows Developer Support
// COPYRIGHT:
//
//   (C) Copyright Microsoft Corp. 1993.  All rights reserved.
//
//   You have a royalty-free right to use, modify, reproduce and
//   distribute the Sample Files (and/or any modified version) in
//   any way you find useful, provided that you agree that
//   Microsoft has no warranty obligations or liability for any
//   Sample Application Files which are modified.
//
//*************************************************************
#define PRINTING                    // needed so constants are defined

#ifndef _GLOBALINC
#include "global.h"
#endif
#include <string.h>
#include <drivinit.h>               // contains printing constants
#include "tt.h"

// globals 
HANDLE  ghInst            = NULL;
HWND    ghWndMain         = NULL;
char    szMainMenu[]      = "MainMenu";
char    szMainClass[]     = "ttdmpClass";
int     iCurrStructChoice = IDM_OTM;

//*************************************************************
//
//  WinMain()
//
//  Purpose:
//
//		Entry point for all Windows applications.
//
//
//  Parameters:
//
//      HANDLE hInstance     - Handle to current instance
//      HANDLE hPrevInstance - Handle to previous instance
//      LPSTR  lpCmdLine     - Command-line arguments
//      int    nCmdShow      - How window is initially displayed
//      
//
//  Return: (int PASCAL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/2/92     Don Miller Created
//
//*************************************************************

int PASCAL WinMain (HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
    MSG msg;

    if (!hPrevInstance && !InitApplication(hInstance))
            return(FALSE);       

    if (!InitInstance(hInstance, nCmdShow))
        return(FALSE);

    while (GetMessage(&msg, NULL, NULL, NULL))
    {
        TranslateMessage(&msg);      
        DispatchMessage(&msg);       
    }
    return(msg.wParam);      

} //*** WinMain

//*************************************************************
//
//  MainWndProc()
//
//  Purpose:
//
//		Main Window procedure.
//
//
//  Parameters:
//
//      HWND     hWnd   - Handle to main window
//      unsigned msg    - Message passed to application
//      WORD     wParam - Additional message information
//      LONG     lParam - Additional message information
//      
//
//  Return: (long FAR PASCAL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/2/92     Don Miller Created
//
//*************************************************************

long FAR PASCAL MainWndProc (HWND hWnd, unsigned msg, WORD wParam, LONG lParam)
{
    FARPROC lpProc;
    static BOOL bTTAvail = FALSE, bTTCheck = FALSE;

    switch (msg) 
    {
        case WM_CREATE:
        {
            HMENU hMenu;

            // set global handles to NULL
            pis.hCurrDevMode  = NULL;
            pis.hCurrDevNames = NULL;
            pis.hPrinterFont  = NULL;

            // check default struct menu choice
            hMenu = GetMenu(hWnd);
            CheckMenuItem(hMenu, iCurrStructChoice, MF_CHECKED);

            // set default text color to black
            sis.crTextColor = RGB(0,0,0);

        }
        break;

        case WM_PAINT:
            {
            PAINTSTRUCT       ps;
            int               iOldMode;
            HFONT             hOldFont;
            DWORD             dwOldColor;

            BeginPaint(hWnd, &ps);

            // set so text background color is same as window backround color
            iOldMode = SetBkMode(ps.hdc, TRANSPARENT);

            // check to see if TT is enabled; do only once
            if (!bTTCheck)
            {
               bTTCheck = TRUE;

               // disable menu options if TT not available
               if (!(bTTAvail = IsTTAvail(ps.hdc)))
               {
                  HMENU hMenu = GetMenu(hWnd);

                  // disable appropriate menu choices
                  EnableMenuItem(hMenu, IDM_PRINT, MF_GRAYED);

                  CheckMenuItem(hMenu, iCurrStructChoice, MF_UNCHECKED);
                  EnableMenuItem(hMenu, IDM_OTM, MF_GRAYED);
                  EnableMenuItem(hMenu, IDM_TM, MF_GRAYED);
                  EnableMenuItem(hMenu, IDM_PANOSE, MF_GRAYED);
                  EnableMenuItem(hMenu, IDM_LF, MF_GRAYED);
               }
               else
               {
                  // tell user to pick a font
                  TextOut(ps.hdc, 10, 10, "Choose a font from the File Fonts menu.", 39);

               }
            }
            // check to see if TT enabled
            if (bTTAvail)
            {
               // if a screen font has been made use it
               if (sis.hScreenFont != NULL)
                  hOldFont = SelectObject(ps.hdc, sis.hScreenFont);

               // set text color to color selected by ChooseFont dialog
               dwOldColor = SetTextColor(ps.hdc, sis.crTextColor);

               // dump struct menu choice
               switch (iCurrStructChoice)
               {
                  case IDM_OTM:
                     // dump OUTLINETEXTMETRIC stuff
                     DumpOTMInfo(hWnd, ps.hdc);
                     break;

                  case IDM_TM:
                     // dump TEXTMETRIC stuff
                     DumpTMInfo(hWnd, ps.hdc);
                     break;

                  case IDM_PANOSE:
                     // dump PANOSE stuff
                     DumpPanoseInfo(hWnd, ps.hdc);
                     break;

                  case IDM_LF:
                     // dump LOGFONT stuff
                     DumpLFInfo(hWnd, ps.hdc);
                     break;

                  default:
                     break;
               }

               // reset text color
               SetTextColor(ps.hdc, dwOldColor);

               // reselect old font back in to dc
               if (sis.hScreenFont != NULL)
                  SelectObject(ps.hdc, hOldFont);

            } // if TTAvail

            // reset background mode back to previous mode
            SetBkMode(ps.hdc, iOldMode);

            EndPaint(hWnd, &ps);
            }
            break;

        case WM_COMMAND: 
            switch ( wParam )
            {
                case IDM_FONTS:
                     // show ChooseFont common dialog box
                     ShowFontBox(hWnd);
                     break;

                case IDM_PRINT:
                     // show Print common dialog box
                     ShowPrintBox(hWnd);
                     break;

                case IDM_PRINTSETUP:
                     // show Print Setup common dialog box
                     ShowPrintSetup(hWnd);
                     break;

                case IDM_EXIT:
                     PostMessage( hWnd, WM_SYSCOMMAND, SC_CLOSE, 0L );
                     break;

                case IDM_ABOUT:
                    // display About dialog box
                    lpProc = MakeProcInstance(About, ghInst);
                    DialogBox(ghInst, "AboutBox", hWnd, lpProc);    
                    FreeProcInstance(lpProc);
                    break;

                case IDM_OTM:
                case IDM_TM:
                case IDM_PANOSE:
                case IDM_LF:
                {
                  HMENU hMenu;

                  hMenu = GetMenu(hWnd);

                  // check appropriate menu choice
                  CheckMenuItem(hMenu, iCurrStructChoice, MF_UNCHECKED);
                  iCurrStructChoice = wParam;
                  CheckMenuItem(hMenu, iCurrStructChoice, MF_CHECKED);

                  // force repaint
                  InvalidateRect(hWnd, NULL, TRUE);
                  }
                  break;
            }
        break;

        case WM_DESTROY:
            // clean up printer font 
            if (pis.hPrinterFont != NULL)
                DeleteObject(pis.hPrinterFont);

            // clean up screen font
            if (sis.hScreenFont != NULL)
                DeleteObject(sis.hScreenFont);

            // had to keep around until app is done using the info 
            if (pis.hCurrDevMode != NULL)
                GlobalFree(pis.hCurrDevMode);

            // had to keep around until app is done using the info 
            if (pis.hCurrDevNames != NULL)
                GlobalFree(pis.hCurrDevNames);

            PostQuitMessage(0);
        break;
    }
    return(DefWindowProc(hWnd, msg, wParam, lParam));

} //*** MainWndProc

//*************************************************************
//
//  About()
//
//  Purpose:
//
//		The About dialog box procedure.
//
//
//  Parameters:
//
//      HWND     hDlg   - Handle to dialog window
//      unsigned msg    - Message passed to dialog window
//      WORD     wParam - Additional message information
//      LONG     lParam - Additional message information
//      
//
//  Return: (BOOL FAR PASCAL)
//
//      TRUE  - About dialog procedure handled the message.
//      FALSE - About dialog procedure did not handle the message.
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/2/92     Don Miller Created
//
//*************************************************************

BOOL FAR PASCAL About (HWND hDlg, unsigned msg, WORD wParam, LONG lParam)
{
    switch (msg) 
    {
        case WM_INITDIALOG: 
            return(TRUE);

        case WM_COMMAND:
            if (wParam == IDOK || wParam == IDCANCEL) 
            {
                EndDialog(hDlg, TRUE);         
                return(TRUE);
            }
        break;
    }
    return(FALSE);      // Didn't process a message

} //*** About


//*************************************************************
//
//  ShowPrintSetup
//
//  Purpose:
//		
//    Function responsible for displaying Print Setup common
//    dialog box.  Stores a handle to the current DEVMODE and a
//    handle to the current DEVNAMES in a global PRINTINFOSTRUCT.
//
//  Parameters:
//      HWND hWnd
//      
//
//  Return: (BOOL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/6/92     Don Miller Created
//*************************************************************

BOOL ShowPrintSetup(HWND hWnd)
{
    PRINTDLG pdPrint;
    BOOL     bResult;

    pdPrint.lStructSize = sizeof(PRINTDLG);           // size of structure
    pdPrint.hwndOwner   = hWnd;                       // common dialog's owner
    pdPrint.hDevMode    = (HANDLE)pis.hCurrDevMode;   // handle to DEVMODE struct
    pdPrint.hDevNames   = (HANDLE)pis.hCurrDevNames;  // handle to DEVNAMES struct
    pdPrint.Flags       = PD_PRINTSETUP;              // flags used to bring up Setup
    pdPrint.hInstance   = (HANDLE)ghInst;             // instance handle

    // display Print Setup common dialog
    bResult = PrintDlg(&pdPrint);

    // set global handles
    pis.hCurrDevMode   = pdPrint.hDevMode;
    pis.hCurrDevNames  = pdPrint.hDevNames;
 
    return bResult;

} //*** ShowPrintSetup


//*************************************************************
//
//  ShowFontBox
//
//  Purpose:
//		
//    Function is responsible for displaying ChooseFont common
//    dialog.  Also, the function creates a printer font and a 
//    screen font and stores them in their respective global
//    structures.
//
//  Parameters:
//      HWND hWnd
//      
//
//  Return: (void)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/6/92     Don Miller Created
//*************************************************************

void ShowFontBox(HWND hWnd)
{
    HDC        hPrnDC;
    LOGFONT    lfTemp;
    int        nyLogPix; 

    if ((hPrnDC = GetPrinterDC ()) != NULL)
    {
        cf.lStructSize      = sizeof(CHOOSEFONT);   // struct size
        cf.hwndOwner        = hWnd;                 // common dialog's owner
        cf.hDC              = hPrnDC;               // Printer DC used for device fonts
        cf.lpLogFont        = &pis.lfCurrFont;      // LOGFONT to initialize font info
        cf.Flags            = CF_TTONLY | CF_SCREENFONTS | CF_EFFECTS | CF_INITTOLOGFONTSTRUCT | CF_USESTYLE;
        cf.lCustData        = 0L;                   // no custom data
        cf.rgbColors        = sis.crTextColor;      // text color
        cf.lpfnHook         = (FARPROC)NULL;        // No hook function
        cf.lpTemplateName   = (LPSTR)NULL;          // No dialog template
        cf.hInstance        = ghInst;               // instance handle
        cf.lpszStyle        = (LPSTR)fi.szFontStyles;  // Buffer contains current font's styles
        cf.nFontType        = SCREEN_FONTTYPE;         // font type
        cf.nSizeMin         = 0;                       // No min size
        cf.nSizeMax         = 0;                       // No max size

        // bring up Font common dialog
        if (ChooseFont(&cf))
        {
            // get rid of old screen font, if one 
            if (sis.hScreenFont != NULL)
            {
                DeleteObject(sis.hScreenFont);
                sis.hScreenFont = NULL;
            }
            // store text color for screen
            sis.crTextColor = (COLORREF)cf.rgbColors;

            // set LOGFONT Precision field to always pick a TT font
            pis.lfCurrFont.lfOutPrecision = OUT_TT_ONLY_PRECIS;

            //create screen font
            sis.hScreenFont = CreateFontIndirect(&pis.lfCurrFont);

            // save LOGFONT in screen info struct
            sis.lfCurrFont = pis.lfCurrFont;

            // create printer font
            // use temp logfont because global logfont is for screen */
            lfTemp = pis.lfCurrFont;

            // adjust font height for correct point size on printer */
            nyLogPix = GetDeviceCaps(hPrnDC, LOGPIXELSY);
            lfTemp.lfHeight = -((cf.iPointSize/10) * nyLogPix)/72;

            // get rid of old printer font, if one
            if (pis.hPrinterFont != NULL)
            {
                DeleteObject(pis.hPrinterFont);
                pis.hPrinterFont = NULL;
            }
            pis.hPrinterFont = CreateFontIndirect(&lfTemp);

            // redraw the client
            InvalidateRect(hWnd, NULL, TRUE);
        }
        DeleteDC(hPrnDC);
    }
} //*** ShowFontBox

//*************************************************************
//
//  ShowPrintBox
//
//  Purpose:
//		
//    Function responsible for displaying Print  common
//    dialog box.  Stores a handle to the current DEVMODE and a
//    handle to the current DEVNAMES in a global PRINTINFOSTRUCT.
//
//  Parameters:
//      HWND hWnd
//      
//
//  Return: (BOOL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/6/92     Don Miller Created
//*************************************************************

BOOL ShowPrintBox(HWND hWnd)
{
    PRINTDLG pdPrint;
    BOOL     bResult;

    pdPrint.lStructSize = sizeof(PRINTDLG);           // struct size
    pdPrint.hwndOwner   = hWnd;                       // dialog's owner
    pdPrint.hDevMode    = (HANDLE)pis.hCurrDevMode;   // handle to DEVMODE struct
    pdPrint.hDevNames   = (HANDLE)pis.hCurrDevNames;  // handle to DEVNAMES struct
    pdPrint.nCopies     = 1;                          // number of copies to print
    pdPrint.Flags       = PD_ALLPAGES | PD_NOPAGENUMS | PD_COLLATE |
                          PD_NOSELECTION | PD_HIDEPRINTTOFILE;

    pdPrint.hInstance   = (HANDLE)ghInst;             // instance handle

    // bring up Print common dialog
    bResult = PrintDlg(&pdPrint);

    // set global handles
    pis.hCurrDevMode   = pdPrint.hDevMode;
    pis.hCurrDevNames  = pdPrint.hDevNames;

    // send printdlg info to printing routines
    if (bResult)
        bResult = PrintIt(hWnd, &pdPrint);

    return bResult;

} //*** ShowPrintBox


//*************************************************************
//
//  GetPrinterDC
//
//  Purpose:
//		
//    Function that retrieves a printer dc.  Retrieves a dc with
//    respect to the data in the current DEVMODE and DEVNAMES.
//
//  Parameters:
//      void
//      
//
//  Return: (HDC)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/6/92     Don Miller Created
//*************************************************************

HDC GetPrinterDC(void)
{
   LPDEVMODE  lpdmDevMode   = NULL;
   LPDEVNAMES lpdnDevNames  = NULL;
   LPSTR      lpstrDevNames = NULL;

   // lock handle to DEVMODE struct and send info to CreateDC 
   // This causes a DC to be created that will have the same
   // DEVMODE attributes as what the user picked in the Print
   // Setup dialog.  

   if (pis.hCurrDevMode != NULL)
       lpdmDevMode = (LPDEVMODE)GlobalLock(pis.hCurrDevMode);

   // lock handle to DEVNAMES struct and send info to CreateDC
   // This causes a DC to be created and linked to the device
   // (printer) that the user selected in the Print Setup 
   // dialog.  If hCurrDevNames is NULL then we create a DC
   // linked to the default printer.

   if (pis.hCurrDevNames != NULL)
   {
       lpdnDevNames = (LPDEVNAMES)GlobalLock(pis.hCurrDevNames);
       pis.szDevice = (LPSTR)lpdnDevNames + lpdnDevNames->wDeviceOffset;
       pis.szDriver = (LPSTR)lpdnDevNames + lpdnDevNames->wDriverOffset;
       pis.szOutput = (LPSTR)lpdnDevNames + lpdnDevNames->wOutputOffset;

       return CreateDC (pis.szDriver, pis.szDevice, pis.szOutput, (LPSTR)lpdmDevMode);
   }
   else    // get default printer from WIN.INI
   {
       GetProfileString ("windows", "device", "", pis.szPrinter, 64) ;

       if ((pis.szDevice = (LPSTR)strtok (pis.szPrinter, "," )) &&
           (pis.szDriver = (LPSTR)strtok (NULL,      ", ")) && 
           (pis.szOutput = (LPSTR)strtok (NULL,      ", ")))
           return CreateDC (pis.szDriver, pis.szDevice, pis.szOutput, (LPSTR)lpdmDevMode);
   }
   return NULL;
} //*** GetPrinterDC


//*************************************************************
//
//  PrintIt
//
//  Purpose:
//		
//    Function that handles all the printing.  Manual banding is
//    not implemented.
//
//  Parameters:
//      HWND hWnd
//      
//
//  Return: (BOOL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/6/92     Don Miller Created
//*************************************************************

BOOL PrintIt(HWND hWnd, LPPRINTDLG lpPD)
{
    HDC        hPrnDC;
    DOCINFO    di;
    FARPROC    lpAbortProc, lpAbortDlg;
    HFONT      hOldFont;
    TEXTMETRIC tm;
    WORD       wMaxXExtent;
    DWORD      dwTextExt;
    int        nxLogPix, nyLogPix, dxPrinter, dyPrinter, iPageNum,
               iCopyNum, iPos, i, iyChar, iyPos;
    char       szBuffer[64], szBuffer2[255];

    // get a printer dc 
    if ((hPrnDC = GetPrinterDC ()) == NULL)
    {
         MessageBox(hWnd, "Cannot create printer DC!", "ERROR",
                    MB_APPLMODAL|MB_ICONSTOP|MB_OK);
         return FALSE;
    }

    // set user abort flag 
    pis.bAbort = FALSE;

    // Initialize the abort procedure. 
    lpAbortProc = MakeProcInstance((FARPROC)PrintAbortProc, ghInst);
    lpAbortDlg  = MakeProcInstance((FARPROC)PrintAbortDlg,  ghInst);

    // disable main application window 
    EnableWindow(hWnd, FALSE);

    // display abort dialog 
    pis.hDlgAbort = CreateDialog(ghInst, "PrintDLG", hWnd, (FARPROC)lpAbortDlg);

    // set abort procedure 
    if (SetAbortProc(hPrnDC, lpAbortProc) < 0)
    {
         MessageBox(NULL, "The SetAbortProc function failed.", "ERROR",
                    MB_APPLMODAL|MB_ICONSTOP|MB_OK);

         DeleteDC(hPrnDC);
         return FALSE;
    }

    // need for StartDoc 
    di.cbSize      = sizeof(DOCINFO);
    di.lpszDocName = (LPSTR)"ttdmp";
    di.lpszOutput  = NULL;

    // issue STARTDOC 
    if (StartDoc(hPrnDC, &di) < 0)
    {
         MessageBox(NULL, "STARTDOC escape problem", "ERROR",
                    MB_APPLMODAL|MB_ICONSTOP|MB_OK);

         DeleteDC(hPrnDC);
         return FALSE;
    }

    // get logical pixels per inch 
    nxLogPix  = GetDeviceCaps(hPrnDC, LOGPIXELSX);
    nyLogPix  = GetDeviceCaps(hPrnDC, LOGPIXELSY);

    // get page resolution 
    dxPrinter = GetDeviceCaps(hPrnDC, HORZRES);
    dyPrinter = GetDeviceCaps(hPrnDC, VERTRES);

    // select font in
    if (pis.hPrinterFont != NULL)
       hOldFont = SelectObject(hPrnDC, pis.hPrinterFont);

    // get text information
    GetTextMetrics(hPrnDC, &tm);
    iyChar = tm.tmHeight + tm.tmExternalLeading;

    // compute max extent  (width of page - 2inches)
    wMaxXExtent = dxPrinter-(2*nxLogPix);

    // do the number of collated copies
    for (iCopyNum=1; iCopyNum <= (int)lpPD->nCopies; iCopyNum++)
    {
      for (iPageNum=1; iPageNum <= 1 && !pis.bAbort; iPageNum++)
      {
         // update abort dialog box page number
         wsprintf((LPSTR)szBuffer, "Now printing Page %d of", iPageNum);
         SetDlgItemText(pis.hDlgAbort, 100, (LPSTR)szBuffer); 

         // select font in printer dc
         if (pis.hPrinterFont != NULL)
            SelectObject(hPrnDC, pis.hPrinterFont);

         // start printing a page
         StartPage(hPrnDC);

         // set position counters
         iPos=0;
         iyPos = nyLogPix;

         // copy facename, styles, and point size to buffer
         wsprintf((LPSTR)szBuffer2, "%s %s %d pt", (LPSTR)pis.lfCurrFont.lfFaceName,
                  (LPSTR)cf.lpszStyle, cf.iPointSize/10); 

         // print facename, styles, and point size 
         TextOut(hPrnDC, nxLogPix, iyPos, (LPSTR)szBuffer2, lstrlen((LPSTR)szBuffer2));

         iyPos += 2*iyChar;

         // write ascii table to printer 
         for (i=1; i < 224; i++)
         {
            // place character into buffer
            szBuffer2[iPos++] = (char)(i+32);

            // place a space in buffer
            szBuffer2[iPos++] = (char)32;

            // get width of string in buffer
            dwTextExt = GetTextExtent(hPrnDC, (LPSTR)szBuffer2, iPos);

            // check to see if buffer is wider than max x extent
            if (LOWORD(dwTextExt) > wMaxXExtent) 
            {
               // print out string in buffer
               TextOut(hPrnDC, nxLogPix, iyPos, (LPSTR)szBuffer2, iPos-2);

               // place the last character back in buffer
               szBuffer2[0] = (char)(i+32);

               // adjust position counters
               iPos=1;
               iyPos += iyChar;
            }
         }
         // print last line
         if (iPos > 0)
            TextOut(hPrnDC, nxLogPix, iyPos, (LPSTR)szBuffer2, iPos);

         // end the page; resets the dc to normal defaults
         EndPage(hPrnDC);
      }
    }
    // only do if user has not aborted 
    if (!pis.bAbort)
    {
      if (EndDoc(hPrnDC) < 0)    // end the document
      {
             MessageBox(NULL, "EndDoc function problem", "ERROR",
                        MB_APPLMODAL|MB_ICONSTOP|MB_OK);

             return FALSE;
      }
    }

    // enable main application window 
    EnableWindow(hWnd, TRUE);

    // get rid of abort dialog 
    DestroyWindow(pis.hDlgAbort);

    // clean up 
    FreeProcInstance(lpAbortProc);
    FreeProcInstance(lpAbortDlg); 

    // deselect font, if changed 
    if (pis.hPrinterFont != NULL)
      SelectObject(hPrnDC, hOldFont);
      
    DeleteDC(hPrnDC);
} //*** PrintIt


// -------------------- Abort Routines --------------------

//*************************************************************
//
//  PrintAbortProc
//
//  Purpose:
//		
//    Function is used to give application some "multi-tasking"
//    through the use of a PeekMessage().  Also, it gives the 
//    application some error control during printing.
//
//  Parameters:
//      HDC hDC
//      short code
//      
//
//  Return: (BOOL FAR PASCAL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/6/92     Don Miller Created
//*************************************************************

BOOL FAR PASCAL PrintAbortProc(HDC hDC, short code)
{
   MSG msg;

   while (!pis.bAbort && PeekMessage(&msg, 0, 0, 0, PM_REMOVE))
      if (!IsDialogMessage(pis.hDlgAbort, &msg))
         {
         TranslateMessage(&msg);
         DispatchMessage(&msg);
         }
   return (!pis.bAbort);
}  //*** PrintAbortProc


//*************************************************************
//
//  PrintAbortDlg
//
//  Purpose:
//		
//    Dialog procedure for print abort dialog.
//
//  Parameters:
//      HWND hWnd
//      unsigned msg
//      WORD wParam
//      LONG lParam
//      
//
//  Return: (int FAR PASCAL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              1/6/92     Don Miller Created
//*************************************************************

int FAR PASCAL PrintAbortDlg(HWND hWnd, unsigned msg, WORD wParam, LONG lParam)
{
   char szMsgBuffer[64];

   switch (msg)
      {
      case WM_INITDIALOG:
         SetFocus(hWnd);
         EnableMenuItem(GetSystemMenu(hWnd, FALSE), SC_CLOSE, MF_GRAYED);

         // set printing info to abort dialog 
         SetDlgItemText(hWnd, 100, "Now printing Page 1 of");
         
         wsprintf((LPSTR)szMsgBuffer, "'%.25s' on the", (LPSTR)"ttdmp"); 
         SetDlgItemText(hWnd, 101, (LPSTR)szMsgBuffer);

         wsprintf((LPSTR)szMsgBuffer, "%.20s on %.20s",(LPSTR)pis.szDevice,(LPSTR)pis.szOutput); 
         SetDlgItemText(hWnd, 102, (LPSTR)szMsgBuffer);

         return TRUE;

      case WM_COMMAND:
         pis.bAbort = TRUE;
         return TRUE;
      }

   return FALSE;
}  // PrintAbortDlg() 


//*************************************************************
//
//  DumpOTMInfo
//
//  Purpose:  This function is responsible for dumping the 
//            OUTLINETEXTMETRICS information to the screen.  The
//            function uses the new 3.1 function
//            GetOutlineTextMetrics().
//		
//
//  Parameters:
//      HWND hWnd
//      HDC hDC
//      
//
//  Return: (void)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              2/14/92    Don Miller Created
//*************************************************************

void DumpOTMInfo(HWND hWnd, HDC hDC)
{
   int                 ixPos, iyPos, ixChar, iyChar, i;   
   LPOUTLINETEXTMETRIC lpOTM;
   HANDLE              hOTM;
   DWORD               dwOTMsize, dwRet;
   char                szBuffer[256];
   HFONT               hTTFont;
   TEXTMETRIC          tm;
   LPSTR               lpstrOTMStr;

   // find size of OTM structure
   if ((dwOTMsize = GetOutlineTextMetrics(hDC, 0, NULL)) == 0)
   {
      // only do if GetOutlineTextMetrics fails
      if (sis.hScreenFont)
      {
         hTTFont = SelectObject(hDC, GetStockObject(SYSTEM_FIXED_FONT));
         TextOut(hDC, 10, 10, "No OTM information!", 19);
         SelectObject(hDC, hTTFont);
      }
      return;
   }

   // allocate buffer for OTM
   hOTM  = GlobalAlloc(GMEM_MOVEABLE, dwOTMsize);

   // lock buffer
   lpOTM = (LPOUTLINETEXTMETRIC) GlobalLock(hOTM);

   // fill OTM structure
   dwRet = GetOutlineTextMetrics(hDC, (UINT)dwOTMsize, lpOTM);

   // use fixed font for field names
   hTTFont = SelectObject(hDC, GetStockObject(SYSTEM_FIXED_FONT));

   // get character width and height
   GetTextMetrics(hDC, &tm);
   ixPos = ixChar = tm.tmAveCharWidth;
   iyPos = iyChar = tm.tmHeight + tm.tmExternalLeading;

   // display header in caption bar
   wsprintf((LPSTR)szBuffer, "%s - OUTLINETEXTMETRIC structure", (LPSTR)sis.lfCurrFont.lfFaceName);
   SetWindowText(hWnd, (LPSTR)szBuffer);

   // display table 
   for (i=0; i < 32; i++)
   {
      // set positions for second column
      if (i == 15)
      {
         ixPos = 10*ixChar + LOWORD(GetTextExtent(hDC, (LPSTR)szOTM[22], lstrlen((LPSTR)szOTM[22])));
         iyPos = iyChar;
      }

      switch(i)
      {
         case 0:
            // size of OUTLINETEXTMETRIC structure
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmSize);
            break;

         case 1:
            // otmTextMetrics field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], (LPSTR)"See Structures menu");
            break;

         case 2:
            // otmFiller field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], (BYTE)lpOTM->otmFiller);
            break;

         case 3:
            // otmPanoseNumber field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], (LPSTR)"See Structures menu");
            break;

         case 4:
         // otmfsSelection field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmfsSelection);
            break;

         case 5:
            // otmfsType field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmfsType);
            break;

         case 6:
            // otmsCharSlopeRise field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmsCharSlopeRise);
            break;

         case 7:
            // otmsCharSlopeRun field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmsCharSlopeRun);
            break;

         case 8:
            // otmItalicAngle field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmItalicAngle);
            break;

         case 9:
            // otmEMSquare field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmEMSquare);
            break;

         case 10:
            // otmAscent field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmAscent);
            break;

         case 11:
            // otmDescent field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmDescent);
            break;

         case 12:
            // otmLineGap field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmLineGap);
            break;

         case 13:
            // otmsXHeight field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmsXHeight);
            break;

         case 14:
            // otmsCapEmHeight field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmsCapEmHeight);
            break;

         case 15:
            // otmrcFontBox field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmrcFontBox);
            break;

         case 16:
            /// otmMacAscent field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmMacAscent);
            break;

         case 17:
            // otmMacDescent field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmMacDescent);
            break;

         case 18:
            // otmMacLineGap field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmMacLineGap);
            break;

         case 19:
            // otmusMinimumPPEM field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmusMinimumPPEM);
            break;

         case 20:
            // otmptSubscriptSize field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmptSubscriptSize.x, lpOTM->otmptSubscriptSize.y);
            break;

         case 21:
            // otmptSubscriptOffset field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmptSubscriptOffset.x, lpOTM->otmptSubscriptOffset.y);
            break;

         case 22:
            // otmptSuperscriptSize field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmptSuperscriptSize.x, lpOTM->otmptSuperscriptSize.y);
            break;

         case 23:
            // otmptSuperscriptOffset field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmptSuperscriptOffset.x, lpOTM->otmptSuperscriptOffset.y);
            break;

         case 24:
            // otmsStrikeoutSize field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmsStrikeoutSize);
            break;

         case 25:
            // otmsStrikeoutPosition field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmsStrikeoutPosition);
            break;

         case 26:
            // otmsUnderscoreSize field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmsUnderscoreSize);
            break;

         case 27:
            // otmsUnderscorePosition field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], lpOTM->otmsUnderscorePosition);
            break;

         case 28:
            // otmpFamilyName field
            lpstrOTMStr = (LPSTR)(lpOTM);
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], (LPSTR)(lpstrOTMStr + (int)lpOTM->otmpFamilyName));
            break;

         case 29:
            // otmpFaceName field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], (LPSTR)(lpstrOTMStr + (int)lpOTM->otmpFaceName));
            break;

         case 30:
            // otmpStyleName field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], (LPSTR)(lpstrOTMStr + (int)lpOTM->otmpStyleName));
            break;

         case 31:
            // otmpFullName field
            wsprintf((LPSTR)szBuffer, (LPSTR)szOTM[i], (LPSTR)(lpstrOTMStr + (int)lpOTM->otmpFullName));
            ixPos = ixChar;
            break;

         default:
            break;
      }
      // print buffer
      TextOut(hDC, ixPos, iyPos, (LPSTR)szBuffer, lstrlen((LPSTR)szBuffer));
      iyPos += iyChar;
   }

   // diplay a line of text in TT font
   SelectObject(hDC, hTTFont);
   wsprintf((LPSTR)szBuffer, "%s %s %d pt - %s", (LPSTR)pis.lfCurrFont.lfFaceName,
            (LPSTR)cf.lpszStyle, cf.iPointSize/10, (LPSTR)"AaBbCcDdEe0123456789"); 

   TextOut(hDC, ixChar, 19*iyChar, (LPSTR)szBuffer, lstrlen((LPSTR)szBuffer));

   // clean up
   GlobalUnlock(hOTM);
   GlobalFree(hOTM);

} //*** DumpTTInfo


//*************************************************************
//
//  DumpTMInfo
//
//  Purpose: This function is responsible for dumping the 
//		       TEXTMETRICS information stored in the
//           OUTLINETEXTMETRICS structure.  The function uses
//           the new 3.1 function GetOutlineTextMetrics().
//
//
//  Parameters:
//      HWND hWnd
//      HDC hDC
//      
//
//  Return: (void)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              2/14/92    Don Miller Created
//*************************************************************

void DumpTMInfo(HWND hWnd, HDC hDC)
{
   int                 ixPos, iyPos, ixChar, iyChar, i;   
   LPOUTLINETEXTMETRIC lpOTM;
   HANDLE              hOTM;
   DWORD               dwOTMsize, dwRet;
   char                szBuffer[128];
   HFONT               hTTFont;
   TEXTMETRIC          tm;
   NEWTEXTMETRIC       ntm;
   FARPROC             lpfnEnumFamProc;

   // find size of OTM structure
   if ((dwOTMsize = GetOutlineTextMetrics(hDC, 0, NULL)) == 0)
   {
      // only do if GetOutlineTextMetrics fails
      if (sis.hScreenFont)
      {
         hTTFont = SelectObject(hDC, GetStockObject(SYSTEM_FIXED_FONT));
         TextOut(hDC, 10, 10, "No OTM information!", 19);
         SelectObject(hDC, hTTFont);
      }
      return;
   }

   // allocate buffer for OTM
   hOTM  = GlobalAlloc(GMEM_MOVEABLE, dwOTMsize);

   // lock buffer
   lpOTM = (LPOUTLINETEXTMETRIC) GlobalLock(hOTM);

   // fill OTM structure in order to get TEXTMETRIC info for current TT
   // font in hDC
   dwRet = GetOutlineTextMetrics(hDC, (UINT)dwOTMsize, lpOTM);

   // use fixed font for field names
   hTTFont = SelectObject(hDC, GetStockObject(SYSTEM_FIXED_FONT));

   // get character width and height
   GetTextMetrics(hDC, &tm);
   ixPos = ixChar = tm.tmAveCharWidth;
   iyPos = iyChar = tm.tmHeight + tm.tmExternalLeading;

   // display header in caption bar
   wsprintf((LPSTR)szBuffer, "%s - NEWTEXTMETRIC structure", (LPSTR)sis.lfCurrFont.lfFaceName);
   SetWindowText(hWnd, (LPSTR)szBuffer);


   // get NEWTEXTMETRIC info from EnumFontFamilies()
   lpfnEnumFamProc = MakeProcInstance(EnumFontFamProc1, ghInst); 
   EnumFontFamilies(hDC, (LPCSTR)sis.lfCurrFont.lfFaceName, (FONTENUMPROC)lpfnEnumFamProc, (LPSTR)&ntm);
   FreeProcInstance(lpfnEnumFamProc);

   // display table 
   for (i=0; i < 24; i++)
   {
      switch(i)
      {
         case 0:
            // tmHeight field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmHeight);
            break;

         case 1:
            // tmAscent field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmAscent);
            break;

         case 2:
            // tmDescent field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmDescent);
            break;

         case 3:
            //tmInternalLeading field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmInternalLeading);
            break;

         case 4:
            // tmExternalLeading field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmExternalLeading);
            break;

         case 5:
            // tmAveCharWidth field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmAveCharWidth);
            break;

         case 6:
            // tmMaxCharWidth field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmMaxCharWidth);
            break;

         case 7:
            // tmWeight field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmWeight);
            break;

         case 8:
            // tmItalic field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmItalic);
            break;

         case 9:
            // tmUnderlined field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmUnderlined);
            break;

         case 10:
            // tmStruckOut field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmStruckOut);
            break;

         case 11:
            // tmFirstChar field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmFirstChar);
            break;

         case 12:
            // tmLastChar field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmLastChar);
            break;

         case 13:
            // tmDefaultChar field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmDefaultChar);
            break;

         case 14:
            // tmBreakChar field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmBreakChar);
            break;

         case 15:
            // tmPitchAndFamily field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmPitchAndFamily);
            break;

         case 16:
            // tmCharSet field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmCharSet);
            break;

         case 17:
            // tmOverhang field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmOverhang);
            break;

         case 18:
            // tmDigitizedAspectX field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmDigitizedAspectX);
            break;

         case 19:
            // tmDigitizedAspectY field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], lpOTM->otmTextMetrics.tmDigitizedAspectY);
            break;

         // NEWTEXTMETRIC stuff is filled in by EnumFontFamProc1
         case 20:
            // ntmFlags field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], ntm.ntmFlags);
            break;

         case 21:
            // ntmEMSize field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], ntm.ntmSizeEM);
            break;

         case 22:
            // ntmCellHeight field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], ntm.ntmCellHeight);
            break;

         case 23:
            // ntmAveWidth field
            wsprintf((LPSTR)szBuffer, (LPSTR)szTextMetric[i], ntm.ntmAvgWidth);
            break;

         default:
            break;

      }
      TextOut(hDC, ixPos, iyPos, (LPSTR)szBuffer, lstrlen((LPSTR)szBuffer));
      iyPos += iyChar; 
   }

   // display a line of text in TT font
   SelectObject(hDC, hTTFont);

   wsprintf((LPSTR)szBuffer, "%s %s %d pt - %s", (LPSTR)pis.lfCurrFont.lfFaceName,
            (LPSTR)cf.lpszStyle, cf.iPointSize/10, (LPSTR)"AaBbCcDdEe0123456789"); 

   TextOut(hDC, ixChar, 26*iyChar, (LPSTR)szBuffer, lstrlen((LPSTR)szBuffer));

   // clean up
   GlobalUnlock(hOTM);
   GlobalFree(hOTM);

} //*** DumpTMInfo


//*************************************************************
//
//  DumpPanoseInfo
//
//  Purpose: This function is responsible for dumping the 
//		       PANOSE information stored in the
//           OUTLINETEXTMETRICS structure.  The function uses
//           the new 3.1 function GetOutlineTextMetrics().
//
//
//  Parameters:
//      HWND hWnd
//      HDC hDC
//      
//
//  Return: (void)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              2/14/92    Don Miller Created
//*************************************************************

void DumpPanoseInfo(HWND hWnd, HDC hDC)
{
   int                 ixPos, iyPos, ixChar, iyChar, i;   
   LPOUTLINETEXTMETRIC lpOTM;
   HANDLE              hOTM;
   DWORD               dwOTMsize, dwRet;
   char                szBuffer[128];
   HFONT               hTTFont;
   TEXTMETRIC          tm;

   // find size of OTM structure
   if ((dwOTMsize = GetOutlineTextMetrics(hDC, 0, NULL)) == 0)
   {
      // only do if GetOutlineTextMetrics fails
      if (sis.hScreenFont)
      {
         hTTFont = SelectObject(hDC, GetStockObject(SYSTEM_FIXED_FONT));
         TextOut(hDC, 10, 10, "No OTM information!", 19);
         SelectObject(hDC, hTTFont);
      }
      return;
   }

   // allocate buffer for OTM
   hOTM  = GlobalAlloc(GMEM_MOVEABLE, dwOTMsize);

   // lock buffer
   lpOTM = (LPOUTLINETEXTMETRIC) GlobalLock(hOTM);

   // fill OTM structure
   dwRet = GetOutlineTextMetrics(hDC, (UINT)dwOTMsize, lpOTM);

   // use fixed font for field names
   hTTFont = SelectObject(hDC, GetStockObject(SYSTEM_FIXED_FONT));

   // get character width and height
   GetTextMetrics(hDC, &tm);
   ixPos = ixChar = tm.tmAveCharWidth;
   iyPos = iyChar = tm.tmHeight + tm.tmExternalLeading;

   // display header in caption bar
   wsprintf((LPSTR)szBuffer, "%s - PANOSE structure", (LPSTR)sis.lfCurrFont.lfFaceName);
   SetWindowText(hWnd, (LPSTR)szBuffer);

   // display table 
   for (i=0; i < 10; i++)
   {
      switch(i)
      {
         case 0:
            // bFamilyType field
            wsprintf((LPSTR)szBuffer, (LPSTR)szPanose[i], lpOTM->otmPanoseNumber.bFamilyType,
                     (LPSTR)szPanFamily[lpOTM->otmPanoseNumber.bFamilyType]);
            break;

         case 1:
            // bSerifStyle field
            wsprintf((LPSTR)szBuffer, (LPSTR)szPanose[i], lpOTM->otmPanoseNumber.bSerifStyle,
                     (LPSTR)szPanSerif[lpOTM->otmPanoseNumber.bSerifStyle]);
            break;

         case 2:
            // bWeight field
            wsprintf((LPSTR)szBuffer, (LPSTR)szPanose[i], lpOTM->otmPanoseNumber.bWeight,
                     (LPSTR)szPanWeight[lpOTM->otmPanoseNumber.bWeight]);
            break;

         case 3:
            // bProportion field
            wsprintf((LPSTR)szBuffer, (LPSTR)szPanose[i], lpOTM->otmPanoseNumber.bProportion,
                     (LPSTR)szPanProportion[lpOTM->otmPanoseNumber.bProportion]);
            break;

         case 4:
            // bContrast field
            wsprintf((LPSTR)szBuffer, (LPSTR)szPanose[i], lpOTM->otmPanoseNumber.bContrast,
                     (LPSTR)szPanContrast[lpOTM->otmPanoseNumber.bContrast]);
            break;

         case 5:
            // bStrokeVariable field
            wsprintf((LPSTR)szBuffer, (LPSTR)szPanose[i], lpOTM->otmPanoseNumber.bStrokeVariation,
                     (LPSTR)szPanStrokeVar[lpOTM->otmPanoseNumber.bStrokeVariation]);
            break;

         case 6:
            // bArmStyle field
            wsprintf((LPSTR)szBuffer, (LPSTR)szPanose[i], lpOTM->otmPanoseNumber.bArmStyle,
                     (LPSTR)szPanArmStyle[lpOTM->otmPanoseNumber.bArmStyle]);
            break;

         case 7:
            // bLetterform field
            wsprintf((LPSTR)szBuffer, (LPSTR)szPanose[i], lpOTM->otmPanoseNumber.bLetterform,
                     (LPSTR)szPanLetterForm[lpOTM->otmPanoseNumber.bLetterform]);
            break;

         case 8:
            // bMidline field
            wsprintf((LPSTR)szBuffer, (LPSTR)szPanose[i], lpOTM->otmPanoseNumber.bMidline,
                     (LPSTR)szPanMidline[lpOTM->otmPanoseNumber.bMidline]);
            break;

         case 9:
            // bXHeight field
            wsprintf((LPSTR)szBuffer, (LPSTR)szPanose[i], lpOTM->otmPanoseNumber.bXHeight,
                     (LPSTR)szPanXHeight[lpOTM->otmPanoseNumber.bXHeight]);
            break;

         default:
            break;

      }
      TextOut(hDC, ixPos, iyPos, (LPSTR)szBuffer, lstrlen((LPSTR)szBuffer));
      iyPos += iyChar;
   }

   // display a line of text in TT font
   SelectObject(hDC, hTTFont);

   wsprintf((LPSTR)szBuffer, "%s %s %d pt - %s", (LPSTR)pis.lfCurrFont.lfFaceName,
            (LPSTR)cf.lpszStyle, cf.iPointSize/10, (LPSTR)"AaBbCcDdEe0123456789"); 

   TextOut(hDC, ixChar, 12*iyChar, (LPSTR)szBuffer, lstrlen((LPSTR)szBuffer));

   // clean up
   GlobalUnlock(hOTM);
   GlobalFree(hOTM);

} //*** DumpPanoseInfo


//*************************************************************
//
//  DumpLFInfo
//
//  Purpose: This function is responsible for dumping the 
//		       LOGFONT information that is returned from the
//           ChooseFont() function.
//
//
//  Parameters:
//      HWND hWnd
//      HDC hDC
//      
//
//  Return: (void)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              2/14/92    Don Miller Created
//*************************************************************

void DumpLFInfo(HWND hWnd, HDC hDC)
{
   HFONT       hTTFont;
   TEXTMETRIC  tm;
   int         i, ixPos, iyPos, ixChar, iyChar;
   char        szBuffer[80];
   FARPROC     lpfnEnumFamProc;
   ENUMLOGFONT elf;

   // get current LOGFONT
   if (sis.hScreenFont)
   {
      // use fixed font for field names
      hTTFont = SelectObject(hDC, GetStockObject(SYSTEM_FIXED_FONT));

      // get character width and height
      GetTextMetrics(hDC, &tm);
      ixPos = ixChar = tm.tmAveCharWidth;
      iyPos = iyChar = tm.tmHeight + tm.tmExternalLeading;

      // display header in caption bar
      wsprintf((LPSTR)szBuffer, "%s - ENUMLOGFONT structure", (LPSTR)sis.lfCurrFont.lfFaceName);
      SetWindowText(hWnd, (LPSTR)szBuffer);

      // get ENUMLOGFONT info 
      lpfnEnumFamProc = MakeProcInstance(EnumFontFamProc2, ghInst); 
      EnumFontFamilies(hDC, (LPCSTR)sis.lfCurrFont.lfFaceName, (FONTENUMPROC)lpfnEnumFamProc, (LPSTR)&elf);
      FreeProcInstance(lpfnEnumFamProc);

      // display table
      for (i=0; i < 16; i++)
      {
         switch(i)
         {
            case 0:
               // lfHeight field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfHeight);
               break;

            case 1:
               // lfWidth field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfWidth);
               break;

            case 2:
               // lfEscapement field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfEscapement);
               break;

            case 3:
               // lfOrientation field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfOrientation);
               break;

            case 4:
               // lfWeight field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfWeight);
               break;

            case 5:
               // lfItalic field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfItalic);
               break;

            case 6:
               // lfUnderline field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfUnderline);
               break;

            case 7:
               // lfStrikeOut field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfStrikeOut);
               break;

            case 8:
               // lfCharSet field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfCharSet);
               break;

            case 9:
               //lfOutPrecision field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfOutPrecision);
               break;

            case 10:
               // lfClipPrecision field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfClipPrecision);
               break;

            case 11:
               // lfQuality field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfQuality);
               break;

            case 12:
               // lfPitchAndFamily field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], sis.lfCurrFont.lfPitchAndFamily);
               break;

            case 13:
               // lfFaceName field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], (LPSTR)sis.lfCurrFont.lfFaceName);
               break;

            // ENUMLOGFONT stuff is filled in by EnumFontFamProc2
            case 14:
               // elfFullName field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], (LPSTR)elf.elfFullName);
               break;

            case 15:
               // elfStyle field
               wsprintf((LPSTR)szBuffer, (LPSTR)szLF[i], (LPSTR)elf.elfStyle);
               break;

            default:
               break;
            
         }     
         TextOut(hDC, ixPos, iyPos, (LPSTR)szBuffer, lstrlen((LPSTR)szBuffer));
         iyPos += iyChar;
      }

      // display a line of text in TT font
      SelectObject(hDC, hTTFont);
      wsprintf((LPSTR)szBuffer, "%s %s %d pt - %s", (LPSTR)pis.lfCurrFont.lfFaceName,
               (LPSTR)cf.lpszStyle, cf.iPointSize/10, (LPSTR)"AaBbCcDdEe0123456789"); 

      TextOut(hDC, ixChar, 18*iyChar, (LPSTR)szBuffer, lstrlen((LPSTR)szBuffer));
   }

} //*** DumpLFInfo


//*************************************************************
//
//  IsTTAvail
//
//  Purpose: This function is responsible for checking to see
//           if TrueType is enabled and that at least one TrueType
//           font is installed on the system.
//		
//
//
//  Parameters:
//      HDC hDC
//      
//
//  Return: (BOOL)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              2/14/92    Don Miller Created
//*************************************************************

BOOL IsTTAvail(HDC hDC)
{
   RASTERIZER_STATUS rs;
   BOOL              bTTAvail = TRUE;
   int               ixPos, iyPos;
   TEXTMETRIC        tm;
   HFONT             hOldFont;


   // select in fixed font
   hOldFont = SelectObject(hDC, GetStockObject(SYSTEM_FIXED_FONT));

   //get fixed font character dims
   GetTextMetrics(hDC, &tm);

   ixPos = tm.tmAveCharWidth;
   iyPos = tm.tmHeight + tm.tmExternalLeading;

   // get TT flags
   GetRasterizerCaps(&rs, sizeof(RASTERIZER_STATUS));

   // check to see if TT fonts are installed
   if (!(rs.wFlags & TT_AVAILABLE))
   {
      TextOut(hDC, ixPos, iyPos, "No TrueType fonts installed!", 28);
      iyPos += iyPos;
      bTTAvail = FALSE;
   }

   // check to see if TT engine is enabled
   if (!(rs.wFlags & TT_ENABLED))
   {
      TextOut(hDC, ixPos, iyPos, "TrueType not enabled!", 21);
      bTTAvail = FALSE;
   }

   // restore the dc
   SelectObject(hDC, hOldFont);

   return (bTTAvail);

} //*** IsTTAvail


//*************************************************************
//
//  EnumFontFamProc1
//
//  Purpose: To enumerate the current TT type face and find a
//           style match.  Once the style match is found, the
//           new NEWTEXTMETRIC stuff is copied to a ptr passed in
//           the lParam parameter.
//		
//
//
//  Parameters:
//      LOGFONT FAR *lpnlf
//      TEXTMETRIC FAR *lpntm
//      int iFontType
//      LPARAM lParam
//      
//
//  Return: (int CALLBACK)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              2/27/92    Don Miller Created
//*************************************************************

int CALLBACK EnumFontFamProc1(LOGFONT FAR *lpnlf, TEXTMETRIC FAR *lpntm, int iFontType, LPARAM lParam)
{
   LPNEWTEXTMETRIC lpNTM, lpVisitingNTM;
   static BYTE bNTMCurrStyleMask = 0;
   static BYTE bNTMSelStyleMask  = 0;

   // get style masks
   bNTMCurrStyleMask = GetStyleMask(lpnlf);
   bNTMSelStyleMask  = GetStyleMask((LOGFONT FAR *)&sis.lfCurrFont);

   // styles match
   if (bNTMCurrStyleMask == bNTMSelStyleMask)
   {
      // fill in NEWTEXTMETRIC stuff
      lpNTM = (LPNEWTEXTMETRIC)lpntm;

      lpVisitingNTM = (LPNEWTEXTMETRIC)lParam;

      lpVisitingNTM->ntmFlags      = lpNTM->ntmFlags;

      lpVisitingNTM->ntmSizeEM     = lpNTM->ntmSizeEM;  

      lpVisitingNTM->ntmCellHeight = lpNTM->ntmCellHeight;  

      lpVisitingNTM->ntmAvgWidth   = lpNTM->ntmAvgWidth;

      // reset masks 
      bNTMCurrStyleMask = 0;
      bNTMSelStyleMask  = 0;

      // quit enumerating
      return 0;
   }
   else
      return 1;  // continue enumerating
}

//*************************************************************
//
//  EnumFontFamProc2
//
//  Purpose: To enumerate the current TT type face and find a
//           style match.  Once the style match is found, the
//           new ENUMLOGFONT stuff is copied to a ptr passed in
//           the lParam parameter.
//		
//
//
//  Parameters:
//      LOGFONT FAR *lpnlf
//      TEXTMETRIC FAR *lpntm
//      int iFontType
//      LPARAM lParam
//      
//
//  Return: (int CALLBACK)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              2/27/92    Don Miller Created
//*************************************************************

int CALLBACK EnumFontFamProc2(LOGFONT FAR *lpnlf, TEXTMETRIC FAR *lpntm, int iFontType, LPARAM lParam)
{
   LPENUMLOGFONT lpELF;
   LPENUMLOGFONT lpVisitingELF;
   static BYTE bELFCurrStyleMask = 0;
   static BYTE bELFSelStyleMask  = 0;

   // get style masks
   bELFCurrStyleMask = GetStyleMask(lpnlf);
   bELFSelStyleMask  = GetStyleMask((LOGFONT FAR *)&sis.lfCurrFont);

   // styles match
   if (bELFCurrStyleMask == bELFSelStyleMask)
   {
      // do ENUMLOGFONT stuff
      lpELF = (LPENUMLOGFONT)lpnlf;
      lpVisitingELF = (LPENUMLOGFONT)lParam;

      lstrcpy((LPSTR)lpVisitingELF->elfFullName, (LPSTR)lpELF->elfFullName);
      lstrcpy((LPSTR)lpVisitingELF->elfStyle, (LPSTR)lpELF->elfStyle);

      // reset masks 
      bELFCurrStyleMask = 0;
      bELFSelStyleMask  = 0;

      // quit enumerating
      return 0;
   }
   else
      return 1;
}

//*************************************************************
//
//  GetStyleMask
//
//  Purpose: To find what kind of styles are in a LOGFONT.  This
//           function returns a BYTE mask that is used in the 
//           EnumFontFamilyProc's
//
//
//  Parameters:
//      LOGFONT FAR *lf
//      
//
//  Return: (BYTE)
//
//
//  Comments:
//
//
//  History:    Date       Author     Comment
//              2/27/92    Don Miller Created
//*************************************************************

BYTE GetStyleMask(LOGFONT FAR *lf)
{
   BYTE bMask = 0;
   // set low bit if bold
   if (lf->lfWeight > 400)
      bMask |= 1;
   
   // set second lowest bit if italic
   if (lf->lfItalic)
      bMask |= 2;

   return bMask;

} //*** GetStyleMask

/*** EOF: ttdmp.c ***/

