/*----------------------------------------------------------------------------*\
|   rledlg.c - Dialog boxes for the RLE app                                    |
\*----------------------------------------------------------------------------*/
#include <windows.h>
#include <mmsystem.h>

#include "rleapp.h"
#include "dib.h"
#include "rle.h"            // for RLE globals

extern double atof(char *pch);

/*----------------------------------------------------------------------------*\
|   fDialog(id,hwnd,fpfn)                                                      |
|                                                                              |
|   Description:                                                               |
|       This function displays a dialog box and returns the exit code.         |
|       the function passed will have a proc instance made for it.             |
|                                                                              |
|   Arguments:                                                                 |
|       id              resource id of dialog to display                       |
|       hwnd            parent window of dialog                                |
|       fpfn            dialog message function                                |
|                                                                              |
|   Returns:                                                                   |
|       exit code of dialog (what was passed to EndDialog)                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
BOOL fDialog(int id,HWND hwnd,FARPROC fpfn)
{
    BOOL        f;
    HANDLE      hInst;

    hInst = GetWindowWord(hwnd,GWW_HINSTANCE);
    fpfn  = MakeProcInstance(fpfn,hInst);
    f = DialogBox(hInst,MAKEINTRESOURCE(id),hwnd,fpfn);
    FreeProcInstance (fpfn);
    return f;
}

/*----------------------------------------------------------------------------*\
|   fDialogParam(id,hwnd,fpfn,lParam)                                          |
|                                                                              |
|   Description:                                                               |
|       This function displays a dialog box and returns the exit code.         |
|       the function passed will have a proc instance made for it.             |
|       Just like fDialog, but calls DialogBoxParam.                           |
|   Arguments:                                                                 |
|       id              resource id of dialog to display                       |
|       hwnd            parent window of dialog                                |
|       fpfn            dialog message function                                |
|       dwParam         Parameter for DialogBoxParam                            |
|   Returns:                                                                   |
|       exit code of dialog (what was passed to EndDialog)                     |
|                                                                              |
\*----------------------------------------------------------------------------*/
BOOL fDialogParam(int id,HWND hwnd,FARPROC fpfn,DWORD dwParam)
{
    BOOL        f;
    HANDLE      hInst;

    hInst = GetWindowWord(hwnd,GWW_HINSTANCE);
    fpfn  = MakeProcInstance(fpfn,hInst);
    f = DialogBoxParam(hInst,MAKEINTRESOURCE(id),hwnd,fpfn,dwParam);
    FreeProcInstance (fpfn);
    return f;
}

/* FrameDlgProc(hwnd, wMsg, wParam, lParam)
 *
 * This function handles messages when changing the Frames/Sec of the movie
 */
BOOL EXPORT                // TRUE iff message has been processed
FrameDlgProc(hwnd, wMsg, wParam, lParam)
HWND            hwnd;           // window handle of dialog box
short           wMsg;           // message number
WORD            wParam;         // message-dependent parameter
LONG            lParam;         // message-dependent parameter
{
    switch (wMsg) 
    {
	char ach[80];
	
	case WM_INITDIALOG:
            wsprintf(ach, "%d.%03d",
                (WORD)(FramesSec/FramesSecScale),
                (WORD)(FramesSec%FramesSecScale) );
	    SetDlgItemText(hwnd, ID_FRAMESSEC, (LPSTR)ach);
	    return TRUE;
		
	case WM_COMMAND:
	    switch (wParam) 
	    {

		case IDOK:
		    GetDlgItemText(hwnd,ID_FRAMESSEC, (LPSTR)ach, 80);
                    FramesSec = (long)(atof(ach) * FramesSecScale);
                    if (FramesSec <= 0 || FramesSec > 100 * FramesSecScale) 
		    {
			ErrMsg("Bad parameter value.  Frame Rate not changed");
			break;
		    }
		    EndDialog(hwnd, TRUE);
		    break;
			
		case IDCANCEL:          // "Done"
		    EndDialog(hwnd, FALSE);
		break;
	    }
	    break;
    }   
    return FALSE;
}

/*---------------------------------------------------------------------*\
|   fnRangeDlg( hDlg, uiMessage, wParam, lParam )                       |
|                                                                       |
|   Arguments:                                                          |
|       hDlg            window handle of about dialog window            |
|       uiMessage       message number                                  |
|       wParam          message-dependent                               |
|       lParam          message-dependent                               |
|                                                                       |
|   Returns:                                                            |
|       TRUE if message has been processed, else FALSE                  |
|                                                                       |
\*---------------------------------------------------------------------*/

BOOL EXPORT fnRangeDlg(HWND hdlg, short msg, WORD wParam, LONG lParam)
{
    int     id;

    static  RANGEDIALOGPARAM rdp;
    static  iPos;

    switch (msg)
    {
	case WM_COMMAND:
	    switch (wParam)
	    {
		case IDOK:
		    EndDialog(hdlg,iPos);
		    break;

		case IDCANCEL:
		    EndDialog(hdlg,(int)rdp.current);
		    break;
	    }
	    break;

	case WM_HSCROLL:
	    id = GetWindowWord(HIWORD(lParam),GWW_ID);

	    switch (wParam)
	    {
		case SB_LINEDOWN:      iPos += 1; break;
		case SB_LINEUP:        iPos -= 1; break;
		case SB_PAGEDOWN:      iPos += 10; break;
		case SB_PAGEUP:        iPos -= 10; break;
		case SB_THUMBTRACK:
		case SB_THUMBPOSITION: iPos = LOWORD(lParam); break;
		default:
		    return TRUE;
	    }

	    iPos = max(rdp.min,iPos);
	    iPos = min(rdp.max,iPos);
	    SetScrollPos(HIWORD(lParam),SB_CTL,iPos,TRUE);
	    SetDlgItemInt(hdlg,ID_TEXT,iPos,FALSE);
	    break;

	case WM_INITDIALOG:
	    rdp = *(LPRANGEDIALOGPARAM) lParam;
	    iPos = rdp.current;
	    iPos = max(rdp.min,iPos);
	    iPos = min(rdp.max,iPos);
	    SetWindowText(hdlg,rdp.lpCaption);
	    SetScrollRange(GetDlgItem(hdlg,ID_SCROLL),SB_CTL,
						(int)rdp.min,(int)rdp.max,TRUE);
	    SetScrollPos  (GetDlgItem(hdlg,ID_SCROLL),SB_CTL,iPos,TRUE);
	    SetDlgItemInt (hdlg,ID_TEXT,iPos,TRUE);
	    return TRUE;
    }
    return FALSE;
}

/*---------------------------------------------------------------------*\
|   fnResizeDlg( hDlg, uiMessage, wParam, lParam )                       |
|                                                                       |
|   Arguments:                                                          |
|       hDlg            window handle of about dialog window            |
|       uiMessage       message number                                  |
|       wParam          message-dependent                               |
|       lParam          message-dependent                               |
|                                                                       |
|   Returns:                                                            |
|       TRUE if message has been processed, else FALSE                  |
|                                                                       |
\*---------------------------------------------------------------------*/

BOOL EXPORT fnResizeDlg(HWND hdlg, short msg, WORD wParam, LONG lParam)
{
    char ach[80];
    int  i;
    BITMAPINFOHEADER bi;

    switch (msg)
    {
	case WM_COMMAND:
	    switch (wParam)
	    {
		case IDOK:
                    i = GetScrollPos(GetDlgItem(hdlg,ID_SCROLL),SB_CTL);
                    EndDialog(hdlg,i);
		    break;

		case IDCANCEL:
                    EndDialog(hdlg,100);
		    break;
	    }
	    break;

	case WM_HSCROLL:
            i = GetScrollPos(HIWORD(lParam),SB_CTL);

	    switch (wParam)
	    {
                case SB_LINEDOWN:      i += 1; break;
                case SB_LINEUP:        i -= 1; break;
                case SB_PAGEDOWN:      i += 10; break;
                case SB_PAGEUP:        i -= 10; break;
		case SB_THUMBTRACK:
                case SB_THUMBPOSITION: i = LOWORD(lParam); break;
		default:
		    return TRUE;
	    }

            SetScrollPos(HIWORD(lParam),SB_CTL,i,TRUE);

            DibInfo(FrameDib(0), &bi);
            wsprintf(ach, "%d%%: %dx%d", i,
                (int)(bi.biWidth * i / 100),
                (int)(bi.biHeight * i / 100));

            SetDlgItemText(hdlg,ID_TEXT,ach);
	    return TRUE;

	case WM_INITDIALOG:
            SetWindowText(hdlg,"Resize");
            SetScrollRange(GetDlgItem(hdlg,ID_SCROLL),SB_CTL,0,500,TRUE);
            PostMessage(hdlg,WM_HSCROLL,SB_THUMBPOSITION, MAKELONG(100,GetDlgItem(hdlg,ID_SCROLL)));
	    return TRUE;
    }
    return FALSE;
}

/*----------------------------------------------------------------------------*\
|   ErrMsg - Opens a Message box with a error message in it.  The user can     |
|            select the OK button to continue                                  |
\*----------------------------------------------------------------------------*/
int ErrMsg (LPSTR sz,...)
{
    char ach[128];

    wvsprintf (ach,sz,(LPSTR)(&sz+1));    /* Format the string */
    MessageBeep(MB_ICONEXCLAMATION);
    MessageBox(NULL,ach,NULL,MB_OK|MB_ICONEXCLAMATION|MB_TASKMODAL);
    return FALSE;
}

/*----------------------------------------------------------------------------*\
|   AppAbout( hDlg, uiMessage, wParam, lParam )                                |
|                                                                              |
|   Description:                                                               |
|       This function handles messages belonging to the "About" dialog box.    |
|       The only message that it looks for is WM_COMMAND, indicating the use   |
|       has pressed the "OK" button.  When this happens, it takes down         |
|       the dialog box.                                                        |
|                                                                              |
|   Arguments:                                                                 |
|       hDlg            window handle of about dialog window                   |
|       uiMessage       message number                                         |
|       wParam          message-dependent                                      |
|       lParam          message-dependent                                      |
|                                                                              |
|   Returns:                                                                   |
|       TRUE if message has been processed, else FALSE                         |
|                                                                              |
\*----------------------------------------------------------------------------*/
BOOL EXPORT AppAbout(HWND hdlg, short msg, WORD wParam, LONG lParam)
{
    switch (msg)
    {
	case WM_COMMAND:
	    if (wParam == IDOK)
		EndDialog(hdlg,TRUE);
	    break;

	case WM_INITDIALOG:
	    return TRUE;
    }
    return FALSE;
}
