/*
 * CLIP.C
 *
 * Functions to interact with the clipboard.
 *
 */
// COPYRIGHT:
//
//   (C) Copyright Microsoft Corp. 1993.  All rights reserved.
//
//   You have a royalty-free right to use, modify, reproduce and
//   distribute the Sample Files (and/or any modified version) in
//   any way you find useful, provided that you agree that
//   Microsoft has no warranty obligations or liability for any
//   Sample Application Files which are modified.

#include <windows.h>
#include <ole.h>
#include "schmoo.h"
#include "oleglobl.h"



/*
 * FEditCut
 *
 * Purpose:
 *  Places a private format, a metafile, and a bitmap of the display
 *  on the clipboard and clears the editor window with the equivalent
 *  to complete the "Cut" operation.  This operation also sets the
 *  dirty flag through a call to FDirtySet.
 *
 * Parameters:
 *  pGlob           LPGLOBALS to the global variable block.
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 *
 */

BOOL FAR PASCAL FEditCut(LPGLOBALS pGlob)
    {
    //Copy editor contents to the clipboard.
    if (!FEditCopy(pGlob, TRUE))
        return FALSE;

    //Clear out the editor contents.
    SendMessage(pGlob->hWndPolyline, PLM_POLYLINENEW, 0, 0L);
    FDirtySet(TRUE);
    return TRUE;
    }



/*
 * FEditCopy
 *
 * Purpose:
 *  Places a private format, a metafile, and a bitmap of the display
 *  on the clipboard.
 *
 *
 * Parameters:
 *  pGlob           LPGLOBALS to the global variable block.
 *  fCut            BOOL indicating if this copy is part of a Cut.
 *
 * Return Value:
 *  BOOL            TRUE if anything was successfully placed on the
 *                  clipboard, FALSE otherwise.
 *
 */

BOOL FAR PASCAL FEditCopy(LPGLOBALS pGlob, BOOL fCut)
    {
    BOOL            fRet=TRUE;
    HANDLE          hMem;

    //Attempt to gain clipboard ownership.
    if (!OpenClipboard(pGlob->hWnd))
        return FALSE;

    //Clean out whatever junk is in the clipboard.
    EmptyClipboard();

    //Copy private data first.
    hMem=HGetPolyline(pGlob->hWndPolyline);

    //Copy private data (not "Native") we used prior to OLE.
    if (NULL!=hMem)
        SetClipboardData(pGlob->cfSchmoo, hMem);
    else
        fRet &=FALSE;

#ifdef MAKEOLESERVER

    //Copy Native data.
    fRet &=FOLECopyNative(pOLE);

    //Copy "OwnerLink" data.
    fRet &=FOLECopyLink(pOLE, TRUE, pGlob->szFile);

#endif //MAKEOLESERVER

    //Send the METAFILEPICT to the clipboard if we have one.
    hMem=HGetMetafilePict(pGlob->hWndPolyline);

    if (NULL!=hMem)
        SetClipboardData(CF_METAFILEPICT, hMem);
    else
        fRet &=FALSE;

    //Send the bitmap to the clipboard if we can get one.
    hMem=HGetBitmap(pGlob->hWndPolyline);

    if (NULL!=hMem)
        SetClipboardData(CF_BITMAP, hMem);
    else
        fRet &=FALSE;

#ifdef MAKEOLESERVER

    //Copy "ObjectLink" data if we have a filename AND are not cutting.
    if (pGlob->fOpenFile && 0!=pGlob->szFile[0] && !fCut)
        {
        fRet &=FOLECopyLink(pOLE, FALSE, pGlob->szFile);
        }

#endif //MAKEOLESERVER

    //Free clipboard ownership.
    CloseClipboard();
    return fRet;
    }






/*
 * FEditPaste
 *
 * Purpose:
 *  Retrieves the private data format from the clipboard and sets it
 *  to the current figure in the editor window.
 *
 *  Note that if this function is called, then the clipboard format
 *  is available because the Paste menu item is only enabled if the
 *  format is present.
 *
 * Parameters:
 *  pGlob           LPGLOBALS to the global variable block.
 *
 * Return Value:
 *  BOOL            TRUE if successful, FALSE otherwise.
 *
 */

BOOL FAR PASCAL FEditPaste(LPGLOBALS pGlob)
    {
    HANDLE          hMem;
    LPPOLYLINE      lppl;

    //Attempt to gain clipboard ownership.
    if (!OpenClipboard(pGlob->hWnd))
        return FALSE;

    hMem=GetClipboardData(pGlob->cfSchmoo);

    if (NULL==hMem)
        {
        CloseClipboard();
        return FALSE;
        }

    lppl=(LPPOLYLINE)GlobalLock(hMem);

    //TRUE in wParam to cause PLN_SIZECHANGE notification
    SendMessage(pGlob->hWndPolyline, PLM_POLYLINESET, TRUE, (LONG)lppl);
    GlobalUnlock(hMem);

    FDirtySet(TRUE);
    CloseClipboard();
    return TRUE;
    }



/*
 * HGetPolyline
 *
 * Purpose:
 *  Allocates global memory and copies the current Polyline into it.
 *
 * Parameters:
 *  hWnd            HWND of the PolyLine window whose data we want.
 *
 * Return Value:
 *  HANDLE          Global handle if successful, NULL otherwise.
 */

HANDLE FAR PASCAL HGetPolyline(HWND hWnd)
    {
    LPPOLYLINE  lppl;
    HANDLE      hMem;

    //Allocate a copy of the POLYLINE structure and set it as private data.
    hMem=GlobalAlloc(GMEM_DDESHARE | GMEM_MOVEABLE, CBPOLYLINE);

    if (NULL!=hMem)
        {
        lppl=(LPPOLYLINE)GlobalLock(hMem);
        SendMessage(hWnd, PLM_POLYLINEGET, 0, (LONG)lppl);
        GlobalUnlock(hMem);
        }

    return hMem;
    }



/*
 * HGetMetafilePict
 *
 * Purpose:
 *  Retrieves a metafile for the current polyline image.
 *
 * Parameters:
 *  hWnd            HWND of the PolyLine window whose image we want.
 *
 * Return Value:
 *  HANDLE          Global handle to a METAFILEPICT structure if
 *                  successful, NULL otherwise.
 */

HANDLE FAR PASCAL HGetMetafilePict(HWND hWnd)
    {
    HANDLE      hMem;
    DWORD       dw;

    //Retrieve a METAFILEPICT structure for this object.
    dw=SendMessage(hWnd, PLM_METAFILEPICTGET, 0, 0L);
    hMem=LOWORD(dw);

    return hMem;
    }




/*
 * HGetBitmap
 *
 * Purpose:
 *  Retrieves a device-dependent bitmap for the current polyline image.
 *
 * Parameters:
 *  hWnd            HWND of the PolyLine window whose image we want.
 *
 * Return Value:
 *  HANDLE          Bitmap handle if successful, NULL otherwise.
 */

HANDLE FAR PASCAL HGetBitmap(HWND hWnd)
    {
    HANDLE      hMem;
    DWORD       dw;

    dw=SendMessage(hWnd, PLM_BITMAPGET, 0, 0L);
    hMem=LOWORD(dw);

    return hMem;
    }
