{++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++}
{                                                                              }
{                              FORMEDIT v1.00                                  }
{                                    BY                                        }
{                               Robert Wittig                                  }
{                         CompuServe ID 76135,2105                             }
{                                                                              }
{                                                                              }
{ FormEditor is a Delphi-written form editor (what else?) that provides much   }
{ of the functionality of Delphi's own form editor.  This includes:            }
{                                                                              }
{            Control selection through use of left mouse button               }
{            Multiple control selection through use of Shift key              }
{            Multiple control selection through use of selection rectangle    }
{            Moving/Sizing of selected controls through use of "handles"      }
{            Snap-to Grid with definable grid resolution (default 8x8)        }
{            Alignment of selected controls within dialog (Left, right, etc.) }
{            Sizing of selected controls relative to each other               }
{            Z-Order (Bring To Front, Send To Back)                           }
{                                                                              }
{ FormEditor has an _extremely_ easy to use interface.  When you want to edit  }
{ a form, just include FormEdit at compile-time and call EditForm(aForm) at    }
{ run-time.  aForm will be put in edit mode and you can move/resize controls   }
{ on the form.  The left mouse button is used to select/deselect controls.     }
{ The right mouse button pops up the form editor menu.  This menu includes the }
{ following options:                                                           }
{                                                                              }
{            Align To Grid - Align selected controls along grid boundaries    }
{            Bring To Front - Bring selected controls to top of Z-Order       }
{            Send To Back - Send selected controls to bottom of Z-Order       }
{            Align Controls - Brings up the alignment dialog allowing you     }
{                              to align controls relative to each other        }
{            Size Controls - Brings up the size dialog allowing you to size   }
{                             controls relative to each or set specific size   }
{            Grid Options - Brings up dialog allowing you to control the      }
{                            sizing grid options.  These include visibility,   }
{                            X and Y resolution, and the type of marker used   }
{                            to indicate the presence of the grid.             }
{            Exit Editor - Return form to run-time mode.                      }
{                                                                              }
{ This first version is intended to duplicate the action of dialog editors.    }
{ Future versions will allow for creation/deletion of controls and for the     }
{ loading/storing of form information.  Long-term is a quest to develop an     }
{ Object Inspector replacement.                                                }
{                                                                              }
{ This unit is provided as limited freeware.  If you distribute it with your   }
{ code, I ask that you send along this read.me file and leave this header at   }
{ the top of FormEdit.Pas.  If you have any suggestions/improvements to make,  }
{ please send them to me (either forum message or email) on CompuServe at      }
{ 76135,2105.  Other than that, all I ask is that you enjoy!                   }
{                                                                              )
{++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++}
unit FormEdit;

interface
Uses Forms;

Type
    tFormEditorAction = ( eaAlignToGrid, eaBringToFront, eaSendToBack,
                          eaAlignControls, eaSizeControls, eaGridOptions,
                          eaExitEditor );

{ This is the function you call if you want to edit a form.  Just pass it the
  form you want to edit, sit back and let FormEditor do the driving! }
Procedure EditForm ( aForm : tForm );

{ If, for some reason, you want to display a sizing dialog without benefit of
  the popup menu, you can use the following functions/variables to perform the
  various actions.  The command functions available from the popup menu
  correspond to the elements in the tFormEditorAction type.  These actions may
  be invoked directly through a call to PerformAction.  This is, in fact, what
  the default dialogs do.  You can set the X and Y grid resolution either both
  at the same time, through use of SetGridResolution, or one at a time, by
  directly setting the value of XGridStep and YGridStep respectively.  By
  setting the value of GridType, you can control the appearance of the Grid.
  And, of course, ShowGrid will control whether or not the grid is visible.
  Please note that even though the grid may be invisible, it _is_ still present.
  If you move/size a control, it will be moved/sized according to the grid,
  unless XGridStep and YGridStep are both 1 or below.  Whenever you make changes
  to the grid, you need to call ResetGrid to redraw the grid on the form. }
Procedure PerformAction ( Action : tFormEditorAction );
Procedure SetGridResolution ( XResolution, YResolution : Byte );
Procedure ResetGrid;

Type
    tGridType = ( gtSmallDots, gtLargeDots, gtLines );

Const
     ShowGrid  : Boolean = True;
     GridType  : tGridType = gtSmallDots;
     XGridStep : Byte = 8;
     YGridStep : Byte = 8;

implementation
Uses SysUtils, Classes, Controls, Dialogs,
     Menus, Messages, WinProcs, WinTypes,
     dAlign, dSize, dGrid;


{ At this point, I make no explanations of what I do in this section of code.
  Well, that's not quite right.  I'll tell you this.  Most of the work goes on
  in FormEditor.HandleMessages.  This is where I perform selection/sizing/moving
  of controls.  HandleMessages is assigned to Application.OnMessage at the time
  the form is edited.  The previous message handler is reassigned when you are
  finished editing.  I also override the edited form's window procedure.  I do
  this because I need the WM_ERASEBKGND message to be able to draw the grid.  I
  do not handle mouse messages here because as far as I can tell, mouse messages
  are dispatched directly from Application to the form's WndProc procedure,
  bypassing Windows normal message handling.  So, if I want to capture mouse
  messages, I have to do it in Application.OnMessage (or override Application's
  window procedure). Other than this, I don't want to explain too much, because
  the code is too liable to change, and you don't necessarily need to know how
  FormEditor works to use it. }

Const
     cHandleSize   = 5;

     hpFirst       = 1;
     hpLast        = 8;

     hpNone        = 0;
     hpTopLeft     = 1;
     hpTop         = 2;
     hpTopRight    = 3;
     hpLeft        = 4;
     hpRight       = 5;
     hpBottomLeft  = 6;
     hpBottom      = 7;
     hpBottomRight = 8;

     eaAlignLefts   = 1;
     eaAlignTops    = 1;
     eaAlignCenters = 2;
     eaAlignRights  = 3;
     eaAlignBottoms = 3;
     eaAlignSpace   = 4;
     eaAlignCenter  = 5;

     eaSizeShrink   = 1;
     eaSizeGrow     = 2;
     eaSizeSet      = 3;

     Cursors : Array[hpFirst..hpLast] Of tCursor = ( crSizeNWSE, crSizeNS, crSizeNESW,
                                                     crSizeWE,             crSizeWE,
                                                     crSizeNESW, crSizeNS, crSizeNWSE );
Type
    tHandlePosition  = hpNone..hpLast;
    tHandleOperation = ( hoCreate, hoParent, hoDestroy, hoMove, hoShow, hoHide );

    tWndProc = Function ( Handle: hWnd; Msg: word;
                          wParam: word; lParam: Longint ) : Longint;

    TFormEditor = Class;

    TControlHandle = Class ( tWinControl )
    Private
           fPosition  : tHandlePosition;
           FormEditor : TFormEditor;

           Procedure WMPaint ( Var Message : tWMPaint ); message WM_PAINT;
           Procedure WMLButtonDown ( Var Message : tWMLButtonDown ); message WM_LBUTTONDOWN;
    Protected
             Procedure SetParent ( aParent : tWinControl ); override;
             Procedure SetPosition ( Value : tHandlePosition );
             Procedure ResetPosition;

             Function GetCenter : tPoint;
    Public
          Constructor Create ( aFormEditor : TFormEditor; aPosition : tHandlePosition );

          Procedure Hide;
          Procedure Show;

          Property Position : tHandlePosition Read fPosition Write SetPosition;
          Property Center   : tPoint          Read GetCenter;
    End;

    TFormEditor = Class
    Private
           fEditForm     : tForm;
           fXGridStep,
           fYGridStep    : Byte;

           EditControls  : tStringList;
           ActiveControl : tWinControl;
           Selection     : tRect;

           PopupMenu     : tPopupMenu;

           OldFormWndProc : tWndProc;

           Handles : Array[hpFirst..hpLast] Of TControlHandle;
    Protected
             Function  IsEditMessage ( Msg : tMsg ) : Boolean;

             Function  FindWinControl ( hWnd : tHandle ) : tWinControl;
             Function  FindControl    ( Parent : tWinControl; P : tPoint ) : tControl;
             Function  GetControlHandle ( Msg : tMsg ) : tHandlePosition;

             Procedure SetEditForm ( Value : tForm );
             Function  CanAddControl ( Value : tControl ) : Boolean;
             Procedure AddEditControl ( Value : tControl );
             Procedure AddEditControlsInSelection;
             Procedure DeleteEditControl ( Value : tControl );
             Procedure ResetSelection;
             Procedure ClearEditControls;

             Function  HandlePosition ( ControlRect : tRect;
                                        Position    : tHandlePosition ) : tPoint;
             Procedure HandleOperation ( Operation : tHandleOperation );
             Procedure DrawControlHandles ( Control : tControl; Visible : Boolean );

             Procedure SetPopupState ( Sender : tObject );
             Procedure PopupActions ( Sender : tObject );

             Procedure AlignToGrid;
             Procedure BringToFront;
             Procedure SendToBack;
             Procedure AlignControls ( XAlign, YAlign : Integer );
             Procedure SizeControls ( XType, YType, XSize, YSize : Integer );

             Procedure OverrideFormWndProc;
             Procedure RestoreFormWndProc;
             Procedure HandleMessages ( Var Msg : tMsg; Var Handled : Boolean );
    Public
          Constructor Create;
          Destructor Destroy; override;

          Property EditForm  : tForm Read fEditForm Write SetEditForm;
    End;

Var
   FormEditor : tFormEditor;



Function Min ( A, B : Integer ) : Integer;
Begin
     If A < B
        Then Result := A
        Else Result := B;
End;

Function Max ( A, B : Integer ) : Integer;
Begin
     If A > B
        Then Result := A
        Else Result := B;
End;

Procedure MapControlPoints ( FromControl, ToControl : tControl; Var Points; NumPoints : Integer );
Type
    tPoints = Array[0..16382] Of tPoint;
Var
   Point : Integer;
Begin
     For Point := 0 To NumPoints-1 Do
         tPoints(Points)[Point] := ToControl.ScreenToClient (
                                   FromControl.ClientToScreen ( tPoints(Points)[Point] ) );
End;

Procedure SetTopRight ( Var Rect : tRect; X, Y : Integer );
Begin
     With Rect Do
     Begin
          Right := X;
          Top   := Y;
     End;
End;

Procedure SetBottomLeft ( Var Rect : tRect; X, Y : Integer );
Begin
     With Rect Do
     Begin
          Left   := X;
          Bottom := Y;
     End;
End;

Procedure OffsetPoint ( Var Point : tPoint; DeltaX, DeltaY : Integer );
Begin
     With Point Do
     Begin
          Inc ( X, DeltaX );
          Inc ( Y, DeltaY );
     End;
End;

Procedure RoundToNearest ( Var Value : Integer; Step : Integer );
Begin
     Value := Round ( Value / Step ) * Step;
End;

Function GetClipDC ( Control : tWinControl ) : tHandle;
Var
   ClipOrg  : tPoint;
   ClipRect : tRect;
   ClipRgn  : tHandle;
Begin
     ClipOrg := Point ( 0, 0 );
     ClientToScreen ( Control.Handle, ClipOrg );
     GetClientRect  ( Control.Handle, ClipRect );
     OffsetRect     ( ClipRect, ClipOrg.X, ClipOrg.Y );
     OffsetPoint    ( ClipRect.BottomRight, 1, 1 );

     Result := GetDC ( 0 );
     SetViewPortOrg ( Result, ClipOrg.X, ClipOrg.Y );
     ClipRgn := CreateRectRgnIndirect ( ClipRect );
     SelectClipRgn ( Result, ClipRgn );
     DeleteObject ( ClipRgn );
End;




Constructor TControlHandle.Create ( aFormEditor : tFormEditor; aPosition : tHandlePosition );
Begin
     Inherited Create ( Nil );

     Visible    := False;
     FormEditor := aFormEditor;
     SetBounds ( 0, 0, cHandleSize, cHandleSize );
     SetPosition ( aPosition );
End;

Procedure TControlHandle.WMPaint ( Var Message : tWMPaint );
Var
   DC : tHandle;
   PS : tPaintStruct;
Begin
     If Message.DC = 0
        Then DC := BeginPaint ( Handle, PS )
        Else DC := Message.DC;

     PatBlt ( DC, 0, 0, Width, Height, BLACKNESS );

     If Message.DC = 0 Then EndPaint ( Handle, PS );
End;

Procedure TControlHandle.WMLButtonDown ( Var Message : tWMLButtonDown );
Begin
     MessageBeep(0);
End;

Procedure TControlHandle.SetParent ( aParent : tWinControl );
Begin
     If Not ( csDestroying in ComponentState ) Then
        Inherited SetParent ( aParent );
End;

Procedure TControlHandle.SetPosition ( Value : tHandlePosition );
Begin
     If Value <> fPosition Then
     Begin
          fPosition := Value;
          Cursor    := Cursors[fPosition];
     End;
End;

Procedure TControlHandle.ResetPosition;
Var
   Center : tPoint;
Begin
     With FormEditor Do
     Begin
          Center := HandlePosition ( Selection, fPosition );
          OffsetPoint ( Center, -cHandleSize Div 2, -cHandleSize Div 2 );
{          MapControlPoints ( tControl(EditControls.Objects[0]).Parent, fEditForm, Center, 1 );}
          SetBounds ( Center.X, Center.Y, cHandleSize, cHandleSize );
     End;
End;

Procedure TControlHandle.Hide;
Var
   Rect : tRect;
Begin
     If Visible And ( Parent <> Nil ) Then
     Begin
          Visible := False;
{          Rect := ClientRect;
          MapWindowPoints ( Handle, Parent.Handle, Rect, 2);
          InvalidateRect ( Parent.Handle, @Rect, False );
          Parent.Update;}
     End;
End;

Procedure TControlHandle.Show;
Begin
     If Parent <> Nil Then
     Begin
          BringToFront;
          Visible := True;
     End;
End;

Function TControlHandle.GetCenter : tPoint;
Begin
     With Result Do
     Begin
          X := Left + Width  Div 2;
          Y := Top  + Height Div 2;
     End;
End;





Function FormEditorWndProc ( Handle : hWnd; Msg : Word;
                             wParam : Word; lParam : LongInt ) : LongInt; export;
Var
   XStep, YStep : Byte;
   Rect         : tRect;
Begin
     If Msg = WM_ERASEBKGND Then
     Begin
          Result := FormEditor.OldFormWndProc ( Handle, Msg, wParam, lParam );

          If ShowGrid And ( XGridStep > 2 ) And ( YGridStep > 2 ) Then
          Begin
               GetClientRect ( Handle, Rect );

               If GridType In [gtSmallDots, gtLargeDots] Then
                  For XStep := 1 to Rect.Right Div XGridStep Do
                      For YStep := 1 to Rect.Bottom Div YGridStep Do
                            If GridType = gtSmallDots
                               Then SetPixel ( wParam, XStep*XGridStep, YStep*YGridStep, 0 )
                               Else PatBlt ( wParam, XStep*XGridStep-1, YStep*YGridStep-1,
                                                     3, 3, BLACKNESS )
               Else
               Begin
                    For XStep := 1 to Rect.Right Div XGridStep Do
                    Begin
                         MoveTo ( wParam, XStep*XGridStep, 0 );
                         LineTo ( wParam, XStep*XGridStep, Rect.Bottom );
                    End;

                    For YStep := 1 to Rect.Bottom Div YGridStep Do
                    Begin
                         MoveTo ( wParam,          0, YStep*YGridStep );
                         LineTo ( wParam, Rect.Right, YStep*YGridStep );
                    End;
               End;
          End;
     End
     Else
         Result := FormEditor.OldFormWndProc ( Handle, Msg, wParam, lParam );
End;





Constructor TFormEditor.Create;
Var
   MenuItem : tMenuItem;

   Procedure AddPopupMenuItem ( Caption : String; Tag : Integer );
   Var
      MenuItem : tMenuItem;
   Begin
        MenuItem := tMenuItem.Create ( PopupMenu );
        MenuItem.Caption := Caption;
        MenuItem.Tag     := Tag;
        MenuItem.OnClick := PopupActions;
        PopupMenu.Items.Add ( MenuItem );
   End;
Begin
     Inherited Create;
     EditControls := tStringList.Create;
     HandleOperation ( hoCreate );

     PopupMenu := tPopupMenu.Create ( Application );
     PopupMenu.OnPopup := SetPopupState;

     { Create Form Editor popup menu }
     AddPopupMenuItem ( 'Align To &Grid',     Ord(eaAlignToGrid) );
     AddPopupMenuItem ( 'Bring To &Front',    Ord(eaAlignToGrid) );
     AddPopupMenuItem ( 'Send To &Back',      Ord(eaAlignToGrid) );
     AddPopupMenuItem ( '-', 0);
     AddPopupMenuItem ( '&Align Controls...', Ord(eaAlignControls) );
     AddPopupMenuItem ( '&Size Controls...',  Ord(eaSizeControls) );
     AddPopupMenuItem ( 'Grid &Options...',   Ord(eaGridOptions) );
     AddPopupMenuItem ( '-', 0 );
     AddPopupMenuItem ( 'E&xit Editor',       Ord(eaExitEditor) );
End;

Destructor TFormEditor.Destroy;
Begin
     EditControls.Free;
     Inherited Destroy;
End;

Procedure TFormEditor.SetEditForm ( Value : tForm );
Const
     OldOnMessage : tMessageEvent = Nil;
Begin
     If Value <> fEditForm Then
     Begin
          ClearEditControls;
          HandleOperation ( hoHide );

          If fEditForm <> Nil Then
          Begin
               fEditForm.ActiveControl := ActiveControl;
               RestoreFormWndProc;
          End;

          fEditForm := Value;

          If fEditForm <> Nil Then
          Begin
               ActiveControl := fEditForm.ActiveControl;
               SetFocus ( fEditForm.Handle );
               OverrideFormWndProc;
          End;


          HandleOperation ( hoParent );

          If fEditForm = Nil Then
             Application.OnMessage := OldOnMessage
          Else
          Begin
               OldOnMessage := Application.OnMessage;
               Application.OnMessage := HandleMessages;
          End;
     End;
End;

Function  TFormEditor.CanAddControl ( Value : tControl ) : Boolean;
Begin
     Result := ( EditControls.Count = 0 ) Or
               ( tControl(EditControls.Objects[0]).Parent = Value.Parent );
End;

Procedure TFormEditor.AddEditControl ( Value : tControl );
Var
   Temp : tRect;
Begin
     If EditControls.IndexOfObject ( Value ) = -1 Then
     Begin
          EditControls.AddObject ( Value.Name, Value );

          If EditControls.Count = 1 Then
          Begin
               Selection := Value.ClientRect;
               MapControlPoints ( Value, Value.Parent, Selection, 2 );
               HandleOperation ( hoParent );
          End
          Else
          Begin
               If EditControls.Count = 2 Then
                  DrawControlHandles ( tControl(EditControls.Objects[0]), True );
               DrawControlHandles ( Value, True );
               Temp := Value.ClientRect;
               MapControlPoints ( Value, Value.Parent, Temp, 2 );
               UnionRect ( Selection, Temp, Selection );
          End;
     End;
End;

Procedure TFormEditor.AddEditControlsInSelection;
Var
   Control : Integer;
   Rect    : tRect;
   Parent  : tWinControl;
   SelectionRectangle : tRect;

   Function FindTopMostParent ( Parent : tWinControl ) : tWinControl;
   Var
      Control : Integer;
      Rect    : tRect;
   Begin
        Result := Nil;

        For Control := 0 to Parent.ControlCount - 1 Do
        With Parent.Controls[Control] Do
        If ClassName <> 'TControlHandle' Then
        Begin
             Rect := ClientRect;
             MapControlPoints ( Parent.Controls[Control], fEditForm, Rect, 2 );

             If IntersectRect ( Rect, Selection, Rect ) > 0 Then
             Begin
                  Result := Parent;
                  Break;
             End;
        End;

        If Result = Nil Then
        For Control := 0 to Parent.ControlCount-1 Do
            If ( Parent.Controls[Control] is tWinControl ) And
               ( Parent.Controls[Control].ClassName <> 'TControlHandle' ) Then
            Begin
                 Result := FindTopMostParent ( tWinControl(Parent.Controls[Control]) );
                 If Result <> Nil Then Break;
            End;
   End;
Begin
     SelectionRectangle := Selection;
     Parent := FindTopMostParent ( fEditForm );
     If Parent <> Nil Then
        For Control := 0 to Parent.ControlCount - 1 Do
        If Parent.Controls[Control].ClassName <> 'TControlHandle' Then
        Begin
             Rect := Parent.Controls[Control].ClientRect;
             MapControlPoints ( Parent.Controls[Control], Parent, Rect, 2 );

             If IntersectRect ( Rect, SelectionRectangle, Rect ) > 0 Then
                AddEditControl ( Parent.Controls[Control] );
        End;
End;


Procedure TFormEditor.DeleteEditControl ( Value : tControl );
Begin
     EditControls.Delete ( EditControls.IndexOfObject ( Value ) );
     ResetSelection;
     DrawControlHandles ( Value, False );
     If EditControls.Count = 1 Then
        DrawControlHandles ( tControl(EditControls.Objects[0]), False );
End;

Procedure TFormEditor.ResetSelection;
Var
   Control : Integer;
   Temp    : tRect;
Begin
     FillChar ( Selection, SizeOf(Selection), 0 );

     For Control := 0 to EditControls.Count-1 Do
     With EditControls Do
     Begin
          Temp := tControl(Objects[Control]).ClientRect;
          MapControlPoints ( tControl(Objects[Control]),
                             tControl(Objects[Control]).Parent, Temp, 2 );
          UnionRect ( Selection, Temp, Selection );
     End;
End;


Procedure TFormEditor.ClearEditControls;
Var
   Control : Integer;
Begin
     If EditControls.Count = 1
        Then HandleOperation ( hoHide )
     Else If EditControls.Count > 1 Then
          For Control := 0 to EditControls.Count-1 Do
              DrawControlHandles ( tControl(EditControls.Objects[Control]), False );

     EditControls.Clear;
     FillChar ( Selection, SizeOf(Selection), 0 );
End;



function TFormEditor.IsEditMessage ( Msg : tMsg ) : Boolean;
Begin
     Result := ( fEditForm <> Nil ) And
               ( ( Msg.hWnd = fEditForm.Handle ) or
                 ( IsChild ( fEditForm.Handle, Msg.hWnd ) ) );
End;

Function TFormEditor.HandlePosition ( ControlRect : tRect;
                                      Position    : tHandlePosition ) : tPoint;
Var
   TopLeft : tPoint;
Begin

     TopLeft := ControlRect.TopLeft;
     OffsetRect ( ControlRect, -TopLeft.X, -TopLeft.Y );

     With ControlRect Do
     Case Position Of
          hpTopLeft     : Result := Point ( 0,           0 );
          hpTop         : Result := Point ( Right Div 2, 0 );
          hpTopRight    : Result := Point ( Right,       0 );
          hpLeft        : Result := Point ( 0,           Bottom Div 2 );
          hpRight       : Result := Point ( Right,       Bottom Div 2 );
          hpBottomLeft  : Result := Point ( 0,           Bottom );
          hpBottom      : Result := Point ( Right Div 2, Bottom );
          hpBottomRight : Result := Point ( Right,       Bottom );
     End;

     OffsetPoint ( Result, TopLeft.X, TopLeft.Y );
End;

Function TFormEditor.GetControlHandle ( Msg : tMsg ) : tHandlePosition;
Var
   Position : tHandlePosition;
Begin
     Result := hpNone;

     For Position := hpFirst to hpLast Do
         If Msg.hWnd = Handles[Position].Handle Then
         Begin
              Result := Position;
              Break;
         End;
End;

Function tFormEditor.FindWinControl ( hWnd : tHandle ) : tWinControl;
Var
   Component : Integer;
Begin
     Result := Nil;

     With fEditForm Do
          For Component := ComponentCount-1 Downto 0 Do
              If ( Components[Component] is tWinControl ) And
                 ( tWinControl(Components[Component]).Handle = hWnd ) Then
                   Begin
                        Result := Components[Component] as tWinControl;
                        Break;
                   End;
End;

Function tFormEditor.FindControl ( Parent : tWinControl; P : tPoint ) : tControl;
Var
   Index   : Integer;
   Control : tControl;
   Rect    : tRect;
Begin
     Result := Parent;

     For Index := Parent.ControlCount-1 Downto 0 Do
     If Parent.Controls[Index].ClassName <> 'TControlHandle' Then
     Begin
          Control := Parent.Controls[Index];
          Rect    := Control.ClientRect;
          MapControlPoints ( Control, Parent, Rect, 2 );

          If PtInRect ( Rect, P ) Then
          Begin
               If Control is tWinControl Then
               Begin
                    MapControlPoints ( Parent, Control, P, 1 );
                    Result := FindControl ( tWinControl(Control), P )
               End
               Else
                   Result := Control;
               Break;
          End;
     End;
End;

Procedure TFormEditor.OverrideFormWndProc;
Begin
     @OldFormWndProc := Pointer ( GetWindowLong ( fEditForm.Handle, GWL_WNDPROC ) );
     SetWindowLong ( fEditForm.Handle, GWL_WNDPROC, Longint ( @FormEditorWndProc ) );
     InvalidateRect ( fEditForm.Handle, Nil, True );
End;

Procedure TFormEditor.RestoreFormWndProc;
Begin
     SetWindowLong ( fEditForm.Handle, GWL_WNDPROC, Longint ( @OldFormWndProc ) );
     InvalidateRect ( fEditForm.Handle, Nil, True );
End;

procedure TFormEditor.HandleMessages ( Var Msg : tMsg; Var Handled : Boolean );
Const
     Sizing         : Boolean = False;
     Moving         : Boolean = False;
     Drawing        : Boolean = False;
     Deleting       : Boolean = False;
     Start          : tPoint = (X:0;Y:0);
     Last           : tPoint = (X:0;Y:0);
     SelectedHandle : tHandlePosition = hpNone;
Var
   Now        : tPoint;
   CursorRect : tRect;
   WinControl : tWinControl;
   Control    : tControl;

   Procedure DrawFocusRect;
   Var
      DC : tHandle;
      OldBrush : tHandle;

      Control     : Integer;

      ClipControl : tWinControl;

      DrawControl : tControl;
      DrawRect    : tRect;
   Begin
        If ( Selection.Left <> Selection.Right ) And
           ( Selection.Top  <> Selection.Bottom ) Then
        Begin
             If EditControls.Count = 0
                Then ClipControl := fEditForm
                Else ClipControl := tControl(EditControls.Objects[0]).Parent;

             DC := GetClipDC ( ClipControl );

             SetROP2 ( DC, R2_NOT );
             OldBrush := SelectObject ( DC, GetStockObject ( HOLLOW_BRUSH ) );

             If EditControls.Count <= 1 Then
                With Selection Do Rectangle ( DC, Left, Top, Right, Bottom )
             Else
                 For Control := 0 to EditControls.Count - 1 Do
                 Begin
                      DrawControl := tControl(EditControls.Objects[Control]);
                      DrawRect := DrawControl.ClientRect;
                      OffsetRect ( DrawRect, DrawControl.Left, DrawControl.Top );
                      With DrawRect Do
                           Rectangle ( DC, Left  + Last.X - Start.X, Top    + Last.Y - Start.Y,
                                           Right + Last.X - Start.X, Bottom + Last.Y - Start.Y );
                 End;

             SelectObject ( DC, OldBrush );
             ReleaseDC ( fEditForm.Handle, DC );
        End;
   End;

   Procedure SetSelection;
   Begin
        If Sizing Or Moving Then
        Begin
             If Sizing Then Last := HandlePosition ( Selection, SelectedHandle );

             If XGridStep > 1 Then RoundToNearest ( Last.X, XGridStep );
             If YGridStep > 1 Then RoundToNearest ( Last.Y, YGridStep );

             If XGridStep > 1 Then RoundToNearest ( Start.X, XGridStep );
             If YGridStep > 1 Then RoundToNearest ( Start.Y, YGridStep );
        End
        Else
            Selection := Rect ( Start.X, Start.Y, Start.X, Start.Y );

        DrawFocusRect;
   End;

   Procedure ChangeSelection;
   Var
      Delta : tPoint;
   Begin
        Delta := Point ( Now.X - Last.X, Now.Y - Last.Y );
        Last := Now;

        If Sizing Then
        With Selection Do
             Case SelectedHandle Of
                  hpTopLeft     : OffsetPoint ( TopLeft, Delta.X, Delta.Y );
                  hpTop         : Inc ( Top, Delta.Y );
                  hpTopRight    : SetTopRight ( Selection, Right + Delta.X, Top + Delta.Y );
                  hpLeft        : Inc ( Left, Delta.X );
                  hpRight       : Inc ( Right, Delta.X );
                  hpBottomLeft  : SetBottomLeft ( Selection, Left + Delta.X, Bottom + Delta.Y );
                  hpBottom      : Inc ( Bottom, Delta.Y );
                  hpBottomRight : OffsetPoint ( BottomRight, Delta.X, Delta.Y );
             End
        Else If Moving Then
             OffsetRect ( Selection, Delta.X, Delta.Y )
        Else
            Selection := Rect ( Min ( Start.X, Now.X ), Min ( Start.Y, Now.Y ),
                                Max ( Start.X, Now.X ), Max ( Start.Y, Now.Y ) );
   End;

   Procedure SetClipRect;
   Var
      CursorRect : tRect;
      Temp       : tPoint;
      ptHandle   : tPoint;

      Function Opposite ( Handle : tHandlePosition ) : tPoint;
      Begin
           Result := HandlePosition ( Selection, 9-Handle );
      End;
   Begin
        If EditControls.Count > 0 Then
        With tControl(EditControls.Objects[0]).Parent Do
        Begin
             CursorRect := ClientRect;
             WinProcs.ClientToScreen ( Handle, CursorRect.TopLeft );
             WinProcs.ClientToScreen ( Handle, CursorRect.BottomRight );
        End
        Else
        Begin
             CursorRect := fEditForm.ClientRect;
             WinProcs.ClientToScreen ( fEditForm.Handle, CursorRect.TopLeft );
             WinProcs.ClientToScreen ( fEditForm.Handle, CursorRect.BottomRight );
        End;

        OffsetPoint ( CursorRect.BottomRight, 1, 1 );


        If Sizing Then
        Begin
             ptHandle := fEditForm.ClientToScreen ( Opposite ( SelectedHandle ) );

             {set top}
             If Not ( SelectedHandle In [hpTopLeft, hpTop, hpTopRight] )
                Then CursorRect.Top := ptHandle.Y;

             {set Left}
             If Not ( SelectedHandle In [hpTopLeft, hpLeft, hpBottomLeft] )
                Then CursorRect.Left := ptHandle.X;

             {set Right}
             If Not ( SelectedHandle In [hpTopRight, hpRight, hpBottomRight] )
                Then CursorRect.Right := ptHandle.X;

             {set bottom}
             If Not ( SelectedHandle In [hpBottomLeft, hpBottom, hpBottomRight] )
                Then CursorRect.Bottom := ptHandle.Y;
        End;

        ClipCursor ( @CursorRect );
   End;

   Procedure AdjustEditControls;
   Var
      Rect    : tRect;
      Prefix  : tPoint;
      Control : Integer;
   Begin
        { account for any scrolling the user has done }
        FillChar ( Prefix, SizeOf(Prefix), 0 );
        Prefix.X := fEditForm.HorzScrollBar.Position;
        Prefix.Y := fEditForm.VertScrollBar.Position;

        { Get the selection rectangle and transform it from the Form's coordinate
          system to the control's parent's coordinate system. }
        Rect := Selection;
        OffsetRect ( Rect, -Prefix.X, -Prefix.Y );

        If Sizing Then
        With tControl(EditControls.Objects[0]) Do
             SetBounds ( Rect.Left, Rect.Top, Rect.Right - Rect.Left, Rect.Bottom - Rect.Top )
        Else
            For Control := 0 to EditControls.Count-1 Do
            With tControl(EditControls.Objects[Control]) Do
                 SetBounds ( Left + Last.X - Start.X, Top + Last.Y - Start.Y, Width, Height );
   End;

begin
     If IsEditMessage ( Msg ) Then
        Case Msg.Message Of
             WM_LBUTTONDOWN : Begin
                                   ReleaseCapture;
                                   SetCapture ( fEditForm.Handle );

                                   Drawing := True;

                                   HandleOperation ( hoHide );

                                   SelectedHandle := GetControlHandle ( Msg );
                                   If SelectedHandle <> hpNone Then
                                      If EditControls.Count = 1
                                         Then Sizing := True
                                         Else Moving := True
                                   Else
                                   Begin
                                        If fEditForm.Handle = Msg.hWnd Then
                                           Control := FindControl ( fEditForm, tPoint(Msg.lParam) )
                                        Else If IsChild ( fEditForm.Handle, Msg.hWnd ) Then
                                        Begin
                                             WinControl := FindWinControl ( Msg.hWnd );
                                             Control := FindControl ( WinControl, tPoint(Msg.lParam) );
                                        End
                                        Else
                                            Control := Nil;

                                        If Control = fEditForm Then
                                           ClearEditControls
                                        Else
                                        With EditControls Do
                                        If IndexOfObject ( Control ) = -1 Then
                                        Begin
                                             Moving := True;
                                             If ( Msg.wParam And MK_SHIFT > 0 ) And
                                                ( CanAddControl ( Control ) ) Then
                                                AddEditControl ( Control )
                                             Else
                                             Begin
                                                  ClearEditControls;
                                                  AddEditControl ( Control );
                                             End;
                                        End
                                        Else If Msg.wParam And MK_SHIFT > 0 Then
                                        Begin
                                             Drawing := False;
                                             DeleteEditControl ( Control );
                                             If EditControls.Count = 1 Then
                                             Begin
{                                                  HandleOperation ( hoHide );}
                                                  HandleOperation ( hoMove );
                                             End;
                                        End
                                        Else
                                            Moving := True;
                                   End;

                                   If Drawing Then
                                   Begin
                                        { Map coordinates to parent of chosen control,
                                          or to form in no control is chosen }
                                        If EditControls.Count > 0 Then
                                           MapWindowPoints ( Msg.hWnd,
                                                             tControl(EditControls.Objects[0]).Parent.Handle,
                                                             Msg.lParam, 1 )
                                        Else
                                            MapWindowPoints ( Msg.hWnd, fEditForm.Handle,
                                                              Msg.lParam, 1 );

                                        Start   := tPoint(Msg.lParam);
                                        Last    := Start;
                                        SetSelection;
                                        SetClipRect;
                                   End;

                                   Handled := True;
                              End;

             WM_MOUSEMOVE   : If Drawing Then
                              Begin
                                   { Map coordinates to parent of chosen control,
                                     if a control is chosen }
                                   If EditControls.Count > 0 Then
                                      MapWindowPoints ( Msg.hWnd,
                                                        tControl(EditControls.Objects[0]).Parent.Handle,
                                                        Msg.lParam, 1 );

                                   Now := tPoint(Msg.lParam);

                                   { Transform Now along grid }
                                   If Sizing or Moving Then
                                   Begin
                                        If XGridStep > 0 Then RoundToNearest ( Now.X, XGridStep );
                                        If YGridStep > 0 Then RoundToNearest ( Now.Y, YGridStep );
                                   End;

                                   If ( ( Now.X <> Last.X ) Or
                                        ( Now.Y <> Last.Y ) ) Then
                                   Begin
                                        DrawFocusRect;
                                        ChangeSelection;
                                        DrawFocusRect;
                                   End;

                                   Handled := True;
                              End;

             { eat a double click }
             WM_LBUTTONDBLCLK : Handled := True;

             WM_LBUTTONUP   : Begin
                                   ClipCursor ( Nil );
                                   ReleaseCapture;

                                   If Drawing Then
                                   Begin
                                        DrawFocusRect;
                                        Drawing   := False;

                                        If Sizing or Moving Then
                                        Begin
                                             AdjustEditControls;
                                             HandleOperation ( hoMove );

                                             SelectedHandle := hpNone;
                                             Sizing := False;
                                             Moving := False;
                                        End
                                        Else
                                        Begin
                                             AddEditControlsInSelection;
                                             HandleOperation ( hoMove );
                                        End;

                                        If EditControls.Count > 0
                                           Then HandleOperation ( hoShow );
                                   End
                                   Else If EditControls.Count = 1
                                        Then HandleOperation ( hoShow );

                                   Handled := True;
                              End;

             WM_RBUTTONDOWN : Begin
                                   If fEditForm.Handle = Msg.hWnd Then
                                      Control := FindControl ( fEditForm, tPoint(Msg.lParam) )
                                   Else If IsChild ( fEditForm.Handle, Msg.hWnd ) Then
                                   Begin
                                        WinControl := FindWinControl ( Msg.hWnd );
                                        Control := FindControl ( WinControl, tPoint(Msg.lParam) );
                                   End
                                   Else
                                       Control := Nil;

                                   If Control = fEditForm Then
                                   Begin
                                        HandleOperation ( hoHide );
                                        ClearEditControls;
                                   End;

                                   If ( EditControls.Count = 0 ) Or
                                      ( EditControls.IndexOfObject ( Control ) > -1 ) Then
                                   Begin
                                        ClientToScreen ( Msg.hWnd, tPoint(Msg.lParam) );
                                        With tPoint(Msg.lParam) Do PopupMenu.Popup ( X, Y );
                                   End;

                                   Handled := True;
                              End;

             WM_RBUTTONUP,
             WM_RBUTTONDBLCLK : Handled := True;
        End;
end;


Procedure TFormEditor.DrawControlHandles ( Control : tControl; Visible : Boolean );
Var
   DrawControl : tWinControl;
   DrawRect    : tRect;
   Width,
   Height      : Integer;

   DC          : tHandle;

   Procedure DrawHandle ( X, Y : Integer );
   Var
      Rect : tRect;
   Begin
        If Visible
           Then PatBlt ( DC, DrawRect.Left + X, DrawRect.Top + Y,
                             cHandleSize, cHandleSize, PATCOPY )
        Else
        Begin
             Rect := Bounds ( DrawRect.Left + X, DrawRect.Top + Y, cHandleSize, cHandleSize );
             InvalidateRect ( DrawControl.Handle, @Rect, True );
        End;
   End;

Begin
     If Control is tWinControl Then
     Begin
          DrawControl := tWinControl(Control);
          DrawRect    := DrawControl.ClientRect;
     End
     Else
     Begin
          DrawControl := Control.Parent;
          DrawRect    := Control.ClientRect;
          OffsetRect ( DrawRect, Control.Left, Control.Top );
     End;

     Width := DrawRect.Right - DrawRect.Left;
     Height := DrawRect.Bottom - DrawRect.Top;

     If Visible Then
     Begin
          DC := GetDC ( DrawControl.Handle );
          SelectObject ( DC, GetStockObject ( GRAY_BRUSH ) );
     End;

     DrawHandle ( 0,                             0 );
     DrawHandle ( ( Width - cHandleSize ) Div 2, 0 );
     DrawHandle ( Width - cHandleSize,           0 );

     DrawHandle ( 0,                             ( Height - cHandleSize ) Div 2 );
     DrawHandle ( Width - cHandleSize,           ( Height - cHandleSize ) Div 2 );

     DrawHandle ( 0,                             Height - cHandleSize );
     DrawHandle ( ( Width - cHandleSize ) Div 2, Height - cHandleSize );
     DrawHandle ( Width - cHandleSize,           Height - cHandleSize );

     If Visible
        Then ReleaseDC ( DrawControl.Handle, DC );

     DrawControl.Update;
End;

Procedure TFormEditor.HandleOperation ( Operation : tHandleOperation );
Var
   Index : tHandlePosition;

   Procedure DrawControls ( Visible : Boolean );
   Var
      Control : Integer;
   Begin
        For Control := 0 to EditControls.Count-1 Do
            DrawControlHandles ( tControl(EditControls.Objects[Control]), Visible );
   End;

Begin
     If ( fEditForm <> Nil ) And
        ( EditControls.Count = 1 ) And
        ( Operation In [hoShow, hoHide] )
        Then LockWindowUpdate ( fEditForm.Handle );

     If ( EditControls.Count > 1 ) And
        ( Operation = hoShow ) Then
        DrawControls ( True );

     For Index := hpFirst to hpLast Do
         Case Operation Of
              hoCreate  : Handles[Index] := TControlHandle.Create ( Self, Index );
              hoParent  : If EditControls.Count = 0
                             Then Handles[Index].Parent := fEditForm
                             Else Handles[Index].Parent := tControl(EditControls.Objects[0]).Parent;
              hoDestroy : Handles[Index].Free;
              hoMove    : Handles[Index].ResetPosition;
              hoShow    : If EditControls.Count = 1 Then Handles[Index].Show;
              hoHide    : Handles[Index].Hide;
         End;

     If ( fEditForm <> Nil ) And
        ( EditControls.Count = 1 ) And
        ( Operation In [hoShow, hoHide] ) Then
     Begin
          LockWindowUpdate ( 0 );
          fEditForm.Update;
     End;
End;

Procedure tFormEditor.AlignToGrid;
Var
   NewLeft, NewTop : Integer;
   Control : Integer;
Begin
     If ( XGridStep > 1 ) And ( YGridStep > 1 ) Then
     Begin
          HandleOperation ( hoHide );

          For Control := 0 to EditControls.Count-1 Do
          With tControl(EditControls.Objects[Control]) Do
          Begin
               NewLeft := Left;
               NewTop  := Top;

               RoundToNearest ( NewLeft, XGridStep );
               RoundToNearest ( NewTop,  YGridStep );

               SetBounds ( NewLeft, NewTop, Width, Height );
          End;

          ResetSelection;
          HandleOperation ( hoMove );
          HandleOperation ( hoShow );
     End;
End;

Procedure tFormEditor.BringToFront;
Var
   Control : Integer;
Begin
     For Control := 0 to EditControls.Count-1 Do
         tControl(EditControls.Objects[Control]).BringToFront;
     HandleOperation ( hoShow );
End;

Procedure tFormEditor.SendToBack;
Var
   Control : Integer;
Begin
     For Control := 0 to EditControls.Count-1 Do
         tControl(EditControls.Objects[Control]).SendToBack;
     HandleOperation ( hoShow );
End;


Procedure tFormEditor.AlignControls ( XAlign, YAlign : Integer );
Var
   Anchor         : tControl;
   AlignControl   : tControl;
   Control        : Integer;
   X, Y           : Integer;

   DeltaX,
   DeltaY         : Integer;

   OriginX,
   OriginY        : Integer;

   Space          : Integer;

   SortList       : tStringList;

   Function SpacePad ( Num : Integer ) : String;
   Begin
        Str ( Num : 5, Result );
   End;
Begin
     If ( fEditForm <> Nil ) And
        ( ( EditControls.Count > 1 ) Or
          ( ( EditControls.Count = 1 ) And
            ( ( XAlign = eaAlignCenter ) Or
              ( YAlign = eaAlignCenter )
             )
           )
         ) Then
     Begin
          HandleOperation ( hoHide );

          { Align controls in X direction }
          If XAlign In [eaAlignSpace,eaAlignCenter] Then
          Begin
               Space := 0;

               SortList := tStringList.Create;
               With SortList Do
               Try
                  { Sort controls }
                  For Control := 0 to EditControls.Count-1 Do
                  With tControl(EditControls.Objects[Control]) Do
                  Begin
                       Inc ( Space, Width );
                       AddObject ( SpacePad(Left), EditControls.Objects[Control] );
                  End;

                  Sort;

                  { Determine spacing }
                  If XAlign = eaAlignSpace Then
                  Begin
                       DeltaX := ( Selection.Right - Selection.Left + 1 - Space ) Div ( Count - 1 );
                       OriginX := tControl(Objects[0]).Left;
                       If DeltaX > 0 Then
                          For Control := 0 to Count-1 Do
                          With tControl(Objects[Control]) Do
                          Begin
                               Left := OriginX;
                               Inc ( OriginX, DeltaX + Width );
                          End;
                  End
                  Else
                  Begin
                       DeltaX := ( tControl(Objects[0]).Parent.ClientWidth -
                                   Selection.Right + Selection.Left -1 ) Div 2 -
                                 tControl(Objects[0]).Left  ;
                       For Control := 0 to Count-1 Do
                       With tControl(Objects[Control]) Do
                            Left := Left + DeltaX;
                  End;

                  { Do the alignment }

               Finally
                      Free;
               End;
          End
          Else
          Begin
               Anchor := tControl(EditControls.Objects[0]);
               For Control := 0 to EditControls.Count - 1 Do
               With tControl(EditControls.Objects[Control]) Do
               Begin
                    X := -1;
                    Case XAlign Of
                         eaAlignLefts   : Left := Anchor.Left;
                         eaAlignCenters : Left := Anchor.Left + Anchor.Width Div 2 - Width Div 2;
                         eaAlignRights  : Left := Anchor.Left + Anchor.Width - Width;
                    End;
               End;
          End;

          { Align controls in Y direction }
          If YAlign In [eaAlignSpace,eaAlignCenter] Then
          Begin
               Space := 0;

               SortList := tStringList.Create;
               With SortList Do
               Try
                  { Sort controls }
                  For Control := 0 to EditControls.Count-1 Do
                  With tControl(EditControls.Objects[Control]) Do
                  Begin
                       Inc ( Space, Height );
                       AddObject ( SpacePad(Top), EditControls.Objects[Control] );
                  End;

                  Sort;

                  { Determine spacing }
                  If YAlign = eaAlignSpace Then
                  Begin
                       DeltaY := ( Selection.Bottom - Selection.Top + 1 - Space ) Div ( Count - 1 );
                       OriginY := tControl(Objects[0]).Top;
                       If DeltaY > 0 Then
                          For Control := 0 to Count-1 Do
                          With tControl(Objects[Control]) Do
                          Begin
                               Top := OriginY;
                               Inc ( OriginY, DeltaY + Height );
                          End;
                  End
                  Else
                  Begin
                       DeltaY := ( tControl(Objects[0]).Parent.ClientHeight -
                                   Selection.Bottom + Selection.Top -1 ) Div 2 -
                                  tControl(Objects[0]).Top;
                       For Control := 0 to Count-1 Do
                       With tControl(Objects[Control]) Do
                            Top := Top + DeltaY;
                  End;

               Finally
                      Free;
               End;
          End
          Else
          Begin
               Anchor := tControl(EditControls.Objects[0]);
               For Control := 0 to EditControls.Count - 1 Do
               With tControl(EditControls.Objects[Control]) Do
               Begin
                    Y := -1;
                    Case YAlign Of
                         eaAlignTops    : Top := Anchor.Top;
                         eaAlignCenters : Top := Anchor.Top + Anchor.Height Div 2 - Height Div 2;
                         eaAlignRights  : Top := Anchor.Top + Anchor.Height - Height;
                    End;
               End;
          End;

          fEditForm.Update;

          { reset the selection window }
          ResetSelection;
          HandleOperation ( hoMove );
          HandleOperation ( hoShow );

     End;
End;

Procedure TFormEditor.SizeControls ( XType, YType, XSize, YSize : Integer );
Var
   NewWidth, NewHeight : Integer;
   SetWidth, SetHeight : Integer;
   Control             : Integer;
Begin
     If ( fEditForm <> Nil ) And
        ( ( EditControls.Count > 1 ) Or
          ( ( EditControls.Count = 1 ) And
            ( ( XType = eaSizeSet ) Or
              ( YType = eaSizeSet )
             )
           )
         ) Then
     Begin
          NewWidth  := -1;
          NewHeight := -1;

          If ( EditControls.Count > 1 ) And
             ( XType in [eaSizeShrink,eaSizeGrow] ) Then
          Begin
               NewWidth := tControl(EditControls.Objects[0]).Width;
               For Control := 1 to EditControls.Count-1 Do
               With tControl(EditControls.Objects[Control]) Do
                   If ( ( XType = eaSizeShrink ) And ( NewWidth > Width ) ) Or
                      ( ( XType = eaSizeGrow   ) And ( NewWidth < Width ) ) Then
                      NewWidth := Width;
          End
          Else If XType = eaSizeSet Then
               NewWidth := XSize;

          If ( EditControls.Count > 1 ) And
             ( YType in [eaSizeShrink,eaSizeGrow] ) Then
          Begin
               NewHeight := tControl(EditControls.Objects[0]).Height;
               For Control := 1 to EditControls.Count-1 Do
               With tControl(EditControls.Objects[Control]) Do
                   If ( ( YType = eaSizeShrink ) And ( NewHeight > Height ) ) Or
                      ( ( YType = eaSizeGrow   ) And ( NewHeight < Height ) ) Then
                      NewHeight := Height;
          End
          Else If YType = eaSizeSet Then
               NewHeight := YSize;

          If ( NewWidth > -1 ) Or ( NewHeight > -1 ) Then
          Begin
               HandleOperation ( hoHide );

               For Control := 0 to EditControls.Count-1 Do
               With tControl(EditControls.Objects[Control]) Do
               Begin
                    If NewWidth = -1
                       Then SetWidth := Width
                       Else SetWidth := NewWidth;

                    If NewHeight = -1
                       Then SetHeight := Height
                       Else SetHeight := NewHeight;

                    SetBounds ( Left, Top, SetWidth, SetHeight );
                    Update;
               End;

               ResetSelection;
               HandleOperation ( hoMove );
               HandleOperation ( hoShow );
          End;
     End;
End;


Procedure TFormEditor.SetPopupState ( Sender : tObject );
Begin
     PopupMenu.Items[0].Enabled := EditControls.Count > 0;
     PopupMenu.Items[1].Enabled := EditControls.Count > 0;
     PopupMenu.Items[2].Enabled := EditControls.Count > 0;
     PopupMenu.Items[4].Enabled := EditControls.Count > 0;
     PopupMenu.Items[5].Enabled := EditControls.Count > 0;
End;

Procedure TFormEditor.PopupActions ( Sender : tObject );
Begin
     If tFormEditorAction ( tMenuItem(Sender).Tag ) = eaExitEditor
        Then SetEditForm ( Nil )
        Else PerformAction ( tFormEditorAction ( tMenuItem(Sender).Tag ) );
End;





Procedure EditForm ( aForm : tForm );
Begin
     FormEditor.EditForm := aForm;
End;

Procedure PerformAction ( Action : tFormEditorAction );
Begin
     If FormEditor.EditForm <> Nil Then
        Case Action Of
             eaBringToFront  : FormEditor.BringToFront;
             eaSendToBack    : FormEditor.SendToBack;
             eaAlignToGrid   : FormEditor.AlignToGrid;
             eaAlignControls : If FormEditor.EditControls.Count > 0 Then
                               With tAlignmentForm.Create ( FormEditor.fEditForm ) Do
                               Try
                                  If ShowModal = idOK
                                     Then FormEditor.AlignControls ( HorizontalAlignment.ItemIndex,
                                                                     VerticalAlignment.ItemIndex );
                               Finally
                                      Free;
                               End;

             eaSizeControls : If FormEditor.EditControls.Count > 0 Then
                              With tSizeForm.Create ( FormEditor.fEditForm ) Do
                              Try
                                 If ShowModal = idOK Then
                                 Begin
                                      If SizeWidth.Text = ''
                                         Then SizeWidth.Text := '0';
                                      If SizeHeight.Text = ''
                                         Then SizeHeight.Text := '0';

                                      FormEditor.SizeControls ( HorizontalSize.ItemIndex,
                                                                VerticalSize.ItemIndex,
                                                                StrToInt ( SizeWidth.Text ),
                                                                StrToInt ( SizeHeight.Text ) );
                                 End
                              Finally
                                     Free;
                              End;
             eaGridOptions : With tGridOptions.Create ( FormEditor.fEditForm ) Do
                             Try
                                XStep.Text := IntToStr ( XGridStep );
                                YStep.Text := IntToStr ( YGridStep );
                                GridMarker.ItemIndex := Ord(GridType);
                                VisibleGrid.Checked := ShowGrid;

                                If ShowModal = idOK Then
                                Begin
                                     XGridStep := StrToInt ( XStep.Text );
                                     YGridStep := StrToInt ( YStep.Text );
                                     GridType  := tGridType ( GridMarker.ItemIndex );
                                     ShowGrid  := VisibleGrid.Checked;
                                     InvalidateRect ( FormEditor.EditForm.Handle, Nil, True );
                                End;
                             Finally
                                    Free;
                             End;
        End;
End;


Procedure SetGridResolution ( XResolution, YResolution : Byte );
Begin
     XGridStep := XResolution;
     YGridStep := YResolution;
End;

Procedure ResetGrid;
Begin
     If ShowGrid And ( FormEditor.EditForm <> Nil ) Then
        InvalidateRect ( FormEditor.EditForm.Handle, Nil, True );
End;

Procedure Cleanup; Far;
Begin
     FormEditor.Free;
End;

Procedure Initialize;
Begin
     FormEditor := tFormEditor.Create;
     SetGridResolution ( 8, 8 );
     GridType := gtSmallDots;
     AddExitProc ( Cleanup );
End;

Begin
     Initialize;
end.
