{                                        }
{   John le Roux: john@majuba.ofs.gov.za }
{                                        }
{   P.O. Box 34292                       }
{   Faunusig                             }
{   Bloemfontein                         }
{   9300                                 }
{   Republic of South Africe             }
{                                        }

unit nstring;

interface

uses DB;

{                                                                    }
{ Define STRBOR to use null terminated strings - not yet implemented }
{                                                                    }

{$IFDEF STRBOR}
const
     StrStart = 0;
type
    TString = PChar;
{$ELSE}
const
     StrStart = 1;
type
     TString = String;
{$ENDIF}

{                                 }
{ general purpose string routines }
{                                 }

function ltrimCh(S: TString; Ch: Char): TString;
         { return a string with leading ch removed }

function ltrim(S: TString): TString;
         { return a string with leading spaces removed }

function rtrimCh(S: TString; Ch:Char): TString;
         { return a string with trailing ch removed }

function rtrim(S: TString): TString;
         { return a string with leading spaces removed }

function trimCh(S: TString; Ch: Char): TString;
         { return a string with leading and trailing ch removed }

function trim(S: TString): TString;
         { return a string with leading and trailing spaces removed }

function rpadCh(S: TString; Ch: Char; Len: Word): TString;
         { return a string right padded to length len with ch }

function rpad(S: TString; Len: Word): TString;
         { return a string right padded to length len with spaces }

function lpadCh(S: TString; Ch: Char; Len: Word): TString;
         { return a string left padded to length len with ch }

function lpad(S: TString; Len: Word): TString;
         { return a string left padded to length len with spaces }

function centerCh(S: TString; Ch: Char; Width: Word): TString;
         { return a string centered with ch with width }

function center(S: TString; Width: Word): TString;
         { return a string centered in spaces with width }

function emptyCh(S: TString; Ch: Char): Boolean;
         { return true if only ch in string or null else false }

function empty(S: TString): Boolean;
         { return true if only space in string or null else false }

function replicate(Ch: Char; Len: Word): TString;
         { return a string filled with ch to len }

function WhiteSpaceCh(S: TString; Ch: Char): Word;
         { return number of Ch to left of S }

function WhiteSpace(S: TString): Word;
         { return number of spaces to left of S }

function AmpersandToTilde(S: TString): TString;
         { replace first &[ch] with ~[ch]~ }

{                             }
{ word/substring manupilation }
{                             }

type
    CharSet = set of char;

function WordCount(S: TString; Delims: CharSet): Byte;
         { return number of words in s delimited by delims }

function WordPosition(N: Byte; S: TString; Delims: CharSet): Byte;
         { return position to n th word in s delimited by delims }

function ExtractWord(N: Byte; S: TString; Delims: CharSet): TString;
         { return the n th word in s delimited by delims }

{                           }
{ decimal number conversion }
{                           }

function atoi(S: TString): Integer;
         { return integer value of string S }

function itoa(I: Integer): TString;
         { return string of integer I }

function atol(S: TString): longint;
         { return longint value of string S }

function ltoa(L: longint): TString;
         { return string of longint L }

function ator(S: TString): Real;
         { return real value of string S }

function rtoa(R: Real; W, D: Integer): TString;
         { return string of real R }

function atow(S: TString): Word;
         { return word value of string S }

function wtoa(W: Word): TString;
         { return string of word W }

{                                       }
{ SQL generate used by Query-by-example }
{                                       }

function GenerateSQL(const Source: String; const ColumnName: String;
                     const ColumnType: TFieldType; var SQL: String): Boolean;

implementation

uses SysUtils;

{                                 }
{ general purpose string routines }
{                                 }

function ltrimCh(S: TString; Ch: Char): TString;
         { return a string with leading ch removed }
var
   I: Word;
begin
     I:=StrStart;
     while (S[I] = Ch) and (I <= length(S)) do inc(I);
     ltrimCh:=Copy(S, I, length(S) - I + 1)
end;

function ltrim(S: TString): TString;
         { return a string with leading spaces removed }
begin
     ltrim:=ltrimCh(S, ' ')
end;

function rtrimCh(S: TString; Ch:Char): TString;
         { return a string with trailing ch removed }
var
   I: Word;
begin
     I:=length(S);
     while (S[I] =Ch) and (I > 0) do dec(I);
     S[0]:=chr(I);
     rtrimCh:=S
end;

function rtrim(S: TString): TString;
         { return a string with leading spaces removed }
begin
     rtrim:=rtrimCh(S, ' ')
end;

function trimCh(S: TString; Ch: Char): TString;
         { return a string with leading and trailing ch removed }
begin
     trimCh:=ltrimCh(rtrimCh(S, Ch), Ch)
end;

function trim(S: TString): TString;
         { return a string with leading and trailing spaces removed }
begin
     trim:=rtrim(ltrim(S))
end;

function rpadCh(S: TString; Ch: Char; Len: Word): TString;
         { return a string right padded to length len with ch }
var
   T: TString;
begin
     if length(S) >=Len then
        rpadCh:=S
     else begin
          FillChar(T, Len + 1, Ch);
          T[0]:=chr(Len);
          rpadCh:=Copy(S + T, 1, Len)
     end
end;

function rpad(S: TString; Len: Word): TString;
         { return a string right padded to length len with spaces }
begin
     rpad:=rpadCh(S, ' ', Len)
end;

function lpadCh(S: TString; Ch: Char; Len: Word): TString;
         { return a string left padded to length len with ch }
var
   T: TString;
begin
     if length(S) >=Len then
        lpadCh:=S
     else begin
          FillChar(T, Len + 1, Ch);
          T[0]:=chr(Len);
          lpadCh:=Copy(T, 1, Len - length(S)) + S
     end
end;

function lpad(S: TString; Len: Word): TString;
         { return a string left padded to length len with spaces }
begin
     lpad:=lpadCh(S, ' ', Len)
end;

function centerCh(S: TString; Ch: Char; Width: Word): TString;
         { return a string centered in ch with width }
var
   I: Word;
begin
     S:=trim(S);
     if length(S) > Width then
        centerCh:=S
     else begin
          I:=(Width - length(S)) div 2;
          centerCh:=replicate(Ch, I) + S + replicate(Ch, Width - (I + length(S)))
     end
end;

function center(S: TString; Width: Word): TString;
         { return a string centered in spaces with width }
begin
     center:=centerCh(S, ' ', Width)
end;

function emptyCh(S: TString; Ch: Char): Boolean;
         { return true if only ch in string else false }
var
   I: Word;
begin
     emptyCh:=length(trimCh(S, Ch)) = 0
end;

function empty(S: TString): Boolean;
         { return true if only spaces in string else false }
begin
     empty:=emptyCh(S, ' ')
end;

function replicate(Ch: Char; Len: Word): TString;
         { return a string filled with ch to len }
var
   S: TString;
begin
     fillchar(S, Len + 1, Ch);
     S[0]:=chr(Len);
     replicate:=S
end;

function WhiteSpaceCh(S: TString; Ch: Char): Word;
         { return number of Ch to left of S }
var
   I, N: Word;
begin
     I:=StrStart;
     N:=0;
     while S[I] = Ch do
     begin
          inc(N);
          inc(I)
     end;
     WhiteSpaceCh:=N
end;

function WhiteSpace(S: TString): Word;
         { return number of spaces to left of S }
begin
     WhiteSpace:=WhiteSpaceCh(S, ' ')
end;

function AmpersandToTilde(S: TString): TString;
         { replace first &[ch] with ~[ch]~ }
var
   P: Word;
begin
     AmpersandToTilde:=S;
     P:=Pos('&', S);
     if P = 0 then Exit;
     S[P]:='~';
     Insert('~', S, P + 2);
     AmpersandToTilde:=S
end;

{                             }
{ word/substring manupilation }
{                             }

function WordCount(S: TString; Delims: CharSet): Byte;
         { return number of words in s delimited by delims }
var
    Count: Byte;
    I: Word;
    Len: Word;
begin
     Count:=0;
     Len:=length(S);
     I:=StrStart;
     while I <=Len do
     begin
          while (I <=Len) and (S[I] in Delims) do Inc(I);
          if I <=Len then Inc(Count);
          while (I <=Len) and not(S[I] in Delims) do Inc(I)
     end;
     WordCount:=Count;
end;

function WordPosition(N: Byte; S: TString; Delims: CharSet): Byte;
         { return position to n th word in s delimited by delims }
var
   Count: Byte;
   I: Word;
   Len: Word;
begin
     Count:=0;
     Len:=length(S);
     I:=StrStart;
     WordPosition:=0;

     while (I <=Len) and (Count <> N) do
     begin
          while (I <=Len) and (S[I] in Delims) do Inc(I);
          if I <=Len then Inc(Count);
          if Count <> N then
             while (I <=Len) and not(S[I] in Delims) do Inc(I)
          else
              WordPosition:=I
     end
end;

function ExtractWord(N: Byte; S: TString; Delims: CharSet): TString;
         { return the n th word in s delimited by delims }
var
   I, Len: Byte;
   SLen: Word;
begin
     Len:=StrStart;
     SLen:=length(S);
     I:=WordPosition(N, S, Delims);
     if I <> 0 then
        while (I <=SLen) and not(S[I] in Delims) do
        begin
             ExtractWord[Len]:=S[I];
             inc(Len);
             Inc(I)
        end;
     dec(Len);
     ExtractWord[0]:=Char(Len)
end;
{                           }
{ decimal number conversion }
{                           }

function atoi(S: TString): Integer;
         { return integer value of string S }
var
   I, Er: Integer;
begin
     val(trim(S), I, Er);
     if Er <> 0 then I:=0;
     atoi:=I
end;

function itoa(I: Integer): TString;
         { return string of integer I }
var
   S: TString;
begin
     Str(I, S);
     itoa:=S
end;

function atol(S: TString): longint;
         { return longint value of string S }
var
   L: longint;
   Er: Integer;
begin
     val(trim(S), L, Er);
     if Er <> 0 then L:=0;
     atol:=L
end;

function ltoa(L: longint): TString;
         { return string of longint L }
var
   S: TString;
begin
     Str(L, S);
     ltoa:=S
end;

function ator(S: TString): Real;
         { return real value of string S }
var
   R: Real;
   Er: integer;
begin
     val(trim(S), R, Er);
     if Er <> 0 then R:=0;
     ator:=R
end;

function rtoa(R: Real; W, D: Integer): TString;
         { return string of real R }
var
   S: TString;
begin
     str(R:W:D, S);
     rtoa:=S
end;

function atow(S: TString): Word;
         { return word value of string S }
var
   I: Word;
   Er: Integer;
begin
     val(trim(S), I, Er);
     if Er <> 0 then I:=0;
     atow:=I
end;

function wtoa(W: Word): TString;
         { return string of word W }
var
   S: TString;
begin
     Str(W, S);
     wtoa:=S
end;

{                                       }
{ SQL generate used by Query-by-example }
{                                       }

type
    TCritTokens = set of Char;
const
     CritTokens: TCritTokens  = ['=', '<', '>', '~', '|'];
var
   ParsePos: Byte;

function TokenizeSQL(Source: String; var Criteria: String; var Value: String): Boolean;
begin
     Criteria:='';
     Value:='';

     { delete white space }
     while (ParsePos <= length(Source)) and (Source[ParsePos] = ' ') do inc(ParsePos);

     if (ParsePos > length(Source)) then
     begin
          Result:=False;
          Exit
     end;

     { get criteria }
     if Source[ParsePos] in CritTokens then
     begin
          while (ParsePos < length(Source)) and (Source[ParsePos] in CritTokens) do
          begin
               Criteria:=Criteria + Source[ParsePos];
               inc(ParsePos)
          end
     end;

     { delete white space }
     while (ParsePos <= length(Source)) and (Source[ParsePos] = ' ') do inc(ParsePos);

     { get value }
     while (ParsePos <= length(Source)) and not (Source[ParsePos] in CritTokens) do
     begin
          Value:=Value + Source[ParsePos];
          inc(ParsePos)
     end
end;

function GenerateSQL(const Source: String; const ColumnName: String;
                     const ColumnType: TFieldType; var SQL: String): Boolean;
var
   Criteria, Value, PrevValue: String;
   IntValue: Longint;
   FloatValue: Extended;
   DateValue: TDateTime;
begin
     SQL:='';
     Result:=True;
     ParsePos:=1;
     while TokenizeSQL(Source, Criteria, Value) do
     begin
          { if empty Criteria then assume = }
          if empty(Criteria) then Criteria := '=';

          { test for validity of Criteria }
          if (Criteria = '=')  or
             (Criteria = '>')  or
             (Criteria = '<')  or
             (Criteria = '>=') or
             (Criteria = '=>') or
             (Criteria = '<=') or
             (Criteria = '=<') or
             (Criteria = '<>') or
             (Criteria = '|')  or
             (Criteria = '&') then
          else begin
               Result:=False;
               break { Error }
          end;

          { test for '|' and '~' as not the first operand }
          if ((Criteria = '|') or (Criteria = '~')) and empty(SQL) then
          begin
               Result:=False;
               break { Error }
          end;

          { test for empty Value - if yes Criteria must be = }
          { Only if SQL is also empty is it Ok }
          if empty(Value) then
          begin
               if (Criteria = '=') and empty(SQL) then
               begin
                    SQL:=trim(ColumnName) + ' is NULL';
                    break { Ok }
               end
               else begin
                    Result:=False;
                    break  { Error }
               end
          end;

          { test for validity of Value }
          case ColumnType of
               ftSmallint,
               ftInteger,
               ftWord: begin
                            try
                               IntValue:=StrToInt(Value)
                            except
                                  Result:=False;
                                  break { Error }
                            end
                       end;
               ftFloat,
               ftBCD: begin
                           try
                              FloatValue:=StrToFloat(Value)
                           except
                                 Result:=False;
                                 break { Error }
                           end
                       end;
               ftDate: begin
                         try
                            DateValue:=StrToDate(Value);
                         except
                               Result:=False;
                               break { Error }
                         end
                    end;
               ftString: begin
                              Value:='"' + Value + '"';
                              if (pos('%', Value) > 0) or
                                 (pos('_', Value) > 0) then
                                    Criteria:='LIKE'
                         end;

          else { assume char }
              Value:= '"' + trim(Value) + '"'; { make char value quoted }
          end;

          { Now build the SQL }
          Value:=trim(Value);
          if Criteria = '|' then
             SQL:=SQL + ' OR ' + ColumnName + ' = ' + Value
          else
              if Criteria = '~' then
                 SQL:=ColumnName + ' >= ' + PrevValue + ' AND ' + ColumnName + ' <= ' + Value
              else
                  if empty(SQL) then
                     SQL:=ColumnName + ' ' + Criteria + ' ' + Value
                  else
                      SQL:=SQL + ' AND ' + ColumnName + ' ' + Criteria + ' ' + Value;
          PrevValue:=Value;
     end;
     if not Result then
        SQL:=''
     else
         if not empty(SQL) then SQL:='(' + SQL + ')'
end;

end.
