{ version 0.1 }

{*******************************************************}
{                                                       }
{       Delphi Visual Component Library                 }
{                                                       }
{       Copyright (c) 1995 Borland International        }
{                                                       }
{*******************************************************}

unit QBEnav;

interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, Forms,
  Graphics, Menus, StdCtrls, ExtCtrls, DB, DBTables, Mask, Buttons,
  InMem;


const
  InitRepeatPause = 400;  { pause before repeat timer (ms) }
  RepeatPause     = 100;  { pause before hint window displays (ms)}
  SpaceSize       =  5;   { size of space between special buttons }

type
  TNavButton = class;
  TQBEDataLink = class;

  TNavGlyph = (ngEnabled, ngDisabled);

  TNavigateBtn = (nbQuery, nbFirst, nbPrior, nbNext, nbLast,
                  nbInsert, nbDelete, nbEdit, nbPost, nbCancel, nbRefresh);
  TButtonSet = set of TNavigateBtn;
  TNavButtonStyle = set of (nsAllowTimer, nsFocusRect);

  ENavClick = procedure (Sender: TObject; Button: TNavigateBtn) of object;

{ TDBQBENav }

  TDBQBENav = class (TCustomPanel)
  private
    FQBETableName: String;
    QBETable: TInMemoryTable;
    QBESaveDataSet: TDataSet;
    dsQuery: Boolean;
    FDataLink: TQBEDataLink;
    FVisibleButtons: TButtonSet;
    FHints: TStrings;
    ButtonWidth: Integer;
    MinBtnSize: TPoint;
    FOnNavClick: ENavClick;
    FocusedButton: TNavigateBtn;
    FConfirmDelete: Boolean;
    function GetDataSource: TDataSource;
    procedure SetDataSource(Value: TDataSource);
    procedure InitButtons;
    procedure InitHints;
    procedure Click(Sender: TObject);
    procedure BtnMouseDown (Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetVisible(Value: TButtonSet);
    procedure AdjustSize (var W: Integer; var H: Integer);
    procedure SetHints(Value: TStrings);
    procedure WMSize(var Message: TWMSize);  message WM_SIZE;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure CMEnabledChanged(var Message: TMessage); message CM_ENABLEDCHANGED;
    procedure QueryActivate;
    procedure QueryCancel;
    procedure QueryAccepted;
    function  QBEColumnValue(FieldName: String): String;
  protected
    Buttons: array[TNavigateBtn] of TNavButton;
    procedure DataChanged;
    procedure EditingChanged;
    procedure ActiveChanged;
    procedure Loaded; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure BtnClick(Index: TNavigateBtn);
  published
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property QBETableName: string read FQBETableName write FQBETableName;
    property VisibleButtons: TButtonSet read FVisibleButtons write SetVisible
      default [nbQuery, nbFirst, nbPrior, nbNext, nbLast, nbInsert, nbDelete,
        nbEdit, nbPost, nbCancel, nbRefresh];
    property Align;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Ctl3D;
    property Hints: TStrings read FHints write SetHints;
    property ParentCtl3D;
    property ParentShowHint;
    property PopupMenu;
    property ConfirmDelete: Boolean read FConfirmDelete write FConfirmDelete default True;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick: ENavClick read FOnNavClick write FOnNavClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnResize;
  end;

{ TNavButton }

  TNavButton = class(TSpeedButton)
  private
    FIndex: TNavigateBtn;
    FNavStyle: TNavButtonStyle;
    FRepeatTimer: TTimer;
    procedure TimerExpired(Sender: TObject);
  protected
    procedure Paint; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
      X, Y: Integer); override;
  public
    destructor Destroy; override;
    property NavStyle: TNavButtonStyle read FNavStyle write FNavStyle;
    property Index : TNavigateBtn read FIndex write FIndex;
  end;

{ TQBEDataLink }

  TQBEDataLink = class(TDataLink)
  private
    FNavigator: TDBQBENav;
  protected
    procedure EditingChanged; override;
    procedure DataSetChanged; override;
    procedure ActiveChanged; override;
  public
    constructor Create(ANav: TDBQBENav);
    destructor Destroy; override;
  end;

procedure Register;

implementation

uses DBIErrs, DBITypes, Clipbrd, DBConsts, Dialogs,
     DBCtrls, Nstring;

{$R QBENav}

{ TDBQBENav }

procedure Register;
begin
  RegisterComponents('Samples', [TDBQBENav]);
end;

const
  BtnStateName: array[TNavGlyph] of PChar = ('EN', 'DI');

  BtnTypeName: array[TNavigateBtn] of PChar = ('QUERY', 'FIRST', 'PRIOR', 'NEXT',
    'LAST', 'INSERT', 'DELETE', 'EDIT', 'POST', 'CANCEL', 'REFRESH');

  BtnHintId: array[TNavigateBtn] of Word = (SFirstRecord, SPriorRecord,
    SNextRecord, SLastRecord, SInsertRecord, SDeleteRecord, SEditRecord,
    SPostEdit, SCancelEdit, SRefreshRecord, 11);

constructor TDBQBENav.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle - [csAcceptsControls, csSetCaption] +
    [csFramed, csOpaque];
  FDataLink := TQBEDataLink.Create(Self);
  FVisibleButtons := [nbQuery, nbFirst, nbPrior, nbNext, nbLast, nbInsert,
    nbDelete, nbEdit, nbPost, nbCancel, nbRefresh];
  FHints := TStringList.Create;
  FHints.Add('Query');
  FHints.Add('First');
  FHints.Add('Prior');
  FHints.Add('Next');
  FHints.Add('Last');
  FHints.Add('Insert');
  FHints.Add('Delete');
  FHints.Add('Edit');
  FHints.Add('Post');
  FHints.Add('Cancel');
  FHints.Add('Refresh');
  ShowHint:=True;
  InitButtons;
  BevelOuter := bvNone;
  BevelInner := bvNone;
  Width := 241;
  Height := 25;
  ButtonWidth := 0;
  FocusedButton := nbFirst;
  FConfirmDelete := True;
end;

destructor TDBQBENav.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
  inherited Destroy;
end;

procedure TDBQBENav.InitButtons;
var
  I: TNavigateBtn;
  Btn: TNavButton;
  X: Integer;
  ResName: array[0..40] of Char;
begin
  MinBtnSize := Point(20, 18);
  X := 0;
  for I := Low(Buttons) to High(Buttons) do
  begin
    Btn := TNavButton.Create (Self);
    Btn.Index := I;
    Btn.Visible := I in FVisibleButtons;
    Btn.Enabled := True;
    Btn.SetBounds (X, 0, MinBtnSize.X, MinBtnSize.Y);
    Btn.Glyph.Handle := LoadBitmap(HInstance,
        StrFmt(ResName, 'QBE_%s', [BtnTypeName[I]]));
    if I = nbQuery then
       Btn.NumGlyphs:=1
    else
        Btn.NumGlyphs := 2;
    Btn.OnClick := Click;
    Btn.OnMouseDown := BtnMouseDown;
    Btn.Parent := Self;
    Buttons[I] := Btn;
    X := X + MinBtnSize.X;
  end;
  InitHints;
  Buttons[nbPrior].NavStyle := Buttons[nbPrior].NavStyle + [nsAllowTimer];
  Buttons[nbNext].NavStyle  := Buttons[nbNext].NavStyle + [nsAllowTimer];
end;

procedure TDBQBENav.InitHints;
var
  I: Integer;
  J: TNavigateBtn;
begin
  for J := Low(Buttons) to High(Buttons) do
    Buttons[J].Hint := LoadStr (BtnHintId[J]);
  J := Low(Buttons);
  for I := 0 to (FHints.Count - 1) do
  begin
    if FHints.Strings[I] <> '' then Buttons[J].Hint := FHints.Strings[I];
    if J = High(Buttons) then Exit;
    Inc(J);
  end;
end;

procedure TDBQBENav.SetHints(Value: TStrings);
begin
  FHints.Assign(Value);
  InitHints;
end;

procedure TDBQBENav.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure TDBQBENav.SetVisible(Value: TButtonSet);
var
  I: TNavigateBtn;
  W, H: Integer;
begin
  W := Width;
  H := Height;
  FVisibleButtons := Value;
  for I := Low(Buttons) to High(Buttons) do
    Buttons[I].Visible := I in FVisibleButtons;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  Invalidate;
end;

procedure TDBQBENav.AdjustSize (var W: Integer; var H: Integer);
var
  Count: Integer;
  MinW: Integer;
  I: TNavigateBtn;
  LastBtn: TNavigateBtn;
  Space, Temp, Remain: Integer;
  X: Integer;
begin
  if (csLoading in ComponentState) then Exit;
  if Buttons[nbFirst] = nil then Exit;

  Count := 0;
  LastBtn := High(Buttons);
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Inc(Count);
      LastBtn := I;
    end;
  end;
  if Count = 0 then Inc(Count);

  MinW := Count * (MinBtnSize.X - 1) + 1;
  if W < MinW then
    W := MinW;
  if H < MinBtnSize.Y then
    H := MinBtnSize.Y;

  ButtonWidth := ((W - 1) div Count) + 1;
  Temp := Count * (ButtonWidth - 1) + 1;
  if Align = alNone then
    W := Temp;

  X := 0;
  Remain := W - Temp;
  Temp := Count div 2;
  for I := Low(Buttons) to High(Buttons) do
  begin
    if Buttons[I].Visible then
    begin
      Space := 0;
      if Remain <> 0 then
      begin
        Dec (Temp, Remain);
        if Temp < 0 then
        begin
          Inc (Temp, Count);
          Space := 1;
        end;
      end;
      Buttons[I].SetBounds (X, 0, ButtonWidth + Space, Height);
      Inc (X, ButtonWidth - 1 + Space);
      LastBtn := I;
    end
    else
      Buttons[I].SetBounds (Width + 1, 0, ButtonWidth, Height);
  end;
end;

procedure TDBQBENav.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  W, H: Integer;
begin
  W := AWidth;
  H := AHeight;
  AdjustSize (W, H);
  inherited SetBounds (ALeft, ATop, W, H);
end;

procedure TDBQBENav.WMSize(var Message: TWMSize);
var
  W, H: Integer;
begin
  inherited;

  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds(Left, Top, W, H);
  Message.Result := 0;
end;

procedure TDBQBENav.Click(Sender: TObject);
begin
  BtnClick (TNavButton (Sender).Index);
end;

procedure TDBQBENav.BtnMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  OldFocus: TNavigateBtn;
  Form: TForm;
begin
  OldFocus := FocusedButton;
  FocusedButton := TNavButton (Sender).Index;
  if TabStop and (GetFocus <> Handle) and CanFocus then
  begin
    SetFocus;
    if (GetFocus <> Handle) then
      Exit;
  end
  else if TabStop and (GetFocus = Handle) and (OldFocus <> FocusedButton) then
  begin
    Buttons[OldFocus].Invalidate;
    Buttons[FocusedButton].Invalidate;
  end;
end;

procedure TDBQBENav.BtnClick(Index: TNavigateBtn);
begin
  if (DataSource <> nil) and (DataSource.State <> dsInactive) then
  begin
       case Index of
            nbQuery: QueryActivate;
            nbPrior: DataSource.DataSet.Prior;
            nbNext: DataSource.DataSet.Next;
            nbFirst: DataSource.DataSet.First;
            nbLast: DataSource.DataSet.Last;
            nbInsert: DataSource.DataSet.Insert;
            nbEdit: DataSource.DataSet.Edit;
            nbCancel: begin
                           DataSource.DataSet.Cancel;
                           if dsQuery then QueryCancel;
                      end;
            nbPost: if not dsQuery then
                       DataSource.DataSet.Post
                    else
                        QueryAccepted;
            nbRefresh: if FDataLink.DataSet is TTable then
                          DataSource.DataSet.Refresh
                       else begin
                            DataSource.DataSet.Close;
                            DataSource.DataSet.Open
                       end;
            nbDelete: begin
                           if not FConfirmDelete or
                              (MessageDlg (LoadStr(SDeleteRecordQuestion), mtConfirmation, mbOKCancel, 0)
                              <> idCancel) then
                              DataSource.DataSet.Delete;
                      end;
       end;
  end;
  if not (csDesigning in ComponentState) and Assigned(FOnNavClick) then
    FOnNavClick(Self, Index);
end;

procedure TDBQBENav.WMSetFocus(var Message: TWMSetFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TDBQBENav.WMKillFocus(var Message: TWMKillFocus);
begin
  Buttons[FocusedButton].Invalidate;
end;

procedure TDBQBENav.KeyDown(var Key: Word; Shift: TShiftState);
var
  NewFocus: TNavigateBtn;
  OldFocus: TNavigateBtn;
begin
  OldFocus := FocusedButton;
  case Key of
    VK_RIGHT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus < High(Buttons) then
            NewFocus := Succ(NewFocus);
        until (NewFocus = High(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_LEFT:
      begin
        NewFocus := FocusedButton;
        repeat
          if NewFocus > Low(Buttons) then
            NewFocus := Pred(NewFocus);
        until (NewFocus = Low(Buttons)) or (Buttons[NewFocus].Visible);
        if NewFocus <> FocusedButton then
        begin
          FocusedButton := NewFocus;
          Buttons[OldFocus].Invalidate;
          Buttons[FocusedButton].Invalidate;
        end;
      end;
    VK_SPACE:
      begin
        if Buttons[FocusedButton].Enabled then
          Buttons[FocusedButton].Click;
      end;
  end;
end;

procedure TDBQBENav.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS;
end;

procedure TDBQBENav.DataChanged;
var
  UpEnable, DnEnable: Boolean;
begin
  UpEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.BOF;
  DnEnable := Enabled and FDataLink.Active and not FDataLink.DataSet.EOF;
  if dsQuery then
     Buttons[nbQuery].Enabled:=False
  else
      Buttons[nbQuery].Enabled:=not FDataLink.Editing and not dsQuery;
  Buttons[nbFirst].Enabled := UpEnable and not FDataLink.Editing;
  Buttons[nbPrior].Enabled := UpEnable and not FDataLink.Editing;
  Buttons[nbNext].Enabled := DnEnable and not FDataLink.Editing;
  Buttons[nbLast].Enabled := DnEnable and not FDataLink.Editing;
  Buttons[nbDelete].Enabled := Enabled and FDataLink.Active and
    FDataLink.DataSet.CanModify and
    not (FDataLink.DataSet.BOF and FDataLink.DataSet.EOF) and
    not FDataLink.Editing;
 end;

procedure TDBQBENav.EditingChanged;
var
  CanModify: Boolean;
begin
  CanModify := Enabled and FDataLink.Active and FDataLink.DataSet.CanModify;
  Buttons[nbInsert].Enabled := CanModify and not FDataLink.Editing;
  Buttons[nbEdit].Enabled := CanModify and not FDataLink.Editing;
  Buttons[nbPost].Enabled := CanModify and FDataLink.Editing;
  Buttons[nbCancel].Enabled := CanModify and FDataLink.Editing;
  Buttons[nbRefresh].Enabled := CanModify and not FDataLink.Editing;
  if dsQuery then
     Buttons[nbQuery].Enabled:=False
  else
      Buttons[nbQuery].Enabled:=not FDataLink.Editing and not dsQuery;
end;

procedure TDBQBENav.ActiveChanged;
var
  I: TNavigateBtn;
begin
  if not (Enabled and FDataLink.Active) then
    for I := Low(Buttons) to High(Buttons) do
      Buttons[I].Enabled := False
  else
  begin
    DataChanged;
    EditingChanged;
  end;
end;

procedure TDBQBENav.CMEnabledChanged(var Message: TMessage);
begin
  inherited;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

procedure TDBQBENav.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  if not (csLoading in ComponentState) then
    ActiveChanged;
end;

function TDBQBENav.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TDBQBENav.Loaded;
var
  W, H: Integer;
begin
  inherited Loaded;
  W := Width;
  H := Height;
  AdjustSize (W, H);
  if (W <> Width) or (H <> Height) then
    inherited SetBounds (Left, Top, W, H);
  InitHints;
  ActiveChanged;
end;

procedure TDBQBENav.QueryActivate;
var
   ColumnName: String;
   I: Integer;
begin
     try
        QBESaveDataSet:=DataSource.DataSet;
        QBETable:=TInMemoryTable.Create(Self);
        QBETable.DataBaseName:='';
        QBETable.TableName:=FQBETableName;
        with DataSource.DataSet do
        begin
             for I:=0 to FieldDefs.Count - 1 do
             begin
                  ColumnName:=FieldDefs.Items[I].Name;
                  QBETable.FieldDefs.Add(ColumnName, ftString, 100, False)
             end
        end;
        QBETable.CreateTable;
        QBETable.ReadOnly:=False;
        DataSource.DataSet:=QBETable;
        QBETable.Open;
        QBETable.Insert;
        dsQuery:=True;
     except
        dsQuery:=False;
        MessageDlg('Could not open Query Table', mtError, [mbOk], 0);
        try
           DataSource.DataSet:=QBESaveDataSet;
           DataSource.DataSet.Cancel;
        except
           MessageDlg('Big error with query active - call John', mtError, [mbOk], 0)
        end
     end
end;

procedure TDBQBENav.QueryCancel;
begin
     dsQuery:=False;
     try
        DataSource.DataSet:=QBESaveDataSet;
        DataSource.DataSet.Cancel;
        QBETable.Close;
        QBETable.Free;
     except
        MessageDlg('Big error with query cancel - call John', mtError, [mbOk], 0)
     end
end;

procedure TDBQBENav.QueryAccepted;
var
   SQLLines: TStringList;
   I: Integer;
   ColumnType: TFieldType;
   ColumnValue: String;
   QBELine: String;
begin
     try
        SQLLines:=TStringList.Create;
        for I:=0 to QBETable.FieldCount - 1 do
        begin
             with QBETable.Fields[I] as TStringField do
             begin
                  ColumnType:=QBESaveDataSet.FieldByName(FieldName).DataType;
                  ColumnValue:=QBEColumnValue(FieldName);
                  if not GenerateSQL(ColumnValue, FieldName, ColumnType, QBELine) then
                  begin
                       FocusControl;
                       SQLLines.Free;
                       raise Exception.Create('Error in field');
                  end
                  else
                      if not empty(QBELine) then
                      begin
                           if SQLLines.Count > 0 then SQLLines.Add('AND');
                           SQLLines.Add(QBELine)
                      end
             end
        end;
     finally
        I:=0
     end;

     with QBESaveDataSet as TQuery do
     begin
          SQL.Clear;
          SQL.Add('select * from ' + QBETableName);
          for I:=0 to SQLLines.Count - 1 do
              SQL.Add(SQLLines.Strings[I]);
          if SQL.Count > 1 then
             SQL.Insert(1, 'where')
     end;

     dsQuery:=False;
     try
        QBETable.Close;
        QBETable.Free;
        DataSource.DataSet:=QBESaveDataSet;
        DataSource.DataSet.Close;
        DataSource.DataSet.Open;
        DataSource.DataSet.Cancel;
        DataSource.DataSet.First
     except
        MessageDlg('Big error with query accept - call John', mtError, [mbOk], 0);
     end;
     SQLLines.Free
end;

{ there must be an easier and better way to do this }
{ reading value sometimes work and sometimes don't  }

function TDBQBENav.QBEColumnValue(FieldName: String): String;
var
   I: Integer;
begin
     Result:='';
     with Owner as TForm do
     begin
          for I:=0 to ComponentCount - 1 do
              if Components[I] is TDBEdit then
              begin
                   with Components[I] as TDBEdit do
                   begin
                        if DataField = FieldName then
                        begin
                             Result:=EditText;
                             break
                        end
                   end
              end
              else if (Components[I] is TDBListBox) then
              begin
                   with Components[I] as TDBListBox do
                   begin
                        if DataField = FieldName then
                        begin
                             try
                                Result:=Items.Strings[ItemIndex];
                             except
                                Result:=''
                             end;
                             break
                        end
                   end
              end
              else if Components[I] is TDBComboBox then
              begin
                   with Components[I] as TDBComboBox do
                   begin
                        if DataField = FieldName then
                        begin
                             try
                                Result:=Items.Strings[ItemIndex];
                             except
                                Result:=''
                             end;
                             break
                        end
                   end
              end
              else if Components[I] is TDBCheckBox then
              begin
                   with Components[I] as TDBCheckBox do
                   begin
                        if DataField = FieldName then
                        begin
                             case State of
                                  cbChecked: Result:=ValueChecked;
                                  cbUnchecked: Result:=ValueUnChecked;
                             end;
                             break
                        end
                   end
              end
              else if Components[I] is TDBRadioGroup then
              begin
                   with Components[I] as TDBRadioGroup do
                   begin
                        if DataField = FieldName then
                        begin
                             try
                                Result:=Values.Strings[ItemIndex];
                             except
                                Result:=''
                             end;
                             break
                        end
                   end
              end
     end
end;

{TNavButton}

destructor TNavButton.Destroy;
begin
  if FRepeatTimer <> nil then
    FRepeatTimer.Free;
  inherited Destroy;
end;

procedure TNavButton.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseDown (Button, Shift, X, Y);
  if nsAllowTimer in FNavStyle then
  begin
    if FRepeatTimer = nil then
      FRepeatTimer := TTimer.Create(Self);

    FRepeatTimer.OnTimer := TimerExpired;
    FRepeatTimer.Interval := InitRepeatPause;
    FRepeatTimer.Enabled  := True;
  end;
end;

procedure TNavButton.MouseUp(Button: TMouseButton; Shift: TShiftState;
                                  X, Y: Integer);
begin
  inherited MouseUp (Button, Shift, X, Y);
  if FRepeatTimer <> nil then
    FRepeatTimer.Enabled  := False;
end;

procedure TNavButton.TimerExpired(Sender: TObject);
begin
  FRepeatTimer.Interval := RepeatPause;
  if (FState = bsDown) and MouseCapture then
  begin
    try
      Click;
    except
      FRepeatTimer.Enabled := False;
      raise;
    end;
  end;
end;

procedure TNavButton.Paint;
var
  R: TRect;
begin
  inherited Paint;
  if (GetFocus = Parent.Handle) and
     (FIndex = TDBQBENav (Parent).FocusedButton) then
  begin
    R := Bounds(0, 0, Width, Height);
    InflateRect(R, -3, -3);
    if FState = bsDown then
      OffsetRect(R, 1, 1);
    DrawFocusRect(Canvas.Handle, R);
  end;
end;

{ TQBEDataLink }

constructor TQBEDataLink.Create(ANav: TDBQBENav);
begin
  inherited Create;
  FNavigator := ANav;
end;

destructor TQBEDataLink.Destroy;
begin
  FNavigator := nil;
  inherited Destroy;
end;

procedure TQBEDataLink.EditingChanged;
begin
  if FNavigator <> nil then FNavigator.EditingChanged;
end;

procedure TQBEDataLink.DataSetChanged;
begin
  if FNavigator <> nil then FNavigator.DataChanged;
end;

procedure TQBEDataLink.ActiveChanged;
begin
  if FNavigator <> nil then FNavigator.ActiveChanged;
end;

end.
