{==============================================================================}
{= RzStrEdt - String List Property Editor and Dialog                          =}
{=                                                                            =}
{= This unit file contains a replacement string list property editor and its  =}
{= associated dialog box.  After installing this unit into the Delphi IDE,    =}
{= this editor will be used to edit all string list properties.  To install   =}
{= this editor, follow the same rules used to install a custom component.     =}
{=                                                                            =}
{= Copyright  1995 by Raize Software Solutions, Inc.                         =}
{==============================================================================}

unit RzStrEdt;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, RzPanel, StdCtrls, Buttons, IniFiles, DsgnIntf,
  Menus, Spin;

type
  TRzStringListProperty = class( TPropertyEditor )
    function GetAttributes : TPropertyAttributes; override;
    function GetValue: string; override;
    procedure Edit; override;
  end;

  TRzStrListEditorDlg = class( TForm )
    BtnOk         : TBitBtn;
    BtnCancel     : TBitBtn;
    BtnHelp       : TBitBtn;
    PnlToolbar    : TPanel;
    BtnIndent: TSpeedButton;
    BtnOpen       : TSpeedButton;
    BtnSave       : TSpeedButton;
    BtnCut        : TSpeedButton;
    BtnCopy       : TSpeedButton;
    BtnPaste      : TSpeedButton;
    BtnUndo       : TSpeedButton;
    BtnFont       : TSpeedButton;
    PnlStringList : TPanel;
    EdtStrings    : TMemo;
    LblCount      : TLabel;
    Label1        : TLabel;
    LblLine       : TLabel;
    Label3        : TLabel;
    LblCol        : TLabel;
    MnuEdit       : TPopupMenu;
    MnuUndo       : TMenuItem;
    MnuCut        : TMenuItem;
    MnuCopy       : TMenuItem;
    MnuPaste      : TMenuItem;
    MnuSep1       : TMenuItem;
    MnuOpen       : TMenuItem;
    MnuSave       : TMenuItem;
    MnuSep2       : TMenuItem;
    MnuIndent: TMenuItem;
    DlgOpen       : TOpenDialog;
    DlgSave       : TSaveDialog;
    DlgFont       : TFontDialog;
    LblCopyright: TLabel;
    BtnCopyright: TButton;
    BtnOutdent: TSpeedButton;
    ChkDefault: TCheckBox;
    BtnTabSize: TSpeedButton;
    BtnSetTabSize: TSpeedButton;
    BtnCancelTabSize: TSpeedButton;
    SpnTabSize: TSpinEdit;
    MnuOutdent: TMenuItem;
    procedure FormCreate( Sender : TObject );
    procedure FormDestroy( Sender : TObject );
    procedure BtnOkClick( Sender : TObject );
    procedure BtnFontClick( Sender : TObject );
    procedure BtnUndoClick( Sender : TObject );
    procedure BtnCutClick( Sender : TObject );
    procedure BtnCopyClick( Sender : TObject );
    procedure BtnPasteClick( Sender : TObject );
    procedure BtnOpenClick( Sender : TObject );
    procedure BtnSaveClick( Sender : TObject );
    procedure BtnIndentClick( Sender : TObject );
    procedure EdtStringsChange( Sender : TObject );
    procedure EdtStringsKeyDown( Sender : TObject; var Key : Word;
                                 Shift : TShiftState );
    procedure EdtStringsKeyUp( Sender : TObject; var Key : Word;
                               Shift : TShiftState );
    procedure EdtStringsClick( Sender : TObject);
    procedure BtnCopyrightClick(Sender: TObject);
    procedure EdtStringsMouseUp( Sender: TObject; Button: TMouseButton;
                                 Shift: TShiftState; X, Y: Integer);
  procedure BtnHelpClick(Sender: TObject);
    procedure BtnTabSizeClick(Sender: TObject);
    procedure BtnOutdentClick(Sender: TObject);
    procedure BtnSetTabSizeClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    SingleLine    : string[ 15 ];
    MultipleLines : string[ 15 ];
    DelphiIni     : TIniFile;
    FTabSize      : Integer;
    FCurLine      : Integer;
    FCurCol       : Integer;
    procedure UpdateClipboardStatus;
    procedure IndentLine( LineNum : Integer );
    function OutdentLine( LineNum : Integer ) : Boolean;
    procedure IndentLines( Indent : Boolean );
    procedure SetTabSize;
    procedure EnableButtons( Enable : Boolean );
  end;


procedure Register;

implementation

{$R *.DFM}

uses
  LibConst, LibHelp, ClipBrd;

const
  Section = 'Raize.StrListEditor';

  fsBoldMask      = 8;              { Constants used to determine font style }
  fsItalicMask    = 4;
  fsUnderlineMask = 2;
  fsStrikeOutMask = 1;
  fsNormal        = 0;


{===================================}
{== TRzStringListProperty Methods ==}
{===================================}

function TRzStringListProperty.GetAttributes: TPropertyAttributes;
begin
  Result := inherited GetAttributes + [ paDialog ] - [ paSubProperties ];
end; {= TRzStringListProperty.GetAttributes =}


function TRzStringListProperty.GetValue : string;
begin
  FmtStr( Result, '(%s)', [ GetPropType^.Name ] );
end; {= TRzStringListProperty.GetValue =}


procedure TRzStringListProperty.Edit;
var
  Dlg : TRzStrListEditorDlg;
begin
  with TRzStrListEditorDlg.Create( Application ) do
  begin
    try
      EdtStrings.Lines := TStringList( GetOrdValue );
      EdtStringsClick( nil );
      if ShowModal = mrOK then
        SetOrdValue( Longint( EdtStrings.Lines ) );
    finally
      Free;
    end;
  end; { with }
end; {= TRzStringListProperty.Edit =}


{=================================}
{== TRzStrListEditorDlg Methods ==}
{=================================}

procedure TRzStrListEditorDlg.FormCreate(Sender: TObject);
var
  StyleBits  : Byte;
begin
  HelpContext := hcDStringListEditor;

  DlgOpen.HelpContext := hcDStringListLoad;
  DlgSave.HelpContext := hcDStringListSave;
  SingleLine := LoadStr( srLine );
  MultipleLines := LoadStr( srLines );

  DelphiIni := TIniFile.Create( 'DELPHI.INI' );
  with EdtStrings.Font do
  begin
    Name := DelphiIni.ReadString( Section, 'FontName', 'MS Sans Serif' );
    Size := DelphiIni.ReadInteger( Section, 'FontSize', 8 );
    Color := DelphiIni.ReadInteger( Section, 'FontColor', clBlack );
    StyleBits := DelphiIni.ReadInteger( Section, 'FontStyle', fsBoldMask );
    Style := [];
    if StyleBits and fsBoldMask = fsBoldMask then
      Style := Style + [ fsBold ];
    if StyleBits and fsItalicMask = fsItalicMask then
      Style := Style + [ fsItalic ];
    if StyleBits and fsUnderlineMask = fsUnderlineMask then
      Style := Style + [ fsUnderline ];
    if StyleBits and fsStrikeOutMask = fsStrikeOutMask then
      Style := Style + [ fsStrikeOut ];
  end;
  FTabSize := DelphiIni.ReadInteger( Section, 'TabSize', 32 );
  UpdateClipboardStatus;
end; {= TRzStrListEditorDlg.FormCreate =}


procedure TRzStrListEditorDlg.FormDestroy(Sender: TObject);
begin
 DelphiIni.Free;
end;


procedure TRzStrListEditorDlg.BtnOkClick(Sender: TObject);
var
  StyleBits : Byte;
begin
  if ChkDefault.Checked then
  begin
    with EdtStrings.Font do
    begin
      DelphiIni.WriteString( Section, 'FontName', Name );
      DelphiIni.WriteInteger( Section, 'FontSize', Size );
      DelphiIni.WriteInteger( Section, 'FontColor', Color );

      StyleBits := 0;
      if fsBold in Style then
        StyleBits := fsBoldMask;
      if fsItalic in Style then
        StyleBits := StyleBits + fsItalicMask;
      if fsUnderline in Style then
        StyleBits := StyleBits + fsUnderlineMask;
      if fsStrikeOut in Style then
        StyleBits := StyleBits + fsStrikeOutMask;
      DelphiIni.WriteInteger( Section, 'FontStyle', StyleBits );
    end;
    DelphiIni.WriteInteger( Section, 'TabSize', FTabSize );
  end;
end; {= TRzStrListEditorDlg.BtnOkClick =}


procedure TRzStrListEditorDlg.BtnFontClick(Sender: TObject);
begin
  DlgFont.Font := EdtStrings.Font;
  if DlgFont.Execute then
  begin
    EdtStrings.Font := DlgFont.Font;
  end;
end;


procedure TRzStrListEditorDlg.BtnUndoClick(Sender: TObject);
begin
  EdtStrings.Perform( wm_Undo, 0, 0 );
end;


procedure TRzStrListEditorDlg.BtnCutClick(Sender: TObject);
begin
  EdtStrings.CutToClipboard;
  UpdateClipboardStatus;
end;


procedure TRzStrListEditorDlg.BtnCopyClick(Sender: TObject);
begin
  EdtStrings.CopyToClipboard;
  UpdateClipboardStatus;
end;


procedure TRzStrListEditorDlg.BtnPasteClick(Sender: TObject);
begin
  EdtStrings.PasteFromClipboard;
end;


procedure TRzStrListEditorDlg.BtnOpenClick(Sender: TObject);
begin
  if DlgOpen.Execute then
    EdtStrings.Lines.LoadFromFile( DlgOpen.FileName );
end;


procedure TRzStrListEditorDlg.BtnSaveClick(Sender: TObject);
begin
  if DlgSave.Execute then
    EdtStrings.Lines.SaveToFile( DlgSave.FileName );
end;


procedure TRzStrListEditorDlg.IndentLine( LineNum : Integer );
begin
  with EdtStrings do
  begin
    { Move to Beginning of line and insert tab }
    SelStart := Perform( em_LineIndex, LineNum, 0 );
    Perform( wm_Char, vk_tab, 0 );
    { Move cursor to end of line }
    SelStart := Perform( em_LineIndex, LineNum + 1, 0 ) - 1;
  end;
end;


function TRzStrListEditorDlg.OutdentLine( LineNum : Integer ) : Boolean;
var
  L : string;
begin
  with EdtStrings do
  begin
    L := Lines[ LineNum ];
    if L[ 1 ] = #9 then
    begin
      { Move to Beginning of line and insert tab }
      SelStart := Perform( em_LineIndex, LineNum, 0 );

      Perform( wm_KeyDown, vk_Delete, 0 );
      Perform( wm_KeyUp, vk_Delete, 0 );
      { Move cursor to end of line }
      SelStart := Perform( em_LineIndex, LineNum + 1, 0 ) - 1;
      Result := True;
    end
    else
      Result := False;
  end;
end;


procedure TRzStrListEditorDlg.IndentLines( Indent : Boolean );
var
  I, StartLine, StopLine : Integer;
  OldSelStart, OldSelLength : Integer;
  LineCount, P : Integer;
begin
  with EdtStrings do
  begin
    StartLine := Perform( em_LineFromChar, SelStart, 0 );
    StopLine := Perform( em_LineFromChar, SelStart + SelLength, 0 );

    SelStart := Perform( em_LineIndex, StartLine, 0 );
    P := Perform( em_LineIndex, StopLine + 1, 0 ) - 1;
    SelLength := P - SelStart;

    OldSelStart := SelStart;
    OldSelLength := SelLength;

    LineCount := 0;
    for I := StartLine to StopLine do
    begin
      if Indent then
        IndentLine( I )
      else
      begin
        if OutdentLine( I ) then
          Inc( LineCount );
      end;
    end;

    SelStart := OldSelStart;
    if Indent then
      SelLength := OldSelLength + StopLine - StartLine
    else
      SelLength := OldSelLength - LineCount;
    
  end;
end; {= TRzStrListEditorDlg.IndentLines =}


procedure TRzStrListEditorDlg.BtnOutdentClick(Sender: TObject);
var
  L : string;
begin
  with EdtStrings do
  begin
    if SelLength <> 0 then
      IndentLines( False )
    else
      OutdentLine( FCurLine );
  end;
end;


procedure TRzStrListEditorDlg.BtnIndentClick(Sender: TObject);
begin
  with EdtStrings do
  begin
    if SelLength <> 0 then
      IndentLines( True )
    else
      IndentLine( FCurLine );
  end;
end;


procedure TRzStrListEditorDlg.EdtStringsChange(Sender: TObject);
var
  Count    : Integer;
  LineText : PString;
begin
  Count := EdtStrings.Lines.Count;
  if Count = 1 then
    LineText := @SingleLine
  else
    LineText := @MultipleLines;
  LblCount.Caption := Format( '%d %s', [ Count, LineText^ ] );
  EdtStringsClick( Sender );
end; {= TRzStrListEditorDlg.EdtStringsChange =}


procedure TRzStrListEditorDlg.EdtStringsKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  EdtStringsClick( Sender );
  if Key = vk_Escape then
    BtnCancel.Click;
end;


procedure TRzStrListEditorDlg.EdtStringsKeyUp(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  EdtStringsClick( Sender );
end;


procedure TRzStrListEditorDlg.EdtStringsClick(Sender: TObject);
begin
  FCurLine := EdtStrings.Perform( em_LineFromChar, EdtStrings.SelStart, 0 );
  FCurCol := EdtStrings.SelStart - EdtStrings.Perform( em_LineIndex, FCurLine, 0 );
  LblLine.Caption := IntToStr( FCurLine + 1 );
  LblCol.Caption := IntToStr( FCurCol + 1 );
  UpdateClipboardStatus;
end;


procedure TRzStrListEditorDlg.UpdateClipboardStatus;
var
  HasText      : Boolean;
  HasSelection : Boolean;
begin
  HasSelection := EdtStrings.SelLength <> 0;
  BtnCut.Enabled := HasSelection;
  MnuCut.Enabled := HasSelection;
  BtnCopy.Enabled := HasSelection;
  MnuCopy.Enabled := HasSelection;
  HasText := Clipboard.HasFormat( cf_Text );
  BtnPaste.Enabled := HasText;
  MnuPaste.Enabled := HasText;
end;


procedure TRzStrListEditorDlg.BtnCopyrightClick(Sender: TObject);
begin
  LblCopyright.Visible := True;
end;


procedure TRzStrListEditorDlg.EdtStringsMouseUp( Sender: TObject;
                                                 Button: TMouseButton;
                                                 Shift: TShiftState;
                                                 X, Y: Integer);
begin
  UpdateClipboardStatus;
end;


procedure TRzStrListEditorDlg.BtnHelpClick(Sender: TObject);
begin
  Application.HelpContext( hcDStringListEditor );
end;

procedure TRzStrListEditorDlg.BtnTabSizeClick(Sender: TObject);
begin
  if BtnTabSize.Down then
  begin
    SpnTabSize.Visible := True;
    BtnSetTabSize.Visible := True;
    BtnCancelTabSize.Visible := True;
    EnableButtons( False );

    SpnTabSize.Value := FTabSize;
    SpnTabSize.SetFocus;
  end
  else
    BtnSetTabSizeClick( BtnCancelTabSize );
end; {= TRzStrListEditorDlg.BtnTabSizeClick =}


procedure TRzStrListEditorDlg.SetTabSize;
var
  TabStop : array[ 1..1 ] of Integer;
begin
  if FTabSize < 0 then
    FTabSize := -FTabSize;
  TabStop[ 1 ] := FTabSize;
  EdtStrings.Perform( em_SetTabStops, 1, Longint( @TabStop[ 1 ] ) );
end;


procedure TRzStrListEditorDlg.BtnSetTabSizeClick(Sender: TObject);
begin
  if Sender = BtnSetTabSize then
  begin
    try
      FTabSize := SpnTabSize.Value;
    except
    end;
    SetTabSize;
  end;

  SpnTabSize.Visible := False;
  BtnSetTabSize.Visible := False;
  BtnCancelTabSize.Visible := False;
  BtnTabSize.Down := False;
  EnableButtons( True );
  EdtStrings.SetFocus;
end;

procedure TRzStrListEditorDlg.EnableButtons( Enable : Boolean );
var
  SysMenu : HMenu;
begin
  BtnOutdent.Enabled := Enable;
  BtnIndent.Enabled := Enable;
  BtnOpen.Enabled := Enable;
  BtnSave.Enabled := Enable;
  BtnUndo.Enabled := Enable;
  BtnFont.Enabled := Enable;
  BtnOK.Enabled := Enable;
  BtnCancel.Enabled := Enable;
  BtnHelp.Enabled := Enable;
  ChkDefault.Enabled := Enable;
  SysMenu := GetSystemMenu( Handle, False );
  if Enable then
    EnableMenuItem( SysMenu, sc_Close, mf_ByCommand or mf_Enabled )
  else
    EnableMenuItem( SysMenu, sc_Close, mf_ByCommand or mf_Disabled or mf_Grayed );
  EdtStrings.Enabled := Enable;

  BtnCut.Enabled := Enable;
  BtnCopy.Enabled := Enable;
  BtnPaste.Enabled := Enable;
  if Enable then
    UpdateClipboardStatus;
end;


procedure TRzStrListEditorDlg.FormShow(Sender: TObject);
begin
  SetTabSize;
end;


{========================}
{== Register Procedure ==}
{========================}

procedure Register;
begin
  RegisterPropertyEditor( TypeInfo( TStrings ), nil, '', TRzStringListProperty );
end;


end.
