unit Tablist;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls;

const
  MaxTabs = 10;
type
  TTabListBox = class(TListBox)
  private
    { Private declarations }
    FTabs : array[1..MaxTabs] of integer;
    procedure SetTabs(Value : string);
    function GetTabs : string;
    procedure SendTabs;
  protected
    { Protected declarations }
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
  public
    { Public declarations }
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
  published
    { Published declarations }
    property TabPercentages : string read GetTabs write SetTabs;
  end;

procedure Register;

implementation

type
  EOrderException = class(Exception);

procedure TTabListBox.CreateParams(var Params: TCreateParams);
begin
inherited CreateParams(Params);
with Params do
  Style := Style or LBS_UseTabStops;
end;

procedure TTabListBox.CreateWnd;
begin
inherited CreateWnd;
SendTabs;
end;

function TTabListBox.GetTabs : string;
{translate FTab array into a string}
var
  Tab, I : integer;
begin
Result := '';
for I := 1 to MaxTabs do
  begin
  Tab := FTabs[I];
  if I > 1 then
    if Tab = 0 then Break
    else Result := Result + ', ';
  Result := Result + IntToStr(Tab);
  end;
end;

function Trim(const S : String) : String;
{Trims leading and trailing spaces, tabs, and control chars from string}
var
  I, Len : Integer;
begin
Result := S;
while (Length(Result) > 0) and (Result[Length(Result)] <= ' ') do
  Dec(Result[0]);   {remove trailing junk}
if Length(Result) > 0 then
  begin
  I := 1;
  while Result[I] <= ' ' do Inc(I); {count leading junk}
  if I>1 then
    begin
    Len := Length(Result) -I+1;  {figure new length}
    Move(Result[I], Result[1], Len);
    Result[0] := chr(Len);
    end;
  end;
end;

procedure TTabListBox.SetTabs(Value : string);
var
  I, J, Last : integer;
  L : LongInt;
  Done : boolean;
  Tmp : array[1..MaxTabs] of Integer;

  function EvalSubStr : LongInt;
  var
    S : string[15];
  begin
  S := Trim(Copy(Value, 1, J-1));
  Delete(Value, 1, J);
  Result := StrToInt(S);
  end;

begin
FillChar(Tmp, Sizeof(Tmp), 0);
I := 1;
Done := False;
Last := -1;
while not Done and (I <= MaxTabs) do
  begin
  Value := Trim(Value);
  J := Pos(',', Value);
  if J > 0 then
    L := EvalSubStr
  else
    begin
    J := Pos(' ', Value);
    if J > 0 then
      L := EvalSubStr
    else
      begin
      L := StrToInt(Value);
      Done := True;
      end;
    end;
  if (L < 0) or (L > 200) then
    Raise EOrderException.Create('Tab percentage values must be '+
          'between 0 and 200');
  Tmp[I] := L;
  if L <= Last then
    Raise EOrderException.Create('Tab percentage values must be in '+
          'increasing order');
  Last := L;
  Inc(I);
  end;
Move(Tmp, FTabs, Sizeof(Tmp)); {only happens if no exception}
SendTabs;
end;

procedure TTabListBox.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
var
  Tmp : integer;
begin
Tmp := Width;
inherited SetBounds(ALeft, ATop, AWidth, AHeight);
if Tmp <> AWidth then
  SendTabs;
end;

procedure TTabListBox.SendTabs;
var
  Tabs : array[1..MaxTabs] of integer;
  DBU, I : integer;
  Test : LongInt;
  SaveFont, SystemFont : TFont;
begin
if HandleAllocated then
  begin
  I := 1;
  DBU := LoWord(GetDialogBaseUnits);
  while (I <= MaxTabs) and (FTabs[I] > 0) do
    begin
    Tabs[I] := (Width*LongInt(FTabs[I])) div DBU div 25;
    Inc(I);
    end;
  {Set the tabs based on standard System font}
  SaveFont := TFont.Create;
  SystemFont := TFont.Create;
  SaveFont.Assign(Font);
  Font.Assign(SystemFont);
  if I = 1 then
    SendMessage(Handle, LB_SetTabStops, 0, 0)
  else
    SendMessage(Handle, LB_SetTabStops, I-1, LongInt(@Tabs));
  Font.Assign(SaveFont);
  SaveFont.Free;
  SystemFont.Free;
  Invalidate;
  end;
end;

procedure Register;
begin
  RegisterComponents('MyStuff', [TTabListBox]);
end;

end.
