/*
 *                 OC Curves for Hypergeometric Sampling
 *
 *                   A Quality Assurance Training Tool:
 *            Statistics Committee of the QA Section of the PMA
 *
 *               Bob Obenchain, CompuServe User [72007,467]
 *
 *                     Usage: DOSprompt>  ochypg
 *
 *      Parameter settings are either...
 *             specified interactively in response to prompts, or
 *             redirected to come from a batch input file.
 */

#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>
#include <math.h>

#define abs(a)      (( (a) >= 0 ) ? (a) : (-a))  /* absolute value of a */
#define sgn(a)      (( (a) >= 0 ) ?  1  :  -1 )  /* numerical sign of a */
#define min(a,b)    (( (a) <= (b) ) ?  (a) : (b) )

/* externals */

FILE    *inpfile, *csvfile;
char    buf[256], inpnam[13], outnam[13];
double  alpha, fraction, term, prob, pmax;
long    xn, xmin, xmax, xold, ymax;
long    sn, nc, pn;
int     yesno;

/* prototypes */

double  gammln( double xx );
char *copyright="Copyright 1987, 1988 Numerical Recipes Software";

void   main( void );
void   handler( int sig );
double hypgprob( long from, long bad, long take, long find );

void handler( int sig ) {     /* Interrupt Handler */
      char c;        /* key scan code,,char */
      
      signal( SIGINT, SIG_IGN );     /* ignore further signals */
      printf("\n\n...Enter a Q to Quit, anything else to Continue...\n\n");
      gets( buf );
      sscanf( buf, "%c", &c );
      if( c == 'Q' || c == 'q' ) {
           exit( 1 );
           }
      /* else...normal return, to continue processing... */
      signal( SIGINT, handler );
      }

/*
 *  gammln() function is the logarithm of gamma function of Press, et.al.
 *  (1988). "Numerical Recipes in C," Cambridge University Press.
 */

double hypgprob( from, bad, take, find )     /* Hypergeometric Probability */
long from, bad, take, find; {

      double hpl;

      if( from < 0 || bad < 0 || take < 0 || find < 0 || from < bad )
             return( -1.0 );       /* error */

      if( find > take || from < take || bad < find ||
          from - bad < take - find )
             return( 0.0 );       /* impossible */

      if( take == from ) {
          if( bad == find )
             return( 1.0 );       /* certain */
          else
             return( 0.0 );       /* impossible */
          }

      if( from == bad ) {
          if( take == find )
             return( 1.0 );       /* certain */
          else
             return( 0.0 );       /* impossible */
          }

      if( bad == 0 ) {
          if( find = 0 )
             return( 1.0 );       /* certain */
          else
             return( 0.0 );       /* impossible */
          }

      hpl =  gammln( (double)( bad + 1 ) )
           - gammln( (double)( find + 1 ) )
           - gammln( (double)( bad - find + 1 ) )
           + gammln( (double)( take + 1 ) )
           - gammln( (double)( from + 1 ) )
           + gammln( (double)( from - take + 1 ) )
           - gammln( (double)( take - find + 1 ) )
           + gammln( (double)( from - bad + 1 ) )
           - gammln( (double)( from - bad - take + find + 1 ) );
      return( exp( hpl ) );
      }

void main() {

      char kb;
      long x;

      if( signal( SIGINT, handler ) == SIG_ERR ) {
             printf("\n\n\tOCHypg: Couldn't set SIGINT...Abort!\n\n");
             exit( 1 );
             }
      
      printf("\n\n     ***    Operating Characteristic Curves for     ***" );
      printf(  "\n                   Hypergeometric Sampling" );
      printf("\n\n                   OCHYPRG....Version 9308");
      printf("\n\n                A Quality Assurance Training Tool:");
      printf(  "\n         Statistics Committee of the QA Section of the PMA");
      printf("\n\n            Bob Obenchain, CompuServe User [72007,467]\n\n" );
      
      printf(      "\tWill Parameter Settings be Input via K = Keyboard ? ...or" );
      printf(    "\n\t                                     B = Batch File ?" );
      printf(  "\n\n\tPress the   K   or   B   key now --> " );
      while( 1 ) {
              gets( buf );
              sscanf( buf, "%c", &kb );
              if( kb == 'k' )
                  kb  = 'K';
              if( kb == 'b' )
                  kb  = 'B';
              if( kb == 'K' || kb == 'B' )
                  break;
              printf( "\r\tPress either K  or  B  ...Try Again --> " );
              }

      strcpy( outnam, "ochypg" );
      strcpy( inpnam, "ochypg" );

      if( kb == 'B' )
          printf( "\n\n\tBatch Input of Parameter Settings Selected..." );
      else
          printf( "\n\n\tKeyboard Input of Parameter Settings Selected..." );
      
      printf( "\n\n\tAt colon Prompts : ...simply press ENTER to get the [default]." );

      if( kb == 'B' ) {
          printf(  "\n\n\tSpecify filename of Batch Input File [%s] : ", inpnam );
          gets( buf );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
              sscanf( buf, "%s", inpnam );
          inpnam[ min( 8, (int)strcspn( inpnam, ". " ) ) ] = '\0';
          if( inpnam[0] == '\0' )
              strcpy( inpnam, "ochypg" );
          strcpy( outnam, inpnam );
          strcat( inpnam, ".inp" );
          printf( "\n\tThe Batch Input file is to be: %s\n", inpnam );
          if( ( inpfile = fopen( inpnam, "r" ) ) == NULL ) {
                  printf(  "\tCannot read Batch Input file: %s\n", inpnam );
                  printf(  "\t...using Keyboard Input from standard Infile, stdin.\n" );
                  kb = 'K';
                  }
          }
      
      if( kb == 'K' )
          inpfile = stdin;

      printf(  "\n\n\tSpecify filename to save primary OCHypg Output [%s] : ",
          outnam );
      fgets( buf, 80, inpfile );
      if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
          sscanf( buf, "%s", outnam );
      outnam[ min( 8, (int)strcspn( outnam, ". " ) ) ] = '\0';
      if( outnam[0] == '\0' )
          strcpy( outnam, "ochypg" );
      strcat( outnam, ".csv" );
      printf(  "\n\tThe OCHypg primary Output file is to be: %s\n",
              outnam );
      if( ( csvfile = fopen( outnam, "w" ) ) == NULL ) {
          printf(  "\tCannot write to Output filename : %s\n", outnam );
          if( ( csvfile = fopen( "ochypg.csv", "w" ) ) == NULL )
                  printf(  "\tCannot write to Output filename : ochypg.csv\n" );
          else {
                  printf(  "\t...using default Outfile name : ochypg.csv\n" );
                  strcpy( outnam, "ochypg.csv" );
                  }
          }

      printf("\n\nOCHypg uses the hypergeometric distribution to calculate values on" );
      printf("\nthe Operating Characteristics curve for sampling without replacement" );
      printf("\nfrom a finite population of 2 million or fewer items.");
      
      pn = 500;
      sn = 100;
      nc =   0;
      fraction = 0.05;

      while( 1 ) {

          printf("\n\nWhat is the Population Size? [N=%ld] : ", pn );
          fgets( buf, 80, inpfile );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
              sscanf( buf, "%ld", &pn );
          if( pn < 2 )
              pn = 2;
          if( pn > 2000000000 )
              pn = 2000000000;
          printf("\n\tPopulation Size: N = %ld", pn );
          if( csvfile != NULL )
              fprintf( csvfile, "\n\nPopulation Size, N = %ld,,", pn );

          printf("\n\nWhat is the Sample Size ? [n=%ld] : ", sn );
          fgets( buf, 80, inpfile );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
              sscanf( buf, "%ld", &sn );
          if( sn < 1 )
              sn = 1;
          if( sn > pn - 1 )
              sn = pn - 1;
          printf("\n\tSample Size: n = %ld", sn);
          if( csvfile != NULL )
              fprintf( csvfile, "\nSample Size, n = %ld,,", sn);

          printf("\n\nWhat is the Acceptance Number ? [c=%ld] : ", nc );
          fgets( buf, 80, inpfile );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
              sscanf( buf, "%ld", &nc );
          if( nc >= sn )
              nc = sn - 1;
          if( nc < 0 )
              nc = 0;
          printf("\n\tAcceptance Number: c = %ld", nc );
          if( csvfile != NULL ) {
              fprintf( csvfile, "\nAcceptance Number, c = %ld,,", nc );
              fprintf( csvfile, "\n\nFraction Nonconforming,Probability of Acceptance,Nonconforms,Population Size" );
              }

          while( 1 ) {

              printf("\n\nTrue Fraction Nonconforming ? [p=%7.5lf] : ",
                  fraction );
              fgets( buf, 80, inpfile );
              if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
                  sscanf( buf, "%lf", &fraction );
              if( fraction > 1.0 )
                  fraction = 1.0;
              if( fraction < 0.0 )
                  fraction = 0.0;

              xn = (long)( fraction * (double)pn + 0.5);
              if( xn < (long)1 )
                  xn = (long)1;
              if( xn == pn )
                  xn = pn - (long)1;
              fraction = (double)xn / (double)pn;
              printf("\n\tNumber of Nonconforming Items in Population = %ld ", xn );
              printf("\n\tTrue Fraction Nonconforming: p=%7.5lf ", fraction );
              prob = 0.0;
              for( x = 0; x <= nc; x++ ) {
                  term = hypgprob( pn, xn, sn, x );
                  if( term < 0.0 || term > 1.0 )
                      break;        /* error */
                  prob += term;
                  }
              printf( "\n\tProbability of Acceptance = %7.5lf.", prob );

              if( csvfile != NULL )
                  fprintf( csvfile, "\n%7.5lf,%7.5lf,%ld,%ld", fraction, prob, xn, pn );

              yesno = 'y';
              printf("\n\nAdditional values of Incoming Fraction Defective [Y|n] ? : ");
              if( kb == 'B' )
                  fgets( buf, 80, inpfile );
              else
                  gets( buf );
              if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
                  sscanf( buf, "%c", &yesno );
              if( yesno == 'n' || yesno == 'N' )
                  break;
              }

          yesno = 'y';
          printf("\n\nAdditional values of Sample Size and/or Accept Number [Y|n] ? : ");
          if( kb == 'B' )
              fgets( buf, 80, inpfile );
          else
              gets( buf );
          if( buf[0] != '\0' && buf[0] != '\r' && buf[0] != '\n' )
              sscanf( buf, "%c", &yesno );
          if( yesno == 'n' || yesno == 'N' )
              break;
          }

      printf( "\n\nExiting OCHypg...\n\n" );
      }
