# Copyright (c) 1997 Sun Microsystems, Inc.
# All rights reserved.
# 
# Permission is hereby granted, without written agreement and without
# license or royalty fees, to use, copy, modify, and distribute this
# software and its documentation for any purpose, provided that the
# above copyright notice and the following two paragraphs appear in
# all copies of this software.
# 
# IN NO EVENT SHALL SUN MICROSYSTEMS, INC. BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
# OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF SUN
# MICROSYSTEMS, INC. HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
# 
# SUN MICROSYSTEMS, INC. SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING,
# BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THE SOFTWARE PROVIDED
# HEREUNDER IS ON AN "AS IS" BASIS, AND SUN MICROSYSTEMS, INC. HAS NO
# OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR
# MODIFICATIONS.
#
# SyncMemo conduit for PilotManager
#  Alan Harder, 4/97
#  Alan.Harder@Sun.COM
#  version 1.1, 7/97
#
#  PilotManager by Bharat.Mediratta@Sun.COM
#
package SyncMemo;

use Tk;
use TkUtils;
use strict;
use Data::Dumper;
use Carp;

#   $gHomeDirectory;		# base directory for files
#   $gOnlyAscii;		# whether to limit files to "ascii text"
#   $gDoPilotToFile;		# whether to do pilot->file transfers
#   $gMemoUpdater;		# script for updating memo files
#   $gUseUpdater;		# whether or not to run update script
#   $gPostUpdater;		# post update script
#   $gUsePostUpdater;		# whether or not to run post update script
#   $gUseFilemerge;		# whether to use filemerge to merge changes
#   $gBlowAwayPilot;		# nuke all pilot memos and copy over files
my ($gConfigDialog);		# configuration dialog
my ($gDismissBtn);		# dismiss button on config dialog
my ($gYesNoDialog);		# yes/no dialog
my ($RCFILE);			# configuration file
my ($DBFILE);			# record of files/checksums
my ($PREFS);			# configuration variables
my ($VERSION) = "1.3_2";	# SyncMemo version

sub conduitInit
{
    $RCFILE = "SyncMemo/SyncMemo.prefs";
    $DBFILE = $PREFS->{"DBFILE"} = "SyncMemo/SyncMemo.db";
    &loadPrefs;

    $PREFS->{"gHomeDirectory"} = "$ENV{HOME}/.pilotmgr/SyncMemo"
	unless (defined($PREFS->{"gHomeDirectory"}));
    $PREFS->{"gOnlyAscii"} = 1
	unless (defined($PREFS->{"gOnlyAscii"}));
    $PREFS->{"gDoPilotToFile"} = 1
	unless (defined($PREFS->{"gDoPilotToFile"}));
    $PREFS->{"gMemoUpdater"} = "$ENV{HOME}/.pilotmgr/SyncMemo/update"
	unless (defined($PREFS->{"gMemoUpdater"}));
    $PREFS->{"gUseUpdater"} = 0
	unless (defined($PREFS->{"gUseUpdater"}));
    $PREFS->{"gPostUpdater"} = ""
	unless (defined($PREFS->{"gPostUpdater"}));
    $PREFS->{"gUsePostUpdater"} = 0
	unless (defined($PREFS->{"gUsePostUpdater"}));
    $PREFS->{"gUseFilemerge"} = 0
	unless (defined($PREFS->{"gUseFilemerge"}));
    $PREFS->{"gBlowAwayPilot"} = 0
	unless (defined($PREFS->{"gBlowAwayPilot"}));
}

sub conduitQuit
{
    &savePrefs;
}

sub conduitInfo
{
    return { "database" =>
		{
		    "name" => "MemoDB",
		    "creator" => "memo",
		    "type" => "DATA",
		    "flags" => 0,
		    "version" => 0,
		},
	     "version" => $VERSION,
	     "author" => "Alan Harder",
	     "email" => "Alan.Harder\@Sun.COM" };
}

sub conduitConfigure
{
    my ($this, $wm) = @_;
    my ($frame, $obj, $subfr, $cb);

    unless (defined($gConfigDialog))
    {
	$gConfigDialog = $wm->Toplevel(-title => "Configuring SyncMemo");
	$gConfigDialog->transient($wm);

	$gYesNoDialog = $wm->Dialog(-title => "Do Something",
				    -text =>
					"Are you sure?      \n(No undo!)     ",
				    -justify => "center",
				    -buttons => ["Yes", "No"],
				    -default_button => "No",
				    -popover => $gConfigDialog,
				    -overanchor => 'c', -popanchor => 'c');
	PilotMgr::setColors($gYesNoDialog);

	$frame = $gConfigDialog->Frame(-relief => "ridge", -bd => 4);
	$obj = TkUtils::Label($frame, "SyncMemo Settings");
	$obj->pack(-anchor => "c");

	$subfr = $frame->Frame;
	$obj = $subfr->Label(-text => "Memo directory: ");
	$obj->pack(-side => "left", -anchor => "nw", -fill => "x");

	$obj = $subfr->Entry(-textvariable => \$PREFS->{"gHomeDirectory"},
			     -relief => "sunken", -width => 60);
	$obj->pack(-side => "right", -anchor => "w", -fill => "x");
	$subfr->pack(-anchor => "e");

	$subfr = $frame->Frame;
	$obj = $subfr->Label(-text => "Pre-Update script: ");
	$obj->pack(-side => "left", -anchor => "nw", -fill => "x");

	$obj = $subfr->Entry(-textvariable => \$PREFS->{"gMemoUpdater"},
			     -relief => "sunken", -width => 60);
	$obj->pack(-side => "right", -anchor => "w", -fill => "x");
	$subfr->pack(-anchor => "e");

	$subfr = $frame->Frame;
	$obj = $subfr->Label(-text => "Post-Update script: ");
	$obj->pack(-side => "left", -anchor => "nw", -fill => "x");

	$obj = $subfr->Entry(-textvariable => \$PREFS->{"gPostUpdater"},
			     -relief => "sunken", -width => 60);
	$obj->pack(-side => "right", -anchor => "w", -fill => "x");
	$subfr->pack(-anchor => "e");

	$subfr = $frame->Frame;
	$obj = TkUtils::Checkbutton($subfr,"Run pre-update script before sync",
				    \$PREFS->{"gUseUpdater"});
	$obj->pack(-side => "left", -fill => "x");

	$obj = TkUtils::Checkbutton($subfr,"Run post-update script after sync",
				    \$PREFS->{"gUsePostUpdater"});
	$obj->pack(-fill => "x");
	$subfr->pack;

	$subfr = $frame->Frame;
	$obj = TkUtils::Checkbutton($subfr, "Only sync text files",
				    \$PREFS->{"gOnlyAscii"});
	$obj->pack(-side => "left", -fill => "x");

	$cb = TkUtils::Checkbutton($subfr, "Use Filemerge",
				   \$PREFS->{"gUseFilemerge"});
	$cb->configure(-state => "disabled")
	    unless ($PREFS->{"gDoPilotToFile"});
	$cb->pack(-side => "right", -fill => "x");

	$obj = TkUtils::Checkbutton($subfr, "Do Pilot->File Transfers",
				    \$PREFS->{"gDoPilotToFile"});
	$obj->configure(-command => sub{
	    $cb->configure(-state => (($cb->cget("-state") eq "normal") ?
					"disabled" : "normal")) });
	$obj->pack(-side => "left", -fill => "x");
	$subfr->pack;

	$subfr = $frame->Frame;
	$gDismissBtn = TkUtils::Button($gConfigDialog, "Dismiss",
		sub{ &savePrefs; $gConfigDialog->withdraw });

	$obj = TkUtils::Button($subfr, "Run Pre-Update Script Now",
		sub{ if (-x $PREFS->{"gMemoUpdater"}) {
			$gDismissBtn->configure(-state => "disabled");
			my ($curs) = $gConfigDialog->cget("-cursor");
			$gConfigDialog->configure(-cursor => "watch");
			$gConfigDialog->update;
			system "$PREFS->{gMemoUpdater}";
			$gConfigDialog->configure(-cursor => $curs);
			$gDismissBtn->configure(-state => "normal");
		     } });
	$obj->pack(-side => "left", -anchor => "w", -fill => "x");

	$obj = TkUtils::Button($subfr, "Run Post-Update Script Now",
		sub{ if (-x $PREFS->{"gPostUpdater"}) {
			$gDismissBtn->configure(-state => "disabled");
			my ($curs) = $gConfigDialog->cget("-cursor");
			$gConfigDialog->configure(-cursor => "watch");
			$gConfigDialog->update;
			system "$PREFS->{gPostUpdater}";
			$gConfigDialog->configure(-cursor => $curs);
			$gDismissBtn->configure(-state => "normal");
		     } });
	$obj->pack(-side => "left", -anchor => "w", -fill => "x");
	$subfr->pack;
	$frame->pack(-side => "top");

	$frame = $gConfigDialog->Frame(-relief => "ridge", -bd => 4);
	$obj = TkUtils::Label($frame, "Special Options");
	$obj->pack(-anchor => "center");
	$obj = TkUtils::Label($frame,
	    "These options allow special actions to be performed on your " .
	    "next hotsync.\nA Reset will cause SyncMemo to reexamine ALL " .
	    "memos on the Pilot instead of just\nthose that have changed.  " .
	    "Blowing away either the Pilot or Files will cause ALL memos\n" .
	    "in that database to be deleted and replaced with the contents " .
	    "of the other database.");
	$obj->configure(-justify => "left");
	$obj->pack(-anchor => "center");

	$subfr = $frame->Frame;
	$obj = TkUtils::Button($subfr, "Reset SyncMemo",
	    sub{ $gYesNoDialog->configure(-title => "Reset SyncMemo");
		 unlink ($PREFS->{"DBFILE"}) if ($gYesNoDialog->Show eq "Yes");
		});
	$obj->pack(-side => "left", -anchor => "w", -fill => "x");
	$obj = TkUtils::Button($subfr, "Blow Away Pilot",
	    sub{ $gYesNoDialog->configure(-title => "Blow Away Pilot");
		 if ($gYesNoDialog->Show eq "Yes")
		 {
		     unlink($PREFS->{"DBFILE"});
		     $PREFS->{"gBlowAwayPilot"} = 1;
		 }
		});
	$obj->pack(-side => "left", -anchor => "w", -fill => "x");
	$obj = TkUtils::Button($subfr, "Blow Away Files",
	    sub{ $gYesNoDialog->configure(-title => "Blow Away Files");
		 if ($gYesNoDialog->Show eq "Yes")
		 {
		     unlink($PREFS->{"DBFILE"});
		     &cleanDir($PREFS->{"gHomeDirectory"}, 1);
		 }
		});
	$obj->pack(-side => "left", -anchor => "w", -fill => "x");
	$subfr->pack;

	$frame->pack(-side => "top", -fill => "x");
	$gDismissBtn->pack(-side => "bottom", -anchor => "c");
	PilotMgr::setColors($gConfigDialog);
    }

    $gConfigDialog->Popup(-popanchor => 'c',
			  -popover => $wm,
			  -overanchor => 'c');
}

sub cleanDir
{
    my ($dir, $top) = @_;
    my ($sub);

    foreach $sub (<$dir/*>)
    {
	# If we're in the top directory, don't clean files only
	# clean directories.
	#
	next if ($top && -f $sub);

	if (-d $sub)
	{
	    &cleanDir($sub, 0);
	    rmdir($sub);
	}

	if (-f $sub)
	{
	    unlink $sub;
	}
    }
}

sub conduitSync
{
    my ($this, $dlp, $info) = @_;
    my (%oldsum, %newsum, %piid, %pifile, $record, %pilot, $fullsync);
    my ($appinfo, $cnt,$ret, $file,$fname, $a1,$a2,$a3, $i, $lastsync);
    my (@cats, %catcase, %catindx, $cat, @cat_ids,
	$updates, $filetext, $pilot_dbhandle);
    my ($dbinfo) = &conduitInfo->{database};
    my ($dbname) = $dbinfo->{name};

    # Special case!  "Blow away Pilot"
    if ($PREFS->{"gBlowAwayPilot"})
    {
	$PREFS->{"gBlowAwayPilot"} = 0;
	&savePrefs;
	PilotMgr::msg(
	    "Blow Away Pilot option: deleting and recreating $dbname!");
	$ret = $dlp->delete($dbname);
	if ($ret < 0)
	{
	    $ret = PDA::Pilot::errorText($ret);  
	    PilotMgr::msg("Error $ret while deleting $dbname.. Aborting!");
	    return;
	}
	$pilot_dbhandle = $dlp->create($dbname, $dbinfo->{creator},
				       $dbinfo->{type}, $dbinfo->{flags},
				       $dbinfo->{version});
    }
    else
    {
	# Open or create DB:
	eval
	{
	    $pilot_dbhandle = $dlp->open($dbname);
	};
	if ($@ =~ /read-only value/)
	{
	    PilotMgr::msg("Pilot database '$dbname' does not exist.\n" .
			  "Creating it...");

	    $pilot_dbhandle = $dlp->create($dbname, $dbinfo->{creator},
					   $dbinfo->{type}, $dbinfo->{flags},
					   $dbinfo->{version});
	}
	elsif ($@)
	{
	    croak($@);
	}
    }

    if (!defined($pilot_dbhandle))
    {
	PilotMgr::msg("Unable to open/create '$dbname'.  Aborting!");
	return;
    }

    # Let the user know what we're doing
    #
    $dlp->getStatus();

    $appinfo = $pilot_dbhandle->getAppBlock();
    @cats = @{$appinfo->{categoryName}};
    @cat_ids = @{$appinfo->{categoryID}};
    foreach $i ($[..$#cats)
    {
	next unless (length($cats[$i])>0);
	$cat = $cats[$i];
	$cat =~ tr/A-Z/a-z/;
	$catcase{$cat} = $cats[$i];	# translate lowercase to anycase
	$catindx{$cats[$i]} = $i;	# translate anycase to index
    }

    # Run pre-update script
    if ($PREFS->{"gUseUpdater"})
    {
	PilotMgr::status("Running pre-update script", 0);
	&runScript($dlp, $PREFS->{"gMemoUpdater"}, "memo update");
	PilotMgr::status("Running pre-update script", 100);
    }

    # Read DBFILE
    ($a1, $a2, $a3, $lastsync) = &readDBFile;

    if ($lastsync > 0)
    {
	if ($lastsync != $info->{"successfulSyncDate"})
	{
	    $i = PilotMgr::askUser("Your last sync was either unsuccessful " .
		"or did not include SyncMemo. If you have done any memo " .
		"sync operations to another machine or restored an old " .
		"MemoDB file you should do a full sync now. Otherwise a " .
		"fast sync is still safe...",
		"Fast Sync", "Full Sync");
	    unless ($i eq "Full Sync")
	    {
		%oldsum = %$a1;   %piid = %$a2;   %pifile = %$a3;
	    }
	}
	else
	{
	    %oldsum = %$a1;   %piid = %$a2;   %pifile = %$a3;
	}
    }

    # Check files
    %newsum = &checkFiles;

    #XXX: Turn watchdog off for now, there's a bug in it that
    # interferes with long downloads.
    #
    PilotMgr::watchdog($dlp, 0);

    # Read pilot changes
    if ($PREFS->{"gDoPilotToFile"})
    {
	if (%oldsum)
	{
	    %pilot = &readChangedMemos($dlp, $pilot_dbhandle);
	    $fullsync = 0;
	} else {
	    %pilot = &readAllMemos($dlp, $pilot_dbhandle);
	    $fullsync = 1;
	}

	# Check for renamed categories
	if (defined($PREFS->{"categories"}))
	{
	    for ($i=0; $i < 16; $i++)
	    {
		if ($cats[$i] && $PREFS->{"categories"}->[$i] &&
		    ($cats[$i] ne $PREFS->{"categories"}->[$i]))
		{
		    # Category renamed! Add records to be processed:
		    #(This could be more efficient by just renaming the dir)
		    PilotMgr::msg("Processing Category Change: " .
				  $PREFS->{"categories"}->[$i] . " -> " .
				  $cats[$i]);
		    %pilot = &readMemosInCategory($pilot_dbhandle,
						  $i, {%pilot});
		}
	    }
	}
	$PREFS->{"categories"} = [@cats];
    }

    #XXX: Turn watchdog back on again...
    #
    PilotMgr::watchdog($dlp, 1);

    #compare and update
    $cnt = 0;
    $updates = "";

    my ($count, $count_max);
    $count = 0;
    $count_max = scalar(keys %newsum);
    foreach $file (keys(%newsum))
    {
	unless ($count % &fake_ceil($count_max / 20))
	{
	    PilotMgr::status("Synchronizing Memos", 
			     int (100 * $count / $count_max));
	}
	$count++;

	if (!defined($oldsum{$file}) ||
	    $newsum{$file} ne $oldsum{$file})
	{
	    # Do update
	    $cnt++;
	    $filetext = `cat '$file'`;
	    if ($?)
	    {
		PilotMgr::msg("Error reading $file.");
		$newsum{$file} = $oldsum{$file};
		delete $oldsum{$file};
		next;
	    }
	    # determine category based on containing directory...
	    if ($file =~ m|/([^/]*)/[^/]*$|)
	    {
		$cat = $a3 = $1;
		$cat =~ tr/A-Z/a-z/;
		if (!defined($catcase{$cat}))
		{
		    for ($a1 = 0; $a1 < 16; $a1++)
		    {
			if (!length($cats[$a1]))
			{
			    $cats[$a1] = $a3;
			    $appinfo->{categoryName} = [@cats];
			    $catcase{$cat} = $a3;
			    $catindx{$a3} = $a1;
			    $a3 = 129;
			    foreach $a2 (@cat_ids)
			    {
				$a3 = ord($a2) + 1 if (ord($a2) >= $a3);
			    }
			    $cat_ids[$a1] = sprintf("%c", $a3);
			    $appinfo->{categoryID} = [@cat_ids];
			    PilotMgr::msg("Creating new Pilot category '" .
				$catcase{$cat} . "' with id " . $a3 . ".");
			    $i = $pilot_dbhandle->setAppBlock($appinfo);
			    if ($i < 0)
			    {
				$i = PDA::Pilot::errorText($i);
				PilotMgr::msg("** Error $i creating new " .
				    " category $catcase{$cat} on Pilot.");
			    }
			    last;
			}
		    }
		}
		$cat = $catcase{$cat};
		$cat = defined($catindx{$cat}) ? $catindx{$cat} : "";
	    }
	    $record = &fileToNeutral($piid{$file}, $cat, $filetext);
	    $fname = $file;
	    $fname =~ s|^$PREFS->{"gHomeDirectory"}/||;
	    if (defined($record->{pilot_id}) &&
		defined($pilot{$record->{pilot_id}}))
	    {
		if ($pilot{$record->{pilot_id}} eq "DELETED")
		{
		    PilotMgr::msg("Memo $fname ($record->{pilot_id}) " .
			"was deleted on the Pilot but modified on the " .
			"workstation.. recreating on Pilot..");
		}
		elsif (&recsDiffer($record, $pilot{$record->{pilot_id}}))
		{
		    $cat = $cats[$pilot{$record->{pilot_id}}->{category}];
		    if ($fname !~ m|$cat/[^/]+$|)
		    {
			# Category changed too!
			PilotMgr::msg("(Category Change: $fname)");
			unlink($file);
			delete $piid{$file};
			delete $oldsum{$file};
			delete $newsum{$file};
			$fname =~ s|^.*/||g;
			$file = "$PREFS->{gHomeDirectory}/$cat";
			$record->{category} = defined($catindx{$cat})
						? $catindx{$cat} : "";
			mkdir($file, 0755) unless (-d $file);
			if (-e "$file/$fname") {
			    $file = &mktemp($file);
			} else {
			    $file .= "/$fname";
			}
			$fname = "$cat/$fname";
			$piid{$file} = $record->{pilot_id};
		    }
		    PilotMgr::msg("Memo $fname ($record->{pilot_id}) " .
			"modified on Pilot and workstation..");
		    &mergeRecs($dlp, $record,
			       $pilot{$record->{pilot_id}});
		    PilotMgr::msg("Update File: $fname");
		    open(FD, ">$file");
		    print FD &neutralToFile($record);
		    close(FD);
		    $ret = `/usr/bin/sum '$file'`;
		    if ($ret =~ /^(\d+)\s+(\d+)/)
		    {
			$newsum{$file} = "$1 $2";
		    }
		    $updates .= "$file\n";
		}
		delete $pilot{$record->{pilot_id}};
	    }
	    if ($fullsync && ($ret = &matchRec(\%pilot, $record)))
	    {
		PilotMgr::msg("Matching $fname and Pilot record $ret.");
		delete $pilot{$ret};
	    } else {
		PilotMgr::msg("Update Pilot: $fname.");
		$ret = &writeMemo($record, $dlp, $pilot_dbhandle);
	    }
	    if ($ret < 0)
	    {
		$ret = PDA::Pilot::errorText($ret);
		PilotMgr::msg("Error $ret writing $file to pilot.");
		$dlp->log("SyncMemo: Error $ret writing $file to pilot.");
		$newsum{$file} = $oldsum{$file};
		delete $oldsum{$file};
		next;
	    }
	    $piid{$file} = $ret;
	}
	delete $oldsum{$file} if (defined($oldsum{$file}));
    }
    # check for files to delete
    foreach $file (keys(%oldsum))
    {
	$fname = $file;
	$fname =~ s|^$PREFS->{"gHomeDirectory"}/||;

	if (defined($pilot{$piid{$file}}))
	{
	    if ($pilot{$piid{$file}} eq "DELETED")
	    {
		# delete on both sides, no action needed..
		delete $pilot{$piid{$file}};
	    } else {
		PilotMgr::msg("Memo $fname ($piid{$file}) " .
		    "was deleted on the workstation but modified on the " .
		    "Pilot.. recreating workstation file..");
	    }
	    next;
	}
	PilotMgr::msg("Delete Pilot: $fname ($piid{$file})");
	$ret = &deleteRecord($dlp, $pilot_dbhandle, $piid{$file});
	if ($ret < 0)
	{
	    $ret = PDA::Pilot::errorText($ret);
	    PilotMgr::msg("Error $ret deleting pilot record $piid{$file}!!");
	    $dlp->log("SyncMemo: Error $ret deleting pilot record " .
		      $piid{$file});
	}
	$cnt++;
    }

    # Check pilot changes
    if ($PREFS->{"gDoPilotToFile"})
    {
	foreach $i (keys(%pilot))
	{
	    $cnt++;
	    if ($pilot{$i} eq "DELETED")
	    {
		if (defined($pifile{$i}))
		{
		    $file = $fname = $pifile{$i};
		    $fname =~ s|^$PREFS->{"gHomeDirectory"}/||;
		    PilotMgr::msg("Delete File: $fname.");
		    unlink($file);
		    delete $newsum{$file};
		    delete $piid{$file};
		    delete $pifile{$i};
		    $updates .= "$file\n";
		} else {
		    PilotMgr::msg("Memo $i deleted on Pilot but file not " .
				  "found.. ignored.");
		}
	    }
	    else
	    {
		$file = "$PREFS->{gHomeDirectory}/" .
			$cats[$pilot{$i}->{category}];
		if (defined($pifile{$i}))
		{
		    $fname = $pifile{$i};
		    $ret = "Update";
		    if ($fname !~ m|^$file/[^/]+$|)
		    {
			# Changed Category!  Delete in old location..
			unlink($fname);
			delete $piid{$fname};
			delete $newsum{$fname};
			$fname =~ s|^$PREFS->{"gHomeDirectory"}/||;
			PilotMgr::msg("Delete File: $fname.");
			$fname =~ s|^.*/||g;
			mkdir($file, 0755) unless (-d $file);
			if (-e "$file/$fname") {
			    $file = &mktemp($file);
			} else {
			    $file .= "/$fname";
			}
			$fname = $file;
			$ret = "New Category/Update";
		    } else {
			$file = $fname;
		    }
		} else {
		    mkdir($file, 0755) unless (-d $file);
		    $fname = &makeFilename($pilot{$i});
		    $fname =~ s/\n.*//g;
		    $fname =~ tr| '"<>()[]/;\n|____________|s;
		    $file = $fname = &mktemp(($a1=$file), $fname);
		    if (open(FD, ">$file"))
		    {
			close(FD);
			unlink($file);
		    } else {
			# in case some weird character causes a filename prob
			$file = $fname = &mktemp($a1);
		    }
		    $ret = "Create new";
		}
		$fname =~ s|^$PREFS->{"gHomeDirectory"}/||;
		unless (open(FD, ">$file"))
		{
		    PilotMgr::msg("$ret File: $fname.\n" .
				  "Error opening $file for write!!");
		    next;
		}
		print FD &neutralToFile($pilot{$i});
		close(FD);
		if ($PREFS->{"gOnlyAscii"} && ($ret eq "Create new"))
		{
		    $a1 = `file '$file'`;
		    unless ($a1 =~ /[ \t]text$/ || $a1 =~ /[ \t]script$/)
		    {
			PilotMgr::msg("Skipping non-text file from pilot (" .
			    $cats[$pilot{$i}->{category}] . ", $i, $fname)");
			unlink($file);
			next;
		    }
		}
		PilotMgr::msg("$ret File: $fname.");
		$ret = `/usr/bin/sum '$file'`;
		if ($ret =~ /^(\d+)\s+(\d+)/)
		{
		    $newsum{$file} = "$1 $2";
		    $piid{$file} = $i;
		    $pifile{$i} = $file;
		}
		$updates .= "$file\n";
	    }
	}
    }
    &cleanupPilot($dlp, $pilot_dbhandle)
	if ($PREFS->{"gDoPilotToFile"});

    $pilot_dbhandle->close();

    PilotMgr::msg("All files up to date.") if ($cnt == 0 && %newsum);
    PilotMgr::msg("No Memo files found.") if ($cnt == 0 && !%newsum);

    # Run post-update script
    if ($PREFS->{"gUsePostUpdater"})
    {
	PilotMgr::status("Running post-update script", 0);
	&runScript($dlp, $PREFS->{"gPostUpdater"}, "post update", $updates);
	PilotMgr::status("Running post-update script", 100);
    }

    # Write back to DBFILE
    unless (open(FD, ">$DBFILE"))
    {
	PilotMgr::msg("Error opening $DBFILE for write.");
	return;
    }
    print FD $info->{"thisSyncDate"}, "\n";
    foreach $file (keys(%newsum))
    {
	print FD "$newsum{$file} $file $piid{$file}\n";
    }
    close(FD);
}

##########################################################################

sub loadPrefs
{
    my ($line);

    open(FD, "<$RCFILE") || return;

    $line = <FD>;

    if ($line =~ /\$SyncMemo::/)
    {
	# Old format (v1.04_01 and before)
	#
	do
	{
	    my ($key, $val);

	    if ($line)
	    {
		$line =~ /\$SyncMemo::(\w+) = '(.*)';$/;
		($key, $val) = ($1, $2);

		$PREFS->{$key} = $val
		    if ($key && $val);
	    }

	    $line = <FD>;
	} while ($line);

	close(FD);
    }
    else
    {
	close(FD);
	eval `cat $RCFILE`;
    }

    # For some reason, we need to reference $PREFS here
    # or the preferences won't get loaded properly.
    #
    $PREFS;
}

sub savePrefs
{
    my ($var);

    $Data::Dumper::Purity = 1;
    $Data::Dumper::Deepcopy = 1;

    if (open(FD, ">$RCFILE"))
    {
	if (defined &Data::Dumper::Dumpxs)
	{
	    print FD Data::Dumper->Dumpxs([$PREFS], ['PREFS']);
	}
	else
	{
	    print FD Data::Dumper->Dump([$PREFS], ['PREFS']);
	}

	print FD "1;\n";
	close(FD);
    }
    else
    {
	PilotMgr::msg("Unable to save preferences to $RCFILE!");
    }
}

sub readDBFile
{
    my (%sum, %id, %fi, $line, @lines);
    my ($lastsync) = 0;

    %sum = %id = %fi = ();
    if (-f $DBFILE)
    {
	@lines = split("\n", `cat $DBFILE`);
	if ($lines[0] =~ /^(\d+)$/)
	{
	    $lastsync = $1;
	    shift(@lines);
	}
	foreach $line (@lines)
	{
	    if ($line =~ /^(\d+ \d+) (.+) (\d+)$/)
	    {
		$sum{$2} = $1;
		$id{$2} = $3;
		$fi{$3} = $2;
	    }
	}
    }

    return ({%sum}, {%id}, {%fi}, $lastsync);
}

sub checkFiles
{
    my (%sum, $line, $file, $type);
    my (@dirlist);

    $file = "$PREFS->{gHomeDirectory}/*/*";
    @dirlist = eval "<$file>";

    my ($count) = 0;
    my ($count_max) = scalar(@dirlist);

    PilotMgr::status("Reading File System Memos", 0);

    foreach $file (@dirlist)
    {
	unless ($count % &fake_ceil($count_max / 20))
	{
	    PilotMgr::status("Reading File System Memos",
			     int(100 * $count / $count_max));
	}
	$count++;
	
	# Don't want 2nd level subdirs...
	#
	next if (-d $file);

	if ($PREFS->{"gOnlyAscii"})
	{
	    $type = `file '$file'`;
	    unless ($type =~ /[ \t]text$/ || $type =~ /[ \t]script$/)
	    {
		my ($fname) = $file;
		$fname =~ s|^$PREFS->{"gHomeDirectory"}/||;
		PilotMgr::msg("Skipping non-textfile $fname.");
		next;
	    }
	}

	my($tmp) = `/usr/bin/sum $file`;
	if ($tmp =~ /^(\d+)\s+(\d+)/)
	{
	    $sum{$file} = "$1 $2";
	}
    }
    PilotMgr::status("Reading File System Memos", 100);

    return (%sum);
}

sub readAllMemos
{
    my ($sock, $dbhandle) = @_;
    my ($i, $record, $id, $count_max);
    my (%db);

    $i = 0;
    PilotMgr::status("Reading Pilot Memos [full sync]", 0);
    $count_max = $dbhandle->getRecords();

    while (1)
    {
	$record = $dbhandle->getRecord($i);
	last if (!defined($record));

	unless ($i % &fake_ceil($count_max / 20))
	{
	    PilotMgr::status("Reading Pilot Memos [full sync]",
			     int(100 * $i / $count_max));
	}

	$i++;

	next if ($record->{"deleted"} || $record->{"archived"} ||
		 $record->{"busy"});

	$id = $record->{"id"};
	$db{$id} = &pilotToNeutral($id, $record->{"category"}, $record);
    }
    PilotMgr::status("Reading Pilot Memos [full sync]", 100);

    return(%db);
}

sub readChangedMemos
{
    my ($sock, $dbhandle) = @_;
    my ($i, $record, $id);
    my (%db);

    PilotMgr::status("Reading Pilot Memos [fast sync]", 0);
    while (1)
    {
	$record = $dbhandle->getNextModRecord();
	last if (!defined($record));

	$id = $record->{"id"};

	if ($record->{"deleted"})
	{
	    $db{$id} = "DELETED";
	}
	else
	{
	    $db{$id} = &pilotToNeutral($id, $record->{"category"}, $record);
	}
    }
    PilotMgr::status("Reading Pilot Memos [fast sync]", 100);

    return(%db);
}

sub readMemosInCategory
{
    my ($dbhandle, $categ, $current_recs) = @_;
    my ($record, $id);
    my (%db) = (%$current_recs);

    while (1)
    {
	$record = $dbhandle->getNextRecord($categ);
	last if (!defined($record));

	$id = $record->{"id"};
	next if (defined($db{$id}));

	if ($record->{"deleted"})
	{
	    $db{$id} = "DELETED";
	}
	else
	{
	    $db{$id} = &pilotToNeutral($id, $record->{"category"}, $record);
	}
    }

    return(%db);
}

sub recsDiffer
{
    my ($a, $b) = @_;
    my (%seen, $key);

    foreach $key (keys %$a, keys %$b)
    {
	next if $seen{$key}++;
	next if (&isSpecialSyncField($key));
	return 1 unless (defined($a->{$key}) && defined($b->{$key}) &&
			 $a->{$key} eq $b->{$key});
    }
    return 0;
}

sub matchRec
{
    my ($pilot, $rec) = @_;
    my ($pi);

    foreach $pi (values(%$pilot))
    {
	return $pi->{pilot_id} unless (&recsDiffer($rec, $pi));
    }
    return 0;
}

sub writeMemo
{
    my ($record, $dlp, $pilot_dbhandle) = @_;
    my ($pi_rec, $id);

    $record->{pilot_id} ||= 0;
    $record->{category} ||= 0;

    $pi_rec = &neutralToPilot($record, $pilot_dbhandle);

    $pi_rec->{attr} ||= 0; # Why isn't this set?
    $pi_rec->{category} ||= 0; # Why isn't this set?

    #XXX: Turn watchdog off for this write.
    #
    PilotMgr::watchdog($dlp, 0);
    $id = $pilot_dbhandle->setRecord($pi_rec);
    PilotMgr::watchdog($dlp, 1);

    return $id;
}

sub deleteRecord
{
    my ($dlp, $pilot_dbhandle, $record_id) = @_;
    my ($ret);

    $ret = $pilot_dbhandle->deleteRecord($record_id);

    return ($ret);
}

sub cleanupPilot
{
    my ($dlp, $pilot_dbhandle) = @_;
    $pilot_dbhandle->purge();
    $pilot_dbhandle->resetFlags();
}

sub fileMerge
{
    my ($dlp, $text1, $text2) = @_;
    my ($file1, $file2, $outfile, $outtext);

    $file1 = &mktemp("/tmp", "File.txt");
    open(FD, ">$file1") || return;
    print FD $text1;
    close(FD);
    $file2 = &mktemp("/tmp", "Pilot.txt");
    open(FD, ">$file2") || do { unlink($file1); return };
    print FD $text2;
    close(FD);

    $outfile = &mktemp("/tmp", "Merge.txt");
    system "filemerge $file1 $file2 $outfile";
    unlink($file1);
    unlink($file2);

    return unless (-f $outfile);
    $outtext = `cat $outfile`;
    unlink($outfile);

    return $outtext;
}

sub runScript
{
    my ($dlp, $script, $name, $input_text) = @_;
    my ($tmp, $tmpfile);

    $name .= " " if ($name);
    if (-x $script)
    {
	PilotMgr::msg("Running ${name}script.");
	if ($input_text)
	{
	    $tmpfile = &mktemp("/tmp");
	    open(FD, ">$tmpfile") || ($tmpfile = "");
	    print FD $input_text;
	    close(FD);
	}
	if ($tmpfile) {
	    system "$script < $tmpfile";
	} else {
	    system "$script < /dev/null";
	}
	unlink($tmpfile) if ($tmpfile);
    } else {
	PilotMgr::msg("Error running ${name}script.");
    }
}

sub mktemp
{
    my ($dir, $basename) = @_;
    my ($fname, $fext) = ("Memo0000", "");

    $basename ||= "";
    return ("$dir/$basename") if ($basename && (! -e "$dir/$basename"));
    $fext = $1 if ($basename =~ /(\.\w+)$/);
    $fext ||= ".txt";

    while (1)
    {
        return ("$dir/$fname$fext") if (! -e "$dir/$fname$fext");
        $fname++;
    }
}

sub fake_ceil
{
    my ($val) = int($_[0]);

    return 1 if ($val == 0);
    return $val;
}

##########################################################################

sub pilotToNeutral
{
    my ($id, $category, $pi_rec) = @_;
    my ($record);

    $record = {
	pilot_id => $id,
	category => $category,
    };
    #Copy all relevant fields here:
    $record->{text} = $pi_rec->{text};

    return($record);
}

sub neutralToPilot
{
    my ($record, $db_handle) = @_;
    my ($pi_rec) = $db_handle->newRecord();

    $pi_rec->{id} = $record->{pilot_id};
    $pi_rec->{category} = $record->{category};
    #Copy all relevant fields here:
    $pi_rec->{text} = $record->{text};

    return $pi_rec;
}

sub fileToNeutral
{
    my ($id, $category, $filetext) = @_;
    my ($record);

    $record = {};
    $record->{pilot_id} = $id if ($id);
    $record->{category} = $category if (length($category) > 0);

    #Parse file and fill all relevant fields here:
    $record->{text} = $filetext;

    return($record);
}

sub neutralToFile
{
    my ($record) = @_;
    my ($filetext) = "";

    #Pull out record fields and encode into filetext:
    $filetext = $record->{text};

    return($filetext);
}

sub isSpecialSyncField
{
    my ($field) = @_;

    # Fields that should NOT be compared between Pilot and file.
    #     (ie info stored only on one side or the other, but not both)
    #     "pilot_id" should always be one of these fields.
    # return ($field eq "pilot_id" || $field eq "createdate");

    return ($field eq "pilot_id");
}

sub mergeRecs
{
    my ($dlp, $record, $pirec) = @_;
    my ($ret);

    #Merge method.  Modify fields in $record to get final result.

    $ret = "";
    if ($PREFS->{"gUseFilemerge"})
    {
	PilotMgr::msg("Running filemerge..");
	$ret = &fileMerge($dlp, $record->{text}, $pirec->{text});
	PilotMgr::msg("Using filemerged text.") if ($ret);
    }
    unless ($ret)
    {
	PilotMgr::msg("Concatenating both versions!  Merge " .
	    "changes before your next hotsync!");
	$ret = $record->{text} .
	       "\n--- ^^ File ^^ === Version Separator === vv Pilot vv ---\n" .
	       $pirec->{text};
    }
    $record->{text} = $ret;
}

sub makeFilename
{
    my ($record) = @_;
    my ($filename) = "";

    #Make a potential filename from the given record:
    if ($record->{text} =~ /^\s*<HTML>/i)
    {
	if ($record->{text} =~ /<TITLE\s*[^>]*>\s*(.*)<\/TITLE\s*[^>]*>/si)
	{
	    # Try to use the title of HTML docs
	    $filename = substr($1, 0, 16);
	}
	elsif ($record->{text} =~ /(\s*<[^>]*>)*\s*([^<]*)\s*/si)
	{
	    # Use first non-html text
	    $filename = substr($2, 0, 16);
	}
	$filename =~ s/\s+$//;
	$filename =~ s/\s+/_/g;
	$filename .= ".html" if (length($filename) > 0);
    }
    unless ($filename)
    {
	$filename = substr($record->{text}, 0, 16);
	$filename =~ s/\n.*//g;
	$filename =~ s/\s+$//;
	$filename =~ s/^\s+//;
	$filename =~ s/\s+/_/g;
	$filename .= ".txt";
    }

    return($filename);
}

1;
