/* gui.c
   Implement functions of the GUI (Graphical User Interface).  
   The GUI is the X window system.  This file is the interface
   to all other X-related functions in panel.c, error.c, save.c,
   text.c, etc. */

     /*---------------------------------------------------------------*/
     /* Moog           version 0.0     27 October 1992                */
     /* Xgopher        version 1.1     20 April 1991                  */
     /*                version 1.0     04 March 1991                  */
     /*                                                               */
     /* X window system client for the University of Minnesota        */
     /*                                Internet Gopher System.        */
     /*                                                               */
     /* Martin Hamilton,  Loughborough University of Technology       */
     /*                   Department of Computer Studies              */
     /*                                                               */
     /* Allan Tuchman,    University of Illinois at Urbana-Champaign  */
     /*                   Computing Services Office                   */
     /*                                                               */
     /* Jonathan Goldman, WAIS project                                */
     /*                   Thinking Machines Corporation               */
     /*                                                               */
     /* Copyright 1992 by                                             */
     /*           the Board of Trustees of the University of Illinois */
     /* Permission is granted to freely copy and redistribute this    */
     /* software with the copyright notice intact.                    */
     /*---------------------------------------------------------------*/

#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include "conf.h"
#include "gui.h"
#include "error.h"
#include "globals.h"
#include "xglobals.h"
#include "resources.h"
#include "util.h"
#include "help.h"
#include "list.h"
#include "cso.h"
#include "index.h"
#include "gopher.h"
#include "cursor.h"
#include "xglobals.h"

Cursor wais_cursors[NUM_CURSORS];

gopherAppResources * getApplicationResources(
);


static	Widget	topLevel;
static	Boolean	windowStarted = False;

static	Boolean errorDialogExists = FALSE;
static	Boolean infoDialogExists = FALSE;
static	Boolean saveDialogExists = FALSE;
static	Boolean csoPanelExists = FALSE;
static	Boolean indexPanelExists = FALSE;

static  XtWorkProcId nullWPID = (XtWorkProcId) 0;
static  XtWorkProcId saveWPID  = (XtWorkProcId) 0,
		     indexWPID = (XtWorkProcId) 0,
		     csoWPID   = (XtWorkProcId) 0,
		     errWPID   = (XtWorkProcId) 0,
		     infoWPID  = (XtWorkProcId) 0;

static char	*emptyList[] = {"<none>", NULL};


/* initGUI
   Initialize the X window system */

BOOLEAN
initGUI(argc, argv)
int	*argc;
char	**argv;
{
	static XtPointer	scr;
	static XtConvertArgRec screenConvertArg[] = {
		{XtAddress, (XtPointer)&scr, sizeof(Screen *)}
		};

        /* initialize X window system */

        topLevel   = XtInitialize (gopherName, GOPHER_CLASS,
                        NULL, 0, argc, argv);

	scr = (XtPointer) XtScreen(topLevel);
		
	if (! getOptions(*argc, argv)) return FALSE;

	makeWaisCursors();

	makeGopherPanel(topLevel);

	XtRealizeWidget(topLevel);

	windowStarted = True;

	return TRUE;
}


/* getOptions
   Use the GUI (X window system) to access application options */

static BOOLEAN
getOptions(argc, argv)
int	argc;
char	**argv;
{
	if ((appResources = getApplicationResources(topLevel, argc, argv)) ==
									NULL) 
		return False;
	else
		return True;
}


/* makeSaveDialogWorkProc
   X work proc to create the Save Dialog during spare cycles. */

static Boolean
makeSaveDialogWorkProc(topLevel)
Widget	topLevel;
{
	makeSaveDialog(topLevel);
	saveDialogExists = TRUE;
	return TRUE;
}


/* makeInfoDialogWorkProc
   X work proc to create the Info Dialog during spare cycles. */

static Boolean
makeInfoDialogWorkProc(topLevel)
Widget	topLevel;
{
	makeInfoDialog(topLevel);
	infoDialogExists = TRUE;
	return TRUE;
}


/* makeErrorDialogWorkProc
   X work proc to create the Error Dialog during spare cycles. */

static Boolean
makeErrorDialogWorkProc(topLevel)
Widget	topLevel;
{
	makeErrorDialog(topLevel);
	errorDialogExists = TRUE;
	return TRUE;
}


/* makeCsoPanelWorkProc
   X work proc to create the CSO (ph) panel during spare cycles. */

static Boolean
makeCsoPanelWorkProc(topLevel)
Widget	topLevel;
{
	makeCsoPanel(topLevel);
	csoPanelExists = TRUE;
	return TRUE;
}


/* makeIndexWorkProc
   X work proc to create the Index Search panel during spare cycles. */

static Boolean
makeIndexWorkProc(topLevel)
Widget	topLevel;
{
	makeIndexPanel(topLevel);
	indexPanelExists = TRUE;
	return TRUE;
}


/* markCurrentDirectory
   set a bookmark at the current directory (by an event other than
   the standard click on the bookmark button) */

void
markCurrentDirectory()
{
	markProc(NULL, NULL, NULL);
}


/* doUserRequests
   The main loop in the GUI process - await user events, and process them. */

void
doUserRequests()
{
	saveWPID = 
	  XtAddWorkProc (makeSaveDialogWorkProc, (XtPointer) topLevel);
	indexWPID = 
	  XtAddWorkProc (makeIndexWorkProc, (XtPointer) topLevel);
	csoWPID = 
	  XtAddWorkProc (makeCsoPanelWorkProc, (XtPointer) topLevel);
	infoWPID = 
	  XtAddWorkProc (makeInfoDialogWorkProc, (XtPointer) topLevel);
	errWPID = 
	  XtAddWorkProc (makeErrorDialogWorkProc, (XtPointer) topLevel);

	XtMainLoop();
}


/* showError
   Cause an error message to be displayed on the user's screen. */

void
showError(message)
char    *message;
{
	if (windowStarted) {
		if (! errorDialogExists) {
			makeErrorDialog(topLevel);
			errorDialogExists = TRUE;
			if (errWPID != nullWPID) XtRemoveWorkProc (errWPID);
		}
		displayError(message, False);
	} else {
		fprintf (stderr, "NOTE!\n%s\n", message);
	}
	LOG (logFP, "ERROR:\n%s\n", message);
}


/* showFatalError
   Cause a fatal error message to be displayed on the user's screen. */

void
showFatalError(message)
char    *message;
{
	if (windowStarted) {
		if (! errorDialogExists) {
			makeErrorDialog(topLevel);
			errorDialogExists = TRUE;
			if (errWPID != nullWPID) XtRemoveWorkProc (errWPID);
		}
		displayError(message, True);
	} else {
		fprintf (stderr, "NOTE: Unrecoverable error!\n%s\n", message);
	}
	LOG (logFP, "FATAL ERROR:\n%s\n", message);
}


/* showInfo
   Cause an information message to be displayed on the user's screen. */

void
showInfo(message)
char    *message;
{
	if (windowStarted) {
		if (! infoDialogExists) {
			makeInfoDialog(topLevel);
			infoDialogExists = TRUE;
			if (infoWPID != nullWPID) XtRemoveWorkProc (infoWPID);
		}
		displayInfo(message);
	} else {
		fprintf (stderr, "NOTE!\n%s\n", message);
	}
}


/* showStatus
   indicate status of the gopher process to the user by messages, icons,
   and/or cursor changes. */

void
showStatus(message, statType)
char    *message;
int     statType;
{

	if (! windowStarted) return;

        changeStatusLabel (message, statType);
}


/* displayCurrent
   display the current directory. */

void
displayCurrent()
{
	gopherItemP	gi;
	int		i, need;
	static	int	dirStringListLen = 0;
	static	char	**dirStringList = NULL;
	gopherDirP      current = getCurrentDirectory();


	LOG(logFP,
	    "Current directory: \'%s'\n\tSelector \'%s\'\n\tat \'%s\' %d\n",
		USER_STRING(current->selectorItem),
		vStringValue(&(current->selectorItem->selector)),
		current->selectorItem->host,
		current->selectorItem->port);

	if (!windowStarted) return;
	if (current->created == NOT_LOADED) updateDirectory(current);

	/* The array of string pointers is allocated a reasonably
	   large amount initially.  After this, if more are needed,
	   the current array is freed, then the new required number
	   is allocated.  */

	need = itemListLength(&(current->contents)) + 1;
	if (need > dirStringListLen) {
		need = need > MIN_DIR_STRING_LIST_LEN ?
				need : MIN_DIR_STRING_LIST_LEN;
		if (dirStringList != NULL) free(dirStringList);
		dirStringList = NULL;
		if ((dirStringList = (char **) malloc(need * sizeof(char *)))
								== NULL) {
			showError(
"Unable to allocate sufficient memory to display this directory.");
			changeDirList(emptyList);
			return;
		}
	}


	changeDirLabel(USER_STRING(current->selectorItem));

	i = 0;
	if ((gi = current->contents.first) != NULL) {
		for ( ; gi != NULL; i++, gi = gi->next)
			dirStringList[i] = USER_STRING_PREFIX(gi);
	}

	dirStringList[i] = (char *) NULL;
	
	changeDirList(dirStringList);

	return;
}


/* displayBookmarks
   display the current bookmark list. */

void
displayBookmarks()
{
	gopherDirP	gd;
	int		i, need;
	static	int	markStringListLen = 0;
	static	char	**markStringList = NULL;

	if (!windowStarted) return;

	/* The array of string pointers is allocated a reasonably
	   large amount initially.  After this, if more are needed,
	   the current array is freed, then the new required number
	   is allocated.  */

	need = markListLength() + 1;
	if (need > markStringListLen) {
		need = need > MIN_MARK_STRING_LIST_LEN ?
				need : MIN_MARK_STRING_LIST_LEN;
		if (markStringList != NULL) free(markStringList);
		markStringList = NULL;
		if ((markStringList = (char **) malloc(need * sizeof(char *)))
								== NULL) {
			showError(
"Unable to allocate sufficient memory to display the bookmark list.");
			changeMarkList(emptyList);
			return;
		}
	}

	i = 0;
	if ((gd = firstMark()) != NULL) {
		for ( ; gd != NULL; i++, gd = gd->next)
			markStringList[i] = USER_STRING(gd->selectorItem);
	}

	markStringList[i] = (char *) NULL;
	
	changeMarkList(markStringList);

	return;
}


/* showFile
   display a file on the screen. */

void
showFile(title, fileName, indexString)
char	*title;
char	*fileName;
char	*indexString;
{

	if (! windowStarted) return;

	if (! saveDialogExists) {
		makeSaveDialog(topLevel);
		saveDialogExists = TRUE;
		if (saveWPID != nullWPID) XtRemoveWorkProc (saveWPID);
	}

	if (indexString == NULL) {
		displayTempFile(topLevel, title, fileName);
	} else {
		displayIndexTempFile(topLevel, title, fileName, indexString);
	}

	return;
}


/* showNameServer
   display a CSO name server panel. */

void
showNameServer(title, s)
char	*title;
int	s;
{

	if (! windowStarted) return;

	LOG (logFP, "CSO name server at:%s\n", title);

	if (! csoPanelExists) {
		makeCsoPanel(topLevel);
		csoPanelExists = TRUE;
		if (csoWPID != nullWPID) XtRemoveWorkProc (csoWPID);
	}
	if (! displayCsoPanel(title, s)) {
		showError(
		"Cannot display CSO Name Server\n(Is one already active?)");
	}

	return;
}


/* showIndex
   display a Index search panel. */

void
showIndex(gi)
gopherItemP	gi;
{

	if (! windowStarted) return;

	LOG (logFP, "Index search:%s\n", USER_STRING(gi));

	if (! indexPanelExists) {
		makeIndexPanel(topLevel);
		indexPanelExists = TRUE;
		if (indexWPID != nullWPID) XtRemoveWorkProc (indexWPID);
	}

	displayIndexPanel((XtPointer) gi, USER_STRING(gi));

	return;
}


/* showHelp
   show a help item in a text window */

void
showHelp(key)
char	*key;
{
        char    *string;
	char	title[HELP_SEC_TITLE_LEN];

        string = getHelpText(key, title);
        if (string == NULL)
                showError ("No help is available.");
        else if (title[0] == '\0') 
		displayTextString(topLevel, "Gopher Assistance", string);
	else 
		displayTextString(topLevel, title, string);
	
	return;
}


void
makeWaisCursors()
{
  int i;
  XColor fore, back;
  Pixmap cursor_pixmap, mask;
  Display *CurDpy = XtDisplay(topLevel);

  back.red = 65535;
  back.green = 65535;
  back.blue = 65535;

  fore.red = 0;
  fore.green = 0;
  fore.blue = 0;

  mask = XCreateBitmapFromData(XtDisplay(topLevel),
					XtScreen(topLevel)->root,
					cursor_mask, cursor_width, cursor_height);
  cursor_pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
					XtScreen(topLevel)->root,
					cursor0_bits, cursor_width, cursor_height);
  wais_cursors[0] = XCreatePixmapCursor(CurDpy, cursor_pixmap, mask, &fore, &back, 0, 0);

  cursor_pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
					XtScreen(topLevel)->root,
					cursor1_bits, cursor_width, cursor_height);
  wais_cursors[1] = XCreatePixmapCursor(CurDpy, cursor_pixmap, mask, &fore, &back, 0, 0);
  cursor_pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
					XtScreen(topLevel)->root,
					cursor2_bits, cursor_width, cursor_height);
  wais_cursors[2] = XCreatePixmapCursor(CurDpy, cursor_pixmap, mask, &fore, &back, 0, 0);
  cursor_pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
					XtScreen(topLevel)->root,
					cursor3_bits, cursor_width, cursor_height);
  wais_cursors[3] = XCreatePixmapCursor(CurDpy, cursor_pixmap, mask, &fore, &back, 0, 0);
  cursor_pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
					XtScreen(topLevel)->root,
					cursor4_bits, cursor_width, cursor_height);
  wais_cursors[4] = XCreatePixmapCursor(CurDpy, cursor_pixmap, mask, &fore, &back, 0, 0);
  cursor_pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
					XtScreen(topLevel)->root,
					cursor5_bits, cursor_width, cursor_height);
  wais_cursors[5] = XCreatePixmapCursor(CurDpy, cursor_pixmap, mask, &fore, &back, 0, 0);
  cursor_pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
					XtScreen(topLevel)->root,
					cursor6_bits, cursor_width, cursor_height);
  wais_cursors[6] = XCreatePixmapCursor(CurDpy, cursor_pixmap, mask, &fore, &back, 0, 0);
}

