/* panel.c
   make and handle callbacks for the main Xgopher panel */

     /*---------------------------------------------------------------*/
     /* Moog           version 0.0     27 October 1992                */
     /* Xgopher        version 1.1     20 April 1991                  */
     /*                version 1.0     04 March 1991                  */
     /*                                                               */
     /* X window system client for the University of Minnesota        */
     /*                                Internet Gopher System.        */
     /*                                                               */
     /* Martin Hamilton,  Loughborough University of Technology       */
     /*                   Department of Computer Studies              */
     /*                                                               */
     /* Allan Tuchman,    University of Illinois at Urbana-Champaign  */
     /*                   Computing Services Office                   */
     /*                                                               */
     /* Jonathan Goldman, WAIS project                                */
     /*                   Thinking Machines Corporation               */
     /*                                                               */
     /* Copyright 1992 by                                             */
     /*           the Board of Trustees of the University of Illinois */
     /* Permission is granted to freely copy and redistribute this    */
     /* software with the copyright notice intact.                    */
     /*---------------------------------------------------------------*/

#include <stdio.h>
#include <setjmp.h>
#include <signal.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>
#include <Xm/PanedW.h>
#include <Xm/PushB.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/Form.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>

#include "gopher.h"
#include "list.h"
#include "misc.h"
#include "help.h"
#include "resources.h"
#include "xglobals.h"
#include "globals.h"

static Widget	dirLabel, dirList,
		markLabel, markList,
		unmarkButton, unmarkAllButton,
		upButton, fetchButton,
		statusLabel,
		gopherPanel,
                gopherPanedWindow,
		topLevel;

Widget		quitButton, helpButton,
		abortButton,
		markButton;

static char	*emptyList[] = {"<none>", NULL};

static  int     previousDir=NULL, previousMark=NULL;
                busy = FALSE;

static jmp_buf jbuf;

/* timer things ... */
#define TIMEOUT 2

static int new, old;
typedef void (voidfunc)();
static void* alarm_signal;


static void
SetCursors(cursor)
Cursor cursor;
{
  Display *CurDpy = XtDisplay(topLevel);

  if(cursor != NULL) {
    XDefineCursor(CurDpy, XtWindow(markList), cursor);
    XDefineCursor(CurDpy, XtWindow(dirList), cursor);
  }
  else {
    XUndefineCursor(CurDpy, XtWindow(markList));
    XUndefineCursor(CurDpy, XtWindow(dirList));
  }
}


static void
alarmhandler(sig, code, scp, addr)
long sig, code;
struct sigcontext *scp;
char *addr;
{
  XEvent event;
  static int cursor = 0;

#ifdef __hpux
  signal(SIGALRM, alarmhandler);
#endif

  SetCursors(wais_cursors[cursor]);
  cursor = (cursor+1)%NUM_CURSORS;

  while(XtPending() != 0) {
    XtNextEvent(&event);
    XtDispatchEvent(&event);
  }
  alarm(new);
  return;
}

/* isUpOK -   decide whether the up button is useful at this point.
   It is not useful when the current directory is the root,
   otherwise it is useful.  Should be used after: push, pop, fetch (goTo)
   (push may happen elsewhere). */

void
isUpOK()
{
	static	Boolean available = True;

	if (atRoot()) 
		if (available) {	/* disable */
			XtVaSetValues(upButton, 
				      XmNsensitive, False,
				      NULL);
			available = False;
		} else ;
	else 
		if (! available) {	/* enable */
			XtVaSetValues(upButton, 
				      XmNsensitive, True,
				      NULL);
			available = True;
		}
}


/* isBookmarkOK
   decide whether the bookmark functions are useful at this point.
   They are not useful when the bookmark list is empty, otherwise 
   they are useful.
   Should be used after: mark, unmark, unmarkAll */

void
isBookmarkOK()
{
  Arg args[1];

	/* must potentially change {unmark, unmarkAll, fetch, markList} */

	if (anyMarks()) 
	  {
		    XtSetArg(args[0], XtNsensitive, True);

		    XtSetValues(unmarkButton, args, 1);
		    XtSetValues(unmarkAllButton, args, 1);
		    XtSetValues(markList, args, 1);
		    XtSetValues(markLabel, args, 1);
		    XtSetValues(markButton, args, 1);
	  }
}


static void
fuzzButtons()
{
  Arg args[1];

  XtVaSetValues(abortButton,
		XtNsensitive, True,
		NULL);

  XtSetArg(args[0], XtNsensitive, False);

  XtSetValues(upButton, args, 1);
  XtSetValues(fetchButton, args, 1);
  XtSetValues(quitButton, args, 1);
  XtSetValues(helpButton, args, 1);
  XtSetValues(markButton, args, 1);
  XtSetValues(unmarkButton, args, 1);
  XtSetValues(unmarkAllButton, args, 1);
}


static void
unfuzzButtons()
{
  Arg args[1];

  XtVaSetValues(abortButton,
		XtNsensitive, False,
		NULL);

  XtSetArg(args[0], XtNsensitive, True);
  XtSetValues(fetchButton, args, 1);
  XtSetValues(quitButton, args, 1);
  XtSetValues(helpButton, args, 1);
  XtSetValues(upButton, args, 1);

  isUpOK();
  isBookmarkOK();
}


/* dirBrowseProc
   callback for a directory browse */

void
dirBrowseProc(w, client_data, call_data)
Widget          w;
XtPointer       client_data, call_data;
{
        XmListCallbackStruct *val = (XmListCallbackStruct *) call_data;

	previousDir = val->item_position - 1;

	return;
}


/* dirSelectProc
   callback for a directory selection */

void
dirSelectProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	XmListCallbackStruct    *val = (XmListCallbackStruct *) call_data;
	gopherItemP		gi;

	if (busy)
	  return;

	XmListDeselectAllItems(markList);
	previousMark = NULL;

	if (appResources->doubleClick 
	    && previousDir == (val->item_position - 1))
	    {
		previousDir = NULL;
		XmListDeselectAllItems(dirList);

		gi = getCurrentItemN(val->item_position - 1);
		if (gi != NULL) 
		  {
		    busy = TRUE;
		    new = TIMEOUT;

		    if (setjmp(jbuf) != 0)
		      {
			busy = FALSE;
			return;
		      }

		    alarm_signal = (void *)signal(SIGALRM, alarmhandler);
		    old = alarm(new);
		    
		    fuzzButtons();
		    SetCursors(NULL);
		    processSelection(gi);
		    SetCursors(NULL);
		    unfuzzButtons();

		    alarm(old);
		    signal(SIGALRM, alarm_signal);
		    busy = FALSE;
		    
		    isUpOK();
		  } else ;
	} else 
		previousDir = val->item_position - 1;

	return;
}


/* markBrowseProc
   callback for a bookmark browse */

void 
markBrowseProc(w, client_data, call_data)
Widget          w;
XtPointer       client_data, call_data;
{
        XmListCallbackStruct    *val = (XmListCallbackStruct *) call_data;

        previousMark = val->item_position - 1;

	return;
      }


/* markSelectProc
   callback for a bookmark selection */

void
markSelectProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	XmListCallbackStruct    *val = (XmListCallbackStruct *) call_data;

	gopherDirP		gd;

	XmListDeselectAllItems(dirList);
	previousDir = NULL;

	if (appResources->doubleClick  
	    &&  previousMark == (val->item_position - 1))
	  {
		previousMark = NULL;
		XmListDeselectAllItems(markList);

		gd = getMarkN(val->item_position - 1);
		if (gd != NULL) {
			clearDirWhenOld(gd);
			goToDir(gd);
			displayCurrent();
			isUpOK();
		} else ;
	} else 
		previousMark = val->item_position - 1;

	return;
}


/* fetchProc
   callback to go to a bookmark or directory item */

void
fetchProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	gopherItemP	gi;
	gopherDirP	gd;

	if (busy) 
	  return;

	if (previousDir != NULL) {
		gi = getCurrentItemN(previousDir);
		previousDir = NULL;
		XmListDeselectAllItems(dirList);

		if (gi != NULL) 
		  {
		    busy = TRUE;
		    new = TIMEOUT;

		    if (setjmp(jbuf) != 0)
		      {
			busy = FALSE;
			unfuzzButtons();
			return;
		      }

		    alarm_signal = (void *)signal(SIGALRM, alarmhandler);
		    old = alarm(new);

		    fuzzButtons();
		    SetCursors(NULL);
		    processSelection(gi);
		    SetCursors(NULL);
		    unfuzzButtons();

		    alarm(old);
		    signal(SIGALRM, alarm_signal);
		    busy = FALSE;
		  }

		isUpOK();
	} else if (previousMark != NULL) {
		gd = getMarkN(previousMark);
		previousMark = NULL;
		if (gd != NULL) {
			XmListDeselectAllItems(markList);

			clearDirWhenOld(gd);
			goToDir(gd);
			displayCurrent();
			isUpOK();
		}
	} else {
		/* nothing selected */
		showError(
			"Something must be selected before it can be fetched");
	}
}


/* quitProc
   exit callback. */

void
quitProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	cleanUpTextProc();
	exit(0);
}


/* helpProc
   help callback. */

void
helpProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	char	*string;

	showHelp("moog help");
}


void
abortProc(w, client_data, call_data)
     Widget w;
     XtPointer client_data, call_data;
{
  if (busy)
    {
      showStatus("aborted!", GSTAT_SELECT);
      SetCursors(NULL);
      unfuzzButtons();

      longjmp(jbuf, 1);
    }
}

/* markProc
   callback to set a bookmark at the current directory */

void
markProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	markDir();
	displayBookmarks();
	isBookmarkOK();
}


/* unmarkProc
   callback to delete a bookmark */

void
unmarkProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
  int *position_list, position_count;

  if (XmListGetSelectedPos(markList, &position_list, &position_count) == FALSE)
    return;

  if ((*position_list - 1) != NULL)
    {
      previousDir = previousMark = NULL;
      unmarkDirN(*position_list - 1);

      displayBookmarks();
      isBookmarkOK();
    }
}


/* unmarkAllProc
   callback to delete all bookmarks */

void
unmarkAllProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	previousMark = NULL;

	unmarkAllDirs();
	displayBookmarks();
	isBookmarkOK();
}


/* upProc
   callback to go up a level */

void
upProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	previousDir = previousMark = NULL;
	XmListDeselectAllItems(markList);

	clearDirWhenOld(getParentDirectory());
	popDir();
	displayCurrent();
	isUpOK();
}


/* changeDirLabel
   change the directory list label */

void
changeDirLabel(newLabel)
char	*newLabel;
{
	XtVaSetValues(dirLabel, 
		      XmNlabelString,
		         XmStringCreateLtoR(newLabel, 
					    XmSTRING_DEFAULT_CHARSET),
		      NULL);
}


/* changeStatusLabel
   change the status label */

void
changeStatusLabel(newLabel, statType)
char	*newLabel;
int	statType;
{
	XtVaSetValues(statusLabel, 
		      XmNlabelString, 
		         XmStringCreateLtoR(newLabel, 
					    XmSTRING_DEFAULT_CHARSET),
		      NULL);
	XSync(XtDisplay(statusLabel), False);
}


/* changeDirList
   change the directory list contents */

void
changeDirList(list)
char	*list[];
{
  int ic = 0;

	if (list == NULL ||   list[0] == NULL)
	  XmListDeleteAllItems(dirList);
	else
	  {
	    XmListDeleteAllItems(dirList);

	    while (list[ic] != NULL)
	      {
		XmListAddItem(dirList,
			      XmStringCreateLtoR(list[ic], 
						 XmSTRING_DEFAULT_CHARSET),
			      ic + 1);
		ic++;
	      }
	    
	  }
}


/* changeMarkList
   change the bookmark list contents */

void
changeMarkList(list)
char	*list[];
{
  int ic = 0;

	if (list == NULL ||   list[0] == NULL)
	  XmListDeleteAllItems(markList);
	else
	  {
	    XmListDeleteAllItems(markList);
	    
	    while (list[ic] != NULL)
	      {
		XmListAddItem(markList,
			      XmStringCreateLtoR(list[ic], 
						 XmSTRING_DEFAULT_CHARSET),
			      ic + 1);
		ic++;
	      }
	  }
}


/* upActionProc
   implement the up action */

void
upActionProc(w, event, params, numParams)
Widget		w[];
XEvent		*event;
String		*params;
Cardinal	*numParams;
{
	if (! atRoot()) upProc(upButton, NULL, NULL);
}


/* makeGopherPanel
   create the elements of the main gopher panel */

void
makeGopherPanel(top)
Widget	top;
{
	Widget		dirFrame, dirForm, dirView,
			markFrame, markForm, markView ;
	Widget		statusForm, goBox;

	static XtActionsRec	actionsTable[] = {
				{"up",  upActionProc}
				};

	topLevel = top;

	XtAddActions(actionsTable, XtNumber(actionsTable));

	/* create main gopher panel */
	gopherPanel = XtCreateManagedWidget("gopherPanel", 
					    xmFormWidgetClass,
					    topLevel, NULL, 0);

	/* create STATUS form */
	statusForm = 
	  XtVaCreateManagedWidget("statusForm", 
				  xmFormWidgetClass, gopherPanel, 
				  XmNtopAttachment, XmATTACH_FORM,
				  XmNleftAttachment, XmATTACH_FORM,
				  XmNrightAttachment, XmATTACH_FORM,
				  NULL);

	/* create a paned window for the other components... */
	gopherPanedWindow = 
	  XtVaCreateManagedWidget("panedWindow", 
				  xmPanedWindowWidgetClass, gopherPanel, 
				  XmNtopAttachment, XmATTACH_WIDGET,
				  XmNtopWidget, statusForm,
				  XmNleftAttachment, XmATTACH_FORM,
			     	  XmNrightAttachment, XmATTACH_FORM,
				  XmNbottomAttachment, XmATTACH_FORM,
				  NULL);

	/* 3d frame for the directory */

	dirFrame = 
	  XtCreateManagedWidget("directoryFrame", 
				  xmFrameWidgetClass, gopherPanedWindow, 
				  NULL, 0);

	/* create DIRECTORY FORM */
	dirForm = XtCreateManagedWidget("directoryForm", xmFormWidgetClass,
					dirFrame, NULL, 0);

	/* 3d frame for the bookmarks */
	markFrame = 
	  XtCreateManagedWidget("bookmarkFrame", xmFrameWidgetClass,
					gopherPanedWindow, NULL, 0);

	/* create BOOKMARK FORM */
	markForm = XtCreateManagedWidget("bookmarkForm", xmFormWidgetClass,
					markFrame, NULL, 0);

	/* create GO box */
	goBox = 
	  XtVaCreateManagedWidget("goBox", 
				  xmRowColumnWidgetClass, markForm, 
				  XmNleftAttachment, XmATTACH_FORM,
				  XmNrightAttachment, XmATTACH_FORM,
				  XmNtopAttachment, XmATTACH_FORM,
				  NULL);

	/* create QUIT button */

	quitButton = XtVaCreateManagedWidget("quit", xmPushButtonWidgetClass,
					     statusForm,
					     XmNleftAttachment, XmATTACH_FORM,
					     XmNtopAttachment, XmATTACH_FORM,
					     NULL);
		XtAddCallback(quitButton, XmNactivateCallback, quitProc, NULL);

	/* create HELP button */

	helpButton = XtVaCreateManagedWidget("help", xmPushButtonWidgetClass,
					     statusForm, 
					     XmNrightAttachment, XmATTACH_FORM,
					     XmNtopAttachment, XmATTACH_FORM,
					     NULL);
		XtAddCallback(helpButton, XmNactivateCallback, helpProc, NULL);

	/* create STATUS label */

	statusLabel = XtVaCreateManagedWidget("status", xmLabelWidgetClass,
					    statusForm,
					    XmNrightAttachment, XmATTACH_FORM,
					    XmNleftAttachment, XmATTACH_FORM,
					    XmNtopAttachment, XmATTACH_WIDGET,
					    XmNtopWidget, quitButton,
					    NULL);

	/* create GOTO button */

	fetchButton = XtCreateManagedWidget("fetch", xmPushButtonWidgetClass,
					goBox, NULL, 0);
		XtAddCallback(fetchButton, XmNactivateCallback, 
			      fetchProc, NULL);

	/* create UP button */

	upButton = XtCreateManagedWidget("up", xmPushButtonWidgetClass,
					goBox, NULL, 0);
		XtAddCallback(upButton, XmNactivateCallback, upProc, NULL);


	abortButton = 
	  XtVaCreateManagedWidget("abort", xmPushButtonWidgetClass, goBox, 
				  XtNsensitive, False,
				  NULL);
	        XtAddCallback(abortButton, XmNactivateCallback, 
			      abortProc, NULL);

	/* create MARK button */

	markButton = XtCreateManagedWidget("mark", xmPushButtonWidgetClass,
					goBox, NULL, 0);
		XtAddCallback(markButton, XmNactivateCallback, markProc, NULL);

	/* create UNMARK button */

	unmarkButton = XtCreateManagedWidget("unmark", xmPushButtonWidgetClass,
					goBox, NULL, 0);
		XtAddCallback(unmarkButton, XmNactivateCallback, 
			      unmarkProc, NULL);

	/* create UNMARK ALL button */

	unmarkAllButton = XtCreateManagedWidget("unmarkAll", 
						xmPushButtonWidgetClass,
						goBox, NULL, 0);
	XtAddCallback(unmarkAllButton, XmNactivateCallback, 
		      unmarkAllProc,NULL);


	/* ============ DIRECTORY PANEL =========== */

	/* the directory panel will have this heirarchy:
	      directoryForm
		 directoryTitle
		 directoryView
		     directory
		     (scrollbar: automatic)
	*/

	/* create DIRECTORY TITLE label */

	dirLabel = XtVaCreateManagedWidget("directoryTitle", 
					   xmLabelWidgetClass, dirForm, 
					   XmNtopAttachment, XmATTACH_FORM,
					   XmNleftAttachment, XmATTACH_FORM,
					   XmNrightAttachment, XmATTACH_FORM,
					   NULL);

	/* create DIRECTORY VIEWPORT */

	dirView = XtVaCreateManagedWidget("directoryView", xmFrameWidgetClass,
					dirForm, 
					XmNtopAttachment, XmATTACH_WIDGET,
					XmNtopWidget, dirLabel,
					XmNleftAttachment, XmATTACH_FORM,
					XmNrightAttachment, XmATTACH_FORM,
					XmNbottomAttachment, XmATTACH_FORM,
					NULL);

	/* create CURRENT DIRECTORY LIST display */

	dirList = XmCreateScrolledList(dirView, "directory", NULL, 0);
	XtManageChild(dirList);

	        XtAddCallback(dirList, XmNdefaultActionCallback, 
			      dirSelectProc, NULL);
	        XtAddCallback(dirList, XmNbrowseSelectionCallback, 
			      dirBrowseProc, NULL);

	/* ============ BOOKMARK PANEL =========== */

	/* the bookmark panel will have this heirarchy:
	      bookmarkForm
		 bookmarkTitle
		 bookmarkView
		     bookmark
		     (scrollbar: automatic)
	*/

	/* create BOOKMARK TITLE label */

	markLabel = XtVaCreateManagedWidget("bookmarkTitle", 
					    xmLabelWidgetClass,	markForm, 
					    XmNtopAttachment, XmATTACH_WIDGET,
					    XmNtopWidget, goBox,
					    XmNrightAttachment, XmATTACH_FORM,
					    XmNleftAttachment, XmATTACH_FORM,
					    NULL);

	/* create BOOKMARK VIEWPORT */

	markView = XtVaCreateManagedWidget("bookmarkView", 
					   xmFrameWidgetClass, markForm, 
					   XmNtopAttachment, XmATTACH_WIDGET,
					   XmNtopWidget, markLabel,
					   XmNleftAttachment, XmATTACH_FORM,
					   XmNrightAttachment, XmATTACH_FORM,
					   XmNbottomAttachment, XmATTACH_FORM,
					   NULL);

	/* create BOOKMARK LIST display */

	markList = XmCreateScrolledList(markView, "bookmark", NULL, 0);
	XtManageChild(markList);

		XtAddCallback(markList, XmNdefaultActionCallback, 
			      markSelectProc,NULL);
	        XtAddCallback(markList, XmNbrowseSelectionCallback, 
			      markBrowseProc, NULL);

	isUpOK();
	isBookmarkOK();

	return;
}




