/* text.c
   make and manage the text popups */

     /*---------------------------------------------------------------*/
     /* Moog           version 0.0     27 October 1992                */
     /* Xgopher        version 1.1     20 April 1991                  */
     /*                version 1.0     04 March 1991                  */
     /*                                                               */
     /* X window system client for the University of Minnesota        */
     /*                                Internet Gopher System.        */
     /*                                                               */
     /* Martin Hamilton,  Loughborough University of Technology       */
     /*                   Department of Computer Studies              */
     /*                                                               */
     /* Allan Tuchman,    University of Illinois at Urbana-Champaign  */
     /*                   Computing Services Office                   */
     /*                                                               */
     /* Jonathan Goldman, WAIS project                                */
     /*                   Thinking Machines Corporation               */
     /*                                                               */
     /* Copyright 1992 by                                             */
     /*           the Board of Trustees of the University of Illinois */
     /* Permission is granted to freely copy and redistribute this    */
     /* software with the copyright notice intact.                    */
     /*---------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Text.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>

#include "conf.h"
#include "resources.h"
#include "xglobals.h"
#include "misc.h"

#define	PERCENT	'%'

typedef struct _textElement {
		Widget		textShell;
		Widget		textDisplay;
		Boolean		used;
		Boolean		deleteFile;
		char		*displayedFile;
		char		**wordList;
		char		*stringValue;
		struct _textElement *next;
	} textElement, *textElementP;

static	textElementP	textElementHead = NULL;


/* textStringToFile
   if a text widget has a string, rather than file value, copy the
   string to a file, and set the structure to reflect this so things
   will be cleaned up. */

static Boolean
textStringToFile(tep)
textElementP	tep;
{
	char	*tempfile = XtMalloc(sizeof(char) * PATH_NAME_LEN);
	FILE	*fp;

	getTempFile(tempfile);
	if ((fp = fopen (tempfile, "w")) == NULL) {
		return False;
	}
	fwrite (tep->stringValue, sizeof(char),
				strlen(tep->stringValue), fp);
	fclose(fp);
	tep->displayedFile = tempfile;
	tep->deleteFile = True;
	free(tep->stringValue);		/* free(), not XtFree() */
	tep->stringValue = NULL;

	return True;
}


/* pageDownProc
   scroll down a page in a text widget */

void
pageDownProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	Widget	tw = (Widget) client_data;

	XtCallActionProc(tw, "next-page", NULL, NULL, 0);
}


/* pageUpProc
   scroll up a page in a text widget */

void
pageUpProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	Widget	tw = (Widget) client_data;

	XtCallActionProc(tw, "previous-page", NULL, NULL, 0);
}


/* printProc
   send the contents of a text widget to the printer */

void
printProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	textElementP	tep = (textElementP) client_data;
	char		*fileName;
	char		printCommand[PATH_NAME_LEN];
	char		errorMessage[MESSAGE_STRING_LEN];
	char		*out, *in, *f;
	Boolean		hasFile = False;

	if (! appResources->allowPrint) {
		return;
	}

	if (tep->displayedFile == NULL) {
		if (! textStringToFile(tep) ) {
			showError(
"Text cannot be printed because a temporary file cannot be created.");
			return;
		}
	}

	fileName = tep->displayedFile;
	
	/* everywhere the print command string contains the letters
	   '%s', the filename is substituted.  Other uses of '%'
	   are not special.  If the print command does not contain
	   at least one '%s', then the filename is added at the end
	   after a space. */

	out = printCommand,
	in  = appResources->printCommand;
	while(*in != NULL) {
		if (*in != PERCENT) {
			*(out++) = *(in++);
		} else {
			++in;
			if (*in == 's') {
				++in;
				f = fileName;
				while (*f != NULL) {
					*(out++) = *(f++);
				}
				hasFile = True;
			} else {
				*(out++) = PERCENT;
				*(out++) = *(in++);
			}
		}
	}

	if (! hasFile) {
		*(out++) = ' ';
		f = fileName;
		while (*f != NULL) {
			*(out++) = *(f++);
		}
	}

	*(out++) = '\0';

	if (system(printCommand) == 127) {
		sprintf (errorMessage,
			"The print command could not be executed:\n%s",
			printCommand);
		showError(errorMessage);
	}

	return;
}



/* saveProc
   save the contents of a text widget to a file */

void
saveProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	textElementP	tep = (textElementP) client_data;

	if (! appResources->allowSave) {
		return;
	}

	if (tep->displayedFile == NULL) {
		if (! textStringToFile(tep) ) {
			showError(
"Text cannot be saved because a temporary file cannot be created.");
			return;
		}
	}

	if (tep == NULL) return;
	
	saveRequest(tep->textShell, tep->displayedFile);
}


/* doneProc
   Done showing text file in a popup, clean up */

void
doneProc(w, client_data, call_data)
Widget		w;
XtPointer	client_data, call_data;
{
	textElementP	tep = (textElementP) client_data;

	XtPopdown(tep->textShell);

	if (tep == NULL) return;

	if (tep->deleteFile)
		if (unlink(tep->displayedFile) != 0) {
			fprintf (stderr,
		"Warning: a gopher internal file could not be removed.\n");
       		fprintf (stderr,
			    "         This may indicate a system error.\n");
		}

	tep->used          = False;
	if (tep->displayedFile != NULL) {
		XtFree(tep->displayedFile);
		tep->displayedFile = NULL;
	}
	freeWordList(tep->wordList);
	tep->wordList      = NULL;
	tep->deleteFile    = False;
	free(tep->stringValue);		/* free(), not XtFree() */
	tep->stringValue   = NULL;

	XmTextReplace(tep->textDisplay, 0, 
		      XmTextGetLastPosition(tep->textDisplay), NULL);
	
	return;
}


/* cleanUpTextProc
   Allow a final chance to remove all the temp files text knows about. */

void
cleanUpTextProc()
{
	textElementP	tep;


	for (tep=textElementHead; tep != NULL; tep = tep->next) 
		if (tep->used && tep->deleteFile)
			(void) unlink(tep->displayedFile);
	
	return;
}


/* createTextShell
   Build a text popup */

textElementP
createTextShell(topLevel)
Widget	topLevel;
{
	Widget		doneButton, pageDownButton, pageUpButton, 
			printButton, saveButton;
	Widget		buttonBox, textForm;
	Arg		args[10];
	Cardinal	n;
	textElementP	tep;


	if ((tep = (textElementP) malloc(sizeof(textElement))) == NULL) {
	    showError("Too many windows are already open to show this file.");
	    return NULL;
	}
	tep->next = textElementHead;
	textElementHead = tep;
	tep->used = True;

	/* create TEXT display popup */
		n=0;
	        XtSetArg(args[n], XtNtitle, "Gopher Text");  n++;
	tep->textShell = XtCreatePopupShell("textShell",
					applicationShellWidgetClass,
					topLevel, args, n);
	
	
	/* create TEXT FORM */
	textForm = XtCreateManagedWidget("textForm", xmFormWidgetClass,
					tep->textShell, NULL, 0);

	/* create BUTTON BOX */
	buttonBox = 
	  XtVaCreateManagedWidget("textButtonBox", xmRowColumnWidgetClass,
				  textForm, 
				  XmNtopAttachment, XmATTACH_FORM,
				  XmNleftAttachment, XmATTACH_FORM,
				  XmNrightAttachment, XmATTACH_FORM,
				  NULL);

	/* create TEXT display */

	XtSetArg(args[n], XmNeditMode, XmMULTI_LINE_EDIT); n++;
	XtSetArg(args[n], XmNeditable, False); n++;
	XtSetArg(args[n], XmNscrollBarDisplayPolicy, XmAS_NEEDED); n++;
	XtSetArg(args[n], XmNwordWrap, True); n++;
	XtSetArg(args[n], XmNrows, 24); n++;
	XtSetArg(args[n], XmNcolumns, 80); n++;

	tep->textDisplay = XmCreateScrolledText(textForm, "textDisplay",
						args, n);

	XtManageChild(tep->textDisplay);

	XtVaSetValues(XtParent(tep->textDisplay),
			       XmNtopAttachment, XmATTACH_WIDGET,
			       XmNtopWidget, buttonBox,
			       XmNleftAttachment, XmATTACH_FORM,
			       XmNrightAttachment, XmATTACH_FORM,
			       XmNbottomAttachment, XmATTACH_FORM,
			       NULL);

	/* create DONE button */	
	doneButton = XtCreateManagedWidget("textDone", xmPushButtonWidgetClass,
					buttonBox, NULL, 0);
		XtAddCallback(doneButton, XmNactivateCallback, doneProc, tep);

	/* create PAGE_DOWN button */
	pageDownButton = XtCreateManagedWidget("textPageDown",
					xmPushButtonWidgetClass,
					buttonBox, NULL, 0);
		XtAddCallback(pageDownButton, XmNactivateCallback,
					pageDownProc, tep->textDisplay);

	/* create PAGE_UP button */
	
	pageUpButton = XtCreateManagedWidget("textPageUp", 
					     xmPushButtonWidgetClass,
					     buttonBox, NULL, 0);
		XtAddCallback(pageUpButton, XmNactivateCallback,
					pageUpProc, tep->textDisplay);

	/* create PRINT button */
	
	if (appResources->allowPrint) {
		printButton = XtCreateManagedWidget("textPrint",
						xmPushButtonWidgetClass,
						buttonBox, NULL, 0);
			XtAddCallback(printButton, XmNactivateCallback,
						printProc, tep);
	}

	/* create SAVE button */
	
	if (appResources->allowSave) {
		saveButton = XtCreateManagedWidget("textSave",
						xmPushButtonWidgetClass,
						buttonBox, NULL, 0);
			XtAddCallback(saveButton, XmNactivateCallback,
						saveProc, tep);
	}

	return tep;
}


/* displayTempFile
   Load a text popup with a temp file */

void
displayTempFile(topLevel, title, fileName)
Widget	topLevel;
char	*title;
char	*fileName;
{
	Cardinal	n;
	char		*textFileName;
	textElementP	tep;
	FILE            *fd;
	char            buff[512];

	for (tep=textElementHead; (tep!=NULL && tep->used); tep = tep->next);

	if (tep == NULL) {
		if ((tep = createTextShell(topLevel)) == NULL) return;
	}

	if ((textFileName = (char *) XtMalloc(strlen(fileName)+1)) ==
								NULL) {
		showError("Unable to allocate additional memory.");
		return;
	}
	strcpy(textFileName, fileName);
	tep->displayedFile = textFileName;
	tep->deleteFile    = True;
	tep->used = True;
	tep->wordList = NULL;
	tep->stringValue = NULL;

	/* set title and file name */

	XtVaSetValues(tep->textShell,
		      XtNtitle, title,
		      NULL);

	fd = fopen(textFileName, "r");

	n = 0;
	while (fgets(buff, sizeof(buff), fd) != NULL)
	  {
	    XmTextReplace(tep->textDisplay, n, n, buff);
	    n += strlen(buff);
	  }
	
	fclose(fd);

	XtPopup(tep->textShell, XtGrabNone);
	return;
}


/* displayIndexTempFile
   Load a text popup with a temp file and index words for highlighting */

void
displayIndexTempFile(topLevel, title, fileName, indexString)
Widget	topLevel;
char	*title;
char	*fileName;
char	*indexString;
{
	Cardinal	n;
	char		*textFileName;
	textElementP	tep;
	FILE            *fd;
	char            buff[512];

	for (tep=textElementHead; (tep!=NULL && tep->used); tep = tep->next);

	if (tep == NULL) {
		if ((tep = createTextShell(topLevel)) == NULL) return;
	}

	if ((textFileName = (char *) XtMalloc(strlen(fileName)+1)) ==
								NULL) {
		showError("Unable to allocate additional memory.");
		return;
	}
	strcpy(textFileName, fileName);
	tep->displayedFile = textFileName;
	tep->deleteFile    = True;
	tep->used = True;
	tep->stringValue = NULL;

	/* set title and file name */

	XtVaSetValues(tep->textShell,
		      XtNtitle, title,
		      NULL);

	fd = fopen(textFileName, "r");

	n = 0;
	while (fgets(buff, sizeof(buff), fd) != NULL)
	  {
	    XmTextReplace(tep->textDisplay, n, n, buff);
	    n += strlen(buff);
	  }

	fclose(buff);

	XtPopup(tep->textShell, XtGrabNone);
	return;
}


/* displayTextString
   Load a text popup with the contents of a text string */

void
displayTextString(topLevel, title, string)
Widget	topLevel;
char	*title;
char	*string;
{
	textElementP	tep;

	for (tep=textElementHead; (tep!=NULL && tep->used); tep = tep->next);

	if (tep == NULL) {
		if ((tep = createTextShell(topLevel)) == NULL) return;
	}

	tep->displayedFile = NULL;
	tep->deleteFile    = False;
	tep->used	   = True;
	tep->wordList	   = NULL;
	tep->stringValue   = string;

	/* set title and file name */

	XtVaSetValues(tep->textShell,
		      XtNtitle, title,
		      NULL);

	XmTextReplace(tep->textDisplay, 0, 0, string);

	XtPopup(tep->textShell, XtGrabNone);
	return;
}


