/* Parse command line, set up command arguments Unix-style, and call function.
 * Note: argument is modified (delimiters are overwritten with nulls)
 * Improved error handling by Brian Boesch of Stanford University
 */
#include <stdio.h>
#include "global.h"
#include "cmdparse.h"
#include "proc.h"

struct boolcmd {
	char *str;	/* Token */
	int val;	/* Value */
};

static struct boolcmd Boolcmds[] = {
	"y",		1,	/* Synonyms for "true" */
	"yes",		1,
	"true",		1,
	"on",		1,
	"1",		1,
	"set",		1,
	"enable",	1,

	"n",		0,	/* Synonyms for "false" */
	"no",		0,
	"false",	0,
	"off",		0,
	"0",		0,
	"clear",	0,
	"disable",	0,
	NULLCHAR
};

int
cmdparse(cmds,line,p)
struct cmds cmds[];
register char *line;
void *p;
{
	struct cmds *cmdp;
	char *argv[NARG],*cp;
	char **pargv;
	int argc,qflag,i;

	/* Remove cr/lf */
	rip(line);

	for(argc = 0;argc < NARG;argc++)
		argv[argc] = NULLCHAR;

	for(argc = 0;argc < NARG;){
		qflag = 0;
		/* Skip leading white space */
		while(*line == ' ' || *line == '\t')
			line++;
		if(*line == '\0')
			break;
		/* Check for quoted token */
		if(*line == '"'){
			line++;	/* Suppress quote */
			qflag = 1;
		}
		argv[argc++] = line;	/* Beginning of token */
		/* Find terminating delimiter */
		if(qflag){
			/* Find quote, it must be present */
			if((line = strchr(line,'"')) == NULLCHAR){
				return -1;
			}
			*line++ = '\0';
		} else {
			/* Find space or tab. If not present,
			 * then we've already found the last
			 * token.
			 */
			if((cp = strchr(line,' ')) == NULLCHAR
			 && (cp = strchr(line,'\t')) == NULLCHAR){
				break;
			}
			*cp++ = '\0';
			line = cp;
		}
	}
	if (argc < 1) {		/* empty command line */
		argc = 1;
		argv[0] = "";
	}
	/* Lines beginning with "#" are comments */
	if(argv[0] == NULLCHAR || argv[0][0] == '#')
		return 0;

	/* Look up command in table; prefix matches are OK */
	for(cmdp = cmds;cmdp->name != NULLCHAR;cmdp++){
		if(strncmp(argv[0],cmdp->name,strlen(argv[0])) == 0)
			break;
	}
	if(cmdp->name == NULLCHAR) {
		if(cmdp->argc_errmsg != NULLCHAR) 
			tprintf("%s\n",cmdp->argc_errmsg);
		return -1;
	} else {
		if(argc < cmdp->argcmin) {
			/* Insufficient arguments */
			tprintf("Usage: %s\n",cmdp->argc_errmsg);
			return -1;
		} else {
			if(cmdp->stksize == 0){
				return (*cmdp->func)(argc,argv,p);
			} else {
				/* Make private copy of argv and args,
				 * spawn off subprocess and return.
				 */
				pargv = (char **)callocw(argc,sizeof(char *));
				for(i=0;i<argc;i++)
					pargv[i] = strdup(argv[i]);
				newproc(cmdp->name,cmdp->stksize,
				(void (*)())cmdp->func,argc,pargv,p);
				return(0);
			}
		}
	}
}

/* Call a subcommand based on the first token in an already-parsed line */
int
subcmd(tab,argc,argv,p)
struct cmds tab[];
int argc;
char *argv[];
void *p;
{
	register struct cmds *cmdp;
	char **pargv;
	int found = 0;
	int i;

	/* Strip off first token and pass rest of line to subcommand */
	if (argc < 2) {
		if (argc < 1)
			tprintf("SUBCMD - Don't know what to do?\n");
		else
			tprintf("\"%s\" - takes at least one argument\n",argv[0]);
		return -1;
	}
	argc--;
	argv++;
	for(cmdp = tab;cmdp->name != NULLCHAR;cmdp++){
		if(strncmp(argv[0],cmdp->name,strlen(argv[0])) == 0){
			found = 1;
			break;
		}
	}
	if(!found){
		tprintf("valid subcommands:");
		for(cmdp = tab;cmdp->name != NULLCHAR;cmdp++)
			if(tprintf(" %s",cmdp->name) == EOF)
				return -1;
		tprintf("\n");
		return -1;
	}
	if(argc < cmdp->argcmin){
		if(cmdp->argc_errmsg != NULLCHAR)
			tprintf("Usage: %s\n",cmdp->argc_errmsg);
		return -1;
	}
	if(cmdp->stksize == 0){
		return (*cmdp->func)(argc,argv,p);
	} else {
		/* Make private copy of argv and args */
		pargv = (char **)callocw(argc,sizeof(char *));
		for(i=0;i<argc;i++)
			pargv[i] = strdup(argv[i]);
		newproc(cmdp->name,cmdp->stksize,
		 (void (*)())cmdp->func,argc,pargv,p);
		return(0);
	}
}
/* Must be called by all command processes before they exit */
void
freeargs(argc,argv)
int argc;
char *argv[];
{
	char **pargv;

	pargv = argv;
	while(argc-- != 0)
		free(*argv++);
	free(pargv);
}

/* Subroutine for setting and displaying boolean flags */
int
setbool(var,label,argc,argv)
int *var;
char *label;
int argc;
char *argv[];
{
	struct boolcmd *bc;

	if(argc < 2){
		tprintf("%s: %s\n",label,*var ? "on":"off");
		return 0;
	}
	for(bc = Boolcmds;bc->str != NULLCHAR;bc++){
		if(strcmpi(argv[1],bc->str) == 0){
			*var = bc->val;
			return 0;
		}
	}
	tprintf("Valid options:");
	for(bc = Boolcmds;bc->str != NULLCHAR;bc++)
		if(tprintf(" %s",bc->str) == EOF)
			return 1;
	tprintf("\n");
	return 1;
}
/* Subroutine for setting and displaying long variables */
int
setlong(var,label,argc,argv)
long *var;
char *label;
int argc;
char *argv[];
{
	if(argc < 2)
		tprintf("%s: %ld\n",label,*var);
	else
		*var = atol(argv[1]);

	return 0;
}
/* Subroutine for setting and displaying short variables */
int
setshort(var,label,argc,argv)
unsigned short *var;
char *label;
int argc;
char *argv[];
{
	if(argc < 2)
		tprintf("%s: %u\n",label,*var);
	else
		*var = atoi(argv[1]);

	return 0;
}
/* Subroutine for setting and displaying integer variables */
int
setint(var,label,argc,argv)
int *var;
char *label;
int argc;
char *argv[];
{
	if(argc < 2)
		tprintf("%s: %u\n",label,*var);
	else
		*var = atoi(argv[1]);

	return 0;
}

