#include <stdio.h>
#include <ctype.h>
#include "global.h"
#include "mbuf.h"
#include "timer.h"
#include "netuser.h"
#include "socket.h"
#include "cmdparse.h"
#include "proc.h"
#include "domain.h"
#include "commands.h"

static struct rr *Dcache = NULLRR;	/* Cache of resource records */
static int Dcache_size = 10;		/* default size limit of cache */
static struct dserver *Dlist = NULLDOM;	/* List of potential servers */
static char *Dsuffix = NULLCHAR;	/* Default suffix for names without periods */
static int Ddebug = 0;
static char *Dtypes[] = {
	"",
	"A",
	"NS",
	"MD",
	"MF",
	"CNAME",
	"SOA",
	"MB",
	"MG",
	"MR",
	"NULL",
	"WKS",
	"PTR",
	"HINFO",
	"MINFO",
	"MX",
	"TXT"
};
static int Ndtypes = 17;
static char delim[] = " \t\r\n";

static void dlist_add __ARGS((struct dserver *dp));
static void dlist_drop __ARGS((struct dserver *dp));
static int doadds __ARGS((int argc,char *argv[],void *p));
static int dodropds __ARGS((int argc,char *argv[],void *p));
static int dolistds __ARGS((int argc,char *argv[],void *p));
static int dosuffix __ARGS((int argc,char *argv[],void *p));
static int dodtrace __ARGS((int argc,char *argv[],void *p));
static int doshowcache __ARGS((int argc,char *argv[],void *p));
static int dosizecache __ARGS((int argc,char *argv[],void *p));

static int compare_rr __ARGS((struct rr *srrp,struct rr *rrp));
static int compare_rr_list __ARGS((struct rr *srrlp,struct rr *rrp));
static struct rr *copy_rr __ARGS((struct rr *rrp));
static void free_rr __ARGS((struct rr *rrp));
static void free_rr_list __ARGS((struct rr *rrlp));
static struct rr *make_rr __ARGS((int action,
	char *dname,int16 class,int16 type,int32 ttl,int16 rdl,void *data));

static void dcache_add __ARGS((struct rr *rrp));
static void dcache_drop __ARGS((struct rr *rrp));
static void dcache_post __ARGS((struct rr *rrp));
static struct rr *dcache_search __ARGS((struct rr *srrlp));

static struct rr *getrr __ARGS((FILE *fp));
static void putrr __ARGS((FILE *fp,struct rr *rrp));
static void dfile_post __ARGS((struct dhdr *dhp));
static struct rr *dfile_search __ARGS((struct rr *srrlp));

static void dumpdomain __ARGS((struct dhdr *dhp,int32 rtt));
static int dns_makequery __ARGS((int16 op,struct rr *srrp,
	char *buffer,int16 buflen));
static int dns_sendquery __ARGS((int32 server,int32 timeout,
	struct rr *srrp,struct mbuf **bpp));
static struct rr *dns_query __ARGS((struct rr *srrp, int post));

static int isaddr __ARGS((char *s));

/**
 **	Domain Server List Utilities
 **/

static struct cmds Dcmds[] = {
	"addserver",	doadds,		0, 2, NULLCHAR,
	"dropserver",	dodropds,	0, 2, NULLCHAR,
	"listservers",	dolistds,	0, 0, NULLCHAR,
	"showcache",	doshowcache,	0, 0, NULLCHAR,
	"sizecache",	dosizecache,	0, 0, NULLCHAR,
	"suffix",	dosuffix,	0, 0, NULLCHAR,
	"trace",	dodtrace,	0, 0, NULLCHAR,
	NULLCHAR,
};

static int
dodtrace(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	return setbool(&Ddebug,"Domain trace",argc,argv);
}
int
dodomain(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	return subcmd(Dcmds,argc,argv,p);
}
static int
dosuffix(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	if(argc < 2){
		if(Dsuffix != NULLCHAR)
			tprintf("%s\n",Dsuffix);
		return 0;
	}
	free(Dsuffix);
	Dsuffix = strdup(argv[1]);
	return 0;
}

static void
dlist_add(dp)
register struct dserver *dp;
{
	dp->prev = NULLDOM;
	dp->next = Dlist;
	if(Dlist != NULLDOM)
		Dlist->prev = dp;
	Dlist = dp;
}

static void
dlist_drop(dp)
register struct dserver *dp;
{
	if(dp->prev != NULLDOM)
		dp->prev->next = dp->next;
	else
		Dlist = dp->next;
	if(dp->next != NULLDOM)
		dp->next->prev = dp->prev;
}

static int
doadds(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct dserver *dp;
	int32 address;

	if((address = resolve(argv[1])) == 0){
		tprintf("Resolver %s unknown\n",argv[1]);
		return 1;
	}
	dp = (struct dserver *)callocw(1,sizeof(struct dserver));
	dp->address = address;
	dp->srtt = (5L * 1000) / MSPTICK; /* About 5 sec */
	dp->mdev = 0;
	dp->timeout = 2 * dp->mdev + dp->srtt + 3;
	dlist_add(dp);
	return 0;
}
static int
dodropds(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct dserver *dp;
	int32 addr;

	addr = resolve(argv[1]);
	for(dp = Dlist;dp != NULLDOM;dp = dp->next)
		if(addr == dp->address)
			break;

	if(dp == NULLDOM){
		tprintf("Not found\n");
		return 1;
	}

	dlist_drop(dp);
	free((char *)dp);
	return 0;
}
static int
dolistds(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	register struct dserver *dp;

	tprintf("Server address      srtt    mdev    timeout\
 queries responses\n");
	for(dp = Dlist;dp != NULLDOM;dp = dp->next){
		tprintf("%-20s%-8lu%-8lu%-8lu%-8lu%-8lu\n",
		 inet_ntoa(dp->address),
		 dp->srtt * MSPTICK,
		 dp->mdev * MSPTICK,
		 dp->timeout * MSPTICK,
		 dp->queries,dp->responses);
	}
	return 0;
}

static int
doshowcache(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct rr *rrp;

	rflush();
	for(rrp=Dcache;rrp!=NULLRR;rrp=rrp->next)
	{
		putrr(stdout,rrp);
	}
	return 0;
}

static int
dosizecache(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	return setint( &Dcache_size, "Resource Cache Size", argc,argv );
}


/**
 **	Domain Resource Record Utilities
 **/

/* Compare two resource records, returning 0 if equal, nonzero otherwise */
static int
compare_rr(srrp,rrp)
register struct rr *srrp,*rrp;
{
	int i;

	if(srrp == NULLRR || rrp == NULLRR)
		return -1;

	if(srrp->action == RR_SKIP)
		return 4;
	if(srrp->class != rrp->class)
		return 3;
	if(srrp->type != rrp->type
	&& !(srrp->action == RR_QUERY && rrp->type == TYPE_CNAME))
		return 2;

	if(srrp->action == RR_NONE || srrp->action == RR_QUERY){
		if((i = strlen(srrp->name)) != strlen(rrp->name))
			return 1;
		if((i = strnicmp(srrp->name,rrp->name,i)) != 0)
			return i;
	}
	if(srrp->action == RR_QUERY)
		return 0;

	/* Note: rdlengths are not compared because they vary depending
	 * on the representation (ASCII or encoded) this record was
	 * generated from.
	 */
	if(srrp->rdlength == 0 || rrp->rdlength == 0)
		return -2;

	switch(srrp->type){
	case TYPE_A:
		i = srrp->rdata.addr != rrp->rdata.addr;
		break;
	case TYPE_SOA:
		i = srrp->rdata.soa.serial != rrp->rdata.soa.serial;
		break;
	case TYPE_HINFO:
		i = strcmp(srrp->rdata.hinfo.cpu,rrp->rdata.hinfo.cpu) ||
			strcmp(srrp->rdata.hinfo.os,rrp->rdata.hinfo.os);
		break;
	case TYPE_MX:
		i = stricmp(srrp->rdata.mx.exch,rrp->rdata.mx.exch);
		break;
	case TYPE_MB:
	case TYPE_MG:
	case TYPE_MR:
	case TYPE_NULL:
	case TYPE_WKS:
	case TYPE_PTR:
	case TYPE_MINFO:
	case TYPE_TXT:
	case TYPE_NS:
	case TYPE_MD:
	case TYPE_MF:
	case TYPE_CNAME:
		i = stricmp(srrp->rdata.data,rrp->rdata.data);
		break;
	}
	return i;
}

static int
compare_rr_list(srrlp,rrp)
register struct rr *srrlp,*rrp;
{
	while(srrlp != NULLRR){
		if(compare_rr(srrlp,rrp) == 0)
			return 0;
		srrlp = srrlp->next;
	}
	return -255;
}

/* Make a new copy of a resource record */
static struct rr *
copy_rr(rrp)
register struct rr *rrp;
{
	register struct rr *newrr;

	if(rrp == NULLRR)
		return NULLRR;

	newrr = (struct rr *)callocw(1,sizeof(struct rr));
	newrr->action =	rrp->action;
	newrr->name =	strdup(rrp->name);
	newrr->type =	rrp->type;
	newrr->class =	rrp->class;
	newrr->ttl =	rrp->ttl;
	if((newrr->rdlength = rrp->rdlength) == 0)
		return newrr;

	switch(rrp->type){
	case TYPE_CNAME:
	case TYPE_MB:
	case TYPE_MG:
	case TYPE_MR:
	case TYPE_NS:
	case TYPE_PTR:
	case TYPE_TXT:
		newrr->rdata.name = strdup(rrp->rdata.name);
		break;
	case TYPE_A:
		newrr->rdata.addr = rrp->rdata.addr;
		break;
	case TYPE_HINFO:
		newrr->rdata.hinfo.cpu = strdup(rrp->rdata.hinfo.cpu);
		newrr->rdata.hinfo.os = strdup(rrp->rdata.hinfo.os);
		break;
	case TYPE_MX:
		newrr->rdata.mx.pref = rrp->rdata.mx.pref;
		newrr->rdata.mx.exch = strdup(rrp->rdata.mx.exch);
		break;
	case TYPE_SOA:
		newrr->rdata.soa.mname = 	strdup(rrp->rdata.soa.mname);
		newrr->rdata.soa.rname = 	strdup(rrp->rdata.soa.rname);
		newrr->rdata.soa.serial = 	rrp->rdata.soa.serial;
		newrr->rdata.soa.refresh = 	rrp->rdata.soa.refresh;
		newrr->rdata.soa.retry = 	rrp->rdata.soa.retry;
		newrr->rdata.soa.expire = 	rrp->rdata.soa.expire;
		newrr->rdata.soa.minimum = 	rrp->rdata.soa.minimum;
		break;
	}
	return newrr;
}

/* Free a resource record */
static void
free_rr(rrp)
register struct rr *rrp;
{
	if(rrp == NULLRR)
		return;
	free(rrp->name);
	if(rrp->rdlength != 0){
		switch(rrp->type){
		case TYPE_CNAME:
		case TYPE_MB:
		case TYPE_MG:
		case TYPE_MR:
		case TYPE_NS:
		case TYPE_PTR:
		case TYPE_TXT:
			free(rrp->rdata.name);
			break;
		case TYPE_A:
			break;	/* Nothing allocated in rdata section */
		case TYPE_HINFO:
			free(rrp->rdata.hinfo.cpu);
			free(rrp->rdata.hinfo.os);
			break;
		case TYPE_MX:
			free(rrp->rdata.mx.exch);
			break;
		case TYPE_SOA:
			free(rrp->rdata.soa.mname);
			free(rrp->rdata.soa.rname);
			break;
		}
	}
	free((char *)rrp);
}

static void
free_rr_list(rrlp)
register struct rr *rrlp;
{
	register struct rr *save_rrp;

	while((save_rrp = rrlp) != NULLRR){
		rrlp = rrlp->next;
		free_rr(save_rrp);
	}
}

static struct rr *
make_rr(action,dname,dclass,dtype,ttl,rdl,data)
int action;
char *dname;
int16 dclass;
int16 dtype;
int32 ttl;
int16 rdl;
void *data;
{
	register struct rr *newrr;

	newrr = (struct rr *)callocw(1,sizeof(struct rr));
	newrr->action = action;
	newrr->name = strdup(dname);
	newrr->class = dclass;
	newrr->type = dtype;
	newrr->ttl = ttl;
	if((newrr->rdlength = rdl) == 0)
		return newrr;

	switch(dtype){
	case TYPE_CNAME:
	case TYPE_MB:
	case TYPE_MG:
	case TYPE_MR:
	case TYPE_NS:
	case TYPE_PTR:
	case TYPE_TXT:
	  {
		newrr->rdata.name = strdup((char *)data);
		break;
	  }
	case TYPE_A:
	  {
		register int32 *ap = (int32 *)data;
		newrr->rdata.addr = *ap;
		break;
	  }
	case TYPE_HINFO:
	  {
		register struct hinfo *hinfop = (struct hinfo *)data;
		newrr->rdata.hinfo.cpu = strdup(hinfop->cpu);
		newrr->rdata.hinfo.os = strdup(hinfop->os);
		break;
	  }
	case TYPE_MX:
	  {
		register struct mx *mxp = (struct mx *)data;
		newrr->rdata.mx.pref = mxp->pref;
		newrr->rdata.mx.exch = strdup(mxp->exch);
		break;
	  }
	case TYPE_SOA:
	  {
		register struct soa *soap = (struct soa *)data;
		newrr->rdata.soa.mname = 	strdup(soap->mname);
		newrr->rdata.soa.rname = 	strdup(soap->rname);
		newrr->rdata.soa.serial = 	soap->serial;
		newrr->rdata.soa.refresh = 	soap->refresh;
		newrr->rdata.soa.retry = 	soap->retry;
		newrr->rdata.soa.expire = 	soap->expire;
		newrr->rdata.soa.minimum = 	soap->minimum;
		break;
	  }
	}
	return newrr;
}

/**
 **	Domain Cache Utilities
 **/

static void
dcache_add(rrp)
register struct rr *rrp;
{
	rrp->last = NULLRR;
	rrp->next = Dcache;
	if(Dcache != NULLRR)
		Dcache->last = rrp;
	Dcache = rrp;
}

static void
dcache_drop(rrp)
register struct rr *rrp;
{
	if(rrp->last != NULLRR)
		rrp->last->next = rrp->next;
	else
		Dcache = rrp->next;
	if(rrp->next != NULLRR)
		rrp->next->last = rrp->last;
}

/* Prepend a non-duplicate resource record to the local cache */
static void
dcache_post(rrp)
register struct rr *rrp;
{
	int count;

	if(rrp == NULLRR)
		return;

	rrp->action = RR_NONE;
	dcache_add(rrp);

	/* ensure we don't already have it on the list */
	for(count = 0; ++count <= Dcache_size;){
		if((rrp = rrp->next) == NULLRR)
			break;
		if(compare_rr(Dcache,rrp) == 0){
			dcache_drop(rrp);
			free_rr(rrp);
			rrp = NULLRR;
			break;
		}
	}

	/* remove any extra off the end */
	if(count > Dcache_size && rrp != NULLRR){
		if(rrp->last != NULLRR)
			rrp->last->next = NULLRR;
		free_rr_list(rrp);
	}
}

/* Search local cache for resource record.
 * Returns NULLRR if record not found.
 */
static struct rr *
dcache_search(srrlp)
register struct rr *srrlp;
{
	register struct rr *rrp;

	for(rrp = Dcache; rrp != NULLRR; rrp = rrp->next){
		if(compare_rr_list(srrlp,rrp) == 0){
			if(rrp->last != NULLRR){
				dcache_drop(rrp);
				dcache_add(rrp);
			}
			break;
		}
	}
	return copy_rr(rrp);
}

/**
 **	File Utilities
 **/

static struct rr *
getrr(fp)
FILE *fp;
{
	char *line,*strtok();
	struct rr *rrp;
	char *name,*ttl,*class,*type,*data;
	int i;

	line = mallocw(256);
	/* Search file */
	while(fgets(line,256,fp),!feof(fp)){
		if(line[0] != '#')
			break;
	}
	if(feof(fp)){
		free(line);
		return NULLRR;
	}
	rrp = (struct rr *)callocw(1,sizeof(struct rr));
	name = strtok(line,delim);
	ttl = strtok(NULLCHAR,delim);
	class = strtok(NULLCHAR,delim);
	type = strtok(NULLCHAR,delim);
	data = strtok(NULLCHAR,delim);

	i = strlen(name);
	if(name[i-1] != '.'){
		/* Tack on a trailing period if it's not there */
		rrp->name = mallocw(i+2);
		strcpy(rrp->name,name);
		strcat(rrp->name,".");
	} else
		rrp->name = strdup(name);

	if(!isdigit(ttl[0])){
		/* Optional ttl field is missing; slide the other fields over */
		data = type;
		type = class;
		class = ttl;
		ttl = NULLCHAR;
	} else {
		rrp->ttl = atol(ttl);
	}
	for(i=0;i<Ndtypes;i++){
		if(strcmp(type,Dtypes[i]) == 0){
			rrp->type = i;
			break;
		}
	}
	if(strcmp(class,"IN") == 0)
		rrp->class = CLASS_IN;

	if(data == NULLCHAR){
		/* Empty record, just return */
		free(line);
		return rrp;
	}
	switch(rrp->type){
	case TYPE_CNAME:
	case TYPE_MB:
	case TYPE_MG:
	case TYPE_MR:
	case TYPE_NS:
	case TYPE_PTR:
	case TYPE_TXT:
		rrp->rdlength = strlen(data);
		rrp->rdata.name = strdup(data);
		break;
	case TYPE_A:
		rrp->rdlength = 4;
		rrp->rdata.addr = aton(data);
		break;
	case TYPE_HINFO:
		rrp->rdlength = strlen(data);
		rrp->rdata.hinfo.cpu = strdup(data);
		if((data = strtok(NULLCHAR,delim)) != NULLCHAR){
			rrp->rdlength += strlen(data);
			rrp->rdata.hinfo.os = strdup(data);
		}
		break;
	case TYPE_MX:
		rrp->rdata.mx.pref = atoi(data);
		rrp->rdlength = 2;

		/* Get domain name of exchanger */
		if((data = strtok(NULLCHAR,delim)) != NULLCHAR){
			rrp->rdlength += strlen(data);
			rrp->rdata.mx.exch = strdup(data);
		}
		break;
	case TYPE_SOA:
		/* Get domain name of master name server */
		rrp->rdlength = strlen(data);
		rrp->rdata.soa.mname = strdup(data);

		/* Get domain name of irresponsible person */
		if((data = strtok(NULLCHAR,delim)) != NULLCHAR){
			rrp->rdata.soa.rname = strdup(data);
			rrp->rdlength += strlen(data);
		}
		data = strtok(NULLCHAR,delim);
		rrp->rdata.soa.serial = atol(data);
		data = strtok(NULLCHAR,delim);
		rrp->rdata.soa.refresh = atol(data);
		data = strtok(NULLCHAR,delim);
		rrp->rdata.soa.retry = atol(data);
		data = strtok(NULLCHAR,delim);
		rrp->rdata.soa.expire = atol(data);
		data = strtok(NULLCHAR,delim);
		rrp->rdata.soa.minimum = atol(data);
		rrp->rdlength += 20;
		break;
	}
	free(line);
	return rrp;
}

/* Print a resource record */
static void
putrr(fp,rrp)
FILE *fp;
struct rr *rrp;
{
	if(fp == NULLFILE || rrp == NULLRR)
		return;

	fprintf(fp,"%s\t%lu",rrp->name,rrp->ttl);
	if(rrp->class == CLASS_IN)
		fprintf(fp,"\tIN");
	else
		fprintf(fp,"\t%u",rrp->class);
	if(rrp->type < Ndtypes)
		fprintf(fp,"\t%s",Dtypes[rrp->type]);
	else
		fprintf(fp,"\t%u",rrp->type);
	if(rrp->rdlength == 0){
		/* Null data portion, indicates nonexistent record */
		fprintf(fp,"\n");
		return;
	}
	switch(rrp->type){
	case TYPE_CNAME:
	case TYPE_MB:
	case TYPE_MG:
	case TYPE_MR:
	case TYPE_NS:
	case TYPE_PTR:
	case TYPE_TXT:
		/* These are all printable text strings */
		fprintf(fp,"\t%s\n",rrp->rdata.data);
		break;
	case TYPE_A:
		fprintf(fp,"\t%s\n",inet_ntoa(rrp->rdata.addr));
		break;
	case TYPE_MX:
		fprintf(fp,"\t%u\t%s\n",rrp->rdata.mx.pref,
		 rrp->rdata.mx.exch);
		break;
	case TYPE_SOA:
		fprintf(fp,"\t%s\t%s\t%lu\t%lu\t%lu\t%lu\t%lu\n",
		 rrp->rdata.soa.mname,rrp->rdata.soa.rname,
		 rrp->rdata.soa.serial,rrp->rdata.soa.refresh,
		 rrp->rdata.soa.retry,rrp->rdata.soa.expire,
		 rrp->rdata.soa.minimum);
		break;
	default:
		fprintf(fp,"\n");
		break;
	}
}

/* Append any non-duplicate resource records to the local file */
static void
dfile_post(dhp)
struct dhdr *dhp;
{
	FILE *fp;
	long ttl = 500;	/* Default TTL for negative records without SOA */
	struct rr *rrp,*frrp;

	pwait(NULL);/* Give up CPU for a while, this is very slow */
	fp = fopen(Dfile,APPEND_TEXT);
	if(fp == NULLFILE){
		printf("Can't append to %s!!\n",Dfile);
		return;
	}

	/* Scan domain file looking for duplicates */
	while((frrp = getrr(fp)) != NULLRR){
		for(rrp = dhp->ans; rrp != NULLRR; rrp = rrp->next){
			if(rrp->type == TYPE_SOA)
				ttl = rrp->ttl;
			if(compare_rr(rrp,frrp) == 0)
				rrp->action = RR_SKIP;
		}
		for(rrp = dhp->ns; rrp != NULLRR; rrp = rrp->next){
			if(rrp->type == TYPE_SOA)
				ttl = rrp->ttl;
			if(compare_rr(rrp,frrp) == 0)
				rrp->action = RR_SKIP;
		}
		for(rrp = dhp->add; rrp != NULLRR; rrp = rrp->next){
			if(rrp->type == TYPE_SOA)
				ttl = rrp->ttl;
			if(compare_rr(rrp,frrp) == 0)
				rrp->action = RR_SKIP;
		}
		free_rr(frrp);
	}

	/* Add negative reply to answers.  This assumes that there was
	 * only one question, which is true for all questions we send.
	 */
	if(dhp->aa && (dhp->rcode == NAME_ERROR || dhp->ancount == 0)){
		if(dhp->qlist != NULLRR){
			rrp = copy_rr(dhp->qlist);
			rrp->ttl = ttl;
			rrp->next = dhp->ans;
			dhp->ans = rrp;
		}
	}

	/* Now append any non-duplicate records */
	fseek(fp,0L,2);
	for(rrp = dhp->ans; rrp != NULLRR; rrp = rrp->next){
		if(rrp->action != RR_SKIP)
			putrr(fp,rrp);
	}
	for(rrp = dhp->ns; rrp != NULLRR; rrp = rrp->next){
		if(rrp->action != RR_SKIP)
			putrr(fp,rrp);
	}
	for(rrp = dhp->add; rrp != NULLRR; rrp = rrp->next){
		if(rrp->action != RR_SKIP)
			putrr(fp,rrp);
	}
	fclose(fp);
	pwait(NULL);/* Give up CPU for a while, this is slow */
}

/* Search local database for resource record.
 * Returns NULLRR if record not found.
 */
static struct rr *
dfile_search(srrlp)
struct rr *srrlp;
{
	register struct rr *rrp;
	FILE *dbase;

	if((dbase = fopen(Dfile,READ_TEXT)) == NULLFILE)
		return NULLRR;

	while((rrp = getrr(dbase)) != NULLRR
	   && compare_rr_list(srrlp,rrp) != 0){
		free_rr(rrp);
	}
	fclose(dbase);
	pwait(NULL);	/* Give up CPU for a while, this is slow */
	dcache_post(rrp);	/* move answer to cache */
	return copy_rr(rrp);
}

/**
 **	Domain Server Utilities
 **/

static void
dumpdomain(dhp,rtt)
struct dhdr *dhp;
int32 rtt;
{
	struct rr *rrp;

	printf("response id %u (rtt %lu sec) qr %u opcode %u aa %u tc %u rd %u ra %u rcode %u\n",
	 dhp->id,((long)rtt * MSPTICK)/1000,
	 dhp->qr,dhp->opcode,dhp->aa,dhp->tc,dhp->rd,
	 dhp->ra,dhp->rcode);
	printf("%u questions:\n",dhp->qdcount);
	for(rrp = dhp->ans; rrp != NULLRR; rrp = rrp->next){
		printf("%s type %u class %u\n",rrp->name,
		 rrp->type,rrp->class);
	}
	printf("%u answers:\n",dhp->ancount);
	for(rrp = dhp->ans; rrp != NULLRR; rrp = rrp->next){
		putrr(stdout,rrp);
	}
	printf("%u authority:\n",dhp->nscount);
	for(rrp = dhp->ns; rrp != NULLRR; rrp = rrp->next){
		putrr(stdout,rrp);
	}
	printf("%u additional:\n",dhp->arcount);
	for(rrp = dhp->add; rrp != NULLRR; rrp = rrp->next){
		putrr(stdout,rrp);
	}
	fflush(stdout);
}

static int
dns_makequery(op,srrp,buffer,buflen)
int16 op;	/* operation */
struct rr *srrp;/* Search RR */
char *buffer;	/* Area for query */
int16 buflen;	/* Length of same */
{
	char *cp,*cp1;
	char *dname, *sname;
	int16 parameter;
	int16 dlen,len;

	cp = buffer;
	cp = put16(cp,(int16)Clock);	/* Use clock for timestamping */
	parameter = (op << 11)
			| 0x0100;	/* Recursion desired */
	cp = put16(cp,parameter);
	cp = put16(cp,1);
	cp = put16(cp,0);
	cp = put16(cp,0);
	cp = put16(cp,0);

	sname = strdup(srrp->name);
	dname = sname;
	dlen = strlen(dname);
	for(;;){
		/* Look for next dot */
		cp1 = strchr(dname,'.');
		if(cp1 != NULLCHAR)
			len = cp1-dname;	/* More to come */
		else
			len = dlen;	/* Last component */
		*cp++ = len;		/* Write length of component */
		if(len == 0)
			break;
		/* Copy component up to (but not including) dot */
		strncpy(cp,dname,len);
		cp += len;
		if(cp1 == NULLCHAR){
			*cp++ = 0;	/* Last one; write null and finish */
			break;
		}
		dname += len+1;
		dlen -= len+1;
	}
	free(sname);
	cp = put16(cp,srrp->type);
	cp = put16(cp,srrp->class);
	return cp - buffer;
}
/* Send a query to a server */
static int
dns_sendquery(server,timeout,srrp,bpp)
int32 server;		/* Server's IP address */
int32 timeout;		/* Query timeout */
struct rr *srrp;	/* Search RR */
struct mbuf **bpp;	/* Place to stash reply */
{
	char *buf;
	int len;
	struct sockaddr_in server_in;
	int s,rval;

	s = socket(AF_INET,SOCK_DGRAM,0);
	buf = mallocw(512);
	len = dns_makequery(0,srrp,buf,512);
	server_in.sin_family = AF_INET;
	server_in.sin_port = IPPORT_DOMAIN;
	server_in.sin_addr.s_addr = server;
	if(Ddebug){
		printf("dns_sendquery: querying server %s for %s\n",
		 inet_ntoa(server),srrp->name);
	}
	sendto(s,buf,len,0,(char *)&server_in,sizeof(server_in));
	free(buf);
	alarm(max(timeout,100));
	/* Wait for something to happen */
	rval = recv_mbuf(s,bpp,0,NULLCHAR,0);
	alarm(0L);
	close_s(s);
	if(Ddebug){
		printf("dns_sendquery: received message length %d, \
errno %d\n", rval, errno );
	}
	return rval;
}

/* domain server resolution loop
 * returns NULLRR when no resolution available.
 *	(future features)
 * srrp is not yet supported as a list.
 * inverse queries not yet supported.
 */
static struct rr *
dns_query(srrp,post)
struct rr *srrp;
int post;
{
	struct dhdr dh;
	struct dserver *dp;	/* server list */
	int32 rtt,abserr;
	struct mbuf *bp;
	struct rr *rrp;
	struct rr *found_rr = NULLRR;

	dp = Dlist;
	for(;;){
		if(dp == NULLDOM){
			return 0;
		}

		dp->queries++;
		if(dns_sendquery(dp->address,dp->timeout,srrp,&bp) > 0)
			break;

		if(errno == EABORT){
			return 0;	/* Killed by "reset" command */
		}
		/* Timeout; back off this one and try another server */
		dp->timeout <<= 1;
		if((dp = dp->next) == NULLDOM)
			dp = Dlist;
	}

	/* got a response */
	dp->responses++;
	ntohdomain(&dh,&bp);    /* Convert to local format */

	/* Compute and update the round trip time */
	rtt = (int32) ((int16)Clock - dh.id);
	abserr = rtt > dp->srtt ? rtt - dp->srtt : dp->srtt - rtt;
	dp->srtt = ((AGAIN-1) * dp->srtt + rtt + (AGAIN/2)) >> LAGAIN;
	dp->mdev = ((DGAIN-1) * dp->mdev + abserr + (DGAIN/2)) >> LDGAIN;
	dp->timeout = 2 * dp->mdev + dp->srtt + 3;

	/* move to top of list for next time */
	if(dp->prev != NULLDOM){
		dlist_drop(dp);
		dlist_add(dp);
	}

	if(Ddebug)
		dumpdomain(&dh,rtt);
	if(post)
		dfile_post(&dh);

	/* Look for an answer in this response */
	while((rrp = dh.ans) != NULLRR){
		dh.ans = rrp->next;
		dcache_post(rrp);	/* move all answers to cache */
		if(compare_rr(srrp,rrp) == 0
		&& found_rr == NULLRR)
			found_rr = copy_rr(rrp);
	}
	free_rr_list(dh.qlist);
	free_rr_list(dh.ns);
	free_rr_list(dh.add);
	if(Ddebug)
		keywait(NULLCHAR,1);
	return found_rr;
}

/**
 **	Resolver Utilities
 **/

/* Return 1 if string appears to be an IP address in dotted decimal;
 * return 0 otherwise (i.e., if string is a domain name)
 */
static int
isaddr(s)
register char *s;
{
	char c;

	if(s == NULLCHAR)
		return 1;	/* Can't happen */

	while((c = *s++) != '\0'){
		if(!isdigit(c) && c != '.')
			return 0;
	}
	return 1;
}

/* Main entry point for domain name -> address resolution.
 * Returns 0 if name is currently unresolvable.
 */
int32
resolve(name)
char *name;
{
	register struct rr *rrp, *arrp;
	char *sname,*tmp;
	int crecurse = 0;
	int32 ip_address;
#define MISSING (0xffffffff)

	if(name == NULLCHAR)
		return 0;

	if(*name == '[')
		return aton(name + 1);

	if(isaddr(name))
		return aton(name);

	sname = strdup(name);
	if(strchr(sname,'.') == NULLCHAR && Dsuffix != NULLCHAR){
		/* Append default suffix */
		tmp = mallocw(strlen(sname)+strlen(Dsuffix)+2);
		sprintf(tmp,"%s.%s",sname,Dsuffix);
		free(sname);
		sname = tmp;
	}
	if(sname[strlen(sname)-1] != '.'){
		/* Append trailing dot */
		tmp = mallocw(strlen(sname)+2);
		sprintf(tmp,"%s.",sname);
		free(sname);
		sname = tmp;
	}

	arrp = make_rr(RR_QUERY,sname,CLASS_IN,TYPE_A,0,0,NULL);
	free(sname);

	for(ip_address = MISSING; ip_address == MISSING;){
		if((rrp=dcache_search(arrp)) == NULLRR
		&& (rrp=dfile_search(arrp)) == NULLRR
		&& (rrp=dns_query(arrp,1)) == NULLRR)
			break;
		if(rrp->rdlength == 0)
			break;
		switch(rrp->type){
		case TYPE_A:
			ip_address = rrp->rdata.addr;
			break;
		case TYPE_CNAME:
		case TYPE_PTR:
			if(++crecurse >= MAXCNAME){
				/* Too many CNAME recursions */
				ip_address = 0;
			} else {
				/* Replace name and try again */
				free(arrp->name);
				arrp->name = strdup(rrp->rdata.name);
				free_rr(rrp);
			}
			break;
		default:
			free_rr(rrp);
		}
	}
	free_rr(arrp);
	free_rr(rrp);
	if(ip_address == MISSING)
		ip_address = 0;
	return ip_address;
}

/* entry point for address -> domain name resolution.
 * Returns NULLCHAR if address is not found.
 */
char *
resolve_a(ip_address)
int32 ip_address;
{
	register struct rr *rrp, *irrp, *prrp;
	char *dname;
	char pname[30];

	if(ip_address == 0L)
		return NULLCHAR;

	sprintf( pname, "%u.%u.%u.%u.IN-ADDR.ARPA.",
			lobyte(loword(ip_address)),
			hibyte(loword(ip_address)),
			lobyte(hiword(ip_address)),
			hibyte(hiword(ip_address)) );

	irrp = make_rr(RR_INQUERY,NULLCHAR,CLASS_IN,TYPE_A,0,4,&ip_address);
	prrp = make_rr(RR_QUERY,pname,CLASS_IN,TYPE_PTR,0,0,NULL);

	irrp->next = prrp;	/* make list to speed search */

	for(dname = NULLCHAR; dname == NULLCHAR;){
		if((rrp=dcache_search(irrp)) == NULLRR
		&& (rrp=dfile_search(irrp)) == NULLRR
		&& (rrp=dns_query(prrp,1)) == NULLRR)
			break;
		if(rrp->rdlength == 0)
			break;
		switch(rrp->type){
		case TYPE_A:
			dname = strdup(rrp->name);
			break;
		case TYPE_PTR:
			dname = strdup(rrp->rdata.name);
			break;
		default:
			free_rr(rrp);
		}
	}
	free_rr(irrp);
	free_rr(prrp);
	free_rr(rrp);
	return dname;
}

