
#include  "qwik.h"

/* the following vars are used by the move_window & relocate_window()
   routines. */

static char *buf2,*h1,*h2,*h3;

/****************************************************************************
 * This module contains the window management routines.                     *
 * Modification History.                                                    *
 * Summer 87 - Original writing (as inspired by J. LeMay's Pascal package.) *
 *                                                                          *
 * 112487 - Added functions relocate_window(), move_window(),               *
 *          and change_border().                                            *
 * 12687 - Added function window_mover(), sys_move_window(),                *
 *         keyboard_handler(), and sys_move_help(), and sys_show_coords().  *
 *         NOTE: These are not yet implemented for AZTEC.                   *
 ***************************************************************************/

#ifdef TURBO
#define Shift_Bits   ((char far *) 0x00000417L)
#define Hot_Combo    ((*Shift_Bits & 12) == 12) /* Our hot key combo */

#define Keyboard 0x09


void interrupt keyboard_handler (void);
void interrupt (* old_keyboard) (); /* previous keyboard int vector */

#endif

static sys_c,sys_direction,w_mover=0;

#ifdef TURBO /* only if compiling under Turbo C */

/****************************************************
 * window_mover() toggles the keyboard vector from  *
 * the system to us and back.                       *
 ***************************************************/

window_mover(x)
int x;
{

      if(x==ON && w_mover==OFF) {
        old_keyboard = getvect(Keyboard);    /* save old vector */
        setvect(Keyboard,keyboard_handler);  /* install us as new vector */
        w_mover=ON;                          /* flag we're installed */
        return(ON);
      }

      if(x==OFF && w_mover==ON) {
         setvect(Keyboard,old_keyboard);     /* restore old vector */
         w_mover=OFF;                        /* flag we're deinstalled */
         return(OFF);
      }

      return(-1); /* passed parm not ON or OFF */

}  /* end of window_mover() */

 /****************************************************************************
 * This is what starts the whole ball rolling!  First we call the old        *
 * keyboard handler, then check for our hot key.  If our hot key has been    *
 * pressed, we toggle our internal request flag.  Next, we call the window   *
 * move driver (sys_move_window()). Upon return we clear the internal request*
 * flag and return to wherever when the first keyboard interrupt occurred.   *
 ****************************************************************************/

void interrupt keyboard_handler()
{
static int in_fl = 0;

   (*old_keyboard) (); /* call orig system handler thus allowing special
                          systems functions (like reset) to function */


   if (! in_fl)             /* if not already busy */
      {

      in_fl = 1;            /* set we're busy flag */
      if(Hot_Combo)         /* if ALT + CTRL */
         sys_move_window(); /* call the move handler */

      in_fl=0;              /* no longer busy */
      }

   return;

} /* end of keyboard_handler() */

/**********************************************************************
 * This is the actual routine which handles moving the window.  It is *
 * called by keyboard_handler().                                      *
 *********************************************************************/

sys_move_window()
{
        beep(); /* 2 beeps going in ... */
        beep();

        while((sys_c=scr_getc()) != 27) {

          switch(sys_c) {
          case home:      restore_window(); /* rstore window to orig coords */
                          continue;

          case cursor_lf: sys_direction=LF;
                          break;

          case cursor_rt: sys_direction=RT;
                          break;

          case cursor_dn: sys_direction=DN;
                          break;

          case cursor_up: sys_direction=UP;
                          break;

          case (int)'?':  sys_show_coord();  /* show present coords */
                          continue;

          default:        sys_move_help();   /* show help screen */
                          continue;
          }
          move_window(sys_direction,1);
       }
       beep();   /* 2 beeps going out. */
       beep();

} /* end of sys_move_window() */

sys_show_coord()
{
int r,c,c2,i;

       i=shadow_effect;
       shadow_effect=nodir;   /* no shadowing */

       make_window(9,25,6,30,0x70,attr(black,cyan),0);
       shadow_effect=i; /* restore shadowing state */

       title_window(center," Current Coordinates ",w[li].wsbattr,TOP);
       title_window(center," Any Key to Exit ",w[li].wsbattr,BOT);
       r=w[li].wsrow+2;
       c=w[li].wscol;
       c2=c+w[li].wscols;
       sprintf(buf,"Top left row = %2d",w[li-1].wsrow);
       qwritecv(r++,c,c2,w[li].wswattr,buf);
       sprintf(buf,"Top left col = %2d",w[li-1].wscol);
       qwritecv(r,c,c2,w[li].wswattr,buf);

       scr_getc(); /* wait for any key ... */

       remove_window();

} /* end of sys_show_coord() */

/*************************************************
 * sys_move_help() displays a help window for    *
 * the sys_move_window() function.               *
 ************************************************/

sys_move_help()
{
int i,r,c;
char bbuf[40];

         i=shadow_effect;
         shadow_effect=nodir;
         make_window(9,15,8,50,0x70,attr(black,cyan),0);
         shadow_effect=i;
         title_window(center," Window-Mover Help ",w[li].wsbattr,TOP);
         title_window(center," Any Key to Exit Help ",w[li].wsbattr,BOT);
         r=w[li].wsrow+2;
         c=w[li].wscol+5;

         bbuf[0]=24; /* insert arrow chars */
         bbuf[1]=25;
         bbuf[2]=26;
         bbuf[3]=27;

         strcpy(&bbuf[4]," - Move Current Window");
         qwrite(r++,c,w[li].wswattr,bbuf);
         qwrite(r++,c,w[li].wswattr,\
                  "HOME - Restore Window to Original Position");
         qwrite(r++,c,w[li].wswattr,\
                  "   ? - Display current coordinates");
         qwrite(r++,c,w[li].wswattr," ESC - Exit Window Move Utility");
         scr_getc();
         remove_window();

} /* end of sys_move_help() */

#endif /* above functions compile only under Turbo C for now */

/*********************************************
 * restore_window() moves the current window *
 * to the screen coords where it was         *
 * originally opened.                        *
 ********************************************/

restore_window()
{
         if((w[li].wsrow != w[li].orig_row) ||
            (w[li].wscol != w[li].orig_col))
           relocate_window(w[li].orig_row,w[li].orig_col);

} /* end of restore_window() */

/****************************************************
 * change_border() redraws the border of the window *
 * currently pointed to by li with a new border type*
 * Titles are preserved.                            *
 * There is a bug in this when the present border   *
 * type is 'blank' or 'none'.                       *
 * *************************************************/

change_border(border)
int border;
{
int r,attr;
register int i;
char ct,cb,clf,crt;

         attr=w[li].wsbattr;
         ct=brdr[w[li].wsbrdr][THC];  /* current horiz line types */
         cb=brdr[w[li].wsbrdr][BHC];  /*    "      "    "     "   */

         get_addr(w[li].wsrow,w[li].wscol); /* top left of window */
         *vid_ram++ = brdr[border][TLC];    /* set corner char */
         *vid_ram++ = attr;                 /* & attrib */

         for(i=0;i<w[li].wscols-2;i++) {    /* do rest of top line while */

           if(*vid_ram == ct) {  /* skipping any non-line */
             *vid_ram++ = brdr[border][THC];        /* chars */
             *vid_ram++ = attr;
           }
           else
             vid_ram += 2;                  /* title char found, skip redraw */
         }

         *vid_ram++ = brdr[border][TRC];    /* top right corner */

         clf=w[li].wscol;
         crt=w[li].wscol+w[li].wscols-1;

         for(i=0,r=w[li].wsrow+1;i<w[li].wsrows-2;i++,r++) { /* do sides */
            get_addr(r,clf);
            *vid_ram++ = brdr[border][LVC];
            *vid_ram = attr;
            get_addr(r,crt);
            *vid_ram++ = brdr[border][RVC];
            *vid_ram = attr;
         }

         get_addr(w[li].wsrow+w[li].wsrows-1,w[li].wscol);  /* bot line */
         *vid_ram++ = brdr[border][BLC];
         *vid_ram++ = attr;

         for(i=0;i<w[li].wscols-2;i++) {

           if(*vid_ram == cb) {
             *vid_ram++ = brdr[border][BHC];
             *vid_ram++ = attr;
           }
           else
             vid_ram += 2;  /* skip any title chars here also */
         }

         *vid_ram++ = brdr[border][BRC];   /* bot right corner */
         w[li].wsbrdr=border;              /* reset border type for this */
                                           /* window */

} /* end of change_border() */

/****************************************************
 * remove_title() removes a title from the window   *
 * pointed to by li.  The border of the window is   *
 * restored.  If pos = TOP the title is removed from*
 * the top of the window, if pos=BOT the title is   *
 * removed from the bottom of the window.           *
 ***************************************************/

remove_title(pos)
int pos;
{
register int r,c;
char cc;

          if(pos==TOP) {
            cc=brdr[w[li].wsbrdr][THC]; /* get border char */
            r=w[li].wsrow;              /* and coord of first row */
          }
          else {
            cc=brdr[w[li].wsbrdr][BHC];    /* get border char */
            r=w[li].wsrow+w[li].wsrows-1;  /* and coord of last row */
          }

          c=w[li].wscol+1;
          qfill(r,c,1,w[li].wscols-2,w[li].wsbattr,cc); /* redraw selected */
                                                        /* border */

} /* end of remove_title() */

/****************************************************
 * relocate_window() moves the window pointed to by *
 * li to a new location on the screen. The screen   *
 * under the current position is restored and the   *
 * screen under the new position is substituted.    *
 * The old heap image is released and a new one is  *
 * allocated for the new screen position.           *
 ***************************************************/

relocate_window(row,col)
int row,col;
{
int shadow,c1,c2,r2;

	if(li==0)
          return(-1); /* no window to move */

        shadow=w[li].wsshadow == right ? right:0; /* get shadow flag */

        if(shadow) {

 	  if(row<0 || row+w[li].wsrows>24 || col<0 || col+w[li].wscols>78)
            return(-2); /* abort - window size/placement invalid */
        }
        else

 	  if(row<0 || row+w[li].wsrows>25 || col<0 || col+w[li].wscols>80)
            return(-2); /* abort - window size/placement invalid */


        /* snapshot the window */

        buf2=qsave(w[li].wsrow,w[li].wscol,w[li].wsrows,w[li].wscols);

	switch(shadow) { /* restore screen at current coords */

        case right:qrestore(w[li].wsrow,w[li].wscol,w[li].wsrows+1, \
                            w[li].wscols+2,wndwptr[li]);
                    c1=col;
                    c2=w[li].wscols+2;
	            r2=w[li].wsrows+1;
                    break;

       /* no shadowing was used */

       default:qrestore(w[li].wsrow,w[li].wscol,w[li].wsrows, \
                        w[li].wscols,wndwptr[li]);
                c1=col;
                c2=w[li].wscols;
                r2=w[li].wsrows;
                break;

       } /* end switch */

       w[li].wsrow=row;              /* save new coords */
       w[li].wscol=col;
       h2=wndwptr[li];               /* release heap image of old contents */
       free(h2);
       wndwptr[li]=qsave(row,c1,r2,c2); /* snapshot screen at new coords */

       /* move snapshot of window to new position */

       qrestore(row,col,w[li].wsrows,w[li].wscols,buf2);

       if(shadow) { /* shadow it if necessary */
         qfill(row+1,col+w[li].wscols,w[li].wsrows-1,2,shadow_attr,' ');
         qfill(row+w[li].wsrows,col+2,1,w[li].wscols,shadow_attr,' ');
       }

       free(buf2);  /* release window snapshot */
       return(0);

} /* end of relocate_window() */

/****************************************************
 * move_window() moves the window pointed to by li  *
 * 1 position. Position may be up/down/left/right.  *
 * If the move would result in the window being     *
 * partially off the screen the move is not         *
 * performed. Returns 0 if move was successful, -1  *
 * if move was not performed.                       *
 ***************************************************/

move_window(direction,count)
int direction,count;
{
int start_row,end_row,start_col,end_col,shadow;
static int sr2,er2,sc2,ec2,er3,ec3,er4,ec4,size,size2,size3,temp,r;
register int i;

          sr2=w[li].wsrow;
          er4=er3=er2=w[li].wsrows;
          sc2=w[li].wscol;
          ec4=ec3=ec2=w[li].wscols;
          start_row=end_row=start_col=end_col=0;
          shadow=w[li].wsshadow ==right ? w[li].wsshadow:0;

          switch(direction) {

          case UP: start_row=w[li].wsrow-count;
                   end_row=start_row+w[li].wsrows;
                   start_col=w[li].wscol;
                   end_col=start_col+w[li].wscols;
                   break;

          case DN: start_row=w[li].wsrow+count;
                   end_row=start_row+w[li].wsrows;
                   start_col=w[li].wscol;
                   end_col=start_col+w[li].wscols;
                   break;

          case LF: start_col=w[li].wscol-count;
                   end_col=start_col+w[li].wscols;
                   start_row=w[li].wsrow;
                   end_row=start_row+w[li].wsrows;
                   break;

          case RT: start_col=w[li].wscol+count;
                   end_col=start_col+w[li].wscols;
                   start_row=w[li].wsrow;
                   end_row=start_row+w[li].wsrows;
                   break;

          } /* switch */

          if(shadow) {

            switch(shadow) {

            case right:if(end_row > 24 || end_col > 78 || start_row < 0 \
                          || start_col < 0)
                         return(-1);

                      end_col+=2;
                      end_row+=1;
                      buf2=qsave(sr2,sc2,er2,ec2);
                      ec2=ec3+=2;
                      er2=er3+=1;
                      break;

            default:  goto _no_shadow; /* no valid value for shadow - ignore */

            } /* end switch */

          }
          else {

_no_shadow:

            if(start_row < 0 || end_row > 25 || start_col < 0 || end_col > 80)
               return(-1);

            buf2=qsave(sr2,sc2,er2,ec2);
          }
          size=ec2<<1;
          size2=size-2;
          h1=wndwptr[li];              /* start of heap block */

          switch(direction) {

          /* move window up 1 line */

          case UP: r=er2-1;
                   temp=r-1;
                   h1=wndwptr[li]+(r * size); /* heap addr of last line */
                   get_addr(sr2+r,sc2);         /* first col of last line */

                   for(i=size;i != 0;i--)
                      *vid_ram++ = *h1++;           /* last line now on screen */

                   if(shadow) {
                     h2=wndwptr[li]+(temp * size); /* heap addr of last line */
                     get_addr(sr2+temp,sc2);         /* first col of last line */

                     for(i=size;i != 0;i--)
                        *vid_ram++ = *h2++;           /* last line now on screen */
                   }

                   h1=wndwptr[li];                  /* 1st byte to move */
                   h2=wndwptr[li]+(r * size); /* last byte to move */
                   h3=wndwptr[li]+size;
                   movmem(h1,h3,h2-h1);             /* move heap image */
                   get_addr(start_row,start_col);   /* new first line */

                   for(i=size;i != 0 ; i--)
                      *h1++ = *vid_ram++;           /* new line now on heap */

                   break;

          /* move window down 1 line */

          case DN: get_addr(sr2,sc2); /* first col of first line */

                   for(i=size;i != 0;i--)
                      *vid_ram++ = *h1++;        /* first line now on screen */

                   if(shadow) {
                     get_addr(sr2+1,sc2); /* first col of first line */
                     h2=wndwptr[li]+size;

                     for(i=size;i != 0;i--)
                        *vid_ram++ = *h2++;        /* first line now on screen */
                   }

                   h1=wndwptr[li]+size;           /* 1st byte to move */
                   h2=wndwptr[li]+(er2 * size); /* last byte to move */
                   h3=wndwptr[li];                /* target in heap */
                   movmem(h1,h3,h2-h1);           /* move heap image */
                   get_addr(end_row-1,start_col);
                   h1 = wndwptr[li]+((er2-1) * size); /* last line heap addr */

                   for(i=size;i != 0 ; i--)
                      *h1++ = *vid_ram++;         /* new line now on heap */

                   break;

          /* move window right 1 column */

          case RT: r=er2-1;
                   if(shadow) {
                     get_addr(sr2+r,sc2);
                     h2=wndwptr[li]+(r * size);
                     *(vid_ram+4)=*(h2+4);
                     *(vid_ram+5)=*(h2+5);
                   }

                   temp=end_col-1;
                   size3=size-1;

                   for(i=0;i<er2;i++) {      /* do er2 lines */
                     r=sr2+i;
                     get_addr(r,sc2);        /* gen video address */
                     *vid_ram++ = *h1;       /* restore char */
                     *vid_ram = *(h1+1);     /* & attrib */
                     movmem(h1+2,h1,size2);  /* move line on heap */
                     get_addr(r,temp);       /* video addr of new end col */
                     *(h1+size2)=*vid_ram++; /* update end of line */
                     *(h1+size3)=*vid_ram;   /* on heap */
                     h1 += size;             /* heap pointer to next line */
                   }

                   break;

          /* move window left 1 column */

          case LF: temp=sc2+ec2-1;
                   size3=size-1;

                   if(shadow) {
                      get_addr(sr2,temp);
                      *(vid_ram-4)=*(h1+size-6);
                      *(vid_ram-3)=*(h1+size-5);
                   }

                   for(i=0;i<er2;i++) {        /* do er2 lines */
                     r=sr2+i;
                     get_addr(r,temp);         /* gen video address */
                     *vid_ram++ = *(h1+size2); /* restore char */
                     *vid_ram = *(h1+size3);   /* & attrib */
                     movmem(h1,h1+2,size2);    /* move line on heap */
                     get_addr(r,start_col);    /* video addr new start col */
                     *h1=*vid_ram++;         /* update start of line */
                     *(h1+1)=*vid_ram;             /* on heap */
                     h1 += size;               /* heap ptr to next line */
                   }

                   break;

          } /* end switch */

          w[li].wsrow=start_row;                      /* update descriptor */
          w[li].wscol=start_col;                      /* for new window coords */
          qrestore(start_row,start_col,er4,ec4,buf2);

          if(shadow) {
             qfill(start_row+1,start_col+ec3-2,er3-1,2,shadow_attr,' ');
             qfill(start_row+er3-1,start_col+2,1,ec3-2,shadow_attr,' ');
          }

          free(buf2);  /* release its storage */
          return(0);

} /* end of move_window() */

/*****************************************************************************
 * clr_window() clears the current window using the atttribute passed in     *
 * color                                                                     *
 ****************************************************************************/

clr_window(colour)
int colour;
{

   	qfill(w[li].wsrow+1,w[li].wscol+1,w[li].wsrows-2,w[li].wscols-2,\
	      colour,clear_char);

} /* end of clr_window() */

/*****************************************************************************
 * init_window initializes the window variables.  Call this routine before   *
 * using make_window, remove_window, or title_window.  The opening screen    *
 * of your application is therefore to be defined as a 80x25 window without  *
 * border.  It is cleared on initialization.                                 *
 ****************************************************************************/


init_window(wattr)
int wattr;
{
	qinit();                /* 'q' routines initialization */
	tattr=wattr;            /* initial text attribute for main window */
	li=0;                   /* first window index */
	licurrent=0;            /* it starts as the current one */
	w[li].orig_row=w[li].wsrow=1;          /* its dimensions and coords */
	w[li].orig_col=w[li].wscol=1;
	w[li].wsrows=25;
	w[li].wscols=80;
	w[li].wswattr=wattr;    /* its text fg/bg attributes */
	w[li].wsbattr=wattr;    /* border (if any) would be the same */
	w[li].wsbrdr=nobrdr;    /* border type */
        w[li].wsshadow=nodir;   /* no shadowing */

        scr_loc(&cur_lin,&cur_col);     /* save current cursor position in
                                           this window */
	w[li].wslastx=cur_col;
	w[li].wslasty=cur_lin;
	scr_clear();	        /* and clear it */

	*(help_line[5]+2)=24;   /* initializes some stuff for the */
        *(help_line[5]+3)=25;   /* directory lister help window       */
        *(help_line[5]+4)=27;
        *(help_line[5]+5)=26;

} /* end of init_window() */

/*****************************************************************************
 * make_window() creates a window on the screen.                             *
 * parms:                                                                    *
 *       row     - first row     (1-screen limit )                           *
 *       col     - first column  (1- screen limit )                          *
 *       rows    - total # of rows (1 - screen limit )                       *
 *       cols    - total # of columns (1 - screen limit )                    *
 *       wattr   - window attribute (1 - 255 )                               *
 *       battr   - border attribute (1 - 255 )                               *
 *       brdrsel - border type select                                        *
 ****************************************************************************/


make_window(row,col,rows,cols,wattr,battr,bs)
int row,col,rows,cols,wattr,battr,bs;
{
register int r1,r2,c1,c2,col_ratio;

	if((li>maxwndw) || (li==maxwndw))
          return(-1); /* cannot allocate anymore windows - already at max
                         - return error code */

	if(row<0 || row+rows>25 || col<0 || col+cols>80)
          return(-2); /* abort - window size/placement invalid */

	switch(shadow_effect) { /* adjust amount to save given shadowing
                                   is being employed */

        case right:c1=col;
                   c2=cols+2;
	           r2=rows+1;
                   break;

        case nodir:
        default:c1=col;
                c2=cols;
                r2=rows;
                break;

        } /* end switch */

	scr_loc(&cur_lin,&cur_col);   /* save the current windows current */
	w[li].wslastx=cur_col;        /* cursor coords */
        w[li].wslasty=cur_lin;
	li++;                         /* bump window index */
	licurrent=li;                 /* make this the current window */
	tattr=wattr;                  /* new text attribute */
	w[li].orig_row=w[li].wsrow=row; /* and its coords and dimensions */
	w[li].orig_col=w[li].wscol=col;
	w[li].wsrows=rows;
	w[li].wscols=cols;
	w[li].wswattr=wattr;          /* its text attributes */
	w[li].wsbattr=battr;          /* its border attributes */
	w[li].wsbrdr=bs;              /* and type  */
	w[li].wsshadow=shadow_effect; /* shadowing type */

	wndwptr[li]=qsave(row,c1,r2,c2); /* snapshot current screen */

	if(zoom_effect==TRUE) {    /* exploding window here */
          r1=row+(rows>>1);        /* we explode the window from the center */
          r2=row+rows-(rows>>1);   /* of the requested area outwards */
	  c1=col+(cols>>1);
          c2=col+cols-(cols>>1);
	  col_ratio=(cols/rows)+1; /* col_ratio is necessary so the col
                                      and row end-points are reached at
                                      the same time. */

	  if(col_ratio>4)
            col_ratio=4;

          do {               /* loop for exploding a window onto the screen */
             if(r1>row)
               --r1;

             if(r2<(row+rows))
               ++r2;

             if(c1>col)
               c1=c1-col_ratio;

             if(c1<col)
               c1=col;

             if(c2<(col+cols))
               c2=c2+col_ratio;

	     if(c2>(col+cols))
               c2=col+cols;

	     qbox(r1,c1,r2-r1,c2-c1,wattr,battr,bs); /* draw successively
                                                     larger boxes simulating
                                                     an exploding box */

             if(qwait==FALSE)
               delay(zoom_delay); /* delay so that zoom is apparent */
				  /* adjust to taste via global zoom_delay */

	     if(c1==col && c2==col+cols && r1==row && r2==row+rows)
               break;
	     }

             while(1);
          }
          else  /* no zoom, just whip the window to the screen */

	     qbox(row,col,rows,cols,wattr,battr,bs);

          switch(shadow_effect) {  /* if shadowing ... */

           case right: if(row+rows<24 && col+cols<79) {
                         qfill(row+1,col+cols,rows-1,2,shadow_attr,' ');
                         qfill(row+rows,col+2,1,cols,shadow_attr,' ');
                       }
                       break;

            } /* end switch */

	return(0);

} /* end of make_window() */

/*****************************************************************************
 * remove_window() removes a window by restoring the screen from the heap    *
 * at the current window coords.  The heap is then deallocated.              *
 ****************************************************************************/

remove_window()
{

	if(li==0)
          return(-1); /* no window to remove */

	switch(w[li].wsshadow) {

        case right:qrestore(w[li].wsrow, \
                            w[li].wscol, \
                            w[li].wsrows+1, \
                            w[li].wscols+2, \
                            wndwptr[li]);
                    break;

       /* no shadowing was used */

       case nodir:
       default:qrestore(w[li].wsrow, \
                        w[li].wscol, \
                        w[li].wsrows, \
                        w[li].wscols, \
                        wndwptr[li]);
                break;

       } /* end switch */

	free(wndwptr[li]); /* deallocate the heap space used */
        li--;              /* point at previous window */
	licurrent=li;      /* it is now the current one */
	tattr=w[li].wswattr; /* we use its text attribute */
	scr_curs(w[li].wslasty,w[li].wslastx); /* reposition cursor */
        return(0);           /* return success flag */

} /* end remove_window() */

/****************************************************************************
 * title_window() will place a title in the top border of the current       *
 * window.  It may be left,or right justified or centered within the        *
 * window margins.  It can have a different attribute than the border.      *
 * Border is used to indicate whether the title is to go on the top or      *
 * bottom border of the current window.                                     *
 ***************************************************************************/

title_window(just,sbuf,attr,border)
int just,attr,border;
char *sbuf;
{
int row;

	if(border==TOP)
          row=w[li].wsrow;
	else  /* must be BOT */
          row=w[li].wsrow+w[li].wsrows-1;

	switch(just) {

        case left: qwrite(row,w[li].wscol+2,attr,sbuf);
                   break;

        case right:qwrite(row, \
                          w[li].wscol+w[li].wscols-(strlen(sbuf))-2,attr,sbuf);
                   break;

        case center:
            default:qwritecv(row, \
                             w[li].wscol, \
                             w[li].wscol+w[li].wscols-1,attr,sbuf);
                    break;

        } /* end switch */

} /* end of title_window() */

/*****************************************************************************
 * scroll_window() scrolls the text area of the current window.  Lines that  *
 * are scrolled off the active window area are NOT preserved.                *
 ****************************************************************************/


scroll_window(begin,endw,dir)
int begin,endw;
int dir;
{
register int brdr_width,r,c,rs,cs;
unsigned char *temp;

	if(w[li].wsbrdr==nobrdr)
          brdr_width=0;
        else
          brdr_width=1;

        r=w[li].wsrow+brdr_width+begin-1;
        c=w[li].wscol+brdr_width;
        rs=endw-begin;
        cs=w[li].wscols-(brdr_width<<1);

	switch(dir) {

        case up:  temp=qsave(r+1,c,rs,cs);
                  qrestore(r,c,rs,cs,temp);
                  qfill(r+rs,c,1,cs,w[li].wswattr,' ');
                  break;

        case down:temp=qsave(r,c,rs,cs);
                  qrestore(r+1,c,rs,cs,temp);
                  qfill(r,c,1,cs,w[li].wswattr,' ');
                  break;

        } /* end switch */

} /* end of scroll_window() */

/****************************************************************************
 * change_window_color() will change the color attributes of the current    *
 * window (both text and border).  If window is passed as 0xff no change    *
 * will be made to the window attributes - only the border.                 *
 ***************************************************************************/

change_wcolor(window,border)
int window,border;
{
register r,c,r1,c1;

	r=w[li].wsrow;
        r1=w[li].wsrows;
        c=w[li].wscol;
        c1=w[li].wscols;
        qattr(r,c,1,c1,border);          /* top border line */
        qattr(r,c,r1,1,border);          /* left border line */
        qattr(r,c+c1-1,r1,1,border);     /* right border line */
	qattr(r+r1-1,c,1,c1,border);     /* bottom border line */

	if(window != 0xff) {
	  w[li].wswattr=window;

	  if(w[li].wsbrdr > 0x0a)
	    qattr(r,c,r1,c1,window);       /* no border type */
	  else
	    qattr(r+1,c+1,r1-2,c1-2,window);
	}
	w[li].wsbattr=border;

} /* end of change_wcolor() */

/* end of file window.c */