/**********************************************************************\
*                                                                      *
*  common.c -- routines common to many modules                         *
*                                                                      *
\**********************************************************************/

#include "defs.h"

int background;

int hidden;
int visual;
int vpage;

int xlimit;
int ylimit;

int old_mode;
int mode;

int mode06;
int mode11;
int mode14;
int mode16;

int mode06_avail;
int mode11_avail;
int mode14_avail;
int mode16_avail;

int mouse;
int xmouse;
int ymouse;
int buttons;

char string[80];

/**********************************************************************\
*                                                                      *
*  abort_program -- called when the program can't be run               *
*                                                                      *
\**********************************************************************/

void abort_program(abort_code)
int abort_code;
{
   /* reset the mode to what it was (probably mode 3) */

   fg_setmode(old_mode);
   fg_reset();

   /* print a line that will stay on the screen after the program exits */

   if (abort_code == 0)
     printf("\nYour system does not have enough memory to run this program.\n");
   else if (abort_code == 1)
     printf("\nThe font file is missing.\n");
   exit(0);
}


/**********************************************************************\
*                                                                      *
*  draw_tbox -- draw non-graphics single line box                      *
*                                                                      *
\**********************************************************************/

void draw_tbox(col1,col2,row1,row2)
int col1, col2, row1, row2;
{
   /* we are working in character space, so use rows and columns */

   register int row, col;

   /* these are the ASCII characters to make the single line box */

   static char *ul = "";
   static char *ur = "";
   static char *ll = "";
   static char *lr = "";
   static char *horz = "";
   static char *vert = "";

   /* draw the sides of the box */

   for (row = row1+1; row < row2; row++)
   {
      put_tstring(vert,row,col1);
      put_tstring(vert,row,col2);
   }

   /* draw the top and the bottom of the box */

   for (col = col1+1; col < col2; col++)
   {
      put_tstring(horz,row1,col);
      put_tstring(horz,row2,col);
   }

   /* put in the corners */

   put_tstring(ul,row1,col1);
   put_tstring(ur,row1,col2);
   put_tstring(ll,row2,col1);
   put_tstring(lr,row2,col2);
}

/**********************************************************************\
*                                                                      *
*  exists -- does the file exist?                                      *
*                                                                      *
\**********************************************************************/

exists(filename)
char *filename;
{
   if (access(filename,0) == 0)
      return(TRUE);
   else
      return(FALSE);
}

/**********************************************************************\
*                                                                      *
*  flushkey -- flush out the keystroke buffer and the mouse buttons    *
*                                                                      *
\**********************************************************************/

void flushkey()
{
   int dummy;
   unsigned char key, aux;

   if (mouse)
   {
      fg_mousebut(1,&dummy,&dummy,&dummy);
      fg_mousebut(2,&dummy,&dummy,&dummy);
   }

   do
      fg_intkey(&key,&aux);
   while (key+aux > 0);
}

/**********************************************************************\
*                                                                      *
*          initialize -- initialize the Fastgraph environment          *
*                                                                      *
\**********************************************************************/

void initialize()
{
   register int page;

   /* Check which video modes are available */

   mode06_avail = fg_testmode(6,PAGES);
   mode11_avail = fg_testmode(11,PAGES);
   mode14_avail = fg_testmode(14,PAGES);
   mode16_avail = fg_testmode(16,PAGES);

   /* Abort the program if none of the video modes are available */

   if (!(mode06_avail | mode11_avail | mode14_avail | mode16_avail))
      abort_program(0);

   /* Save the current video mode */

   old_mode = fg_getmode();

   /* Propose the default video mode */

   if (mode16_avail)
      mode = 16;
   else if (mode14_avail)
      mode = 14;
   else if (mode11_avail)
      mode = 11;
   else if (mode06_avail)
      mode = 6;

   /* Let the user choose one of the supported modes */

   mode = select_mode();

   /* Switch to the selected video mode */

   fg_setmode(mode);

   /* Allocate virtual pages as needed */

   for (page = 1; page < PAGES; page++)
      fg_allocate(page);

   /* Set up visual and hidden pages */

   fg_sethpage(1);
   visual = 0;
   hidden = 1;
   background = 0;

   /* Determine maximum screen space coordinates */

   xlimit = fg_getmaxx();
   ylimit = fg_getmaxy();

   /* For Hercules mode, we'll only use the first 640 of 720 columns */

   if (xlimit > 639) xlimit = 639;

   /* For 640x350 EGA mode, use only as much vertical resolution as Hercules */

   if (ylimit > 347) ylimit = 347;

   /* Initialize the mouse, if present */

   init_mouse();
}

/**********************************************************************\
*                                                                      *
*  init_mouse -- initialize the mouse if present                       *
*                                                                      *
\**********************************************************************/

void init_mouse()
{
   if (fg_mouseini() > 0)
   {
      mouse = TRUE;
      fg_mouselim(6,xlimit-5,0,ylimit);
      xmouse = xlimit / 2;
      ymouse = ylimit / 2;
      fg_mousemov(xmouse,ymouse);
      fg_mousevis(ON);
   }
   else
      mouse = FALSE;
}

/**********************************************************************\
*                                                                      *
*  printer_ready -- is the printer on line?                            *
*                                                                      *
\**********************************************************************/

printer_ready()
{
   static union REGS registers;

   registers.h.ah = 2;
   registers.x.dx = 0;
   int86(0x17,&registers,&registers);
   return(registers.h.ah == 0x90);
}

/**********************************************************************\
*                                                                      *
*  scale -- the y in the low-res modes is 4/7th the y in hi-res modes  *
*                                                                      *
\**********************************************************************/

scale(y)
int y;
{
   if (mode06 || mode14)
      return((y * 4) / 7);
   else
      return(y);
}

/**********************************************************************\
*                                                                      *
* select_mode -- let the user choose a supported video mode            *
*                                                                      *
\**********************************************************************/

select_mode()
{
   register int i;
   unsigned char key, aux;
   int current, new;

   static int max = 4;

   static char *menu_string[] = {
      " CGA, MCGA, or Tandy 1000          ",
      " EGA with RGB display              ",
      " EGA with enhanced display, or VGA ",
      " Hercules monochrome               "};

   static int menu_row[] = {10,11,12,13};

   /* use setmode to clear the screen and initialize Fastgraph */

   fg_setmode(old_mode);

   /* turn off the cursor and the NumLock key */

   fg_cursor(OFF);
   fg_setnum(OFF);

   /* draw the box */

   fg_setattr(7,7,0);
   fg_rect(21,59,9,14);
   fg_setattr(0,7,0);
   draw_tbox(22,58,9,14);

   mode06 = FALSE;
   mode11 = FALSE;
   mode14 = FALSE;
   mode16 = FALSE;

   /* start out pointing to preferred mode for this system */

   switch(mode)
   {
      case 6:
         current = 0;
         break;
      case 11:
         current = 3;
         break;
      case 14:
         current = 1;
         break;
      case 16:
         current = 2;
   }

   new = current;

   for (i = 0; i < max; i++)
   {
      fg_setattr(0,7,0);
      put_tstring(menu_string[i],menu_row[i],23);
   }
   fg_setattr(15,0,0);
   put_tstring(menu_string[current],menu_row[current],23);
   put_tstring("Best choice -->",menu_row[current],5);

   /* get input and let the user select the mode they want */

   while (TRUE)
   {
      fg_getkey(&key,&aux);

      if (key == ESC)
          terminate();

      else if (aux == DOWN_ARROW)
      {
          new = current + 1;
          if (new >= max) new = 0;
      }

      else if (aux == UP_ARROW)
      {
          new = current - 1;
          if (new < 0) new = max - 1;
      }

      else if (key == CR)
      {
         switch(current)
         {
            case 0:                /* CGA 2-color mode */
               if (mode06_avail)
               {
                  mode06 = TRUE;
                  return(6);
               }
               else
               {
                  fg_waitfor(1);
                  fg_sound(1200,1);
                  break;
               }
            case 1:               /* 640X200 16 color EGA & VGA */
               if (mode14_avail)
               {
                  mode14 = TRUE;
                  return(14);
               }
               else
               {
                  fg_waitfor(1);
                  fg_sound(1200,1);
                  break;
               }
            case 2:               /* 640X350 16 color EGA & VGA */
               if (mode16_avail)
               {
                  mode16 = TRUE;
                  return(16);
               }
               else
               {
                  fg_waitfor(1);
                  fg_sound(1200,1);
                  break;
               }
            case 3:               /* 720 X 348 Hercules */
               if (mode11_avail)
               {
                  mode11 = TRUE;
                  return(11);
               }
               else
               {
                  fg_waitfor(1);
                  fg_sound(1200,1);
                  break;
               }
         }
      }

      /* unhighlight current string */

      fg_setattr(0,7,0);
      put_tstring(menu_string[current],menu_row[current],23);

      /* highlight the new string */

      fg_setattr(15,0,0);
      put_tstring(menu_string[new],menu_row[new],23);

      current = new;
   }
}

/**********************************************************************\
*                                                                      *
*  terminate -- perform necessary cleanup and return control to DOS    *
*                                                                      *
\**********************************************************************/

void terminate()
{
   register int page;

   /* Release Fastgraph's virtual video pages */

   for (page = 1; page < PAGES; page++)
      fg_freepage(page);

   /* Restore the original video mode and screen attributes */

   fg_setmode(old_mode);
   fg_reset();

   /* Return control to DOS */

   exit(0);
}

/**********************************************************************\
*                                                                      *
*  wait_for_keystroke -- wait for a keystroke or mouse button          *
*                                                                      *
\**********************************************************************/

void wait_for_keystroke()
{
   int buttons;
   int count;
   int x, y;
   unsigned char key, aux;

   flushkey();

   /* if the mouse is loaded, must loop and wait for button or keystroke */

   if (mouse)
   {
      fg_mousevis(ON);
      while (TRUE)
      {
         fg_waitfor(1);
         fg_intkey(&key,&aux);
         if (key+aux > 0) break;
         fg_mousebut(1,&count,&x,&y);
         if (count > 0) break;
         fg_mousebut(2,&count,&x,&y);
         if (count > 0) break;
      }
      do
         fg_mousepos(&x,&y,&buttons);
      while (buttons&3);
      fg_mousevis(OFF);
   }

   /* if the mouse is not loaded, just wait for a key */

   else
      fg_getkey(&key,&aux);
}


/**********************************************************************\
*                                                                      *
*  wait_for_mouse_buttons -- wait until no mouse buttons are pressed   *
*                                                                      *
\**********************************************************************/

void wait_for_mouse_buttons()
{
   int buttons;
   int x, y;

   if (mouse)
   {
      do
         fg_mousepos(&x,&y,&buttons);
      while (buttons&3);
   }
}
