/**********************************************************************\
*                                                                      *
*  fundmtls.c -- graphics fundamentals: points, lines, rects, etc.     *
*                                                                      *
\**********************************************************************/

#include "defs.h"

/**********************************************************************\
*                                                                      *
*  do_boxes -- draw a bunch of hollow rectangles                       *
*                                                                      *
\**********************************************************************/

do_boxes()
{
   register int i,j;
   int x0,x1,x2,y1,y2;
   int xinc,yinc;
   int color;

   x0 = 5;
   x1 = x0;
   xinc = (xlimit - x0 - 1) / 10;
   x2 = x1 + xinc;

   yinc = (ylimit - menu_bottom) / 10;
   y1 = menu_bottom;
   y2 = y1 + yinc;

   color = 0;

   /* clear bottom part of screen */

   fg_mousevis(OFF);
   fg_setcolor(15);
   fg_rect(0,xlimit,menu_bottom,ylimit);

   /* draw 100 hollow rectangles */

   fg_mousevis(OFF);
   fg_boxdepth(2,2);
   for (i = 0; i < 10; i++)
   {
      for (j = 0; j < 10; j++)
      {
         fg_setcolor(color);
         fg_box(x1,x2-1,y1,y2-1);

         color++;
         if (color > 14) color = 0;

         x1 = x2;
         x2 = x1 + xinc;
      }
      x1 = x0;
      x2 = x1 + xinc;

      y1 = y2;
      y2 = y1 + yinc;
   }
   fg_boxdepth(1,1);

   /* wait for a keystroke or mouse button */

   fg_mousevis(ON);
   wait_for_keystroke();

   /* restore the screen and return to the menu */

   fg_mousevis(OFF);
   fg_restore(0,xlimit,menu_bottom,ylimit);

   fg_mousevis(ON);
   redraw = TRUE;

   return(OK);
}

/**********************************************************************\
*                                                                      *
*  do_circles -- draw a bunch of circles                               *
*                                                                      *
\**********************************************************************/

do_circles()
{
   register int i;
   int x1,x2,y1,y2;

   /* establish clipping limits */

   x1 = 0;
   x2 = xlimit;
   y1 = menu_bottom;
   y2 = ylimit;

   fg_setclip(x1,x2,y1,y2);

   /* clear bottom part of screen */

   fg_mousevis(OFF);
   if (mode06 || mode11)
     fg_setcolor(0);
   else
     fg_setcolor(1);
   fg_rect(x1,x2,y1,y2);

   /* move to the center of the screen */

   x1 = xlimit / 2;
   y1 = (ylimit + menu_bottom) / 2;
   fg_move(x1,y1);

   /* draw concentric circles */

   x2 = 4;
   fg_setcolor(15);
   for (i = 0; i < 25; i++)
   {
      fg_circle(x2);
      x2 += 8;
   }

   /* wait for a keystroke */

   fg_mousevis(ON);
   wait_for_keystroke();

   /* restore clipping limits */

   fg_setclip(0,fg_getmaxx(),0,fg_getmaxy());

   /* restore the screen and return to the menu */

   fg_mousevis(OFF);
   fg_restore(0,xlimit,menu_bottom,ylimit);

   fg_mousevis(ON);
   redraw = TRUE;

   return(OK);
}

/**********************************************************************\
*                                                                      *
*  do_clip -- do those lines, clipped to a small rectangle             *
*                                                                      *
\**********************************************************************/

int do_clip()
{
   register int i,j;
   int x1,x2,y1,y2;
   int xinc;

   static int lcolor[] = {2,1,9,11,11,9,1,2};

   /* establish clipping limits - small rectangle in middle of screen */

   fg_setclip(135,510,menu_bottom+55,ylimit-55);

   /* clear bottom of screen */

   fg_mousevis(OFF);
   fg_restore(0,xlimit,menu_bottom,ylimit);
   fg_setcolor(15);
   fg_rect(135,510,menu_bottom+55,ylimit-55);

   /* draw horizontal lines */

   fg_setcolor(0);
   for (i = menu_bottom; i < ylimit; i+=40)
   {
      for (j = 0; j < 8; j++)
      {
         if (mode14 || mode16)
            fg_setcolor(lcolor[j]);

         y1 = i + 3*j;
         fg_move(0,y1);
         fg_draw(xlimit,y1);

      }
   }

   /* draw vertical lines */

   y1 = menu_bottom;
   y2 = ylimit;
   for (i = 0; i < 640; i+=60)
   {
      for (j = 0; j < 8; j++)
      {
         if (mode14 || mode16)
            fg_setcolor(lcolor[j]);

         x1 = i + 3*j;
         fg_move(x1,y1);
         fg_draw(x1,y2);
      }
   }

   /* draw red diagonal lines */

   y1 = menu_bottom;
   y2 = ylimit;
   xinc = ylimit - menu_bottom;
   fg_setcolor(12);
   for (x1 = -640; x1 < 640; x1+=60)
   {
      x2 = x1 + xinc;
      fg_move(x1,y1);
      fg_draw(x2,y2);
   }

   y1 = menu_bottom;
   y2 = ylimit;
   xinc = ylimit - menu_bottom;

   /* draw red diagonal lines */

   fg_setcolor(4);
   for (x1 = 0; x1 < 1280; x1+=60)
   {
      x2 = x1-xinc;
      fg_move(x1,y1);
      fg_draw(x2,y2);
   }

   /* wait for a keystroke or mouse button */

   fg_mousevis(ON);
   wait_for_keystroke();

   /* clear the screen and return to the menu */

   fg_mousevis(OFF);
   fg_setclip(0,fg_getmaxx(),0,fg_getmaxy());
   fg_restore(0,xlimit,menu_bottom,ylimit);

   fg_mousevis(ON);
   redraw = TRUE;
   return(OK);
}

/**********************************************************************\
*                                                                      *
*  do_ellipses -- draw a bunch of elipses                              *
*                                                                      *
\**********************************************************************/

do_ellipses()
{
   register int i;
   int x1,x2,x3,y1;

   /* clear the screen */

   fg_mousevis(OFF);
   if (mode06 || mode11)
     fg_setcolor(1);
   else
     fg_setcolor(9);
   fg_rect(0,xlimit,menu_bottom,ylimit);

   /* move to the center of the screen */

   x1 = xlimit / 2;
   y1 = (ylimit + menu_bottom) / 2;
   fg_move(x1,y1);

   /* draw concentric ellipses */

   x2 = 4;
   x3 = 1;
   fg_setcolor(0);
   for (i = 0; i < 80; i++)
   {
      fg_ellipse(x2,x3);
      x2 += 3;
      x3++;
   }

   /* wait for a keystroke or mouse button */

   fg_mousevis(ON);
   wait_for_keystroke();

   /* restore the screen and return to the menu */

   fg_mousevis(OFF);
   fg_restore(0,xlimit,menu_bottom,ylimit);

   fg_mousevis(ON);
   redraw = TRUE;

   return(OK);
}

/**********************************************************************\
*                                                                      *
*  do_lines -- draw lines to create a plaid pattern                    *
*                                                                      *
\**********************************************************************/

do_lines()
{
   register int i,j;
   int x1,x2,y1,y2;
   int xinc;

   static int lcolor[] = {2,1,9,11,11,9,1,2};

   /* clear bottom part of screen */

   fg_mousevis(OFF);
   fg_setcolor(15);
   fg_rect(0,xlimit,menu_bottom,ylimit);

   /* draw horizontal lines */

   fg_setcolor(0);
   for (i = menu_bottom; i < ylimit; i+=40)
   {
      for (j = 0; j < 8; j++)
      {
         if (mode14 || mode16)
            fg_setcolor(lcolor[j]);

         y1 = i + 3*j;
         fg_move(0,y1);
         fg_draw(xlimit,y1);

      }
   }

   /* draw vertical lines */

   y1 = menu_bottom;
   y2 = ylimit;
   for (i = 0; i < 640; i+=60)
   {
      for (j = 0; j < 8; j++)
      {
         if (mode14 || mode16)
            fg_setcolor(lcolor[j]);

         x1 = i + 3*j;
         fg_move(x1,y1);
         fg_draw(x1,y2);
      }
   }

   /* draw red diagonal lines */

   y1 = menu_bottom;
   y2 = ylimit;
   xinc = ylimit - menu_bottom;
   fg_setcolor(12);
   for (x1 = -640; x1 < 640; x1+=60)
   {
      x2 = x1 + xinc;
      fg_move(x1,y1);
      fg_draw(x2,y2);
   }

   y1 = menu_bottom;
   y2 = ylimit;
   xinc = ylimit - menu_bottom;

   /* draw red diagonal lines */

   fg_setcolor(12);
   for (x1 = 0; x1 < 1280; x1+=60)
   {
      x2 = x1 - xinc;
      fg_move(x1,y1);
      fg_draw(x2,y2);
   }

   /* wait for a keystroke or mouse button */

   fg_mousevis(ON);
   wait_for_keystroke();

   /* restore the screen and return to the menu */

   fg_mousevis(OFF);
   fg_restore(0,xlimit,menu_bottom,ylimit);

   fg_mousevis(ON);
   redraw = TRUE;

   return(OK);
}

/**********************************************************************\
*                                                                      *
*  do_paint -- draw a quartered circle, then paint around it           *
*                                                                      *
\**********************************************************************/

do_paint()
{
   int x1,x2,y1,y2;

   /* restor the screen */

   fg_mousevis(OFF);
   fg_restore(0,xlimit,menu_bottom,ylimit);

   /* draw a rectangle */

   y1 = menu_bottom + scale(20);
   y2 = ylimit - scale(20);
   x1 = 40;
   x2 = xlimit - 40;

   if (mode06 || mode11)
      fg_setcolor(0);
   else
      fg_setcolor(11);

   fg_rect(x1,x2,y1,y2);

   /* outline the rectangle */

   if (mode06 || mode11)
      fg_setcolor(1);
   else
      fg_setcolor(0);
   fg_box(x1,x2,y1,y2);

   y1 = (ylimit + menu_bottom) / 2;
   x1 = xlimit / 2;
   fg_move(x1,y1);

   /* draw the circle */

   if (mode06 || mode11)
      fg_setcolor(1);
   else
      fg_setcolor(0);
   fg_circle(80);

   /* draw cross bars in the circle */

   y2 = scale(80);
   fg_move(x1,y1-y2);
   fg_draw(x1,y1+y2);

   x2 = 100;
   fg_move(x1-x2,y1);
   fg_draw(x1+x2,y1);

   /* paint each quarter of the circle */

   if (!mode06 && !mode11)
   {
      fg_setcolor(1);
      fg_paint(x1-6,y1-6);

      fg_setcolor(2);
      fg_paint(x1+6,y1+6);

      fg_setcolor(3);
      fg_paint(x1+6,y1-6);

      fg_setcolor(4);
      fg_paint(x1-6,y1+6);
   }

   /* paint the box */

   x1 = 41;
   y1 = menu_bottom + scale(20) + 1;

   if (mode06 || mode11)
      fg_setcolor(1);
   else
      fg_setcolor(14);
   fg_paint(x1,y1);

   /* wait for a keystroke or a mouse button */

   fg_mousevis(ON);
   wait_for_keystroke();

   /* restore the screen and return to the menu */

   fg_mousevis(OFF);
   fg_restore(0,xlimit,menu_bottom,ylimit);

   fg_mousevis(ON);
   redraw = TRUE;

   return(OK);
}

/**********************************************************************\
*                                                                      *
*  do_points -- draw a nice pattern of points                          *
*                                                                      *
\**********************************************************************/

do_points()
{
   register int i,j;
   int yinc;

   /* clear the bottom part of the screen */

   fg_mousevis(OFF);
   if (mode06 || mode11)
     fg_setcolor(0);
   else
     fg_setcolor(1);
   fg_rect(0,xlimit,menu_bottom,ylimit);

   yinc = scale(8);

   /* draw the patterns of points */

   fg_setcolor(15);
   for (i = 5; i < xlimit; i+=20)
   {
      for (j = menu_bottom+2; j < ylimit; j+=yinc)
         fg_point(i,j);
   }

   for (i = 15; i < xlimit; i+=20)
   {
      for (j = menu_bottom+yinc/2+2; j < ylimit; j+=yinc)
         fg_point(i,j);
   }

   /* wait for a keystroke or mouse button */

   fg_mousevis(ON);
   wait_for_keystroke();

   /* restore the screen and return to the menu */

   fg_mousevis(OFF);
   fg_restore(0,xlimit,menu_bottom,ylimit);

   fg_mousevis(ON);
   redraw = TRUE;

   return(OK);
}

/**********************************************************************\
*                                                                      *
*  do_rects -- draw a bunch of solid rectangles                        *
*                                                                      *
\**********************************************************************/

do_rects()
{
   register int i,j;
   int x0,x1,x2,y1,y2;
   int xinc,yinc;
   int color;

   x0 = 5;
   x1 = x0;
   xinc = (xlimit - x0 - 1) / 10;
   x2 = x1 + xinc;

   yinc = (ylimit - menu_bottom) / 10;
   y1 = menu_bottom;
   y2 = y1 + yinc;

   color = 0;

   /* draw 100 solid rectangles */

   fg_mousevis(OFF);
   for (i = 0; i < 10; i++)
   {
      for (j = 0; j < 10; j++)
      {
         fg_setcolor(color);
         fg_rect(x1,x2,y1,y2);

         color++;
         if (color > 14) color = 0;

         x1 = x2;
         x2 = x1 + xinc;
      }
      x1 = x0;
      x2 = x1 + xinc;

      y1 = y2;
      y2 = y1 + yinc;
   }

   /* wait for a keystroke or mouse button */

   fg_mousevis(ON);
   wait_for_keystroke();

   /* restore the screen and return to the menu */

   fg_mousevis(OFF);
   fg_restore(0,xlimit,menu_bottom,ylimit);

   fg_mousevis(ON);
   redraw = TRUE;

   return(OK);
}

/**********************************************************************\
*                                                                      *
*  do_text -- demonstrate ROM text                                     *
*                                                                      *
\**********************************************************************/

do_text()
{
   register int i;
   register int row;
   int x1,x2,y1,y2;

   static char *string[] = {
   "Fastgraph allows you to display",
   "ROM text, a stroke character font,",
   "or define your own bitmapped font.",
   "This message is displayed using",
   "ROM text.  The menus use bitmapped",
   "characters."
   };

   fg_mousevis(OFF);

   fg_restore(0,xlimit,menu_bottom,ylimit);

   /* convert rows and columns to x's and y's */

   x1 = fg_xconvert(20);
   x2 = fg_xconvert(60);

   y1 = fg_yconvert(5);
   y2 = fg_yconvert(13);

   /* draw a small rectangle */

   if (mode06 || mode11)
     fg_setcolor(0);
   else
     fg_setcolor(1);
   fg_rect(x1,x2,y1,y2);

   /* draw a white boarder around the box */

   fg_setcolor(15);
   fg_box(x1+1,x2-1,y1+1,y2-1);

   /* put ROM text in the box */

   row = 6;
   for (i = 0; i < 6; i++)
   {
      fg_locate(row,23);
      fg_text(string[i],strlen(string[i]));
      row++;
   }

   /* wait for a keystroke or mouse button */

   fg_mousevis(ON);
   wait_for_keystroke();

   /* restore the screen and return to the menu */

   fg_mousevis(OFF);
   fg_restore(0,xlimit,menu_bottom,ylimit);

   fg_mousevis(ON);
   redraw = TRUE;

   return(OK);
}
