unit TPDBDate;
                           (***********************************)
                           (*               TPDB              *)
                           (***********************************)
                           (*         Object -Oriented        *)
                           (*     Turbo Pascal 6.0 Units      *)
                           (*    for Accessing dBASE III      *)
                           (*             files.              *)
                           (*        Copyright 1991           *)
                           (*          Brian Corll            *)
                           (*       All Rights Reserved       *)
                           (*     dBASE is a registered       *)
                           (* trademark of Ashton-Tate, Inc.  *)
                           (*   Version 3.20  October, 1991   *)
                           (***********************************)
                           (*   Portions Copyright 1984,1991  *)
                           (*    Borland International Corp.  *)
                           (***********************************)


interface

uses
    Dos;

type
    DayStr = string [9];
    DateType = word;
    DateStr = string [8];
    TimeStr = string [13];
    Str9 = string [9];

function CalcDate(InDate: DateStr; Days, Months, Years: integer): DateStr;
(* Add or subtract days,months, or years from two dates. *)

function CDOW(InDate: DateStr): DayStr;
(* Returns character day of week - i.e. 'Monday','Tuesday',etc. *)

function CMonth(InDate: DateStr): Str9;
(* Returns character month - i.e. 'March' *)

function CompDates(Date1, Date2: DateStr): word;
(* Compares two dates and calculates the number of days between them. *)

function CTOD(InDate: DateStr): DateType;
(* Converts a .DBF compatible date field to a word date type. *)

function DTOC(Julian: DateType): DateStr;
(* Converts a word date type to a string compatible with .DBF date fields. *)


function Mon(InDate: DateStr): byte;
(* Returns numeric value for the month in a date. *)

function TimeNow: TimeStr;
(* Returns current time in formatted string. *)

function Today: DateStr;
(* Returns current date in .DBF date field compatible format. *)

function ValidDate(InDate: DateStr): boolean;
(* Checks whether a date is valid. *)

function FormDate(InDate: DateStr): string;
(* Formats a date as 'MM/DD/YY' *)




implementation

const
    Months: array [1..12] of Str9 = ('January  ', 'February ', 'March    ', 'April    ', 'May      ', 'June     ',
            'July     ', 'August   ', 'September', 'October  ', 'November ', 'December ');

var
    Temp, Month, Day, Year, ErrCode: integer;
    MM, DD: string [2];
    YY: string [4];


function CDOW(InDate: DateStr): DayStr;
(* Returns the name of the day of the week represented by
   a date. *)

var
    DayOfWeek, DOW: integer;

begin
    YY := Copy(InDate, 1, 4);
    MM := Copy(InDate, 5, 2);
    DD := Copy(InDate, 7, 2);
    Val(MM, Month, ErrCode);
    Val(DD, Day, ErrCode);
    Val(YY, Year, ErrCode);
    if month <= 2 then begin
        month := month + 12;
        year := year - 1;
    end;

    DayOfWeek := (Day + month * 2 + (month + 1) * 6 div 10 + year + year div 4 - year div 100 + year div 400 + 2) mod 7;

    if DayOfWeek = 0 then
        DOW := 7
    else
        DOW := DayOfWeek;

    case DOW of
        1: CDOW := 'Sunday';
        2: CDOW := 'Monday';
        3: CDOW := 'Tuesday';
        4: CDOW := 'Wednesday';
        5: CDOW := 'Thursday';
        6: CDOW := 'Friday';
        7: CDOW := 'Saturday';
    end;
end;

function CTOD(InDate: DateStr): DateType;
(* Convert from a date string to a word date type. *)

var
    Julian: DateType;

begin
    YY := Copy(InDate, 1, 4);
    MM := Copy(InDate, 5, 2);
    DD := Copy(InDate, 7, 2);

    Val(YY, Year, ErrCode);
    Val(MM, Month, ErrCode);
    Val(DD, Day, ErrCode);

    if (Year = 1900) and (Month < 3) then
        if Month = 1 then
            Julian := Pred(Day)
        else
            Julian := Day + 30
    else begin
        if Month > 2 then
            Dec(Month, 3)
        else begin
            Inc(Month, 9);
            Dec(Year)
        end;
        Dec(Year, 1900);
        Julian := (1461 * longint(Year) div 4) + ((153 * Month + 2) div 5) + Day + 58
    end;
    CTOD := Julian;
end;

function DTOC(Julian: DateType): DateStr;
(* Convert from a word date type to a date string. *)

var
    LongTemp: longint;

begin
    if Julian <= 58 then begin
        Year := 1900;
        if Julian <= 30 then begin
            Month := 1;
            Day := Succ(Julian)
        end else begin
            Month := 2;
            Day := Julian - 30
        end
    end else begin
        LongTemp := 4 * longint(Julian) - 233;
        Year := LongTemp div 1461;
        Temp := LongTemp mod 1461 div 4 * 5 + 2;
        Month := Temp div 153;
        Day := Temp mod 153 div 5 + 1;
        Inc(Year, 1900);
        if Month < 10 then
            Inc(Month, 3)
        else begin
            Dec(Month, 9);
            Inc(Year)
        end
    end;
    Str(Month: 2, MM);
    Str(Day: 2, DD);
    Str(Year: 4, YY);
    if Month < 10 then
        MM := '0' + Copy(MM, 2, 1);
    if Day < 10 then
        DD := '0' + Copy(DD, 2, 1);;
    DTOC := YY + MM + DD;
end;

function ValidDate(InDate: DateStr): boolean;
(* Check whether a date field contains a valid date. *)

begin
    YY := Copy(InDate, 1, 4);
    MM := Copy(InDate, 5, 2);
    DD := Copy(InDate, 7, 2);
    Val(DD, Day, ErrCode);
    Val(MM, Month, ErrCode);
    Val(YY, Year, ErrCode);
    if (Day = 0) and (Year - 1900 = 0) and (Month = 0) then begin
        ValidDate := True;
        Exit;
    end;
    if (Day < 1) or (Year < 1900) or (Year > 2078) then
        ValidDate := False
    else
        case Month of
            1, 3, 5, 7, 8, 10, 12: ValidDate := Day <= 31;
            4, 6, 9, 11: ValidDate := Day <= 30;
            2: ValidDate := Day <= 28 + Ord((Year mod 4) = 0) * Ord(Year <> 1900) else ValidDate := False
        end
end;

function CalcDate(InDate: DateStr; Days, Months, Years: integer): DateStr;
(* Add or subtract days, months , and years from a specific date string,
 as stored in a .DBF record. *)

var
    Julian: DateType;
    TempDate: DateStr;

begin
    YY := Copy(InDate, 1, 4);
    MM := Copy(InDate, 5, 2);
    DD := Copy(InDate, 7, 2);
    Val(MM, Month, ErrCode);
    Val(DD, Day, errCode);
    Val(YY, Year, ErrCode);
    Month := Month + Months - 1;
    Year := Year + Years + (Month div 12) - Ord(Month < 0);
    Month := (Month + 12000) mod 12 + 1;
    Str(Month: 2, MM);
    Str(Day: 2, DD);
    Str(Year: 4, YY);
    if Month < 10 then
        MM := '0' + Copy(MM, 2, 1);
    if Day < 10 then
        DD := '0' + Copy(DD, 2, 1);
    TempDate := YY + MM + DD;
    Julian := CTOD(TempDate) + Days;
    CalcDate := DTOC(Julian);
end;

function CompDates(Date1, Date2: DateStr): word;
(* Compare two dates and calculate the number of
 days between them. *)

begin
    if CTOD(Date1) > CTOD(Date2) then
        CompDates := CTOD(Date1) - CTOD(Date2)
    else
        CompDates := CTOD(Date2) - CTOD(Date1);
end;

function CMonth(InDate: DateStr): Str9;
(* Returns the month name for any date. *)

begin
    MM := Copy(InDate, 5, 2);
    Val(MM, Month, ErrCode);
    CMonth := Months[Month]
end;

function TimeNow: TimeStr;
(* Returns a formatted string for the current time. *)

var
    Hour, Minute, Second, Sec100: word;
    HH, MM, SS: string [2];
    Temp: string [8];
    Code: integer;

begin
    GetTime(Hour, Minute, Second, Sec100);
    Str(Minute, MM);
    Str(Second, SS);
    if Minute < 10 then
        MM := '0' + MM;
    if Second < 10 then
        SS := '0' + SS;
    if Hour > 12 then begin
        Str(Hour - 12, HH);
    end else
        Str(Hour, HH);
    if Hour >= 12 then
        TimeNow := HH + ':' + MM + ':' + SS + ' p.m.'
    else
        TimeNow := HH + ':' + MM + ':' + SS + ' a.m.';
end;

function Today: DateStr;
(* Returns today's date in dBASE III date format. *)

var
    mMonth, mDay, mYear, mDayOfWk: word;

begin
    GetDate(mYear, mMonth, mDay, mDayOfWk);
    Str(mMonth, MM);
    Str(mDay, DD);
    Str(mYear, YY);
    if mMonth < 10 then
        Insert('0', MM, 1);
    if mDay < 10 then
        Insert('0', DD, 1);
    Today := YY + MM + DD;
end;

function Mon(InDate: DateStr): byte;
(* Returns number of month in a date. *)

var
    Temp: byte;

begin
    MM := Copy(InDate, 5, 2);
    Val(MM, Temp, ErrCode);
    Mon := Temp;
end;

function FormDate(InDate: DateStr): string;
(* Formats dBASE date field as MM/DD/YY *)

var
    OutDate: string [8];

begin
    OutDate := Copy(InDate, 5, 2) + '/' + Copy(InDate, 7, 2) + '/' + Copy(InDate, 3, 2);
    FormDate := OutDate;
end;

end.                                                        (* TPDBDate *)
