
 ; Program Name: DosView
 ;      Version: 1.0
 ;         Date: January, 1995
 ;       Author: Steven H. Fisk
 ;    Assembler: A86
 ; Distribution: See page 1 of the DOSVIEW.DOC file for warranty and
 ;               distribution policies.
 ;  Description: A TSR program to limit output to only a certain range of
 ;               lines on the screen. This program will only work correctly
 ;               with programs that output data to the screen using DOS
 ;               interrupt calls (Int 21h). BIOS calls, custom output routines
 ;               and direct screen writes will not be affected by the program.


 ;                     A SHORT NOTE ABOUT THIS SOURCE CODE
 ;
 ; This source code was written using the A86 V3.22 macro assembler. I've
 ; attempted to document the source code below as thoroughly and clearly as
 ; possible. Almost every line of code has at least some sort of comment. The
 ; only exception to this is that lines containing jump instructions have
 ; been deliberately left undocumented. The idea behind this is to keep the
 ; comments as managable as possible and such lines are only used to lead to
 ; another section of code which has it's own documentation anyway. Also note
 ; that because of the size of this source code, I was forced to create
 ; labels whose only purpose is to lead to a JMP statement. This was done so
 ; that conditional jumps greater than 128 bytes can take place. I've tried
 ; to eliminate as many of these as possible, but some cases couldn't be
 ; avoided. For your convenience, each label that marks the beginning of a
 ; major routine has been given a code followed by a detailed description for
 ; that routine. The code consists of a set of numbers separated by periods.
 ; A main section of code would have a single number with no periods (eg. 2).
 ; While a sub-section might have a code such as 2.1, 2.1.1, etc.



 ; Initialize.

        .MODEL  TINY
        ORG     100h


 ; Code Segment.

        .CODE



 ; _____________________________ START OF MACROS _____________________________
 ;
 ; Section: 1
 ;
 ; The macros below are used throughout the three main sections of DosView.

        push_a  macro                   ; PUSH_A - Push all registers on to
                pushf                   ;          the stack.
                push    bp
                push    ds
                push    es
                push    si
                push    di
                push    dx
                push    cx
                push    bx
                push    ax
                #em

        pop_a   macro                   ; POP_A - Pop all registers from the
                pop     ax              ;         the stack.
                pop     bx
                pop     cx
                pop     dx
                pop     di
                pop     si
                pop     es
                pop     ds
                pop     bp
                popf
                #em

        print   macro                   ; PRINT - Print string (CR).
                mov     ah,9h
                mov     dx,offset #1
                int     21h
                #em

        cr      macro                   ; CR - Move cursor down a line.
                mov     ah,6h
                mov     dl,0dh
                int     21h
                mov     ah,6h
                mov     dl,0ah
                int     21h
                #em

        bye     macro                   ; BYE - Exit program.
                mov     ah,4ch
                mov     al,0h
                int     21h
                #em

 ; ______________________________ END OF MACROS ______________________________



 ; _____________________ START OF COMMAND-LINE INTERFACE _____________________
 ;
 ; Section: 2
 ;
 ; This section takes care of the command-line parameters that can be passed
 ; to DosView from the DOS prompt.


 check_install:
 ;
 ; Section: 2.1
 ;
 ; This routine is used to find out if the TSR portion of DosView is
 ; installed. If it isn't, the command-line interface is skipped and control
 ; is passed to the TSR installation routine. This routine works by attempting
 ; to send an interrupt service request to the TSR portion of DosView. If the
 ; TSR is installed, a return code of ABCDh is returned in register BX.

        mov     ah,[service]            ; Select TSR's control service.
        mov     al,7                    ; Select function to request return
        int     21h                     ; code.
        cmp     bx,0abcdh               ; Was the return code given?
        je      get_params
        jmp     install_tsr


 get_params:
 ;
 ; Section: 2.2
 ;
 ; This routine retrieves command-line parameters.

        mov     si,82h                  ; Source first byte of params.
        mov     di,offset params        ; Destination variable.
        mov     cx,0                    ; Clear length register.
        mov     cl,[80h]                ; Put param length into CL.
        rep     movsb                   ; Move parameters to variable.
        dec     di                      ; Remove garbage character.

 exist_params:                          ; WERE PARAMETERS GIVEN?
        cmp     b[offset params],0      ; Do params exist?
        jne     remove_space
        jmp     usage

 remove_space:                          ; REMOVE SPACES ON END OF PARAM.
        dec     di                      ; Decrement parameter size.
        cmp     b[di],20h               ; Is the last character a space?
        je      remove_space
        inc     di                      ; Get ready for end-of-variable.
        mov     al,0                    ; Select end-of-variable char.
        stosb                           ; Append end-of-variable char.

                                        ; CONVERT PARAMETERS TO UPPERCASE.
        mov     si,offset params        ; Select parameters for conversion.

 sc_loop:                               ; CONVERT LOOP.
        lodsb                           ; Get byte from block.
        cmp     al,0                    ; If at the end of the string,
        je      parse_params            ; end convert loop.
        mov     bl,al                   ; Make a copy of byte.
        sbb     bl,61h                  ; Is byte < than ASCII 97?
        jc      sc_loop                 ; If so, don't convert.
        mov     bl,al                   ; Make a copy of byte.
        adc     bl,85h                  ; Is byte > than ASCII 122?
        jc      sc_loop                 ; If so, don't convert.
        and     al,0dfh                 ; Convert byte to uppercase.
        mov     di,si                   ; Copy converted bytes.
        dec     di                      ; Reposition string pointer.
        stosb                           ; Move byte back into string.
        jmp     sc_loop                 ; Continue loop.


 parse_params:
 ;
 ; Section: 2.3
 ;
 ; The collection of routines below scan for and act on parameters given at
 ; the command line. Each available parameter has it's own routine. Each
 ; routine carries out it's own scan of the parameters given. If a parameter
 ; request is present for a given routine, the remainder of the code to
 ; execute that request is used, after which control is passed on to the next
 ; parameter routine. If the parameter in question is not given, the code
 ; for execution is skipped and control is passed to the next routine.

        mov     b[param_flag],'0'       ; Clear parameter flag. This flag is
                                        ; set to equal '1' if the command-
                                        ; line parameters given have at least
                                        ; 1 correct parameter in them. If the
                                        ; flag remains equal to '0', an error
                                        ; message is displayed.


 parse_remove:
 ;
 ; Section: 2.3.1
 ;
 ; This routine is used to remove the DosView TSR from memory when the '/R'
 ; parameter is given. Since the removal of the TSR eliminates the need to
 ; execute any other parameters that might be given, this routine terminates
 ; the program after it executes instead of handing control over to the next
 ; routine in sequence.

        mov     di,offset params        ; Start scan at beginning of params.

 parse_remove_loop:                     ; SEARCH FOR A '/' CHARACTER IN PARAMS.
        cmp     b[di],'/'               ; Has a '/' character been found?
        je      parse_remove_act
        cmp     b[di],0                 ; Are we at the end of the string?
        je      parse_top
        inc     di                      ; Goto next byte in parameters.
        jmp     parse_remove_loop

 parse_remove_act:                      ; EXECUTE A TSR REMOVAL.
        inc     di                      ; Goto next byte in parameters.
        cmp     b[di],'R'               ; Has an 'R' character been found?
        jne     parse_remove_loop
        mov     ah,[service]            ; Call TSR removal service.
        mov     al,6
        int     21h
        cr                              ; Print TSR removal message.
        print   usage_1
        cr
        print   remove_msg
        cr
        jmp     end_program             ; End the program.


 parse_top:
 ;
 ; Section: 2.3.2
 ;
 ; This routine is used to set the top line of the DosView's display window
 ; when the '/T' parameter is given. The parameter is followed by a number
 ; from 1 - 25 to indicate the top line. The initial value for the top line
 ; is 5. For this routine to execute properly, the top line must not be
 ; greater or equal to the bottom line.

        mov     di,offset params        ; Start scan at beginning of params.

 parse_top_loop:                        ; SEARCH FOR A '/' CHARACTER.
        cmp     b[di],'/'               ; Has a '/' character been found?
        je      parse_top_act
        cmp     b[di],0                 ; Are we at the end of the string?
        je      parse_top_parse_bottom
        inc     di                      ; Goto next byte in parameters.
        jmp     parse_top_loop

 parse_top_parse_bottom:
        jmp     parse_bottom

 parse_top_act:                         ; EXECUTE A TOP LINE MODIFICATION.
        inc     di                      ; Goto next byte in parameters.
        cmp     b[di],'T'               ; Has a 'T' character been found?
        jne     parse_top_loop

 parse_top_get_line:                    ; GET LINE NUMBER FROM PARAMETERS.
        inc     di                      ; Goto next byte in parameters.
        mov     b[top_line_st],[di]     ; Move first digit into string.
        inc     di                      ; Goto next byte in parameters.
        mov     b[top_line_st+1],[di]   ; Move second digit into string.
        cmp     b[top_line_st+1],'0'    ; Is the second digit < ASCII '0'?
        jl      parse_top_single
        cmp     b[top_line_st+1],'9'    ; Is the second digit > ASCII '9'?
        jg      parse_top_single

 parse_top_double:                      ; DOUBLE DIGIT CONVERSION.
        cmp     b[top_line_st],'0'      ; Is the first digit < ASCII '0'?
        jl      parse_top_error_range
        cmp     b[top_line_st],'9'      ; Is the first digit > ASCII '9'?
        jg      parse_top_error_range
        jmp     parse_top_convert

 parse_top_single:                      ; SINGLE DIGIT CONVERSION.
        cmp     b[top_line_st],'0'      ; Is the first digit < ASCII '0'?
        jl      parse_top_error_range
        cmp     b[top_line_st],'9'      ; Is the first digit > ASCII '9'?
        jg      parse_top_error_range
        mov     al,[top_line_st]        ; Move first digit to second digit.
        mov     [top_line_st+1],al
        mov     [top_line_st],'0'       ; Zero out first digit.

 parse_top_convert:                     ; CONVERT STRING TO TOP LINE.
        mov     al,[top_line_st]        ; Get first byte of string.
        sub     al,30h                  ; Subtract 48 to get true value.
        mov     bl,0ah                  ; BL = Ah (10).
        mul     bl                      ; Multiply first byte by 10.
        mov     ah,[top_line_st+1]      ; Get second byte of string.
        sub     ah,30h                  ; Subtract 48 to get true value.
        add     al,ah                   ; Add the first and second bytes.

 parse_top_int:                         ; CALL THE TOP LINE SERVICE FROM TSR.
        mov     bl,al                   ; BL = New top line position.
        cmp     bl,1                    ; Is the top line < 1?
        jl      parse_top_error_range
        cmp     bl,19h                  ; Is the top line > 25?
        jg      parse_top_error_range
        mov     ah,[service]            ; AH = TSR service.
        mov     al,1                    ; AL = Top line modify function.
        int     21h
        cmp     bh,1                    ; Did an error occur?
        je      parse_top_error_bottom
        mov     [param_flag],'1'        ; Update parameter flag.
        jmp     parse_bottom

 parse_top_error_range:                 ; ERROR MESSAGE DISPLAYED WHEN THE
        cr                              ; TOP LINE SPECIFIED IS OUT OF THE
        print   usage_1                 ; ALLOWABLE RANGE OF 1 - 25.
        cr
        print   range_msg_1
        cr
        jmp     end_program

 parse_top_error_bottom:                ; ERROR MESSAGE DISPLAYED WHEN THE
        cr                              ; TOP LINE SPECIFIED IS GREATER OR
        print   usage_1                 ; EQUAL TO THE BOTTOM LINE.
        cr
        print   bottom_msg
        cr
        jmp     end_program


 parse_bottom:
 ;
 ; Section: 2.3.3
 ;
 ; This routine is used to set the bottom line of the DosView's display
 ; window when the '/B' parameter is given. The parameter is followed by a
 ; number from 1 - 25 to indicate the bottom line. The initial value for the
 ; bottom line is 19. For this routine to execute properly, the bottom line
 ; must not be less than or equal to the top line.

        mov     di,offset params        ; Start scan at beginning of params.

 parse_bottom_loop:                     ; SEARCH FOR A '/' CHARACTER.
        cmp     b[di],'/'               ; Has a '/' character been found?
        je      parse_bottom_act
        cmp     b[di],0                 ; Are we at the end of the string?
        je      parse_bottom_parse_pause
        inc     di                      ; Goto next byte in parameters.
        jmp     parse_bottom_loop

 parse_bottom_parse_pause:
        jmp     parse_pause

 parse_bottom_act:                      ; EXECUTE A BOTTOM LINE MODIFICATION.
        inc     di                      ; Goto next byte in parameters.
        cmp     b[di],'B'               ; Has a 'B' character been found?
        jne     parse_bottom_loop

 parse_bottom_get_line:                 ; GET LINE NUMBER FROM PARAMETERS.
        inc     di                      ; Goto next byte in parameters.
        mov     b[bottom_line_st],[di]  ; Move first digit into string.
        inc     di                      ; Goto next byte in parameters.
        mov     b[bottom_line_st+1],[di]; Move second digit into string.
        cmp     b[bottom_line_st+1],'0' ; Is the second digit < ASCII '0'?
        jl      parse_bottom_single
        cmp     b[bottom_line_st+1],'9' ; Is the second digit > ASCII '9'?
        jg      parse_bottom_single

 parse_bottom_double:                   ; DOUBLE DIGIT CONVERSION.
        cmp     b[bottom_line_st],'0'   ; Is the first digit < ASCII '0'?
        jl      parse_bottom_error_range
        cmp     b[bottom_line_st],'9'   ; Is the first digit > ASCII '9'?
        jg      parse_bottom_error_range
        jmp     parse_bottom_convert

 parse_bottom_single:                   ; SINGLE DIGIT CONVERSION.
        cmp     b[bottom_line_st],'0'   ; Is the first digit < ASCII '0'?
        jl      parse_bottom_error_range
        cmp     b[bottom_line_st],'9'   ; Is the first digit > ASCII '9'?
        jg      parse_bottom_error_range
        mov     al,[bottom_line_st]     ; Move first digit to second digit.
        mov     [bottom_line_st+1],al
        mov     [bottom_line_st],'0'    ; Zero out first digit.

 parse_bottom_convert:                  ; CONVERT STRING TO BOTTOM LINE.
        mov     al,[bottom_line_st]     ; Get first byte of string.
        sub     al,30h                  ; Subtract 48 to get true value.
        mov     bl,0ah                  ; BL = Ah (10).
        mul     bl                      ; Multiply first byte by 10.
        mov     ah,[bottom_line_st+1]   ; Get second byte of string address.
        sub     ah,30h                  ; Subtract 48 to get true value.
        add     al,ah                   ; Add the first and second bytes.

 parse_bottom_int:                      ; CALL THE BOTTOM LINE SERVICE FROM TSR
        mov     bl,al                   ; BL = New bottom line position.
        cmp     bl,1                    ; Is the top line < 1?
        jl      parse_bottom_error_range
        cmp     bl,19h                  ; Is the top line > 25?
        jg      parse_bottom_error_range
        mov     ah,[service]            ; AH = TSR service.
        mov     al,2                    ; AL = Bottom line function.
        int     21h
        cmp     bh,1                    ; Did an error occur?
        je      parse_bottom_error_top
        mov     [param_flag],'1'        ; Update parameter flag.
        jmp     parse_pause

 parse_bottom_error_range:              ; ERROR MESSAGE DISPLAYED WHEN THE
        cr                              ; BOTTOM LINE SPECIFIED IS OUT OF
        print   usage_1                 ; THE ALLOWABLE RANGE OF 1 - 25.
        cr
        print   range_msg_2
        cr
        jmp     end_program

 parse_bottom_error_top:                ; ERROR MESSAGE DISPLAYED WHEN THE
        cr                              ; BOTTOM LINE SPECIFIED IS LESS THAN
        print   usage_1                 ; OR EQUAL TO THE TOP LINE.
        cr
        print   top_msg
        cr
        jmp     end_program


 parse_pause:
 ;
 ; Section: 2.3.4
 ;
 ; This routine is used to temporarily pause DosView's display window when
 ; the '/P' parameter is given. When DosView is in pause mode, all output to
 ; the screen resulting from DOS interrupt calls is not confined to the
 ; display window. It does this by stopping DosView from intercepting DOS
 ; output routines as it normally does and the effect is the equivalent of
 ; not having the DosView TSR in memory at all. Pause mode can be deativated
 ; using the unpause (/U) parameter.
 ;
 ; NOTE: The DosView TSR initially starts out in pause mode when first loaded.

        mov     di,offset params        ; Start scan at beginning of params.

 parse_pause_loop:                      ; SEARCH FOR A '/' CHARACTER.
        cmp     b[di],'/'               ; Has a '/' character been found?
        je      parse_pause_act
        cmp     b[di],0                 ; Are we at the end of the string?
        je      parse_unpause
        inc     di                      ; Goto next byte in parameters.
        jmp     parse_pause_loop

 parse_pause_act:                       ; EXECUTE A PAUSE.
        inc     di                      ; Goto next byte in parameters.
        cmp     b[di],'P'               ; Has an 'P' character been found?
        jne     parse_pause_loop
        mov     ah,[service]            ; Call TSR pause function.
        mov     al,3
        int     21h
        mov     [param_flag],'1'        ; Update parameter flag.
        jmp     parse_unpause


 parse_unpause:
 ;
 ; Section: 2.3.5
 ;
 ; This routine deactivates the pause mode of DosView when the '/U' parameter
 ; is given. It is the opposite of the '/P' parameter above.

        mov     di,offset params        ; Start scan at beginning of params.

 parse_unpause_loop:                    ; SEARCH FOR A '/' CHARACTER.
        cmp     b[di],'/'               ; Has a '/' character been found?
        je      parse_unpause_act
        cmp     b[di],0                 ; Are we at the end of the string?
        je      parse_settings
        inc     di                      ; Goto next byte in parameters.
        jmp     parse_unpause_loop

 parse_unpause_act:                     ; EXECUTE AN UNPAUSE.
        inc     di                      ; Goto next byte in parameters.
        cmp     b[di],'U'               ; Has an 'U' character been found?
        jne     parse_unpause_loop
        mov     ah,[service]            ; Call TSR unpause function.
        mov     al,4
        int     21h
        mov     [param_flag],'1'        ; Update parameter flag.
        jmp     parse_settings


 parse_settings:
 ;
 ; Section: 2.3.6
 ;
 ; This routine displays the current DosView TSR settings when the '/S'
 ; parameter is given. Four pieces of information are displayed which
 ; include: the top and bottom lines for the display window, the current
 ; interrupt service code assigned to DosView and the current pause status.

        mov     di,offset params        ; Start scan at beginning of params.

 parse_settings_loop:                   ; SEARCH FOR A '/' CHARACTER.
        cmp     b[di],'/'               ; Has a '/' character been found?
        je      parse_settings_act
        cmp     b[di],0                 ; Are we at the end of the string?
        je      parse_settings_clear
        inc     di                      ; Goto next byte in parameters.
        jmp     parse_settings_loop

 parse_settings_clear:
        jmp     parse_clear

 parse_settings_act:                    ; SHOW SETTINGS.
        inc     di                      ; Goto next byte in parameters.
        cmp     b[di],'S'               ; Has an 'S' character been found?
        jne     parse_settings_loop
        cr                              ; Display status header.
        print   usage_1
        cr
        print   set_msg_1
        cr
        print   set_msg_2               ; Display top line message.
        mov     ah,[service]            ; Call TSR status function.
        mov     al,5
        int     21h
        push    ax                      ; Save status.
        push    bx
        push    cx
        mov     al,ah                   ; Get top line.
        call    byte_to_dec             ; Convert top line to a string.
        print   decimal                 ; Display top line.
        cr
        print   set_msg_3               ; Display bottom line message.
        pop     cx                      ; Restore status.
        pop     bx
        pop     ax
        push    bx                      ; Save status.
        push    cx
        call    byte_to_dec             ; Convert bottom line to a string.
        print   decimal                 ; Display bottom line.
        cr
        print   set_msg_4               ; Display service code message.
        pop     cx                      ; Restore status.
        pop     bx
        push    cx                      ; Save status.
        mov     al,bl                   ; Get service code.
        call    byte_to_hex             ; Convert service code to a string.
        print   hexadecimal
        cr
        print   set_msg_5
        pop     cx
        cmp     cl,0                    ; Is the TSR paused?
        jne     parse_settings_npause

 parse_settings_pause:                  ; DISPLAY THE PAUSED MESSAGE.
        print   set_msg_6
        cr
        mov     [param_flag],'1'        ; Update parameter flag.
        jmp     parse_clear

 parse_settings_npause:                 ; DISPLAY THE NOT PAUSED MESSAGE.
        print   set_msg_7
        cr
        mov     [param_flag],'1'        ; Update parameter flag.
        jmp     parse_clear


 parse_clear:
 ;
 ; Section: 2.3.7
 ;
 ; This routine is used to clear the display window of DosView when the '/C'
 ; parameter is given. Since most screen clearing routines such as the DOS
 ; 'CLS' command clear the whole screen no matter what, this routine was
 ; added to allow just the display window to be cleared.

        mov     di,offset params        ; Start scan at beginning of params.

 parse_clear_loop:                      ; SEARCH FOR A '/' CHARACTER.
        cmp     b[di],'/'               ; Has a '/' character been found?
        je      parse_clear_act
        cmp     b[di],0                 ; Are we at the end of the string?
        je      parse_flag
        inc     di                      ; Goto next byte in parameters.
        jmp     parse_clear_loop

 parse_clear_act:                       ; CLEAR WINDOW.
        inc     di                      ; Goto next byte in parameters.
        cmp     b[di],'C'               ; Has a 'C' character been found?
        jne     parse_clear_loop
        mov     ah,0ffh                 ; Execute TSR clear window service.
        mov     al,8
        int     21h
        mov     [param_flag],'1'        ; Update parameter flag.
        jmp     parse_flag


 parse_flag:
 ;
 ; Section: 2.3.8
 ;
 ; This routine is used after all of the above parameters have been retrieved
 ; and executed to act on the current condition of the parameter flag. If the
 ; flag was set by one of the above routines, it means at least one parameter
 ; was given correctly and the program will terminate normally. On the other
 ; hand, if the flag wasn't set, it means that none of the parameters given
 ; by the user were correct and an error message is displayed to indicate so.
 ; This routine is here as a simple convenience to the user. It isn't
 ; absolutely required since any non-existant parameters given will be
 ; ignored by the above routines anyway.

        cmp     [param_flag],'0'        ; Were there valid parameters?
        je      parse_bad_params
        jmp     end_program

 parse_bad_params:                      ; PRINT BAD PARAMETER ERROR MESSAGE.
        cr
        print   usage_1
        cr
        print   badparam
        cr
        print   usage_2
        cr
        print   usage_3
        print   usage_4
        print   usage_5
        print   usage_6
        print   usage_7
        print   usage_8
        print   usage_9
        print   usage_10
        print   usage_11
        print   usage_12
        cr
        jmp     end_program


 usage:
 ;
 ; Section: 2.4
 ;
 ; This routine prints a welcome message and a command-line parameter summary
 ; when the user activates DosView without parameters.

        cr
        print   usage_1
        cr
        print   usage_2
        cr
        print   usage_3
        print   usage_4
        print   usage_5
        print   usage_6
        print   usage_7
        print   usage_8
        print   usage_9
        print   usage_10
        print   usage_11
        print   usage_12
        cr
        jmp     end_program


 end_program:
 ;
 ; Section: 2.5
 ;
 ; This routine terminates the command-line interface of DosView and returns
 ; control to DOS.

        bye


 ; START OF SUB-ROUTINES FOR COMMAND-LINE INTERFACE.
 ;
 ; Section: 2.6
 ;
 ; The routines below are used by section 2.3.6 (parse_settings:) to convert
 ; the settings information into strings that can be displayed on the screen.


 byte_to_dec:
 ;
 ; Section: 2.6.1
 ;
 ; This sub-routine takes a byte given in register AL and converts it to a
 ; decimal string stored in the 'decimal' variable. It does this by setting
 ; up a simple counting routine. Since the range a byte can represent is 0
 ; through 255 (0h - FFh), three registers (AH,BL and BH) are used to hold
 ; the three digits possible in a decimal string. The registers all start out
 ; equal to ASCII character 30h (0). To convert the byte to a string, this
 ; routine simple increments each register in sequence until the registers
 ; are set to the proper ASCII values (simply put, this routine can be
 ; compared to the odometer on the dashboard in your car).

        mov     ah,30h                  ; Initialize first digit of string.
        mov     bl,30h                  ; Initialize second digit of string.
        mov     bh,30h                  ; Initialize third digit of string.

 byte_to_dec_1:                         ; GET FIRST DIGIT OF STRING.
        cmp     ah,39h                  ; Is the ASCII character > 9?
        jg      byte_to_dec_2
        cmp     al,0                    ; Are we down to 0 yet?
        je      byte_to_dec_4
        inc     ah                      ; Increment ASCII character by 1.
        dec     al                      ; Decrement address by 1.
        jmp     byte_to_dec_1

 byte_to_dec_2:                         ; GET SECOND DIGIT OF STRING.
        inc     bl                      ; Increment BL by 1.
        mov     ah,30h                  ; Reset AH.
        cmp     bl,39h                  ; Is BL > 9?
        jg      byte_to_dec_3
        jmp     byte_to_dec_1

 byte_to_dec_3:                         ; GET THIRD DIGIT OF STRING.
        inc     bh                      ; Increment BH by 1.
        mov     bl,30h                  ; Reset BL.
        jmp     byte_to_dec_1

 byte_to_dec_4:                         ; TRANSFER STRING TO VARIABLE.
        mov     [decimal],bh            ; 1st byte of string = BH.
        mov     [decimal+1],bl          ; 2nd byte of string = BL.
        mov     [decimal+2],ah          ; 3rd byte of string = AH.
        cmp     b[decimal],'0'          ; Was the 1st byte of string needed?
        jne     byte_to_dec_6
        cmp     b[decimal+1],'0'        ; Was the 2nd byte of string needed?
        jne     byte_to_dec_5
        mov     al,[decimal+2]          ; Get third byte and...
        mov     [decimal],al            ; move it to the first position.
        mov     [decimal+1],'$'         ; Mark end of string.
        jmp     byte_to_dec_6

 byte_to_dec_5:
        mov     al,[decimal+1]          ; Get second byte and...
        mov     [decimal],al            ; move it to the first position.
        mov     al,[decimal+2]          ; Get third byte and...
        mov     [decimal+1],al          ; move it to the second position.
        mov     [decimal+2],'$'         ; Mark end of string.
        jmp     byte_to_dec_6

 byte_to_dec_6:
        ret


 byte_to_hex:
 ;
 ; Section: 2.6.2
 ;
 ; This sub-routine converts a byte given in register AL and converts it to a
 ; hexadecimal string stored in the 'hexadecimal' variable. The routine works
 ; by first separating the high and low nibbles of the byte. The high and low
 ; nibbles are then increased by a certain amount to obtain the equivalent
 ; ASCII characters (48 is added if the nibble is between 0 - 9 or 55 is
 ; added if the nibble is between A - F).


        mov     ah,0                    ; Initialize AH.

 byte_to_hex_1:
        mov     bl,10h                  ; Divide AL by 10h.
        div     bl
        push    ax                      ; Save result of division.
        cmp     al,0ah                  ; Is the quotient >= 10?
        jge     byte_to_hex_2
        add     al,30h                  ; Add 48 to quotient to get ASCII char.
        jmp     byte_to_hex_3

 byte_to_hex_2:
        add     al,37h                  ; Add 55 to quotient to get ASCII char.
        jmp     byte_to_hex_3

 byte_to_hex_3:
        mov     [hexadecimal],al        ; Set first byte of string.
        pop     ax                      ; Restore result of division.
        cmp     ah,0ah                  ; Is the remainder >= 10?
        jge     byte_to_hex_4
        add     ah,30h                  ; Add 48 to remainder to get ASCII chr.
        jmp     byte_to_hex_5

 byte_to_hex_4:
        add     ah,37h                  ; Add 55 to remainder to get ASCII chr.
        jmp     byte_to_hex_5

 byte_to_hex_5:
        mov     [hexadecimal+1],ah      ; Set second byte of string.
        ret

 ; ______________________ END OF COMMAND-LINE INTERFACE ______________________



 ; ______________________ START OF TSR INSTALLATION CODE _____________________
 ;
 ; Section: 3
 ;
 ; This section is used to install the TSR portion of DosView in memory. It
 ; also reroutes the vector for interrupt 21h and sets up software interrupt
 ; routines for interrupts 10h and 21h.

 install_tsr:


 set_int_10:
 ;
 ; Section: 3.1
 ;
 ; This routine is used to set up a sub-routine for the TSR portion of
 ; DosView that executes service calls to interrupt 10h. Since it's not
 ; possible to directly call an interrupt using the INT statement within a
 ; TSR or any other routine that was called using the INT statement, the CALL
 ; statement must be used in it's place. Since call statements use a memory
 ; address instead of an interrupt number as a parameter, we must set up the
 ; sub-routine by retrieving the memory address for interrupt 10h from the
 ; interrupt vector table and saving it for later use. The method used for
 ; saving this address is rather unique. The most common method for saving
 ; the address is to store it in a variable. Instead of doing this, a
 ; self-modification routine is used. The routine retrieves the address from
 ; the interrupt vector table and stores it at a memory location in the TSR
 ; code itself starting at the 4th byte from the program label 'int_10'. By
 ; examining the code at this label, we can see that the 4th byte points to
 ; the second byte in a call statement. The call statement initially reads
 ; 'CALL 0000:0000'. Once this routine has completed executing, '0000:0000'
 ; will be replaced by the actual address for interrupt 10h. Overall, this
 ; technique is about equal to using variables. It has a slight advantage of
 ; not requiring the setup of variables and the disadvantage that care must
 ; be taken to make sure that the call statement remains just after the label
 ; specified.

        mov     ds,0                    ; Source segment = 0.
        mov     si,40h                  ; Source offset = 40h (Int 10h).
        mov     es,cs                   ; Destination segment = program.
        mov     di,offset int_10+3      ; Destination offset = Int 10h+3.
        mov     cx,4                    ; Copy 4 bytes.
        rep     movsb


 set_int_21:
 ;
 ; Section: 3.2
 ;
 ; This routine is used to set up a sub-routine for the TSR portion of
 ; DosView that executes service calls to interrupt 21h. It works in the
 ; same manner as the routine in section 3.1 above.

        mov     ds,0                    ; Source segment = 0.
        mov     si,84h                  ; Source offset = 84h (Int 21h).
        mov     es,cs                   ; Destination segment = program.
        mov     di,offset int_21+3      ; Destination offset = Int 21h+3.
        mov     cx,4                    ; Copy 4 bytes.
        rep     movsb


 get_vector:
 ;
 ; Section: 3.3
 ;
 ; This routine is used to retrieve and save the old interrupt vector address
 ; for 21h before it's modified. It uses the same self-modification routine
 ; as the last two routines above to save the address to a JMP instruction
 ; located at the offset of the 'cont' label. This allows the original
 ; interrupt vector to be executed after the TSR is finished doing it's work.
 ; This routine also makes a copy of interrupt 21h for use if a request is
 ; made to remove the TSR from memory.

        mov     ds,0                    ; Source segment = 0.
        mov     si,84h                  ; Source offset = 84h (Int 21h).
        mov     es,cs                   ; Destination segment = program
        mov     di,offset cont+1        ; Destination offset = end of tsr.
        mov     cx,4                    ; Copy 4 bytes.
        rep     movsb
        mov     si,84h                  ; Reset source offset to 84h.
        mov     di,offset old_vector    ; Destination offset = old_vector.
        mov     cx,4                    ; Copy 4 bytes.
        rep     movsb                   ; Make a copy for deinstallation.


 set_vector:
 ;
 ; Section: 3.4
 ;
 ; Once the old vector is saved, this routine is used to reset interrupt
 ; vector 21h to point towards the TSR portion of DosView. Service 25h of
 ; interrupt 21h is used to alter the vector.

        mov     ah,25h                  ; DOS Service 25h (Set vector)
        mov     al,21h                  ; Alter interrupt 21h (DOS)
        mov     ds,cs                   ; New segment = program.
        mov     dx,offset tsr           ; New offset = beginning of tsr.
        int     21h


 terminate:
 ;
 ; Section: 3.5
 ;
 ; This routine uses Interrupt 27h to make DosView into a TSR. The pointer
 ; to the last byte of the program is set to 5 bytes after the program label
 ; 'last_byte'.

        cr                              ; Print TSR installed message.
        print   usage_1
        cr
        print   installed_1
        print   installed_2
        cr
        mov     dx,offset last_byte+5   ; End of TSR.
        int     27h                     ; Terminate.

 ; _______________________ END OF TSR INSTALLATION CODE ______________________



 ; ______________________________ START OF TSR _______________________________
 ;
 ; Section: 4
 ;
 ; This is the actual TSR portion of DosView. It works by intercepting DOS
 ; interrupt 21h and modifying the output from the output services (2h, 6h,
 ; 9h and 40h). The TSR also features functions to modify parameters and
 ; return status information. These functions are implimented by creating a
 ; new interrupt 21h service code that can be called by the DosView command-
 ; line interface or from another program. Initially, the new service code is
 ; assigned a value of FFh. This code is kept in a variable named "service"
 ; and can be modified to any value desired. Since DOS uses most of the lower
 ; service codes for itself, it is recomended to keep the code between F0h
 ; and FFh.

 tsr:
        push_a                          ; Save registers from calling program.


 control_scan:
 ;
 ; Section: 4.1
 ;
 ; This routine checks register AH to see if the Interrupt 21h has been called
 ; to execute a DosView service function.

        cmp     ah,cs:[service]         ; Does AH = the service function?
        je      controls


 read_pause_flag:
 ;
 ; Section: 4.2
 ;
 ; This routine checks to see if DosView is currently in pause mode. If it
 ; is, the output modification routines are skipped.

        cmp     cs:b[pause_flag],'1'    ; Are we currently paused?
        je      intercept_services
        jmp     restore_registers


 intercept_services:
 ;
 ; Section: 4.3
 ;
 ; This routine checks to see if one of the DOS output services was requested
 ; by the calling program. If so, control is passed to the output modification
 ; routines. If not, the output modification routines are ignored.

        cmp     ah,40h                  ; Scan for DOS file write service.
        je      modify_outputj
        cmp     ah,9h                   ; Scan for DOS String output.
        je      modify_outputj
        cmp     ah,6h                   ; Scan for DOS Console I/O.
        je      modify_outputj
        cmp     ah,2h                   ; Scan for DOS character output.
        je      modify_outputj
        jmp     restore_registers

 modify_outputj:
        jmp     modify_output


 controls:
 ;
 ; Section: 4.4
 ;
 ; If a request has been made to execute A DosView service function, this
 ; routine is used to find out which function has been requested by examining
 ; register AL.

        cmp     al,1                    ; Scan for top row set.
        je      top_row
        cmp     al,2                    ; Scan for bottom row set.
        je      bottom_row
        cmp     al,3                    ; Scan for pause.
        je      pause
        cmp     al,4                    ; Scan for unpause.
        je      unpause
        cmp     al,5                    ; Scan for status request.
        je      status
        cmp     al,6                    ; Scan for remove TSR
        je      deinstallj
        cmp     al,7                    ; Scan for send install flag.
        je      send_flagj
        cmp     al,8                    ; Scan for clear window.
        je      clearj
        jmp     interrupt_return_fail

 deinstallj:
        jmp     deinstall

 send_flagj:
        jmp     send_flag

 clearj:
        jmp     clear


 tsr_parameters:
 ;
 ; Section: 4.5
 ;
 ; The routines below are used to alter the TSR's parameters and send back
 ; status information.


 top_row:
 ;
 ; Section: 4.5.1
 ;
 ; This routine is used to modify the current setting for the top line of
 ; the DosView display window. The routine expects the top line to be
 ; specified in register BL. Valid values for BL are 1 - 25 (1h - 19h). Once
 ; completed, this routine stores the new top line value in the 'top_line'
 ; variable.

        cmp     bl,1                    ; Is the top line < 1?
        jl      top_row_error
        cmp     bl,19h                  ; Is the top line > 25?
        jg      top_row_error
        dec     bl                      ; Subtract 1 from top line (0 base).
        cmp     bl,cs:[bottom_line]     ; Is the top line >= the bottom line?
        jge     top_row_error
        mov     cs:[top_line],bl        ; Save top line.
        jmp     interrupt_return_ok

 top_row_error:                         ; ERROR WHILE MODIFYING TOP ROW.
        jmp     interrupt_return_fail


 bottom_row:
 ;
 ; Section: 4.5.2
 ;
 ; This routine is used to modify the current setting for the bottom line of
 ; the DosView display window. The routine expects the bottom line to be
 ; specified in register BL. Valid values for BL are 1 - 25 (1h - 19h). Once
 ; completed, this routine stores the new top line value in the 'bottom_line'
 ; variable.

        cmp     bl,1                    ; Is the bottom line < 1?
        jl      bottom_row_error
        cmp     bl,19h                  ; Is the bottom line > 25?
        jg      bottom_row_error
        dec     bl                      ; Subtract 1 from bottom line.
        cmp     bl,cs:[top_line]        ; Is the bottom line <= the top line?
        jle     bottom_row_error
        mov     cs:[bottom_line],bl     ; Save bottom line.
        jmp     interrupt_return_ok

 bottom_row_error:                      ; ERROR WHILE MODIFYING THE BOTTOM ROW.
        jmp     interrupt_return_fail


 pause:
 ;
 ; Section: 4.5.3
 ;
 ; This routine is used to activate the pause mode. In this mode, DosView's
 ; windowing routines are deactivated. The pause mode is activated by simply
 ; altering the variable named 'pause_flag' to equal '0'.

        mov     cs:[pause_flag],'0'
        jmp     interrupt_return_ok


 unpause:
 ;
 ; Section: 4.5.4
 ;
 ; This routine deactivates the pause mode so that DosView's windowing
 ; routines are reactivated.

        mov     cs:[pause_flag],'1'
        jmp     interrupt_return_ok


 status:
 ;
 ; Section: 4.5.5
 ;
 ; This routine sends back information to the calling program concerning the
 ; current parameter settings of DosView. The following information is sent
 ; back in the indicated registers:
 ;
 ; AH = Top line of window.             AL = Bottom line of window.
 ; BL = Interrupt service code.         CL = Pause status.  '1' = activated
 ;                                                          '0' = deactivated

        pop_a
        mov     ah,cs:[top_line]        ; AH = Top line.
        inc     ah                      ; Add 1 to top line (remove 0 base).
        mov     al,cs:[bottom_line]     ; AL = Bottom line.
        inc     al                      ; Add 1 to bottom line (remove 0 base).
        mov     bh,0                    ; BH = Return code (success).
        mov     bl,cs:[service]         ; BL = Service code.
        mov     cl,cs:[pause_flag]      ; CL = Pause status.
        sub     cl,30h                  ; Subtract 48 to get true value.
        iret


 deinstall:
 ;
 ; Section: 4.5.6
 ;
 ; This routine removes the DosView TSR from memory. It was designed to be
 ; quick and dirty and makes no attempt to see whether it's possible to
 ; remove the TSR from memory at the current time. The routine simply resets
 ; the old vector for interrupt 21h using the data stored in the 'old_vector'
 ; variable and then frees up the memory allocated to the TSR by calling
 ; interrupt 21h, service 49h. Whether or not the memory used by the TSR
 ; actually is recovered, the TSR will be permanently deactivated since the
 ; interrupt vector to the TSR will be lost.

        mov     ds,cs                   ; Set data segment to TSR.
        mov     si,offset old_vector    ; Source offset = old_vector.
        mov     es,0                    ; Destination segment = 0.
        mov     di,84h                  ; Destination offset = 84h (Int 21h)
        mov     cx,4                    ; Copy 4 bytes.
        rep     movsb                   ; Reset vector.
        mov     ah,49h                  ; DOS Service 49h (Free allocated mem).
        mov     es,cs                   ; Free memory used by TSR.
        call    int_21
        jmp     interrupt_return_ok


 send_flag:
 ;
 ; Section: 4.5.7
 ;
 ; This routine provides a method of determining whether or not the TSR is
 ; present in memory. The routine below sends back the address ABCDh in
 ; register BX. If the TSR is not in memory, neither is this routine and
 ; thus, the above address wouldn't be sent back. This routine is mainly
 ; used by the command-line interface portion of DosView to stop the TSR
 ; portion from being installed more than once.

        pop_a
        mov     bx,0abcdh               ; Flag = ABCD.
        iret


 clear:
 ;
 ; Section: 4.5.8
 ;
 ; This routine clears the lines of the screen that are currently within
 ; DosView's display window. Interrupt 21h, service 6 (BIOS scroll window up
 ; routine) is used to clear the display window. This routine was chosen
 ; since it seemed to be an easy and standardized way of clearing the screen.

        mov     ah,0fh                  ; BIOS Service 0Fh (get video state).
        call    int_10
        dec     ah                      ; Sub 1 from screen width (0 base).
        mov     cs:[screen_width],ah    ; Save width of screen.
        mov     ah,8                    ; Read char attribute at cursor pos.
        mov     bh,0
        call    int_10
        mov     bh,ah                   ; Move screen attribute to BH.
        mov     ah,6                    ; Select BIOS service 6.
        mov     ch,cs:[top_line]        ; Top line of window to scroll.
        mov     dh,cs:[bottom_line]     ; Bottom line of window to scroll.
        mov     al,dh                   ; Number of lines to scroll = ...
        sub     al,ch                   ; bottom line - top line ...
        inc     al                      ; + 1
        mov     cl,0                    ; Left-most column = 1.
        mov     dl,cs:[screen_width]    ; Right-most column = screen width.
        call    int_10
        jmp     interrupt_return_ok


 modify_output:
 ;
 ; Section: 4.6
 ;
 ; The routines below represent the main function of DosView - to confine
 ; output from DOS-based display routines to only certain lines on the
 ; screen. At the current time, DosView will intercept output from any
 ; program that sends it's output to the screen through the DOS output
 ; services accessable through interrupt 21h. The services intercepted are:
 ; 2h (output character), 6h (direct console I/O), 9h (output string) and 40h
 ; (write file). To keep things as simple as possible, all output from any of
 ; these services is rerouted through service 40h (write file). If 2h, 6h or
 ; 9h are used, the data to be output is reformatted so that it can be
 ; handled by service 40h. This allows all of the screen modifications to be
 ; centered around a single routine instead of four individual ones.


 modify_output_redir:
 ;
 ; Section 4.6.1
 ;
 ; Interrupt 21h, service 40h (write file) is used for more than just writing
 ; output to the screen. It is also used for writing to disk files, printers
 ; and other output devices. Furthermore, the other services are capable of
 ; having their output redirected towards other devices besides the screen.
 ; It is the job of this routine to only allow the remaining output
 ; modification routines to execute if the output is actually going to the
 ; screen. To do this, interrupt 21h, service 44h (get device information) is
 ; used to examine the file handle for the device that the output is being
 ; sent to (unless DOS service 40h is being used, this file handle is always
 ; equal to 1 which is STDOUT). Information about this file handle is
 ; returned in DX. If Bit 0 (console device specifier) and bit 7 (disk
 ; file/device indicator) of register DL are set correctly, the remaining
 ; routines are allowed to execute.

        push_a                          ; Save registers.
        cmp     ah,40h                  ; Was DOS service 40h called?
        je      modify_output_redir2
        mov     bx,1                    ; Set file handle to 1.
 modify_output_redir2:
        mov     ah,44h                  ; Select DOS service 44h...
        mov     al,0                    ; Function 0 (Get device information).
        call    int_21
        and     dl,82h                  ; Separate out console and file bits.
        cmp     dl,82h                  ; Is this a console device, no file?
        jne     modify_output_restore
        pop_a                           ; Restore registers.
        cmp     ah,2                    ; Was DOS service 2 used for output?
        je      modify_output_2_6
        cmp     ah,6                    ; Was DOS service 6 used for output?
        je      modify_output_2_6
        cmp     ah,9                    ; Was DOS service 9 used for output?
        je      modify_output_9
        jmp     modify_output_40

 modify_output_restore:
        pop_a
        jmp     restore_registers


 modify_output_2_6:
 ;
 ; Section: 4.6.2
 ;
 ; This routine makes the necessary conversions needed for DOS services 2h
 ; and 6h to be executed as service 40h. Since service 6h (direct console
 ; I/O) can also be used for input, the routine will also reroute input
 ; requests.

        cmp     ah,6                    ; Is the service direct console I/O?
        jne     modify_output_2_6b
        cmp     dl,0ffh                 ; Is this an input request?
        je      modify_output_2_6r

 modify_output_2_6b:                    ; CONVERT TO 40H.
        mov     cx,1                    ; Write 1 byte to string.
        mov     ds,cs                   ; Set string code segment to TSR.
        mov     [char_string],dl        ; Save character to string.
        mov     dx,offset char_string   ; Set string offset to variable.
        jmp     modify_output_40

 modify_output_2_6r:
        jmp     restore_registers


 modify_output_9:
 ;
 ; Section: 4.6.3
 ;
 ; This routine makes the necessary conversions needed for DOS service 9h
 ; to be executed as service 40h.

        mov     di,dx                   ; Initialize string position.
        mov     cx,0                    ; Initialize bytes to write.

 modify_output_9_loop:                  ; Search for end of string.
        cmp     b[di],'$'               ; Are we at the end of the string?
        je      modify_output_40
        inc     cx                      ; Increment bytes to write by 1.
        inc     di                      ; Increment string position by 1.
        jmp     modify_output_9_loop


 modify_output_40:
 ;
 ; Section: 4.6.4
 ;
 ; This is the primary routine for modification of output from DOS interrupt
 ; calls. It handles service 40h (write file) as well as the output from
 ; services 2h, 6h and 9h which by now have been converted by the previous
 ; routines to work as though they were originally a call to service 40h.
 ; This routine works by taking apart the string that was to be output by DOS
 ; service 40h and sending the text to the screen character-by-character.
 ; This method results in slower output, but it appears to be the best method
 ; available.

        push_a                          ; Save registers.
        mov     ah,0fh                  ; BIOS Service 0Fh (Get video state).
        call    int_10
        mov     cs:[display_page],bh    ; Save current display page.
        dec     ah                      ; Sub 1 from screen width (0 base).
        mov     cs:[screen_width],ah    ; Save width of screen.
        pop_a                           ; Restore registers.
        mov     di,dx                   ; Initialize string pointer.

 modify_output_40_loop:                 ; OUTPUT STRING.
        cmp     cx,0                    ; Are we at the end of the string?
        je      modify_output_40_end
        cmp     b[di],0ah               ; Has a linefeed character been found?
        je      modify_output_40_lf
        push_a                          ; Save registers.
        mov     ah,3                    ; BIOS Service 3 (Get cursor position).
        mov     bh,cs:[display_page]    ; Video page = current.
        call    int_10
        cmp     b[di],0dh               ; Is the next byte for output a CR?
        je      modify_output_40_loop2
        cmp     dl,cs:[screen_width]    ; Is the cursor at the last column?
        je      modify_output_40_wrap
 modify_output_40_loop2:
        pop_a                           ; Restore registers.
        mov     ah,2                    ; DOS Service 2 (output character).
        mov     dl,[di]                 ; Character to output.
        call    int_21
        inc     di                      ; Update string pointer.
        dec     cx                      ; Update bytes to write.
        jmp     modify_output_40_loop

 modify_output_40_end:                  ; END OF OUTPUT.
        pop_a                           ; Restore registers.
        cmp     ah,40h                  ; Was the original DOS service 40h?
        jne     modify_output_40_iret
        mov     cx,0                    ; Zero out bytes to write.
        jmp     cont

 modify_output_40_iret:
        iret

 modify_output_40_wrap:                 ; HANDLE COLUMN WRAPPING.
        push    dx                      ; Save cursor position information.
        mov     ah,0ah                  ; BIOS Service 0Ah (Write character).
        mov     al,b[di]                ; Character to write.
        mov     bh,cs:[display_page]    ; Video page = current.
        mov     cx,1                    ; Write 1 character.
        call    int_10
        pop     dx                      ; Restore cursor position information.
        mov     ah,2                    ; BIOS Service 2 (Set cursor position).
        mov     bh,cs:[display_page]    ; Video page = current.
        mov     dl,0                    ; Cursor column = 1.
        call    int_10
        pop_a                           ; Restore registers.
        mov     b[di],0ah               ; Change current byte of string to LF.

 modify_output_40_lf:                   ; HANDLE LINEFEEDS.
        push_a                          ; Save registers.
        mov     ah,3                    ; BIOS Service 3 (read cursor pos).
        mov     bh,cs:[display_page]    ; Video page = current.
        call    int_10
        cmp     dh,cs:[bottom_line]     ; Is dh >= bottom line of window?
        jge     modify_output_40_scroll
        pop_a                           ; Restore registers.
        mov     ah,2                    ; DOS Service 2 (output character).
        mov     dl,[di]                 ; Character to output.
        call    int_21
        inc     di                      ; Update string pointer.
        dec     cx                      ; Update bytes to write.
        jmp     modify_output_40_loop

 modify_output_40_scroll:               ; SCROLL WINDOW.
        mov     ah,8                    ; BIOS Service 8 (read attribute).
        mov     bh,cs:[display_page]    ; Video page = current.
        call    int_10
        mov     bh,ah                   ; Move attribute byte to BH.
        mov     ah,6                    ; BIOS Service 6 (scroll window up).
        mov     al,1                    ; Scroll 1 line.
        mov     ch,cs:[top_line]        ; Top row of window.
        mov     cl,0                    ; Left column = 1.
        mov     dh,cs:[bottom_line]     ; Bottom row of window.
        mov     dl,cs:[screen_width]    ; Right column = width of screen.
        call    int_10
        mov     ah,3                    ; BIOS Service 3 (get cursor position).
        mov     bh,cs:[display_page]    ; Video page = current.
        call    int_10
        mov     ah,2                    ; BIOS Service 2 (set cursor position).
        mov     bh,cs:[display_page]    ; Video page = current.
        mov     dh,cs:[bottom_line]     ; Cursor row = bottom row of window.
        dec     dh                      ; Subtract 1 from cursor row.
        call    int_10
        pop_a                           ; Restore registers.
        mov     ah,2                    ; DOS Service 2 (output character).
        mov     dl,[di]                 ; Character to output.
        call    int_21
        inc     di                      ; Update string pointer.
        dec     cx                      ; Update bytes to write.
        jmp     modify_output_40_loop

 int_10:
 ;
 ; Section: 4.7
 ;
 ; This sub-routine allows the execution of a call to interrupt 10h from
 ; within a TSR.

        pushf                           ; Save flag register.
        sti                             ; Set interrupt flag.
        call 0000:0000                  ; Call interrupt.
        ret                             ; Continue.


 int_21:
 ;
 ; Section: 4.8
 ;
 ; This sub-routine allows the execution of a call to interrupt 21h from
 ; within a TSR.

        pushf                           ; Save flag register.
        sti                             ; Set interrupt flag.
        call 0000:0000                  ; Call interrupt.
        ret                             ; Continue.


 interrupt_return_ok:
 ;
 ; Section: 4.9
 ;
 ; This routine is used by the parameter setting options of the TSR to return
 ; control to the calling program. A return code of 0 in BH is sent to
 ; indicate that no errors occurred.

        pop_a
        mov     bh,0                    ; Return code = 0 (success).
        iret


 interrupt_return_fail:
 ;
 ; Section: 4.10
 ;
 ; This routine is used by the parameter setting options of the TSR to return
 ; control to the calling program. A return code of 1 in BH is sent to
 ; indicate that an error occurred.

        pop_a
        mov     bh,1                    ; Return code = 1 (failure).
        iret


 restore_registers:
 ;
 ; Section: 4.11
 ;
 ; This routine is used to restore the registers to their original form before
 ; the TSR was called.

        pop_a
        jmp     cont


 cont:
 ;
 ; Section: 4.12
 ;
 ; This routine hands control over to the original interrupt routine that
 ; would have been executed if the TSR wasn't installed.

        jmp     0000:0000

 ; _______________________________ END OF TSR ________________________________



 ; Data Segment.

        .DATA



 ; ____________________________ START OF VARIABLES ___________________________
 ;
 ; Section: 5
 ;
 ; The variables below are used throughout most sections of DosView. The
 ; first three variables below and their position next to each other are
 ; important to note. The 'service' variable is used to specify the Int 21h
 ; service code that the TSR portion of DosView will look for to execute
 ; parameter alteration requests (initially set to FFh). Surrounding this are
 ; the variables 'service_start' and 'service_end'. You won't find these
 ; variables used anywhere in this source code, but it is used by the
 ; VIEWSET.COM program included with DosView. ViewSet is used to alter the
 ; service code inside of the assembled .COM file version of DosView. ViewSet
 ; will use these variables to find the position of the service code within
 ; the .COM file.

 service_start  db      13,10,'(SERVICE>'
 service        db      0ffh            ; Interrupt service code.
 service_end    db      '<SERVICE)',13,10
 display_page   db      0               ; Active display page.
 screen_width   db      0               ; Width of screen in columns.
 params         db      255 dup(0)      ; Command line parameters.
 param_flag     db      '0'             ; Parameter flag.
 pause_flag     db      '0'             ; Pause indicator flag.
 top_line       db      4               ; Top of display (line 5, base 0).
 top_line_st    db      '00'            ; Top of display (string version).
 bottom_line    db      18              ; Bottom of display (line 19, base 0).
 bottom_line_st db      '00'            ; Bottom of display (string version).
 decimal        db      '000$'          ; Converted dec string (byte_to_dec).
 hexadecimal    db      '00h$'          ; Converted hex string (byte_to_hex).
 old_vector     db      4 dup(0)        ; Old Int 21h vector for deinstall.
 char_string    db      0               ; String to hold data from char output.
 installed_1    db      ' DOSVIEW TSR INSTALLED - Enter the dosview command again by itself to see a',13,10,'$'
 installed_2    db      '                         list of available command-line options.',13,10,'$'
 remove_msg     db      ' DOSVIEW TSR REMOVED FROM MEMORY.',13,10,'$'
 usage_1        db      ' DosView V1.0 - by Steven H. Fisk - January, 1995',13,10,'$'
 usage_2        db      ' USAGE: dosview <PARAMETERS>',13,10,'$'
 usage_3        db      ' Available <PARAMETERS>:',13,10,'$'
 usage_4        db      '  /t<#> = Specify top line of window. Replace <#> with a column number from',13,10,'$'
 usage_5        db      '          1-25. (eg. /t5 sets the top line to 5).',13,10,'$'
 usage_6        db      '  /b<#> = Specify bottom line of window. Replace <#> with a column number',13,10,'$'
 usage_7        db      '          from 1-25. (eg. /b19 Sets the bottom line to 19).',13,10,'$'
 usage_8        db      '     /p = Temporarily pause DosView',39,'s windowing routines.',13,10,'$'
 usage_9        db      '     /u = Unpause DosView',39,'s windowing routines.',13,10,'$'
 usage_10       db      '     /s = Display the current DosView settings.',13,10,'$'
 usage_11       db      '     /c = Clear the window area of the screen.',13,10,'$'
 usage_12       db      '     /r = Remove DosView TSR from memory.',13,10,'$'
 set_msg_1      db      ' DosView',39,'s parameters are currently set as follows:',13,10,'$'
 set_msg_2      db      '        TOP LINE OF DISPLAY: $'
 set_msg_3      db      '     BOTTOM LINE OF DISPLAY: $'
 set_msg_4      db      ' INTERRUPT 21H SERVICE CODE: $'
 set_msg_5      db      '               PAUSE STATUS: $'
 set_msg_6      db      'WINDOW DEACTIVATED',13,10,'$'
 set_msg_7      db      'WINDOW ACTIVATED',13,10,'$'
 badparam       db      ' ERROR - NO VALID PARAMETERS WERE GIVEN.',13,10,'$'
 range_msg_1    db      ' ERROR - THE TOP LINE MUST BE SPECIFIED AS A NUMBER BETWEEN 1 AND 25.',13,10,'$'
 range_msg_2    db      ' ERROR - THE BOTTOM LINE MUST BE SPECIFIED AS A NUMBER BETWEEN 1 AND 25.',13,10,'$'
 bottom_msg     db      ' ERROR - THE TOP LINE SPECIFIED CAN NOT BE > OR = TO THE BOTTOM LINE.',13,10,'$'
 top_msg        db      ' ERROR - THE BOTTOM LINE SPECIFIED CAN NOT BE < OR = TO THE TOP LINE.',13,10,'$'

 ; ______________________________END OF VARIABLES ____________________________

 last_byte:                             ; USED TO INDICATE THE LAST BYTE
                                        ; POSITION IN THE SOURCE CODE. NEEDED
                                        ; TO INSTALL DOSVIEW AS A TSR.

 end

