/*----------------------------------------------------------------------------
/ VDiff.C - file compare and change-bar for text files
/
/     author:   D. Krantz / Craig J. Kim
/       date:   8 Jan 87
/   language:   K&R C with VAX/VMS C extention
/
/    history:   Original: 08-Jan-87 > D. Krantz - Dr. Dobb's Journal 8/87
/               revision: 04-Aug-87 > Craig J. Kim for speed
/
/---------------------------------------------------------------------------*/

#ifdef VAXC

#   include stdio
#   include ctype

#else

#   include <stdio.h>
#   include <ctype.h>

#   define TRUE     1
#   define FALSE    0

#endif

#ifdef DEBUGGING
#   define DBG(x)   x
#else
#   define DBG(x)
#endif

#define OPT_FLAG     '/'                /* command option swithc */

#ifdef VAXC
#   define MAXLINE  15                  /* maximum characters in input file */
#   define FNAMLEN  255                 /* file name maximum length */
#else
#   define MAXLINE  85
#   define FNAMLEN  64
#endif

#define FIRST       0
#define SECOND      1

#define FORMFEED    12
#define DFTPAGELEN  66
#define DFTTOPSKIP  0
#define DFTBTMSKIP  0
#define DFTBARCOL   78
#define DFTRESYNC   5

extern char *strchr();
extern char *strupr();

struct _line {
        int linenum;                    /* what line number of page */
        int pagenum;                    /* what page line is from */
        struct _line *link;             /* linked list pointer */
        char text[MAXLINE];             /* text of line */
        char dup[MAXLINE];              /* uppercase of line text */
};

struct _line root[2];

FILE *msg;

int line_count[2] = { 1, 1 };           /* file's line counter */
int page_count[2] = { 1, 1 };           /* file's page counter */
int command_error = 0;                  /* any command error */
int files = 0;                          /* how many files */
char infile_name[2][FNAMLEN];
char outfile_name[FNAMLEN];
FILE *infile[2];                        /* input file handles */
FILE *outfile;
struct _line *at[2] = {
        &(root[FIRST]), &(root[SECOND])
};

int bar_col = DFTBARCOL;                /* where '|' should be */
int top_skip = DFTTOPSKIP;              /* lines to skip at top */
int btm_skip = DFTBTMSKIP;              /* lines to skip at bottom */
int page_len = DFTPAGELEN;              /* page length */
int up_case = FALSE;                    /* case sensitivity */
int re_sync = DFTRESYNC;                /* resynch count */
int output = FALSE;                     /* write output */
int blanks = FALSE;                     /* blank line significant */
int lookahead = 200;                    /* how many lines to look ahead */
int skip1 = 0;                          /* pages to skip */
int skip2 = 0;                          /* pages to skip */


main(argc, argv)
int argc;
char *argv[];
{
        int i;

        DBG(callstack("main");)
        if (argc < 2)
            usage();
        msg = stdout;
        for (i = 1; i < argc; i++)
            strip_opt(argv[i]);

        if (files < 2) {
            printf("Error:  Must specify two files");
            exit(1);
        }
        open_files();
        if (command_error)
            exit(2);
        page_skip();
        diff();
        DBG(callpop();)
}

/*---------------------------------------------------- dont_look() -----------
/ tells whether or not this line should be considered for comparison or is a
/ filler (e.g.  header, blank) line
/---------------------------------------------------------------------------*/
dont_look(line)
struct _line *line;
{
        DBG(callstack("don't_look");)
        if (!line) {
            DBG(callpop();)
            return(FALSE);
        }
        if (line->linenum <= top_skip) {
            DBG(callpop();)
            return(TRUE);
        }
        if (line->linenum > page_len - btm_skip) {
            DBG(callpop();)
            return(TRUE);
        }
        if (!blanks) {
            int i;

            for (i = 0; i < MAXLINE; i++)
                switch (line->text[i]) {
                    case '\0':
                    case '\n':
                        DBG(callpop();)
                        return(TRUE);
                    case '\t':
                    case ' ':
                        break;
                    default:
                        DBG(callpop();)
                        return(FALSE);
                }
        }
        DBG(callpop();)
        return(FALSE);
}

/*---------------------------------------------------- equal() ---------------
/ tells if the pointers 'a' and 'b' point to line buffers containing equivalent
/ text or not
/---------------------------------------------------------------------------*/
equal(a, b)
struct _line *a, *b;
{
        DBG(callstack("equal");)
        if (!a || !b) {
            DBG(callpop();)
            return(FALSE);
        }
        DBG(callpop();)
        if (up_case)
            return(!strcmp(a->dup, b->dup));
        else
            return(!strcmp(a->text, b->text));
}

/*---------------------------------------------------- position() ------------
/ moves the input pointer for file 'f' such that the next line to be read will
/ be 'where'
/---------------------------------------------------------------------------*/
position(f, where)
int f;
struct _line *where;
{
        DBG(callstack("position");)
        at[f] = &root[f];
        while (at[f]->link != where)
            at[f] = at[f]->link;
        DBG(callpop();)
}

#ifdef VAXC
/*---------------------------------------------------- fix() -----------------
/ fixes the end-of-line sequence on a VAX to be just a newline instead of a
/ carriage-return/newline
/---------------------------------------------------------------------------*/
char *fix(str)
register char *str;
{
        char *strsave;

        DBG(callstack("fix");)
        strsave = str;
        if (!str) {
            DBG(callpop();)
            return(NULL);
        }
        while (*str)
            if (*str++ == '\r' && *str == '\n') {
                *str-- = NULL;
                *str = '\n';
                break;
            }
        DBG(callpop();)
        return(strsave);
}
#endif

/*---------------------------------------------------- next_line() -----------
/ allocates, links, and returns the next line from file 'f' if no lines are
/ buffered, otherwise, returns the next buffered line from file 'f' and
/ updates the link pointer to the next buffered line.
/---------------------------------------------------------------------------*/
struct _line *next_line(f)
int f;
{
        extern char *malloc();
        struct _line *temp, *place_hold;

        DBG(callstack("next_line");)
        if (at[f]->link) {
            at[f] = at[f]->link;
            DBG(callpop();)
            return(at[f]);
        }
        else {
            at[f]->link = (struct _line *) malloc(sizeof (struct _line));
            if (!at[f]->link) {
                printf("\nOut of memory");
                exit(2);
            }
            place_hold = at[f];
            at[f] = at[f]->link;
            at[f]->link = NULL;
#ifdef VAXC
            if (!fix(fgets(at[f]->text, MAXLINE, infile[f]))) {
#else
            if (!fgets(at[f]->text, MAXLINE, infile[f])) {
#endif
                free(at[f]);
                at[f] = place_hold;
                at[f]->link = NULL;
                DBG(callpop();)
                return(NULL);
            }
#ifdef EMBEDDED_FORMFEEDS
            if (strchr(at[f]->text, FORMFEED)
#else
            if (*at[f]->text == FORMFEED
#endif
                    || line_count[f] > page_len) {
                page_count[f]++;
                line_count[f] = 1;
            }
            at[f]->linenum = line_count[f]++;
            at[f]->pagenum = page_count[f];
            if (up_case) {
                strcpy(at[f]->dup, at[f]->text);
                strupr(at[f]->dup);
            }
            DBG(callpop();)
            return(at[f]);
        }
}

/*---------------------------------------------------- discard() -------------
/ deallocates all buffered lines from the root up to and including 'to' for
/ file 'f'
/---------------------------------------------------------------------------*/
discard(f, to)
register int f;
struct _line *to;
{
        register struct _line *temp;

        DBG(callstack("discard");)
        while (temp = root[f].link) {
            root[f].link = root[f].link->link;
            free(temp);
            if (temp == to)
                break;
        }
        at[f] = &root[f];
        DBG(callpop();)
}

#ifdef VAXC
/*---------------------------------------------------- vfputs() --------------
/ for VAX, un-fixes newline at end-of-line to be carriage-return/newline
/---------------------------------------------------------------------------*/
vfputs(str, file)
char *str;
FILE *file;
{
        register int i;
        register char *cp;

        DBG(callstack("vfputs");)
        for (cp = str, i = 0; i < MAXLINE; i++)
            if (*cp++ == '\n') {
                strcpy(--cp, "\r\n");
                break;
            }
        fputs(str, file);
}
#endif

/*---------------------------------------------------- put() -----------------
/ if changed-barred output file is turned on, prints all lines from the root
/ of file 1 up to and including 'line'.  this is called only if a match exists
/ for each significant line in file 2.
/---------------------------------------------------------------------------*/
put(line)
struct _line *line;
{
        register struct _line *temp;

        DBG(callstack("put");)
        if (output)
            for (temp = root[FIRST].link; ; ) {
                if (!temp) {
                    DBG(callpop();)
                    return;
                }
#ifdef VAXC
                vfputs(temp->text, outfile);
#else
                fputs(temp->text, outfile);
#endif
                if (temp == line) {
                    DBG(callpop();)
                    return;
                }
                temp = temp->link;
            }
        DBG(callpop();)
}

/*---------------------------------------------------- change_bar() ----------
/ inserts a change-bar into the passed text and returns the pointer to it
/---------------------------------------------------------------------------*/
char *change_bar(str)
register char *str;
{
        char *base;

        DBG(callstack("change_bar");)
        base = str;
        if (bar_col) {
            register int i;

            for (i = 0; *str && *str != '\n'; i++, str++)
                if (*str == '\r' && *(str + 1) != '\n')
                    i = 0;
            while (i++ < bar_col)
                *str++ = ' ';
            strcpy(str, "|\n");
        }
        else {
            char temp[MAXLINE + 1];

            if (*str != ' ') {
                strcpy(temp, str);
                *str++ = '|';
                strcpy(str, temp);
            }
        }
        DBG(callpop();)
        return(base);
}

/*---------------------------------------------------- added() ---------------
/ prints a change summary for all significant lines from the root of file 1 up
/ to and including 'line'.  if output is enabled, adds a change bar to the
/ text and outputs the line to the output file.
/---------------------------------------------------------------------------*/
added(line)
register struct _line *line;
{
        register struct _line *temp;

        DBG(callstack("added");)
        for (temp = root[FIRST].link; temp; temp = temp->link) {
            if (!dont_look(temp))
                fprintf(msg, "+%d:%-2d -> %s", temp->pagenum,
                        temp->linenum, temp->text);
            if (output)
                if (dont_look(temp))
#ifdef VAXC
                    vfputs(temp->text, outfile);
#else
                    fputs(temp->text, outfile);
#endif
                else
#ifdef VAXC
                    vfputs(change_bar(temp->text), outfile);
#else
                    fputs(change_bar(temp->text), outfile);
#endif
            if (temp == line)
                break;
        }
        DBG(callpop();)
}

/*---------------------------------------------------- deleted() -------------
/ outputs a change summary for all lines in file 2 from the root up to and
/ including 'line'
/---------------------------------------------------------------------------*/
deleted(line)
register struct _line *line;
{
        register struct _line *temp;

        DBG(callstack("deleted");)
        for (temp = root[SECOND].link; temp; temp = temp->link) {
            if (!dont_look(temp))
                fprintf(msg, "-%d:%-2d -> %s",
                        temp->pagenum, temp->linenum, temp->text);
            if (temp == line)
                break;
        }
        DBG(callpop();)
}

/*---------------------------------------------------- resync() --------------
/ resynchronizes file 1 and file 2 after a difference is detected, and outputs
/ changed line and change summaries via added() and deleted().  exits with the
/ file inputs pointing at the next two lines that match, unless it is
/ impossible to sync up again, in which case all lines in file 1 are printed
/ via added().  deallocates all lines printed by this function
/---------------------------------------------------------------------------*/
resync(first, second)
struct _line *first, *second;
{
        struct _line *file1_start, *file2_start, *last_bad1, *last_bad2;
        struct _line *t1, *t2;
        int i, j, k, moved1, moved2;

        DBG(callstack("resync");)
        moved1 = 0;
        file1_start = first;
        position(FIRST, first);
        for (k = 0; k < lookahead; k++) {
            while (dont_look(file1_start = next_line(FIRST)))
                ;
            if (!file1_start) {         /* no sync */
                position(FIRST, first);
                while (first = next_line(FIRST)) {
                    added(first);
                    discard(FIRST, first);
                }
                DBG(callpop();)
                return;
            }
            moved2 = 0;
            file2_start = second;
            position(SECOND, second);
            for (j = 0; j < lookahead; j++) {
                while (dont_look(file2_start = next_line(SECOND)))
                    ;
                if (!file2_start)
                    break;
                t1 = file1_start;
                t2 = file2_start;
                for (i = 0; i < re_sync && equal(t1, t2); i++) {
                    while (dont_look(t1 = next_line(FIRST)))
                        ;
                    while (dont_look(t2 = next_line(SECOND)))
                        ;
                    if (!t1 || !t2)
                        break;
                }
                if (i == re_sync) {     /* synced! */
                    if (moved1) {
                        added(last_bad1);
                        discard(FIRST, last_bad1);
                    }
                    position(FIRST, file1_start);
                    if (moved2) {
                        deleted(last_bad2);
                        discard(SECOND, last_bad2);
                    }
                    position(SECOND, file2_start);
                    fprintf(msg, "\n");
                    DBG(callpop();)
                    return;
                }
                last_bad2 = file2_start;
                position(SECOND, file2_start);
                while (dont_look(file2_start = next_line(SECOND)))
                    ;
                moved2++;
            }
            last_bad1 = file1_start;
            position(FIRST, file1_start);
            while (dont_look(file1_start = next_line(FIRST)))
                ;
            moved1++;
        }
        printf("\n*** Error: lost sync in file %s at page %d line d",
                infile_name[FIRST], first->pagenum, first->linenum);
        fclose(outfile);
        exit(2);
}

/*---------------------------------------------------- diff() ----------------
/ differencing executive.  prints and deallocates all lines up to where a
/ difference is detected, at which point resync() is called.  exits on
/ end-of-file condition for file 1
/---------------------------------------------------------------------------*/
diff()
{
        register struct _line *first, *second;

        DBG(callstack("diff");)
        do {
            while (dont_look(first = next_line(FIRST)))
                ;
            if (!first) {
                put(first);
                break;
            }
            while (dont_look(second = next_line(SECOND)))
                ;
            if (equal(first, second)) {
                put(first);
                discard(FIRST, first);
                discard(SECOND, second);
            }
            else
                resync(first, second);
        } while (second);
        DBG(callpop();)
}

/*---------------------------------------------------- page_skip() -----------
/ skips the first 'skip1' pages of file 1, and then the first 'skip2' pages of
/ file 2.  this is useful to jump over tables of contents, etc.
/---------------------------------------------------------------------------*/
page_skip()
{
        struct _line *first, *second;

        DBG(callstack("page_skip");)
        for ( ; ; ) {
            first = next_line(FIRST);
            if (!first || first->pagenum > skip1)
                break;
            put(first);
            discard(FIRST, first);
        }
        if (first)
            position(FIRST, first);

        for ( ; ; ) {
            second = next_line(SECOND);
            if (!second || second->pagenum > skip2)
                break;
            discard(SECOND, second);
        }
        if (second)
            position(SECOND, second);
        DBG(callpop();)
}

/*---------------------------------------------------- usage() ---------------
/ prints the program usage
/---------------------------------------------------------------------------*/
static char *help_message[] = {
        "DIFF - File compare and change-bar",
        "Usage:  DIFF [option[option...]] newfile oldfile [output]",
        "",
        "    newfile - latest revision of text file",
        "    oldfile - baseline to compare against",
        "    output  - newfile with change-bars",
        "",
        "Options:",
        "    /BAR_COL=n   colume of output file in which change bar will appear",
        "    /TOP_SKIP=n  lines at top of page to skip (default = 0)",
        "    /BTM_SKIP=n  lines at bottom of page to skip (default = 0)",
        "    /PAGE_LEN=n  lines per page (default = 66)",
        "    /UP_CASE     case-insensitive",
        "    /NOUP_CASE   case-sensitive (default)",
        "    /RE_SYNC=n   lines that must match before files are considered synced",
#ifdef VAXC
        "    /OUTPUT=file file to redirect difference summary to",
#endif
        "    /BLANKS      blank lines are considered significant",
        "    /NOBLANKS    ignore blank lines (default)",
        "    /LOOKAHEAD=n lines to look ahead in each file to resync",
        "    /SKIP1=n     pages in 'newfile' to skip before compare (default = 0)",
        "    /SKIP2=n     pages in 'oldfile' to skip before compare (default = 0)",
        "",
        NULL
};

usage()
{
        register int i = 0;

        for ( ; help_message[i]; i++)
            printf("%s\n", help_message[i]);
        exit(1);
}

/*---------------------------------------------------- open_files() ----------
/ opens the input and output files
/---------------------------------------------------------------------------*/
open_files()
{
        register int i;

        DBG(callstack("open_file");)
        for (i = 0; i < 2; i++)
            if (!(infile[i] = fopen(infile_name[i], "r"))) {
                printf("Error: Can't open %s\n", infile_name[i]);
                command_error++;
            }
        if (files > 2)
            if (!(outfile = fopen(outfile_name, "w"))) {
                printf("Error: Can't create %s\n", outfile_name);
                command_error++;
            }
        DBG(callpop();)
}

/*---------------------------------------------------- redirect() ------------
/ performs output redirection under VAX/VMS - for MS-DOS or UNIX, use '>'
/---------------------------------------------------------------------------*/
redirect(str)
char *str;
{
        char filename[FNAMLEN], *ptr, *dest;

        DBG(callstack("redirect");)
        dest = filename;
        if (!(ptr = strchr(str, '='))) {
            printf("Error: option %s", str);
            command_error++;
        }
        ptr++;
        while (*ptr != OPT_FLAG && (*dest++ = *ptr++))
            ;
        *dest = NULL;
        if (!(msg = fopen(filename, "w"))) {
            printf("Error: can't create %s", filename);
            command_error++;
        }
        DBG(callpop();)
}

/*---------------------------------------------------- strip_opt() -----------
/ processes each command line option
/---------------------------------------------------------------------------*/
strip_opt(str)
char *str;
{
        DBG(callstack("strip_opt");)
        strupr(str);
        if (*str == OPT_FLAG) {
            char *option;

            option = str + 1;
            if (match(option, "BAR_COL"))
                bar_col = num(str);
            else if (match(option, "TOP_SKIP"))
                top_skip = num(str);
            else if (match(option, "BOTTOM_SKIP"))
                btm_skip = num(str);
            else if (match(option, "PAGE_LEN"))
                page_len = num(str);
            else if (match(option, "UP_CASE"))
                up_case = TRUE;
            else if (match(option, "NOUP_CASE"))
                up_case = FALSE;
            else if (match(option, "RE_SYNC"))
                re_sync = num(str);
            else if (match(option, "BLANKS"))
                blanks = TRUE;
            else if (match(option, "NOBLANKS"))
                blanks = FALSE;
            else if (match(option, "LOOKAHEAD"))
                lookahead = num(str);
            else if (match(option, "SKIP1"))
                skip1 = skip2 = num(str);
            else if (match(option, "SKIP2"))
                skip2 = num(str);
            else if (match(option, "OUTPUT"))
                redirect(str);
            else {
                printf("Unrecognized option %s", str);
                command_error++;
            }
        }
        else {
            switch (files) {
                case 0:
                    strcpy(infile_name[FIRST], str);
                    break;
                case 1:
                    strcpy(infile_name[SECOND], str);
                    break;
                case 2:
                    strcpy(outfile_name, str);
                    output = TRUE;
                    break;
                default:
                    printf("Too many args at %s", str);
                    command_error++;
            }
            files++;
        }
        if (strchr(str + 1, OPT_FLAG))
            strip_opt(str + 1, OPT_FLAG);
        DBG(callpop();)
}

/*---------------------------------------------------- match() ---------------
/ looks for a match of 'str' with the first (strlen(str)) characters of
/ 'pattern'.  returns 0 for no match, nonzero on match
/---------------------------------------------------------------------------*/
match(str, pattern)
register char *str, *pattern;
{
        DBG(callstack("match");)
        do {
            if (*str++ != *pattern++) {
                DBG(callpop();)
                return(FALSE);
            }
        } while (*pattern && *str && *str != '=');
        DBG(callpop();)
        return(TRUE);
}

/*---------------------------------------------------- num() -----------------
/ returns the integer associated with a command line option.  an equal sign
/ must appear in the option
/---------------------------------------------------------------------------*/
num(str)
register char *str;
{
        register char *where;

        DBG(callstack("num");)
        DBG(callpop();)
        if (where = strchr(str, '='))
            return(atoi(++where));
        else
            return(0);
}

/*---------------------------------------------------- debugging -------------
/ debugging routines
/---------------------------------------------------------------------------*/

DBG(int stack = 0;                                          )

DBG(callstack(str)                                          )
DBG(char *str;                                              )
DBG({                                                       )
DBG(        int i;                                          )
DBG(        char c;                                         )
DBG(        names[stack++] = str;                           )
DBG(        for (i = 0; i < stack; i++)                     )
DBG(            printf("   ");                              )
DBG(        printf("Entering %s\n", str);                   )
DBG(}                                                       )

DBG(callpop()                                               )
DBG({                                                       )
DBG(        int i;                                          )
DBG(        for (i = 0; i < stack; i++)                     )
DBG(            printf("   ");                              )
DBG(        printf("Leaving %s\n", str);                    )
DBG(        stack--;                                        )
DBG(}                                                       )

/*----------------------------- End of VDiff.C -----------------------------*/
