{$R-}
UNIT GroupItem;
(**) INTERFACE (**)
USES GroupExceptions, GroupType, ShellApi, SysUtils, 
  Classes, WinTypes, WinProcs;
TYPE
  String25 = String[25];
  String40 = String[40];
  String128 = String[128];

  TItemDataObj = Class(TObject)
  private
    fActualTID  : TItemData;
    fTIRH       : TIconResourceHeader;
    fANDPlane   : PByteArray;
    fXORPlane   : PByteArray;
    fItemName   : String40;
    fCommand    : String128;
    fIconPath   : String128;
    fWorkDir    : String128;
    fMinimized  : Boolean;
    fHotKey     : Word;
  protected
    FUNCTION GetHotKeyStr : String25;
    FUNCTION GetIcon : THandle;
    PROCEDURE SetPoint(CONST NuPoint : TPoint);
    PROCEDURE SetItemName(NuItemName : String40);
    PROCEDURE SetCommand(NuCommand : String128);
    PROCEDURE SetIconPath(NuIconPath : String128);
    PROCEDURE SetWorkDir(NuWorkDir : String128);
    PROCEDURE SetMinimized(CONST NuMinimized : Boolean);
    PROCEDURE SetHotKey(CONST NuHotKey : Word);
  public
    Changed            : Boolean;
    CONSTRUCTOR Create(TID : TItemData; TGFO : TObject);
    DESTRUCTOR Destroy; Override;
    {read-only properties}
    PROPERTY ActualTID : TItemData Read fActualTID;
    PROPERTY TIRH      : TIconResourceHeader Read fTIRH;
    PROPERTY ANDPlane  : PByteArray Read fAndPlane;
    PROPERTY XORPlane  : PByteArray Read fXorPlane;
    PROPERTY HotKeyStr : String25 Read GetHotKeyStr;
    PROPERTY Icon      : hIcon Read GetIcon;
    PROPERTY IconIndex : Word Read fActualTID.iIcon;
    {read-write properties}
    PROPERTY ItemName  : String40 Read fItemName Write SetItemName;
    PROPERTY Command   : String128 Read fCommand Write SetCommand;
    PROPERTY IconPath  : String128 Read fIconPath Write SetIconPath;
    PROPERTY WorkDir   : String128 Read fWorkDir Write SetWorkDir;
    PROPERTY Point     : TPoint Read fActualTID.pt Write SetPoint;
    PROPERTY Minimized : Boolean Read fMinimized Write SetMinimized;
    PROPERTY HotKey    : Word Read fHotKey Write SetHotKey;
  END;

(**) IMPLEMENTATION (**)
USES GroupFile;

  CONSTRUCTOR TItemDataObj.Create(TID : TItemData; TGFO : TObject);
  BEGIN
    Inherited Create;
    fActualTID := TID;
    fMinimized := FALSE;
    fWorkDir := '';
    fHotKey := 0;
    WITH TGFO AS TGroupFileObj DO
      BEGIN
        fItemName := StringFmOffset(TID.pName);
        fCommand:= StringFmOffset(TID.pCommand);
        fIconPath := StringFmOffset(TID.pIconPath);
        IF TID.cbHeader <> SizeOf(fTIRH) THEN
          Raise EGrpError.Create(msg_IconHeaderBad);
        FillBuffer(TID.pHeader, TID.cbHeader, @fTIRH);
        GetMem(fANDPlane, TID.cbANDPlane);
        FillBuffer(TID.pANDPlane, TID.cbANDPlane, fANDPlane);
        GetMem(fXORPlane, TID.cbXORPlane);
        FillBuffer(TID.pXORPlane, TID.cbXORPlane, fXORPlane);
      END;
    Changed := FALSE;
  END;

  DESTRUCTOR TItemDataObj.Destroy;
  BEGIN
    IF Assigned(fANDPlane) THEN
      FreeMem(fANDPlane, fActualTID.cbAndPlane);
    IF Assigned(fXORPlane) THEN
      FreeMem(fXORPlane, fActualTID.cbXorPlane);
    Inherited Destroy;
  END;

  FUNCTION TItemDataObj.GetIcon : THandle;
  VAR Buff : ARRAY[0..128] OF Char;
  BEGIN
    StrPCopy(Buff, fIconPath);
    GetIcon := ExtractIcon(0, Buff, ActualTID.iIcon);
  END;

  FUNCTION TItemDataObj.GetHotKeyStr : String25;
  BEGIN
    Result := '';
    IF Hi(fHotKey) AND 2 = 2 THEN
      Result := 'Ctrl+';
    IF Hi(fHotKey) AND 1 = 1 THEN
      Result := Result + 'Shift+';
    IF Hi(fHotKey) AND 4 = 4 THEN
      Result := Result + 'Alt+';
    Result := Result + Char(Lo(fHotKey));
  END;

  PROCEDURE TItemDataObj.SetPoint(CONST NuPoint : TPoint);
  BEGIN
    IF (fActualTID.pt.X = NuPoint.X) AND
       (fActualTID.pt.Y = NuPoint.Y) THEN Exit;
    fActualTID.pt := NuPoint;
    Changed := TRUE;
  END;

  PROCEDURE TItemDataObj.SetItemName(NuItemName : String40);
  BEGIN
    IF fItemName = NuItemName THEN Exit;
    fItemName := NuItemName;
    Changed := TRUE;
  END;

  PROCEDURE TItemDataObj.SetCommand (NuCommand : String128);
  BEGIN
    IF fCommand = NuCommand THEN Exit;
    fCommand := NuCommand;
    Changed := TRUE;
  END;

  PROCEDURE TItemDataObj.SetIconPath(NuIconPath : String128);
  BEGIN
    IF fIconPath = NuIconPath THEN Exit;
    fIconPath := NuIconPath;
    Changed := TRUE;
  END;

  PROCEDURE TItemDataObj.SetWorkDir (NuWorkDir : String128);
  BEGIN
    IF fWorkDir = NuWorkDir THEN Exit;
    fWorkDir := NuWorkDir;
    Changed := TRUE;
  END;

  PROCEDURE TItemDataObj.SetMinimized(CONST NuMinimized : Boolean);
  BEGIN
    IF fMinimized = NuMinimized THEN Exit;
    fMinimized := NuMinimized;
    Changed := TRUE;
  END;

  PROCEDURE TItemDataObj.SetHotKey(CONST NuHotKey : Word);
  BEGIN
    IF fHotKey = NuHotKey THEN Exit;
    fHotKey := NuHotKey;
    Changed := TRUE;
  END;

END.
