unit EDSPrint;
  {unit to programmatically set printer options so that user does not}
  {have to go to the Printer Options Dialog Box}
  {Revision 2.1}
interface
uses
  Classes, Graphics, Forms, Printers, SysUtils, Print, WinProcs, WinTypes, Messages;
            {see the WinTypes unit for constant declarations such as}
            {dmPaper_Letter, dmbin_Upper, etc}

const
  CCHBinName  = 24;  {Size of bin name (should have been in PRINT.PAS}
  CBinMax     = 256; {Maximum number of bin sources}
  CPaperNames = 256; {Maximum number of paper sizes}

type
  TWordArray = array[0..255] of Word;
  PWordArray = ^TWordArray;

  TPrintSet = class (TComponent)
  private
    { Private declarations }
    FDevice:     PChar;
    FDriver:     PChar;
    FPort:       PChar;
    FHandle:     THandle;
    FDeviceMode: PDevMode;
    FPrinter:    integer;      {same as Printer.PrinterIndex}
    FBinArray:   PWordArray;   {array of bin sources}
    FNumBins:    byte;         {number of bins}
    FPaperArray: PWordArray;   {array of paper sizes}
    FNumPapers:  byte;         {number of paper sizes}
    procedure    CheckPrinter;
      {-checks to see if the printer has changed and calls SetDeviceMode if it has}
    procedure    SetBinArray;
      {-sets the bin array}
    procedure    SetPaperArray;
      {-sets the paper array}
    function    DefaultPaperName (PaperID: word): String;
      {-returns the default name for the specified paper}
  protected
    { Protected declarations }
    procedure   SetOrientation (Orientation: integer);
    function    GetOrientation: integer;
      {-sets/gets the paper orientation}
    procedure   SetPaperSize (Size: integer);
    function    GetPaperSize: integer;
      {-sets/gets the paper size}
    procedure   SetPaperLength (Length: integer);
    function    GetPaperLength: integer;
      {-sets/gets the paper length}
    procedure   SetPaperWidth (Width: integer);
    function    GetPaperWidth: integer;
      {-sets/gets the paper width}
    procedure   SetScale (Scale: integer);
    function    GetScale: integer;
      {-sets/gets the printer scale (whatever that is)}
    procedure   SetCopies (Copies: integer);
    function    GetCopies: integer;
      {-sets/gets the number of copies}
    procedure   SetBin (Bin: integer);
    function    GetBin: integer;
      {-sets/gets the paper bin}
    procedure   SetPrintQuality (Quality: integer);
    function    GetPrintQuality: integer;
      {-sets/gets the print quality}
    procedure   SetColor (Color: integer);
    function    GetColor: integer;
      {-sets/gets the color (monochrome or color)}
    procedure   SetDuplex (Duplex: integer);
    function    GetDuplex: integer;
      {-sets/gets the duplex setting}
    procedure   SetYResolution (YRes: integer);
    function    GetYResolution: integer;
      {-sets/gets the y-resolution of the printer}
    procedure   SetTTOption (Option: integer);
    function    GetTTOption: integer;
      {-sets/gets the TrueType option}
    function    GetPrinterName: string;
      {-returns the name of the current printer}
    function    GetPrinterPort: string;
      {-returns the port of the current printer}
    function    GetPrinterDriver: string;
      {-returns the printer driver name of the current printer}
    procedure   SetBinFromList (BinNum: byte);
      {-sets the bin for the current item from the bin source list}
    function    GetBinIndex: byte;
      {-returns the current bin from the bin list}
    procedure   SetPaperFromList (PaperNum: byte);
      {-sets the paper for the current item from the paper list}
    function    GetPaperIndex: byte;
      {-returns the current paper size from the paper list}
  public
    { Public declarations }
    constructor Create (AOwner: TComponent); override;
      {-initializes object}
    destructor  Destroy;  override;
      {-destroys class}
    function    GetBinSourceList: TStringList;
      {-returns the current list of bins}
    function    GetPaperList: TStringList;
      {-returns the current list of paper sizes}
    procedure   SetDeviceMode;
      {-sets the internal pointer to the printers TDevMode structure}
    procedure   UpdateDeviceMode;
      {-updates the printers TDevMode structure}
    procedure   SaveToDefaults;
      {-updates the default settings for the current printer}
    procedure   SavePrinterAsDefault;
      {-saves the current printer as the Window's default}
    procedure   ResetPrinterDialogs;
      {-resets the printer dialogs to insure they come up}

    { Property declarations }
    property Orientation: integer     read   GetOrientation
                                      write  SetOrientation;
    property PaperSize: integer       read   GetPaperSize
                                      write  SetPaperSize;
    property PaperLength: integer     read   GetPaperLength
                                      write  SetPaperLength;
    property PaperWidth: integer      read   GetPaperWidth
                                      write  SetPaperWidth;
    property Scale: integer           read   GetScale
                                      write  SetScale;
    property Copies: integer          read   GetCopies
                                      write  SetCopies;
    property DefaultSource: integer   read   GetBin
                                      write  SetBin;
    property PrintQuality: integer    read   GetPrintQuality
                                      write  SetPrintQuality;
    property Color: integer           read   GetColor
                                      write  SetColor;
    property Duplex: integer          read   GetDuplex
                                      write  SetDuplex;
    property YResolution: integer     read   GetYResolution
                                      write  SetYResolution;
    property TTOption: integer        read   GetTTOption
                                      write  SetTTOption;
    property PrinterName: String      read   GetPrinterName;
    property PrinterPort: String      read   GetPrinterPort;
    property PrinterDriver: String    read   GetPrinterDriver;
    property BinIndex: byte           read   GetBinIndex
                                      write  SetBinFromList;
    property PaperIndex: byte         read   GetPaperIndex
                                      write  SetPaperFromList;
  end;  { TPrintSet }

procedure CanvasTextOutAngle (OutputCanvas: TCanvas; X,Y: integer;
                              Angle: Word; St: string);
  {-prints text at the desired angle}
  {-current font must be TrueType!}
procedure SetPixelsPerInch;
  {-insures that PixelsPerInch is set so that text print at the desired size}
function GetResolution: TPoint;
  {-returns the resolution of the printer}

procedure Register;
  {-registers the printset component}

implementation

constructor TPrintSet.Create (AOwner: TComponent);
  {-initializes object}
begin
  inherited Create (AOwner);
  FBinArray := nil;
  FPaperArray := nil;
  if not (csDesigning in ComponentState) then
  begin
    GetMem (FDevice, 255);
    GetMem (FDriver, 255);
    GetMem (FPort, 255);
    SetDeviceMode;
  end {:} else
  begin
    FDevice := nil;
    FDriver := nil;
    FPort   := nil;
  end;  { if... }
end;  { TPrintSet.Create }

procedure TPrintSet.CheckPrinter;
  {-checks to see if the printer has changed and calls SetDeviceMode if it has}
begin
  if FPrinter <> Printer.PrinterIndex then
  begin
    Printer.GetPrinter (FDevice, FDriver, FPort, FHandle);
    Printer.SetPrinter (FDevice, FDriver, FPort, 0);
    SetDeviceMode;
  end;  { if... }
end;  { CheckPrinter }

procedure TPrintSet.SetBinArray;
  {-sets the bin array}
var
  NumBinsRec:   Longint;      {number of bins received}
  DevCaps:      TFarProc;
  DrvHandle:    THandle;
  DriverName:   String;
begin
  if FBinArray <> nil then
    FreeMem (FBinArray, FNumBins * SizeOf (Word));
  DrvHandle := LoadLibrary (FDriver);
  if DrvHandle <> 0 then
  begin
    DevCaps := GetProcAddress (DrvHandle, 'DeviceCapabilities');
    if DevCaps<>nil then
    begin
      FNumBins := TDeviceCapabilities (DevCaps)(FDevice, FPort, DC_Bins,
                                                nil, FDeviceMode^);
      if FNumBins > 0 then
      begin
        GetMem (FBinArray, FNumBins * SizeOf (Word));
        NumBinsRec := TDeviceCapabilities (DevCaps)(FDevice, FPort, DC_Bins,
                                                    PChar (FBinArray), FDeviceMode^);
        if NumBinsRec <> FNumBins then
        begin
          {raise an exception}
          Raise EPrinter.Create ('Error retrieving Bin Source Info');
        end;  { if... }
      end {:} else
        FBinArray := nil;
    end;  { if... }
    FreeLibrary (DrvHandle);
  end {:} else
  begin
    {raise an exception}
    DriverName := StrPas (FDriver);
    Raise EPrinter.Create ('Error loading driver '+DriverName);
  end;  { else }
end;  { TPrintSet.SetBinArray }

procedure TPrintSet.SetPaperArray;
  {-sets the paper array}
var
  NumPapersRec: Longint;      {number of papers received}
  DevCaps:      TFarProc;
  DrvHandle:    THandle;
  DriverName:   String;
begin
  if FPaperArray <> nil then
    FreeMem (FPaperArray, FNumPapers * SizeOf (Word));
  DrvHandle := LoadLibrary (FDriver);
  if DrvHandle <> 0 then
  begin
    DevCaps := GetProcAddress (DrvHandle, 'DeviceCapabilities');
    if DevCaps<>nil then
    begin
      FNumPapers:= TDeviceCapabilities (DevCaps)(FDevice, FPort, DC_Papers,
                                                  nil, FDeviceMode^);
      if FNumPapers > 0 then
      begin
        GetMem (FPaperArray, FNumPapers * SizeOf (Word));
        NumPapersRec := TDeviceCapabilities (DevCaps)(FDevice, FPort, DC_Papers,
                                                      PChar (FPaperArray), FDeviceMode^);
        if NumPapersRec <> FNumPapers then
        begin
          {raise an exception}
          Raise EPrinter.Create ('Error retrieving Paper Source Info');
        end;  { if... }
      end {:} else
        FPaperArray := nil;
    end;  { if... }
    FreeLibrary (DrvHandle);
  end {:} else
  begin
    {raise an exception}
    DriverName := StrPas (FDriver);
    Raise EPrinter.Create ('Error loading driver '+DriverName);
  end;  { else }
end;  { TPrintSet.SetPaperArray }

function TPrintSet.DefaultPaperName (PaperID: word): String;
  {-returns the default name for the specified paper}
begin
  {these constants are taken straight from WinTypes.INT}
  case PaperID of
    dmpaper_Letter        : Result := 'Letter 8 1/2 x 11 in';
    dmpaper_LetterSmall   : Result := 'Letter Small 8 1/2 x 11 in';
    dmpaper_Tabloid       : Result := 'Tabloid 11 x 17 in';
    dmpaper_Ledger        : Result := 'Ledger 17 x 11 in';
    dmpaper_Legal         : Result := 'Legal 8 1/2 x 14 in';
    dmpaper_Statement     : Result := 'Statement 5 1/2 x 8 1/2 in';
    dmpaper_Executive     : Result := 'Executive 7 1/2 x 10 in';
    dmpaper_A3            : Result := 'A3 297 x 420 mm';
    dmpaper_A4            : Result := 'A4 210 x 297 mm';
    dmpaper_A4Small       : Result := 'A4 Small 210 x 297 mm';
    dmpaper_A5            : Result := 'A5 148 x 210 mm';
    dmpaper_B4            : Result := 'B4 250 x 354';
    dmpaper_B5            : Result := 'B5 182 x 257 mm';
    dmpaper_Folio         : Result := 'Folio 8 1/2 x 13 in';
    dmpaper_Quarto        : Result := 'Quarto 215 x 275 mm';
    dmpaper_10X14         : Result := '10x14 in';
    dmpaper_11X17         : Result := '11x17 in';
    dmpaper_Note          : Result := 'Note 8 1/2 x 11 in';
    dmpaper_Env_9         : Result := 'Envelope #9 3 7/8 x 8 7/8 in';
    dmpaper_Env_10        : Result := 'Envelope #10 4 1/8 x 9 1/2 in';
    dmpaper_Env_11        : Result := 'Envelope #11 4 1/2 x 10 3/8 in';
    dmpaper_Env_12        : Result := 'Envelope #12 4 \276 x 11 in';
    dmpaper_Env_14        : Result := 'Envelope #14 5 x 11 1/2 in';
    dmpaper_CSheet        : Result := 'C size sheet';
    dmpaper_DSheet        : Result := 'D size sheet';
    dmpaper_ESheet        : Result := 'E size sheet';
    dmpaper_User          : Result := 'User Defined Size';
    else Result := 'Unknown Paper Size';
  end;  { case }
end;  { TPrintSet.DefaultPaperName }

function TPrintSet.GetBinSourceList: TStringList;
  {-returns the current list of bins (returns nil for none)}
type
  TcchBinName = array[0..CCHBinName-1] of Char;
  TBinArray   = array[1..cBinMax] of TcchBinName;
  PBinArray   = ^TBinArray;
var
  NumBinsRec:   Longint;      {number of bins received}
  BinArray:     PBinArray;
  BinList:      TStringList;
  BinStr:       String;
  i:            Longint;
  DevCaps:      TFarProc;
  DrvHandle:    THandle;
  DriverName:   String;
begin
  CheckPrinter;
  Result   := nil;
  BinArray := nil;
  if FNumBins = 0 then Exit;
  try
    DriverName := 'C:\WINDOWS\SYSTEM\' + StrPas (FDriver) + '.DRV';
    DrvHandle := LoadLibrary (FDriver);
    if DrvHandle <> 0 then
    begin
      DevCaps := GetProcAddress (DrvHandle, 'DeviceCapabilities');
      if DevCaps<>nil then
      begin
        GetMem (BinArray, FNumBins * SizeOf (TcchBinName));
        NumBinsRec := TDeviceCapabilities (DevCaps)(FDevice, FPort, DC_BinNames,
                                                    PChar (BinArray), FDeviceMode^);
        if NumBinsRec <> FNumBins then
        begin
          {raise an exception}
          Raise EPrinter.Create ('Error retrieving Bin Source Info');
        end;  { if... }
        {now convert to TStringList}
        BinList := TStringList.Create;
        for i := 1 to NumBinsRec do
        begin
          BinStr := StrPas (BinArray^[i]);
          BinList.Add (BinStr);
        end;  { next i }
      end;  { if... }
      FreeLibrary (DrvHandle);
      Result := BinList;
    end {:} else
    begin
      {raise an exception}
      DriverName := StrPas (FDriver);
      Raise EPrinter.Create ('Error loading driver '+DriverName);
    end;  { else }
  finally
    if BinArray <> nil then
      FreeMem (BinArray, FNumBins * SizeOf (TcchBinName));
  end;  { try }
end;  { TPrintSet.GetBinSourceList }

function TPrintSet.GetPaperList: TStringList;
  {-returns the current list of paper sizes (returns nil for none)}
type
  TcchPaperName = array[0..CCHPaperName-1] of Char;
  TPaperArray   = array[1..cPaperNames] of TcchPaperName;
  PPaperArray   = ^TPaperArray;
var
  NumPaperRec:   Longint;      {number of paper types received}
  PaperArray:    PPaperArray;
  PaperList:     TStringList;
  PaperStr:      String;
  i:             Longint;
  DevCaps:       TFarProc;
  DrvHandle:     THandle;
  DriverName:    String;
begin
  CheckPrinter;
  Result     := nil;
  PaperArray := nil;
  if FNumPapers = 0 then Exit;
  try
    DrvHandle := LoadLibrary (FDriver);
    if DrvHandle > HInstance_Error then
    begin
      DevCaps := GetProcAddress (DrvHandle, 'DeviceCapabilities');
      if DevCaps<>nil then
      begin
        GetMem (PaperArray, FNumPapers * SizeOf (TcchPaperName));
        NumPaperRec := TDeviceCapabilities (DevCaps)(FDevice, FPort, DC_PaperNames,
                                                     PChar (PaperArray), FDeviceMode^);
        if NumPaperRec <> FNumPapers then
        begin
          {construct the list as best we can}
          PaperList := TStringList.Create;
          for i := 1 to FNumPapers do
          begin
            PaperStr := DefaultPaperName (FPaperArray^[i - 1]);
            PaperList.Add (PaperStr);
          end;  { next i }
        end {:} else
        begin
          {now convert to TStringList}
          PaperList := TStringList.Create;
          for i := 1 to NumPaperRec do
          begin
            PaperStr := StrPas (PaperArray^[i]);
            PaperList.Add (PaperStr);
          end;  { next i }
        end;  { if... }
      end;  { if... }
      FreeLibrary (DrvHandle);
      Result := PaperList;
    end {:} else
    begin
      {raise an exception}
      DriverName := StrPas (FDriver);
      Raise EPrinter.Create ('Error loading driver '+DriverName);
    end;  { else }
  finally
    if PaperArray <> nil then
      FreeMem (PaperArray, FNumPapers * SizeOf (TcchPaperName));
  end;  { try }
end;  { TPrintSet.GetPaperList }

procedure TPrintSet.SetDeviceMode;
begin
  Printer.GetPrinter (FDevice, FDriver, FPort, FHandle);
  if FHandle = 0 then
  begin  {driver not loaded}
    Printer.PrinterIndex := Printer.PrinterIndex;
      {-forces Printer object to load driver}
    Printer.GetPrinter (FDevice, FDriver, FPort, FHandle);
  end;  { if... }
  if FHandle<>0 then
  begin
    FDeviceMode := Ptr (FHandle, 0);
      {-PDeviceMode now points to Printer.DeviceMode}
    FDeviceMode^.dmFields := 0;
    UpdateDeviceMode;
  end {:} else
  begin
    FDeviceMode := nil;
    Raise EPrinter.Create ('Error retrieving DeviceMode');
  end;  { if... }
  SetBinArray;
  SetPaperArray;
  FPrinter := Printer.PrinterIndex;
end;  { TPrintSet.SetDeviceMode }

procedure TPrintSet.UpdateDeviceMode;
  {-updates the loaded TDevMode structure}
var
  DrvHandle:   THandle;
  ExtDevCaps:  TFarProc;
  DriverName:  String;
  ExtDevCode:  Integer;
  OutDevMode:  PDevMode;
begin
  DrvHandle := LoadLibrary (FDriver);
  if DrvHandle <> 0 then
  begin
    ExtDevCaps := GetProcAddress (DrvHandle, 'ExtDeviceMode');
    if ExtDevCaps<>nil then
    begin
      ExtDevCode := TExtDeviceMode (ExtDevCaps)
        (0, DrvHandle, FDeviceMode^, FDevice, FPort,
         FDeviceMode^, nil, DM_IN_BUFFER or DM_OUT_BUFFER);
      if ExtDevCode <> IDOK then
      begin
        {raise an exception}
        raise EPrinter.Create ('Error updating printer driver.');
      end;  { if... }
    end;  { if... }
    FreeLibrary (DrvHandle);
  end {:} else
  begin
    {raise an exception}
    DriverName := StrPas (FDriver);
    Raise EPrinter.Create ('Error loading driver '+DriverName);
  end;  { else }
end;  { TPrintSet.UpdateDeviceMode }

procedure TPrintSet.SaveToDefaults;
  {-updates the default settings for the current printer}
var
  DrvHandle:   THandle;
  ExtDevCaps:  TFarProc;
  DriverName:  String;
  ExtDevCode:  Integer;
  OutDevMode:  PDevMode;
begin
  CheckPrinter;
  DrvHandle := LoadLibrary (FDriver);
  if DrvHandle <> 0 then
  begin
    ExtDevCaps := GetProcAddress (DrvHandle, 'ExtDeviceMode');
    if ExtDevCaps<>nil then
    begin
      ExtDevCode := TExtDeviceMode (ExtDevCaps)
        (0, DrvHandle, FDeviceMode^, FDevice, FPort,
         FDeviceMode^, nil, DM_IN_BUFFER OR DM_UPDATE);
      if ExtDevCode <> IDOK then
      begin
        {raise an exception}
        raise EPrinter.Create ('Error updating printer driver.');
      end {:} else
        SendMessage ($FFFF, WM_WININICHANGE, 0, 0);
    end;  { if... }
    FreeLibrary (DrvHandle);
  end {:} else
  begin
    {raise an exception}
    DriverName := StrPas (FDriver);
    Raise EPrinter.Create ('Error loading driver '+DriverName);
  end;  { else }
end;  { TPrintSet.SaveToDefaults }

procedure TPrintSet.SavePrinterAsDefault;
  {-saves the current printer as the Window's default}
var
  DeviceStr: String;
begin
  CheckPrinter;  {make sure new printer is loaded}
  {set the new device setting in the WIN.INI file}
  DeviceStr := StrPas (FDevice) + ',' + StrPas (FDriver) + ',' + StrPas (FPort) + #0;
  WriteProfileString ('windows', 'device', @DeviceStr[1]);
  {force write to WIN.INI}
  WriteProfileString (nil, nil, nil);
  {broadcast to everyone that WIN.INI changed}
  SendMessage ($FFFF, WM_WININICHANGE, 0, 0);
end;  { TPrintSet.SavePrinterAsDefault }

procedure TPrintSet.ResetPrinterDialogs;
  {-resets the printer dialogs to insure they come up}
begin
  Printer.GetPrinter (FDevice, FDriver, FPort, FHandle);
  Printer.SetPrinter (FDevice, FDriver, FPort, 0);
  SetDeviceMode;
end;  { TPrintSet.ResetPrinterDialogs }

procedure TPrintSet.SetOrientation (Orientation: integer);
  {-sets the paper orientation}
begin
  CheckPrinter;
  FDeviceMode^.dmOrientation := Orientation;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_ORIENTATION;
end;  { TPrintSet.SetOrientation }

function TPrintSet.GetOrientation: integer;
  {-gets the paper orientation}
begin
  CheckPrinter;
  Result := FDeviceMode^.dmOrientation;
end;  { TPrintSet.GetOrientation }

procedure TPrintSet.SetPaperSize (Size: integer);
  {-sets the paper size}
begin
  CheckPrinter;
  FDeviceMode^.dmPaperSize := Size;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_PAPERSIZE;
end;  { TPrintSet.SetPaperSize }

function TPrintSet.GetPaperSize: integer;
  {-gets the paper size}
begin
  CheckPrinter;
  Result := FDeviceMode^.dmPaperSize;
end;  { TPrintSet.GetPaperSize }

procedure TPrintSet.SetPaperLength (Length: integer);
  {-sets the paper length}
begin
  CheckPrinter;
  FDeviceMode^.dmPaperLength := Length;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_PAPERLENGTH;
end;  { TPrintSet.SetPaperLength }

function TPrintSet.GetPaperLength: integer;
  {-gets the paper length}
begin
  CheckPrinter;
  Result := FDeviceMode^.dmPaperLength;
end;  { TPrintSet.GetPaperLength }

procedure TPrintSet.SetPaperWidth (Width: integer);
  {-sets the paper width}
begin
  CheckPrinter;
  FDeviceMode^.dmPaperWidth := Width;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_PAPERWIDTH;
end;  { TPrintSet.SetPaperWidth }

function TPrintSet.GetPaperWidth: integer;
  {-gets the paper width}
begin
  CheckPrinter;
  Result := FDeviceMode^.dmPaperWidth;
end;  { TPrintSet.GetPaperWidth }

procedure TPrintSet.SetScale (Scale: integer);
  {-sets the printer scale (whatever that is)}
begin
  CheckPrinter;
  FDeviceMode^.dmScale := Scale;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_SCALE;
end;  { TPrintSet.SetScale }

function TPrintSet.GetScale: integer;
  {-gets the printer scale}
begin
  CheckPrinter;
  Result := FDeviceMode^.dmScale;
end;  { TPrintSet.GetScale }

procedure TPrintSet.SetCopies (Copies: integer);
  {-sets the number of copies}
begin
  CheckPrinter;
  FDeviceMode^.dmCopies := Copies;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_COPIES;
end;  { TPrintSet.SetCopies }

function TPrintSet.GetCopies: integer;
  {-gets the number of copies}
begin
  CheckPrinter;
  Result := FDeviceMode^.dmCopies;
end;  { TPrintSet.GetCopies }

procedure TPrintSet.SetBin (Bin: integer);
  {-sets the paper bin}
begin
  CheckPrinter;
  FDeviceMode^.dmDefaultSource := Bin;
  FDeviceMode^.dmFields  := FDeviceMode^.dmFields or DM_DEFAULTSOURCE;
end;  { TPrintSet.SetBin }

function TPrintSet.GetBin: integer;
  {-gets the paper bin}
begin
  CheckPrinter;
  Result := FDeviceMode^.dmDefaultSource;
end;  { TPrintSet.GetBin }

procedure TPrintSet.SetPrintQuality (Quality: integer);
  {-sets the print quality}
begin
  CheckPrinter;
  FDeviceMode^.dmPrintQuality := Quality;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_PRINTQUALITY;
end;  { TPrintSet.SetPrintQuality }

function TPrintSet.GetPrintQuality: integer;
  {-gets the print quality}
begin
  CheckPrinter;
  Result := FDeviceMode^.dmPrintQuality;
end;  { TPrintSet.GetPrintQuality }

procedure TPrintSet.SetColor (Color: integer);
  {-sets the color (monochrome or color)}
begin
  CheckPrinter;
  FDeviceMode^.dmColor := Color;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_ORIENTATION;
end;  { TPrintSet.SetColor }

function TPrintSet.GetColor: integer;
  {-gets the color}
begin
  CheckPrinter;
  Result := FDeviceMode^.dmColor;
end;  { TPrintSet.GetColor }

procedure TPrintSet.SetDuplex (Duplex: integer);
  {-sets the duplex setting}
begin
  CheckPrinter;
  FDeviceMode^.dmDuplex := Duplex;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_DUPLEX;
end;  { TPrintSet.SetDuplex }

function TPrintSet.GetDuplex: integer;
  {-gets the duplex setting}
begin
  CheckPrinter;
  Result := FDeviceMode^.dmDuplex;
end;  { TPrintSet.GetDuplex }

procedure TPrintSet.SetYResolution (YRes: integer);
  {-sets the y-resolution of the printer}
var
  PrintDevMode: Print.PDevMode;
begin
  CheckPrinter;
  PrintDevMode := @FDeviceMode^;
  PrintDevMode^.dmYResolution := YRes;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_YRESOLUTION;
end;  { TPrintSet.SetYResolution }

function  TPrintSet.GetYResolution: integer;
  {-gets the y-resolution of the printer}
var
  PrintDevMode: Print.PDevMode;
begin
  CheckPrinter;
  PrintDevMode := @FDeviceMode^;
  Result := PrintDevMode^.dmYResolution;
end;  { TPrintSet.GetYResolution }

procedure TPrintSet.SetTTOption (Option: integer);
  {-sets the TrueType option}
var
  PrintDevMode: Print.PDevMode;
begin
  CheckPrinter;
  PrintDevMode := @FDeviceMode^;
  PrintDevMode^.dmTTOption := Option;
  FDeviceMode^.dmFields := FDeviceMode^.dmFields or DM_TTOPTION;
end;  { TPrintSet.SetTTOption }

function TPrintSet.GetTTOption: integer;
  {-gets the TrueType option}
var
  PrintDevMode: Print.PDevMode;
begin
  CheckPrinter;
  PrintDevMode := @FDeviceMode^;
  Result := PrintDevMode^.dmTTOption;
end;  { TPrintSet.GetTTOption }

function TPrintSet.GetPrinterName: string;
  {-returns the name of the current printer}
begin
  CheckPrinter;
  Result := StrPas (FDevice);
end;  { TPrintSet.GetPrinterName }

function TPrintSet.GetPrinterPort: string;
  {-returns the port of the current printer}
begin
  CheckPrinter;
  Result := StrPas (FPort);
end;  { TPrintSet.GetPrinterPort }

function TPrintSet.GetPrinterDriver: string;
  {-returns the printer driver name of the current printer}
begin
  CheckPrinter;
  Result := StrPas (FDriver);
end;  { TPrintSet.GetPrinterDriver }

procedure TPrintSet.SetBinFromList (BinNum: byte);
  {-sets the bin for the current item from the bin source list}
begin
  CheckPrinter;
  if FNumBins = 0 then Exit;
  if BinNum > FNumBins then
    Raise EPrinter.Create ('Index out of range setting bin.')
  else
    DefaultSource := FBinArray^[BinNum];
end;  { TPrintSet.SetBinFromList }

function TPrintSet.GetBinIndex: byte;
  {-returns the current bin from the bin list}
var
  i: byte;
begin
  Result := 0;
  for i := 0 to FNumBins do
    if FBinArray^[i] = FDeviceMode^.dmDefaultSource then
    begin
      Result := i;
      Break;
    end;  { if... }
end;  { TPrintSet.GetBinIndex }

procedure TPrintSet.SetPaperFromList (PaperNum: byte);
  {-sets the paper for the current item from the paper list}
begin
  CheckPrinter;
  if FNumPapers = 0 then Exit;
  if PaperNum > FNumPapers then
    Raise EPrinter.Create ('Index out of range setting paper.')
  else
    PaperSize := FPaperArray^[PaperNum];
end;  { TPrintSet.SetPaperFromList }

function TPrintSet.GetPaperIndex: byte;
  {-returns the current paper size from the paper list}
var
  i: byte;
begin
  Result := 0;
  for i := 0 to FNumPapers do
    if FPaperArray^[i] = FDeviceMode^.dmPaperSize then
    begin
      Result := i;
      Break;
    end;  { if... }
end;  { TPrintSet.GetPaperIndex }

destructor TPrintSet.Destroy;
  {-destroys class}
begin
  if FBinArray <> nil then
    FreeMem (FBinArray, FNumBins * SizeOf (Word));
  if FPaperArray <> nil then
    FreeMem (FPaperArray, FNumPapers * SizeOf (Word));
  if FDevice <> nil then
    FreeMem (FDevice, 255);
  if FDriver <> nil then
    FreeMem (FDriver, 255);
  if FPort <> nil then
    FreeMem (FPort, 255);
  inherited Destroy;
end; { TPrintSet.Destroy }

procedure CanvasTextOutAngle (OutputCanvas: TCanvas; X,Y: integer;
                              Angle: Word; St: string);
  {-prints text at the desired angle}
  {-current font must be TrueType!}
var
  LogRec:        TLogFont;
  NewFontHandle: HFont;
  OldFontHandle: HFont;
begin
  GetObject (OutputCanvas.Font.Handle, SizeOf (LogRec), Addr (LogRec));
  LogRec.lfEscapement := Angle;
  NewFontHandle := CreateFontIndirect (LogRec);
  OldFontHandle := SelectObject (OutputCanvas.Handle, NewFontHandle);
  OutputCanvas.TextOut (x, y, St);
  NewFontHandle := SelectObject (OutputCanvas.Handle, OldFontHandle);
  DeleteObject (NewFontHandle);
end; { CanvasTextOutAngle }

procedure SetPixelsPerInch;
  {-insures that PixelsPerInch is set so that text print at the desired size}
var
  FontSize: integer;
begin
  FontSize := Printer.Canvas.Font.Size;
  Printer.Canvas.Font.PixelsPerInch := GetDeviceCaps (Printer.Handle, LOGPIXELSY );
  Printer.Canvas.Font.Size := FontSize;
end;  { SetPixelsPerInch }

function GetResolution: TPoint;
  {-returns the resolution of the printer}
begin
  Result.X := GetDeviceCaps(Printer.Handle, LogPixelsX);
  Result.Y := GetDeviceCaps(Printer.Handle, LogPixelsY);
end;  { GetResolution }

procedure Register;
  {-registers the printset component}
begin
  RegisterComponents('Domain', [TPrintSet]);
end;  { Register }

end.  { EDSPrint }
