--************************************************************************
--
--  SERVTASK.ADB
--
--  A copyright-reserved, free use program.
--  (c)John H. McCoy, 1993, Sam Houston St. Univ., TX 77341-2206
--************************************************************************

with drivers; use drivers;
with video, cursor;
with text_io; use text_io;
with unchecked_deallocation;

package body ServerTasks is

  BlockSize  : constant := 2048;     -- assume cooked mode only
  MaxBlocks  : constant := 31;       -- NETBIOS limits dta to 64K -1

  type dtaAccess is access bytes;
  procedure ZapDta is new unchecked_deallocation(bytes,dtaAccess);

  CBMax      : constant := 131;      -- max command block size

task body Consoles is

  type UserEntries is record
     SessionNumber: byte;
     Client       : String16;
  end record;

  MaxEntries: integer;
  LastUser  : integer := 0;
  NewUser   : boolean;
  I         : integer;

begin

  accept Init(MaxSessions: integer) do
    MaxEntries := MaxSessions;
  end Init;

  declare
    Users: array (0..MaxEntries) of UserEntries;
  begin
    put_line("Console started.");
    loop
      NewUser := False;
      select
        accept SignIn(SessionNumber: byte;
                      Client       : String16) do
          LastUser       := LastUser + 1;
          Users(LastUser).SessionNumber:= SessionNumber;
          Users(LastUser).Client:= Client;
          NewUser := True;
        end SignIn;
      or
        accept SignOut(SessionNumber: byte) do
          Users(0).SessionNumber := SessionNumber;
          I := LastUser;
          loop
            exit when Users(I).SessionNumber = SessionNumber;
            I := I - 1;
          end loop;
          LastUser := LastUser-1;
          for j in I..LastUser loop
            Users(j) := Users(j + 1);
          end loop;
        end SignOut;
      end select;

      if NewUser then
        if LastUser = 1 then
           video.clear_screen;
           cursor.move(0,0);
           put_line("Session  Client");
        end if;
        put(byte'image(Users(LastUser).SessionNumber));
        set_col(To => 10);put_line(Users(LastUser).Client);
      else
        video.clear_screen;
        cursor.move(0,0);
        put_line("Session  User Name");
        if LastUser < 1 then
           put_line("No Users.");
        else
           for j in 1..LastUser loop
             put(byte'image(Users(j).SessionNumber));
             set_col(To => 10);put_line(Users(j).Client);
           end loop;
        end if;
      end if;
    end loop;
  end;
end Consoles;

task body Sessions is

  Stop         : boolean := False;
  rh           : rhs;
  pkt          : pkts;
  dta          : dtaAccess;
  NCB, NCBinit : ncbAccess := new NetBiosCmdBlks;
  Net          : NetAccess;
  SessionActive: boolean;
  LocalSession : byte;
  ServerName   : string16;
  DtaSave      : DW;
  CB           : bytes(1..CBMax);
  CbSave       : DW;
  CD           : CDAccess;
  CdSubUnits   : byte;
  Hub          : SchedulerAccess;
  AllocError   : boolean;

  procedure Wait is
  begin
    loop
      exit when NCB.CommandStatus /= NB_CommandPending;
      delay (0.0);                               -- give other tasks a chance
    end loop;
  end Wait;

  procedure SendRH is
  begin
     -- send rh back to client
     pkt := Rhs_to_Pkts(rh);
     pkt(3..pkts'last-1) := pkt(4..pkts'last);
     NCB                  := NCBinit;
     NCB.Command          := NB_Send_NoWait;
     NCB.BufferPtr        := pkt(1)'address;
     NCB.BufferLength     := word(rh.length);
     Net.Call(NCB);
     Wait;
  end SendRH;

begin
  accept Start(Net       : NetAccess;
               LocalName : string16;
               CD        : CDAccess;
               SubUnits  : byte;
               Scheduler : SchedulerAccess ) do
    Sessions.Net  := Net;
    ServerName    := LocalName;
    Sessions.CD   := CD;
    CdSubUnits    := SubUnits;
    Hub           := Scheduler;
  end Start;

loop                                  -- new sessions start here

  SessionActive := False;
  NCB             := NCBinit;
  NCB.Command     := NB_Listen_NoWait;
  NCB.CallName(1) := '*';             -- listen for any caller
  NCB.Name        := ServerName;      -- calling this listener's name
  Hub.Listen(Net, NCB);               -- get in queue for a call
                                      -- won't return until a session request
                                      -- is received or NET is terminated.


  exit when NCB.ReturnCode /= NB_Ok;  -- this will terminate session que;

  SessionActive        := True;
  LocalSession         := NCB.LocalSession;
  NCBinit.LanAdapter   := NCB.LanAdapter;
  NCBinit.LocalSession := NCB.LocalSession;
  NCBinit.CallName     := NCB.CallName;
  NCBinit.Name         := NCB.Name;
  NCBinit.NameNumber   := NCB.NameNumber;
  Console.SignIn(SessionNumber => LocalSession,
                 Client        => NCBinit.CallName);

session: loop                       -- intra session loop starts here

  -- get request header

  NCB                  := NCBinit;
  NCB.Command          := NB_Receive_NoWait;
  NCB.BufferPtr        := pkt(1)'address;
  NCB.BufferLength     := pkts'last;
  Net.Call(NCB);
  Wait;
  exit session when NCB.ReturnCode /= NB_Ok;      -- abort session

  pkt(4..pkts'last) := pkt(3..pkts'last-1);
  pkt(3)    := 0;
  rh        := Pkts_to_Rhs(pkt);

  case rh.rhX.command is
    when DeviceReadLong =>
      if W_to_Word(rh.rhX.ReadLong.SectorsToRead) > MaxBlocks then
        -- request to big, chop to fit.
        rh.rhX.ReadLong.SectorsToRead := Word_to_W(MaxBlocks);
      end if;
      if rh.rhX.ReadLong.SectorsToRead = Word_to_W(0) then
        rh.rhX.ReadLong.Status := DeviceDone;
        -- send rh back to client
        SendRH;
        exit session when NCB.ReturnCode /= NB_Ok;
      else
        loop -- retry storage allocation
          begin
          -- local block for exception handler
          dta := new bytes(1..W_to_Word(rh.rhX.ReadLong.SectorsToRead)*BlockSize);
          AllocError := false;
          --  pass rh on to the CD
          dtaSave := rh.rhX.ReadLong.DtaPtr;       -- save remote dta ptr
          rh.rhX.ReadLong.dtaPtr:=SA_to_DW(dta(1)'address);  -- point to local
          CD.Call(rh);                             -- device status now in rhX
          rh.rhX.ReadLong.DtaPtr := dtaSave;       -- restore for return
          -- send rh back to client
          SendRH;
          if NCB.ReturnCode /= NB_Ok then
            raise NBX_GeneralError;
          end if;
          -- send dta back to client
          NCB := NCBinit;
          NCB.BufferLength:=BlockSize*W_to_Word(rh.rhX.ReadLong.SectorsToRead);
          if rh.rhX.ReadLong.status = DeviceDone
             and then NCB.BufferLength /= 0 then
            NCB.Command          := NB_Send_NoWait;
            NCB.BufferPtr        := dta(1)'address;
            Net.Call(NCB);
            Wait;
            if NCB.ReturnCode /= NB_Ok then
              raise NBX_GeneralError;
            end if;
          end if;
          exception
            when NBX_GeneralError => ZapDta(dta);
                                     exit session;
            when storage_error    => AllocError := true;
          end;  -- end of local block
          exit when not AllocError;
        end loop;
      end if;
      ZapDta(dta);
    when DeviceIoctlInput =>
        --  get the Command Block from client
        NCB                  := NCBinit;
        NCB.Command          := NB_Receive_NoWait;
        NCB.BufferPtr        := CB(1)'address;
        NCB.BufferLength     := W_to_word(rh.rhX.IoctlIn.TransferCount);
        Net.Call(NCB);              -- get the CB
        Wait;                               -- until command completes
        exit session when NCB.ReturnCode /= NB_Ok;  -- abort session
        --  pass it on to the CD
        CBSave       := rh.rhX.IoctlIn.CBPtr;            -- save remote CB ptr
        rh.rhX.IoctlIn.CBPtr := SA_to_DW(CB(1)'address); -- point to local CB
        CD.Call(rh);     -- device status now in rhX
        -- send rh back to client
        rh.rhX.IoctlIn.CBPtr := CBSave;        -- set dta back for return
        SendRH;
        exit session when NCB.ReturnCode /= NB_Ok;   -- aborts session
        -- send CB back to client
        NCB                  := NCBinit;
        NCB.Command          := NB_Send_NoWait;
        NCB.BufferPtr        := CB(1)'address;
        NCB.BufferLength     := W_to_Word(rh.rhX.IoctlIn.TransferCount);
        Net.Call(NCB);
        Wait;
        exit session when NCB.ReturnCode /= NB_Ok;   -- aborts session
    when DeviceIoctlOutput =>
        --  get the Command Block from client
        NCB                  := NCBinit;
        NCB.Command          := NB_Receive_NoWait;
        NCB.BufferPtr        := CB(1)'address;
        NCB.BufferLength     := W_to_word(rh.rhX.IoctlOut.TransferCount);
        Net.Call(NCB);              -- get the CB
        Wait;                               -- until command completes
        exit session when NCB.ReturnCode /= NB_Ok;  -- abort session
        --  pass it on to the CD
        CBSave       := rh.rhX.IoctlOut.CBPtr;   -- save remote CB ptr
        rh.rhX.IoctlOut.CBPtr := SA_to_DW(CB(1)'address); -- point to local CB
        CD.Call(rh);     -- device status now in rhX
        -- send rh back to client
        rh.rhX.IoctlOut.CBPtr         := CBSave;       -- set dta back for return
        SendRH;
        exit session when NCB.ReturnCode /= NB_Ok;   -- aborts session
    when DeviceInit =>
        rh.rhX.Init.NumberUnits := CDSubUnits;  -- for MSCDEX only, client
        rh.rhX.Init.Status := DeviceDone;       -- always tells DOS 1
        -- send rh back to client
        SendRH;
        exit session when NCB.ReturnCode /= NB_Ok;   -- aborts session
    when DeviceSeek | DeviceReadLongPrefetch =>
        --  pass it on to the CD
        CD.Call(rh);     -- device status now in rhX
        -- send rh back to client
        SendRH;
        exit session when NCB.ReturnCode /= NB_Ok;   -- aborts session

    when Others =>      -- should never come here, but !!!
        rh.rhX.Other.Status := DeviceDone OR DeviceUnknownCommand;
        SendRH;          -- just send it back

  end case;


end loop session;

  -- session aborted to get here
  if SessionActive then
    Console.SignOut(SessionNumber => LocalSession);
  end if;

end loop;

end Sessions;

task body Schedulers is

begin

loop
  accept Listen(Net : NetAccess; Ncb : ncbAccess ) do
    Net.Call(NCB);
    loop
      exit when NCB.CommandStatus /= NB_CommandPending;
      delay (0.0);
    end loop;
  end Listen;
  delay(0.0);
end loop;

end Schedulers;

task body Nets is
  Stop             : boolean := False;
begin
  accept Start(Name: string16) do
    NetAddName(NetName => Name);
  end Start;
loop
  select
    accept Call(Ncb: ncbAccess) do
      NetBiosCall (Ncb => Ncb);
    end Call;
  or
    accept ShutDown do
      Stop := True;
    end ShutDown;
  or
    when Stop => terminate;
  end select;
end loop;
end Nets;

task body CDRoms is
  Stop            : boolean := False;
  DriverStrategy  : system.address;
  DriverInterrupt : system.address;
  pkt             : pkts;
begin
  accept Init(DeviceStrategy  : system.address;
              DeviceInterrupt : system.address ) do
    DriverStrategy  := DeviceStrategy;
    DriverInterrupt := DeviceInterrupt;
  end Init;
loop
  select
    accept Call (rh: in out rhs) do
      pkt := Rhs_to_Pkts(rh);
      pkt(3..pkts'last-1) := pkt(4..pkts'last);
      CallDriver (rh              => pkt(1)'address,
                  DeviceStrategy  => DriverStrategy,
                  DeviceInterrupt => DriverInterrupt);
      pkt(4..pkts'last) := pkt(3..pkts'last-1);
      pkt(3)    := 0;
      rh        := Pkts_to_Rhs(pkt);
    end Call;
  or
    accept ShutDown do
      Stop := True;
    end ShutDown;
  or
    when Stop => terminate;
  end select;
end loop;
end CDRoms;

end ServerTasks;