//--------------------------------------------------------------------------
//
//      TSR.H: header file for DOS TSR class.
//      Copyright (c) J.English 1993.
//      Author's address: je@unix.brighton.ac.uk
//
//      Permission is granted to use copy and distribute the
//      information contained in this file provided that this
//      copyright notice is retained intact and that any software
//      or other document incorporating this file or parts thereof
//      makes the source code for the TSR class of which this file
//      is a part freely available.
//
//--------------------------------------------------------------------------
//
//      Note: this class is highly DOS specific and hence non-portable.
//      It also involves the use of assembly language and interrupt
//      functions, so it is also very compiler-specific and will need
//      modification for use with compilers other than Borland C++ 3.0
//      or later.
//
//      Revision history:
//      1.0     March 1993      Initial coding
//
//--------------------------------------------------------------------------

#ifndef __TSR_H
#define __TSR_H

//--------------------------------------------------------------------------
//
//      Class TSR: abstract base class for DOS resident programs.
//      ---------------------------------------------------------
//
//      TSRs written using this class will require a PC/AT compatible
//      machine running DOS version 3 or higher.  See the file TSR.DOC
//      for detailed documentation.
//
//      Public interface:
//      -----------------
//      TSR (const char* name,
//           unsigned stacksize = 1024);
//                      -- Class constructor.  "Name" is a unique name
//                         for the TSR instance, and "stacksize" is the
//                         number of bytes of stack space to allocate.
//      virtual ~TSR ();
//                      -- Class destructor.  This is only invoked when
//                         the TSR is NOT made resident.
//      void run (int hotkey,
//                unsigned timeslice = 0);
//                      -- Make the TSR resident.  "Hotkey" is the hotkey
//                         to be used to wake up the TSR and "timeslice"
//                         is the number of timer ticks between timed
//                         activations.  Use value from the enumeration
//                         TSR::hotkey_code to specify the hotkey value.
//                         A timeslice of 0 means that timeslicing is not
//                         required.  This function only returns if there
//                         is an error preventing the TSR from loading.
//      int unload ();
//                      -- Unload a previously-loaded copy of the TSR from
//                         memory.  Returns zero if the TSR is successfully
//                         unloaded, or a non-zero error code if it could
//                         not be unloaded.
//      int loaded ();
//                      -- Returns non-zero if a copy of the TSR has already
//                         been loaded.
//      int request (int& fn, int& p1, int& p2);
//                      -- This function allows a foreground copy of the
//                         program to communicate with a resident copy.
//                         "Fn" is an application-specific function code
//                         (00 - 7F) and "p1" and "p2" are function-specific
//                         parameters.  Returns zero if there is no resident
//                         copy of the TSR; otherwise the virtual function
//                         "respond" will be called in the resident copy to
//                         process the function.  "Respond" can update the
//                         values of the three parameters to return a result
//                         to the foreground program.
//      const char* name ();
//                      -- Returns the TSR's identifying string.
//      int status ();
//                      -- Returns a status code showing if any errors have
//                         occurred during TSR construction or loading (as
//                         defined by the enumeration type TSR::status_code).
//
//      Protected interface:
//      --------------------
//      virtual void main (int hotkey) = 0;
//                      -- The main body of the TSR, executed whenever it
//                         is woken up.  "Hotkey" will be non-zero if the
//                         TSR was woken up by the hotkey, and zero if it
//                         was woken up by a timeslice interrupt.  This is
//                         a pure virtual function and must be defined by
//                         all derived TSR classes.
//
//      Functions for use by "main".
//      ----------------------------
//      void pause ();
//                      -- allow other TSRs to run if lengthy processing
//                         is being performed by "main".
//      void sync ();
//                      -- resynchronise the timer.  Timeslice interrupts
//                         occur every N timer ticks after "run" is executed.
//                         This function resets the clock so that the next
//                         interrupt will occur N timer ticks from the time
//                         it is called.
//      int userbreak ();
//                      -- returns non-zero if Ctrl-Break has been pressed
//                         since the last time this function was called.
//
//      Virtual functions:
//      ------------------
//      (Note: these functions should not be called directly; they will be
//       called automatically when necessary.)
//
//      virtual int startup ();
//                      -- Performs initialisation (like a constructor) when
//                         the TSR is being made resident.  Returns non-zero
//                         if an error occurs.
//      virtual int shutdown ();
//                      -- Performs finalisation (like a destructor) when
//                         the TSR is being unloaded from memory.  Returns
//                         non-zero if an error occurs.
//      virtual int respond (int fn, int& p1, int& p2);
//                      -- Respond to a call to "request" in a foreground
//                         copy of the program.  "Fn" is a function code
//                         between 00 and 7F, and "p1" and "p2" are function
//                         specific parameters.  The return value will be
//                         stored in the caller's function code (the first
//                         parameter to "request") and any changes to "p1"
//                         or "p2" will be reflected in the corresponding
//                         parameters of the foreground program's call to
//                         "request".
//      virtual critical_code critical_error (int n);
//                      -- Called when a critical errors occurs during
//                         execution of "main".  "N" is the critical error
//                         code; the result is a value from the enumeration
//                         TSR::critical_code (indicating how DOS should
//                         proceed).
//      virtual void dos_error (int fn, int ce, int cs, int ip);
//                      -- Called if an invalid DOS function is called by
//                         "main" or "critical_error".  The default action
//                         is to clear the screen and print an error message.
//                         Parameter "fn" is the function code from register
//                         AH; "ce" is non-zero if the error occurred while
//                         handling a critical error; "cs" and "ip" are the
//                         segment and offset of the return address from the
//                         offending interrupt.  Do not attempt to use any
//                         DOS services from within this function!
//
//--------------------------------------------------------------------------



class TSR
{
    friend void activate ();

  public:
    enum hotkey_code {
        NONE   = 0x0000,
        ALT    = 0x0800,  CTRL   = 0x0400,  LSHIFT = 0x0200,  RSHIFT = 0x0100,
        KEY_A  = 0x001E,  KEY_B  = 0x0030,  KEY_C  = 0x002E,  KEY_D  = 0x0020,
        KEY_E  = 0x0012,  KEY_F  = 0x0021,  KEY_G  = 0x0022,  KEY_H  = 0x0023,
        KEY_I  = 0x0017,  KEY_J  = 0x0024,  KEY_K  = 0x0025,  KEY_L  = 0x0026,
        KEY_M  = 0x0032,  KEY_N  = 0x0031,  KEY_O  = 0x0018,  KEY_P  = 0x0019,
        KEY_Q  = 0x0010,  KEY_R  = 0x0013,  KEY_S  = 0x001F,  KEY_T  = 0x0014,
        KEY_U  = 0x0016,  KEY_V  = 0x002F,  KEY_W  = 0x0011,  KEY_X  = 0x002D,
        KEY_Y  = 0x0015,  KEY_Z  = 0x002C,  KEY_0  = 0x0078,  KEY_1  = 0x0079,
        KEY_2  = 0x007A,  KEY_3  = 0x007B,  KEY_4  = 0x007C,  KEY_5  = 0x007D,
        KEY_6  = 0x007E,  KEY_7  = 0x007F,  KEY_8  = 0x0080,  KEY_9  = 0x0081,
        F1     = 0x003B,  F2     = 0x003C,  F3     = 0x003D,  F4     = 0x003E,
        F5     = 0x003F,  F6     = 0x0040,  F7     = 0x0041,  F8     = 0x0042,
        F9     = 0x0043,  F10    = 0x0044,  SPACE  = 0x0039,  ENTER  = 0x001C
    };
    enum status_code {
        SUCCESS     = 0,                                        // general
        DOS_VERSION = 1,  MULTI_COPIES = 2,  STACK_FAIL = 3,    // run/c'tor
        NO_MUX_FUNC = 4,  RELOAD_FAIL  = 5,  LOAD_FAIL  = 6,
        STARTUP_ERR = 7,
        NOT_LOADED  = 1,  UNHOOK_FAIL  = 2,  MEM_FROZEN = 3,    // unload
        ENV_FROZEN  = 4
    };
    enum critical_code { IGNORE = 0, RETRY = 1, FAIL = 3 };

    TSR (const char* name,                  // TSR identifier
         unsigned stacksize = 1024);        // stack size in bytes
    virtual ~TSR ();

    void run (int hotkey,                   // hotkey used to wake up TSR
              unsigned timeslice = 0);      // length of timeslice in ticks
    int unload ();                          // unload TSR from memory
    int loaded ();                          // is TSR already loaded?
    const char* name ();                    // get TSR identifier
    int status ();                          // current TSR status

    virtual int startup ();                 // TSR startup code
    virtual int shutdown ();                // TSR shutdown code
    int request (int&, int&, int&);         // request to TSR resident copy
    virtual int respond (int, int far&, int far&);
                                            // response from TSR resident copy
    virtual critical_code critical_error (int n);
                                            // critical error handler
    virtual void dos_error (int fn, int ce, int cs, int ip);
                                            // DOS error handler

  protected:
    void pause ();                          // let other TSRs run
    void sync ();                           // resychronise timer
    int userbreak ();                       // has ctrl-break been pressed?
    virtual void main (int hotkey) = 0;     // main body of TSR, called when
                                            // hotkey pressed or timesliced
  private:
    int stat;
};



//--------------------------------------------------------------------------
//
//      Inline functions
//
inline TSR::~TSR ()                                     { }
inline int TSR::startup  ()                             { return 0; }
inline int TSR::shutdown ()                             { return 0; }
inline int TSR::status   ()                             { return stat; }
inline int TSR::respond (int, int far&, int far&)       { return 0; }
inline TSR::critical_code TSR::critical_error (int n)   { return TSR::FAIL; }

#endif
