//--------------------------------------------------------------------------
//
//      XMS.CPP: body of XMS interface library.
//      Copyright (c) J.English 1993.
//      Author's address: je@unix.brighton.ac.uk
//
//      Permission is granted to use copy and distribute the
//      information contained in this file provided that this
//      copyright notice is retained intact and that any software
//      or other document incorporating this file or parts thereof
//      makes the source code for the library of which this file
//      is a part freely available.
//
//--------------------------------------------------------------------------
//
//      Revision history:
//      1.0     Jun 1993        Initial coding
//
//--------------------------------------------------------------------------

#include "xms.h"
#include <dos.h>
#include <stdlib.h>

//--------------------------------------------------------------------------
//
//      Constants.
//
const int USED      = 1,    // cache line is in use
          DIRTY     = 2;    // cache line has been written to

//--------------------------------------------------------------------------
//
//      Data structures.
//
struct XMSbuffer            // cache structure
{
    int   mask;             // ... mask to extract item number in buffer
    long  where;            // ... offset to start of XMS block
    int   width;            // ... unit for XMS transfers in bytes
    char* cache;            // ... buffer for XMS transfers
    long  tag;              // ... current index held in buffer
    char  flags;            // ... buffer state (in use, dirty)
};

struct XMSheader            // header for XMS block
{
    long size;              // ... block size in bytes
    int  free;              // ... is it a free block?
};

struct XMSmove              // descriptor for XMS moves
{
    long count;             // ... number of bytes to move
    int  srchandle;         // ... source handle (0 for real memory)
    long srcoffset;         // ... source offset (or far pointer)
    int  dsthandle;         // ... destination handle (0 for real memory)
    long dstoffset;         // ... destination offset (or far pointer)
};

//--------------------------------------------------------------------------
//
//      Globals (statics).
//
static XMSmove descriptor;  // why use more than one?

static int  XMSinited = 0;  // has XMSinit been called already?
static int  XMShandle = 0;  // handle allocated by XMSinit
static int  XMSstatus = 0;  // last error code
static long XMSsize   = 0;  // current allocation size, internal reckoning
                            // (XMS driver rounds up to nearest 16K or so)
static void far (*XMSdriver)() = 0;
                            // pointer to XMS driver routine

//--------------------------------------------------------------------------
//
//      Function to return the number of bits required to represent "n",
//      used for rounding up to powers of 2 and generating masks.
//
static int bits (unsigned n)
{
    int b = 0;
    for (unsigned i = 1; i > 0 && i < n; i <<= 1)
        b++;
    return b;
}

//--------------------------------------------------------------------------
//
//      XMSclose: release the XMS handle (if any).
//
//      This is registered as an "atexit" function by XMSinit.
//
static void XMSclose ()
{
    if (XMShandle != 0)
    {   _DX = XMShandle;
        _AH = 0x0A;
        XMSdriver ();
    }
}


//--------------------------------------------------------------------------
//
//      XMSinit: initialise XMS system (first time only).
//
//      The first time it is called, this function locates the XMS driver
//      and allocates a handle for a zero-byte block.  It returns TRUE if
//      a handle was successfully allocated.  Subsequent calls return the
//      same value.
//
static int XMSinit ()
{
    //--- check if this is the first time XMSinit has been called
    if (XMSinited != 0)
        return (XMShandle != 0);

    //--- now it isn't!
    XMSinited = 1;

    //--- check that an XMS driver present (magic spell)
    REGS r;
    r.x.ax = 0x4300;
    int86 (0x2F, &r, &r);
    if (r.h.al != 0x80)
        return 0;

    //--- get the XMS driver's address (magic spell)
    SREGS s;
    r.x.ax = 0x4310;
    int86x (0x2F, &r, &r, &s);
    XMSdriver = (void far (*)()) MK_FP(s.es,r.x.bx);

    //--- allocate a handle
    _DX = 0;
    _AH = 0x09;
    XMSdriver ();

    //--- check result (XMShandle only set if no errors occurred)
    if (_AX == 0)
        XMSstatus = _BL;
    else
        XMShandle = _DX;

    //--- register exit handler
    atexit (XMSclose);

    //--- return success indicator
    return XMShandle != 0;
}

//--------------------------------------------------------------------------
//
//      XMSget: get block from XMS.
//
//      This function copies "size" bytes from offset "src" in XMS into
//      address "dst" in conventional memory.  It does nothing if XMS
//      initialisation fails.
//
static void XMSget (void far* dst, long src, int size)
{
    if (XMSinit ())
    {   descriptor.count     = size;
        descriptor.srchandle = XMShandle;
        descriptor.srcoffset = src;
        descriptor.dsthandle = 0;
        descriptor.dstoffset = (long) dst;
        _SI = FP_OFF (&descriptor);
        _AH = 0x0B;
        XMSdriver ();
        if (_AX == 0)
            XMSstatus = _BL;
    }
}


//--------------------------------------------------------------------------
//
//      XMSget: put block into XMS.
//
//      This function copies "size" bytes from address "src" in conventional
//      memory to offset "dst" in XMS.  It does nothing if XMS initialisation
//      fails.
//
static void XMSput (const void far* src, long dst, int size)
{
    if (XMSinit ())
    {   descriptor.count     = size;
        descriptor.srchandle = 0;
        descriptor.srcoffset = (long) src;
        descriptor.dsthandle = XMShandle;
        descriptor.dstoffset = dst;
        _SI = FP_OFF (&descriptor);
        _AH = 0x0B;
        XMSdriver ();
        if (_AX == 0)
            XMSstatus = _BL;
    }
}

//--------------------------------------------------------------------------
//
//      XMSlocate: locate a free block of XMS.
//
//      This is used by XMSalloc to check if there is a block of suitable
//      size which has been returned to the free list.  If not, it returns
//      0 (which is never a legitimate offset) and if so, it divides the
//      block as necessary, marks the allocated block as in use and returns
//      its offset.  XMS blocks are prefaced by an XMSheader giving their
//      size and status (free/in use).  Offsets refer to the first byte
//      of the allocation (i.e. the byte after the header); the size then
//      gives the address of the next block.  Adjacent free blocks are merged
//      while scanning the list.  If the last block is free, it can't be
//      merged with anything and so XMSsize is adjusted to remove it from
//      the internal record of the allocation size.
//
static long XMSlocate (long size)
{
    long p = 0, oldp = 0;
    XMSheader h, oldh = {0,0};

    //--- scan block list from offset 0
    while (p < XMSsize)
    {
        //--- get header of next block
        XMSget (&h, p, sizeof(h));

        //--- merge adjacent free blocks by changing size of previous block
        if (oldh.free && h.free)
        {   oldh.size += h.size + sizeof(h);
            h = oldh, p = oldp;
            XMSput (&h, p, sizeof(h));
        }

        //--- check if block (after possible merging) is big enough
        if (h.free && h.size >= size + sizeof(XMSheader))
            break;

        //--- keep value and offset for block and get offset of next header
        oldh = h, oldp = p;
        p += h.size + sizeof(h);
    }

    //--- no suitable block available: remove last block if it is free
    if (p >= XMSsize)
    {   if (oldh.free)
            XMSsize = oldp;
        return 0;
    }

    //--- divide block if necessary
    if (h.free && h.size >= size + sizeof(XMSheader))
    {
        XMSheader t = {h.size - size - sizeof(t), 1};
        XMSput (&t, p + sizeof(h) + size, sizeof(t));
        h.size = size;
    }

    //--- write new block's header and return offset of first usable byte
    h.free = 0;
    XMSput (&h, p, sizeof(h));
    return p + sizeof(h);
}

//--------------------------------------------------------------------------
//
//      XMSalloc: allocate a block of XMS.
//
//      This function tries to allocate "size" bytes of XMS and returns 0
//      if it fails.
//
static long XMSalloc (long size)
{
    //--- check XMS initialised before proceeding
    if (!XMSinit ())
        return 0;

    //--- try to find a suitable free block in the current allocation
    long pos = XMSlocate (size);
    if (pos != 0)
        return pos;

    //--- calculate total allocation required in K and try to get it
    _BX = (XMSsize + size + sizeof(XMSheader) + 1023) / 1024;
    _DX = XMShandle;
    _AH = 0x0F;
    XMSdriver ();

    //--- return if allocation failed
    if (_AX == 0)
    {   XMSstatus = _BL;
        return 0;
    }

    //--- write the header for the new block at the end of current allocation
    XMSheader h = {size,0};
    XMSput (&h, XMSsize, sizeof(h));

    //--- get offset of first usable byte of new block
    pos = XMSsize + sizeof(h);

    //--- increase allocation size to accomodate it and return its offset
    XMSsize += size + sizeof(h);
    return pos;
}


//--------------------------------------------------------------------------
//
//      XMSfree: mark a block of XMS as free.
//
//      This returns a block to the free list by rewriting its header.
//
static void XMSfree (long offset)
{
    XMSheader h;
    XMSget (&h, offset - sizeof(h), sizeof(h));
    h.free = 1;
    XMSput (&h, offset - sizeof(h), sizeof(h));
}

//--------------------------------------------------------------------------
//
//      XMS::available.
//
//      Returns the size of the largest unallocated XMS block available.
//
long XMS::available ()
{
    if (XMSinit ())
    {   _AH = 0x08;
        XMSdriver ();
        return _AX * 1024L;
    }
    else
        return 0;
}


//--------------------------------------------------------------------------
//
//      XMS::status.
//
//      Return the last-recorded XMS error code, resetting the code to zero.
//
long XMS::status ()
{
    int s = XMSstatus;
    XMSstatus = 0;
    return s;
}

//--------------------------------------------------------------------------
//
//      XMS::XMS.
//
//      Constructor for base class XMS, called by constructor for XMSarray.
//      Constructs an XMS array containing the specified number of items of
//      the specified size and also a one-line cache of the specified size.
//
XMS::XMS (long items, int size, int cachesize)
{
    //--- initial state is "unallocated"
    state = 0;

    //--- create the cache
    buffer = new XMSbuffer;
    if (buffer == 0)
        return;

    //--- set up the item offset mask (items in line - 1)
    buffer->mask = (size > cachesize) ? 0 : 
                   (1 << (bits(cachesize) - bits(size))) - 1;

    //--- calculate number of items needed (1 extra rounded up to nearest line)
    if (items < 0)
        items = 0;
    items += buffer->mask;
    items &= ~buffer->mask;

    //--- try to allocate the XMS block
    buffer->where = XMSalloc (items * size);

    //--- calculate cache size in bytes and allocate it
    buffer->width = (buffer->mask + 1) * size;
    buffer->cache = new char [buffer->width];
    if (buffer->cache == 0)
        return;

    //--- mark cache line as unused
    buffer->tag   = 0;
    buffer->flags = 0;

    //--- record if allocation succeeded
    state = (buffer->where != 0);
}


//--------------------------------------------------------------------------
//
//      XMS::XMS.
//
//      Copy constructor used by XMSitem to clone a reference to the
//      base array.
//
XMS::XMS (const XMS& base)
{
    state  = base.state;
    buffer = base.buffer;
}

//--------------------------------------------------------------------------
//
//      XMS::free.
//
//      Try to free an array.  Check that it's there before doing anything,
//      then free the XMS block and delete the cache.  This is not a written
//      as a destructor so that XMSitem doesn't call it.  Unfortunately
//      Borland C++ won't let you nominate all possible instantiations of
//      a template as friends of a non-template class, i.e.
//          class X { template<class T> friend class Y<T>; ... };
//      gives a compilation error.
//
void XMS::free ()
{
    if (buffer == 0)
        return;
    if (buffer->where != 0)
        XMSfree (buffer->where);
    delete buffer->cache;
    delete buffer;
}


//--------------------------------------------------------------------------
//
//      XMS::mask.
//
//      Return the item number within the cache line for a given subscript.
//
int XMS::mask (int n)
{
    if (!state)
        return 0;
    return n & buffer->mask;
}

//--------------------------------------------------------------------------
//
//      XMS::get.
//
//      Get a block from XMS into conventional memory.  If it's not in the
//      cache, load the relevant line (writing back the old line if it was
//      dirty).  Return a pointer to the cached copy of the item.  "Base"
//      is the base address of the line to load and "offset" is the offset
//      into the cache buffer.
//
void* XMS::get (long base, int offset)
{
    if (!state)
        return 0;
    if (buffer->tag != base || !(buffer->flags & USED))
    {   if (buffer->flags & DIRTY)
            XMSput (buffer->cache, buffer->where + buffer->tag, buffer->width);
        XMSget (buffer->cache, buffer->where + base, buffer->width);
        buffer->flags = USED;
        buffer->tag = base;
    }
    return buffer->cache + offset;
}


//--------------------------------------------------------------------------
//
//      XMS::put.
//
//      Gets a copy of the relevant cache line into memory and then marks
//      the line as dirty (about to be written to).
//
void* XMS::put (long base, int offset)
{
    if (!state)
        return 0;
    void* addr = get (base, offset);
    buffer->flags |= DIRTY;
    return addr;
}
