//--------------------------------------------------------------------------
//
//      XMS.H: header file for XMS interface library.
//      Copyright (c) J.English 1993.
//      Author's address: je@unix.brighton.ac.uk
//
//      Permission is granted to use copy and distribute the
//      information contained in this file provided that this
//      copyright notice is retained intact and that any software
//      or other document incorporating this file or parts thereof
//      makes the source code for the library of which this file
//      is a part freely available.
//
//--------------------------------------------------------------------------
//
//      Revision history:
//      1.0     Jun 1993        Initial coding
//
//--------------------------------------------------------------------------

#ifndef __XMS_H
#define __XMS_H

#include <mem.h>

//--------------------------------------------------------------------------
//
//      Class XMS: base class for XMS array classes.
//
//      This class encapsulates the XMS interface required by XMSarray
//      and XMSitem.  The public member functions provide a limited set
//      of XMS status tests but the class is not otherwise available to
//      outside users.  The status tests are:
//          * XMS::available() -- return the number of bytes of XMS
//                                that are currently available
//          * XMS::status()    -- get the code for the last XMS error
//                                that occurred
//
class XMS
{
    friend class XMSmanager;

  public:
    static long available ();           // test amount of XMS available
    static long status    ();           // status of last XMS operation
    int         allocated ()            // was object allocated successfully?
                                        { return state; }
  protected:
    XMS       (const XMS& array);       // constructor used by XMSitem
    XMS       (long items, int size, int cachesize);
                                        // constructor used by XMSarray
    void free ();                       // free an XMS array
    int  mask (int n);                  // mask out offset field of address

    void* get (long base, int offset);  // get an XMS value for reading
    void* put (long base, int offset);  // get an XMS value for writing

  private:
    struct XMSbuffer* buffer;
    int               state;
};

//--------------------------------------------------------------------------
//
//      Class XMSarray: XMS array class.
//
//      The constructor parameters specify the number of array elements
//      required and (optionally) the size of the internal cache in bytes
//      to be used for buffering accesses to small items.  The cache size
//      will be rounded up to a power of 2 times the item size.  The only
//      things you can do with an XMSarray are:
//          * Subscript it (e.g. x[i])
//          * Check if it has been allocated successfully (e.g.
//            x.allocate())
//
//      The template parameter T must be a type capable of being assigned
//      to which can be constructed without supplying any parameters to
//      its constructor.
//
template<class T> class XMSarray : public XMS
{
  public:
    XMSarray (long size, int cachesize = 2048)
                                          : XMS (size, sizeof(T), cachesize)
                                          { items = size < 0 ? 1 : size + 1; }
    virtual    ~XMSarray ()               { free (); }
    long       size ()                    { return items - 1; }
    XMSitem<T> operator[] (long index);   // subscript operator

  private:
    long items;
};


//--------------------------------------------------------------------------
//
//      Class XMSitem: XMS data item class.
//
//      This is a helper class which represents a single element of an
//      XMSarray.  Users cannot create objects of this type directly;
//      XMSitems are only created as a result of subscripting an XMSarray.
//      You can perform the following operations on an XMSitem:
//          * Assign a new value to it (e.g. x[i] = y)
//          * Get its value (e.g. y = x[i])
//
template<class T> class XMSitem : public XMS
{
    friend class XMSarray<T>;

  public:
    XMSitem& operator= (XMSitem& value);  // copy one array element to another
    XMSitem& operator= (const T& value);  // assign new value to array element
    operator T ();                        // get value of array element

  private:
    XMSitem  (long i, XMSarray<T>& b);
    long     base;
    int      offset;
};

//--------------------------------------------------------------------------
//
//      Template functions.
//
template<class T>
inline XMSitem<T> XMSarray<T>::operator[] (long index)
{
    long i = (index >= 0 && index < items) ? index : items;
    return XMSitem<T> (i, *this);
}

template<class T>
inline XMSitem<T>::XMSitem (long i, XMSarray<T>& b)
    : XMS (b)
{
    offset = mask(i) * sizeof(T);
    base   = i * sizeof(T) - offset;
}

template<class T>
inline XMSitem<T>& XMSitem<T>::operator= (XMSitem<T>& value)
{
    if (allocated())
    {   T temp;
        memcpy (&temp, value.get(value.base,value.offset), sizeof(T));
        *(T*) put(base,offset) = temp;
    }
    return *this;
}

template<class T>
inline XMSitem<T>& XMSitem<T>::operator= (const T& value)
{
    if (allocated())
        *(T*) put(base,offset) = value;
    return *this;
}

template<class T>
inline XMSitem<T>::operator T ()
{
    return *(T*) get(base,offset);      // this is for DOS -- dereferencing
}                                       // (void*)0 is okay, gives garbage
                                        // (I'd prefer exceptions...)
#endif
