//-------------------------------------------------------------------------
//			ActionMedia II Programmer's Toolkit
//			
//			Windows Sample Code Shared Functions
//
// Module Name:	DispErr
//
// Description:	This module contains functions for displaying messages and
//				errors in Windows Message Boxes.
//
// Copyright:	Copyright (c) Intel Corporation, 1992
//
//-------------------------------------------------------------------------
//
// Exported functions from this module:
//
//		InitErrSystem
//		CloseErrSystem
//		DispMsg
//		DispErr
//		DispAvkErr
//		TimeString
//		DateString
//
//-------------------------------------------------------------------------
// NOTES on DispErr
//	
//	These functions are used in most of the sample applications to 
//	display messages, generic errors and AVK errors in Windows Message 
//	Boxes.
//
//	They can be made to log error messages to a log file by #defining
//	the manifest constant LOGGING.  Messages will be logged provided
//	that the application has previously called LogOpen().
//	
//	Also included are two functions, TimeString() and DateString(), 
//	which get the system time and data and format them into strings.
//	
//-------------------------------------------------------------------------
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <dos.h>
#include "avkapi.h"
#include "bitflags.h"
#include "log.h"

// 	Undocumented AVK function to return an error description string for
//	a given AVK error value.  This is not supported and should not be 
//	used in production code.

extern AVK_CALL AvkGetErrString(U16 ErrCode, I16 Len, char far *pDestin);

#define	DE_INIT			0x01
#define	DE_LOG_ON		0x02
#define DE_INCL_DATE	0x04
#define	DE_INCL_TIME	0x08

static unsigned short	DeStatus = 0;

static HAVK 	hAvk = (HAVK)0;

char	   *TimeString(VOID);
char	   *DateString(VOID);


//-------------------------------------------------------------------------
//FUNCTION:
//
//	void InitErrSystem(hAvkSession, bLogging, bShowDate, bShowTime)
//	
//PARMS IN:
//	
//	HAVK	hAvkSession		AVK session handle returned from AvkBeginMsg().
//	BOOL	bLogging		If TRUE then logging is desired
//	BOOL	bShowDate		If TRUE then logging includes a date-stamp
//	BOOL	bShowTime		If TRUE then logging includes a time-stamped
//	
//DESCRIPTION:
//
//	Initialize message display function.  This allows the application to
//	register the AVK session handle for use when calling AvkGetErrString()
//	in DispAvkErr().  Boolean flags can also be set to turn logging on
//	and select date- and time-stamping in the log file.
//
//	NO MESSAGE BOXES WILL BE DISPLAYED UNTIL THIS FUNCTION HAS BEEN CALLED
//	TO INITIALIZE THE SYSTEM.
//
//	To turn on logging, the application needs to pass the AVK session
//	handle retrieved from AvkBeginMsg() or AvkBeginCallback().  Either
//	of these functions must be called before InitErrSystem() to obtain
//	the handle.  If no logging is desired, the InitErrSystem can be
//	called before the AVK session is begon.
//
//	Note that InitErrSystem can be called more than once in an application
//	to change the logging status.  Thus, an application can call 
//	InitErrSystem() with a null AVK handle and FALSE for the bLogging
//	parameter at the beginning of the application to enable error reporting
//	and then call it again after AvkBeginMsg() to enable logging, and
//	again periodically to turn logging off and on.
//
//	InitErrSystem() can also be used in conjunction with CloseErrSystem()
//	to turn error reporting on and off in an application.
//
//-------------------------------------------------------------------------

void
InitErrSystem(HAVK hAvkSession, BOOL bLogging, BOOL bShowDate, BOOL bShowTime)
{
	//	If a null AVK session handle has
	if (hAvkSession == (HAVK)0)
		bLogging = FALSE;
	else
		hAvk = hAvkSession;

	if (bLogging)
	{
		SetFlag(DeStatus, DE_LOG_ON);
	
		if (bShowDate)
			SetFlag(DeStatus, DE_INCL_DATE);
		if (bShowTime)
		 	SetFlag(DeStatus, DE_INCL_TIME);
	}

	SetFlag(DeStatus, DE_INIT);
}

//-------------------------------------------------------------------------
//FUNCTION:
//
//	void CloseErrSystem(VOID)
//	
//DESCRIPTION:
//
//	Turns off error reporting.  The error call functions will still
//	return as normal, but will not display any Message Box or log
//	any errors to the log file.
//
//-------------------------------------------------------------------------

void
CloseErrSystem()
{
	ClearFlag(DeStatus, DE_INIT);
}

//-------------------------------------------------------------------------
//FUNCTION:
//
//	BOOL DispMsg(pMsg)
//
//PARMS IN:
//
//	char *pMsg;			pointer to message to be displayed
//	
//DESCRIPTION:
//	
//	Display a message in a message box and wait for the user to press
//	the OK button.
//	
//RETURN:
//	Always returns TRUE.
//
//-------------------------------------------------------------------------

BOOL
DispMsg(char *pMsg)
{
	if (TestFlag(DeStatus, DE_INIT))
	{
		//	Display the message in a window message box and return 'ok'.

		MessageBox(NULL, pMsg, "Application Message", 
		  MB_ICONINFORMATION | MB_OK);

		if (TestFlag(DeStatus, DE_LOG_ON))
		{
			if (TestFlag(DeStatus, DE_INCL_DATE))
				LogWrite("%s ", DateString());
			if (TestFlag(DeStatus, DE_INCL_TIME))
				LogWrite("%s ", TimeString());
			LogWrite("%s"CR, pMsg);
		}
	}
	return TRUE;
}


//-------------------------------------------------------------------------
//FUNCTION:
//
//	BOOL DispErr(pLoc, pErr)
//
//PARMS IN:
//
//	char *pLog;			pointer to a string with the name of the function
//							that produced the error.
//	char *pErr;			pointer to an error message to be displayed.
//	
//DESCRIPTION:
//	
//	Display a generic error message in a Window's Message Box and wait 
//	for the user to press the OK button. This function always returns
//	the Boolean error value, FALSE, so an application can include a
//	call to DispErr() within a return call, as follows:
//
//		if (MyFunc() == FALSE)
//			return DispErr("MyFunc", "My function failed");
//	
//RETURN:
//
// Always returns FALSE for error.



//	Always returns TRUE.
//
//-------------------------------------------------------------------------

BOOL
DispErr(char *pLoc, char *pErr)
{
	char Buf[128];

	if (TestFlag(DeStatus, DE_INIT))
	{

		//	Make up a message and display it in a window message box

		sprintf(Buf, "ERROR: %s", pErr);
		MessageBox(NULL, Buf, pLoc, MB_OK | MB_ICONEXCLAMATION);

		//	Log it and return an error value.

		if (TestFlag(DeStatus, DE_LOG_ON))
		{
			if (TestFlag(DeStatus, DE_INCL_DATE))
				LogWrite("%s ", DateString());
			if (TestFlag(DeStatus, DE_INCL_TIME))
				LogWrite("%s ", TimeString());
			LogWrite("%s: %s"CR, pLoc, Buf);
		}
	}
	return FALSE;
}


//-------------------------------------------------------------------------
//FUNCTION:
//
//	BOOL DispAvkErr(AvkRet, pFunc)
//
//PARMS IN:
//
//	int		AvkRet		The error value returned from a failed AVK API
//							function.
//	char 	*pFunc		Pointer to string with the failed function's name.
//	
//DESCRIPTION:
//	
//	Display an AVK error message in a Window's Message Box and wait 
//	for the user to press the OK button.  This function uses an 
//	undocumented AVK API call, AvkGetErrString() to retrieve a string
//	from AVK describing the error.  This call should not be used in
//	production code, since, as an unsupported function, it is not
//	guaranteed to return the correct error description in future
//	version (if it exists at all).  DispAvkErr() also extracts the
//	AVK subsystem error, if any.  If an AVK API function fails in
//	a lower-level 'subsystem', a subsystem error will give additional
//	information on the exact location and nature of the failure.
//	Like DispErr(), this function always returns the Boolean error value, 
//	FALSE, so an application can include a call to DispAvkErr() within a 
//	return call, as follows:
//
//	if ((AvkRet = AvkBeginMsg(hWnd, &hAvk, AVK_SESSION_DEFAULT) != AVK_ERR_OK)
//		return DispAvkErr(AvkRet, "AvkBeginMsg");
//	
//RETURN:
//
// Always returns FALSE for error.
//
//-------------------------------------------------------------------------

BOOL
DispAvkErr(int AvkRet, char *pFunc)
{
	char 	Buf1[128];
	char	Buf2[128];
	U16		SubSysErr = 0;

	if (TestFlag(DeStatus, DE_INIT))
	{

		//	Get error string and subsystem error from AVK.  If we don't
		//	have access to the AVK session handle, set the subsystem
		//	error to 0xffff.

		if (AvkGetErrString(AvkRet, 80, (char *)Buf1))
		 	sprintf(Buf1, "AVK ERROR %x", AvkRet);

		if (hAvk != (HAVK)0)
			AvkGetSubSysErrCode(hAvk, &SubSysErr);
		else
			SubSysErr = 0xffff;

		//	Make up a message and display it in a window message box

		sprintf(Buf2, "%s [SubSys 0x%04x]", Buf1, SubSysErr);

		MessageBox(GetFocus(), Buf2, pFunc, MB_OK | MB_ICONHAND);

		//	Log it and return the error value.

		if (TestFlag(DeStatus, DE_LOG_ON))
		{
			if (TestFlag(DeStatus, DE_INCL_DATE))
				LogWrite("%s ", DateString());
			if (TestFlag(DeStatus, DE_INCL_TIME))
				LogWrite("%s ", TimeString());

			LogWrite("%s - %s"CR, pFunc, Buf2);
		}
	}
	return FALSE;
}

//-------------------------------------------------------------------------
//FUNCTION:
//	
//	char *TimeString(VOID)
//
//DESCRIPTION:
//	
//	Extracts the computer's system time and builds a string in the format:
//	
//		hh:mm:ss.xx 
//	
//	where hh=hours, mm=minutes, ss=seconds and xx=hundredths of a second.
//
//
//RETURN:
//
//	Pointer to an internal buffer which holds the formatted time string.
//
//-------------------------------------------------------------------------
	
char *
TimeString(VOID)
{
	static char 		Buf[16];
	struct dostime_t	Tm;

	_dos_gettime(&Tm);
	sprintf(Buf, "%02u:%02u:%02u.%02u", Tm.hour, Tm.minute, Tm.second, 
	  Tm.hsecond);
	return Buf;
}

//-------------------------------------------------------------------------
//FUNCTION:
//	
//	char *DateString(VOID)
//
//DESCRIPTION:
//	
//	Extracts the computer's system date and builds a string in the format:
//	
//		mm/dd/yy 
//	
//	where mm=month, dd=day and yy=year.
//
//
//RETURN:
//
//	Pointer to an internal buffer which holds the formatted date string.
//
//-------------------------------------------------------------------------
	
char *
DateString(VOID)
{
	static char Buf[16];
	struct dosdate_t	Dt;

	_dos_getdate(&Dt);
	sprintf(Buf, "% 2u/%02u/%02u", Dt.month, Dt.day, Dt.year % 100);
	return Buf;
}



