/****************************************************************
| 
| H_MAIN.C -- 
|     Main Module for DDJ Recognition Engine Test Harness.
+-----------------------------------------------------------------
|     Original Mac version by Ron Avitzur.
|     Modifications and port to DOS/Windows by Ray Valdes.
|     See READ.ME for description of files, code, architecture,
|     configuration, and coding conventions.
|     
|     FileName          Size      Contents
|     -----------    ---------   ------------------------------
|     RECOG    C       27975     Sample recognizer code.
|     RECOG    H         345     Sample recognizer private header file.
|     H_RECOG  H         766     Interface between recognizer and harness.
|     H_MAIN   C       13755     Main source code module of test harness.
|     H_CONFIG H        4050     Global configuration file.
|     H_STDDEF H         994     Standard typedefs.
|     H_MEM    H         535     Public interface to memory allocation layer.
|     H_MEM    C        3474     Implementation   of memory allocation layer.
|     H_LIST   H         496     Declarations related to List data structure.
|     
\****************************************************************/

#include "h_config.h"

#include <time.h>

#ifdef TURBO_C
#include <stdio.h>
#include <setjmp.h>
#endif

#ifdef MSOFT_C7
#include <stdio.h>
#include <setjmp.h>
#include <process.H>
#endif

#ifdef LIGHTSPEED_C
#include <console.h>
#include <profile.h>
#endif

#include "h_stddef.h"
#include "h_list.h"
#include "h_recog.h"
#include "h_mem.h"

/****************************************************************
|    DATA STRUCTURES -- see header file RECOG.H for the typedefs
+-----------------------------------------------------------------
|    Character data is read from a file on disk (such as RON.DAT)
|    into in-memory data structures.  The top level structure is
|    an array of List structs which holds the entire character set.
|    The array is indexed by the ASCII code of a character.
|    Each item in the array is List of prototypes for that character.
|    Characters are loaded into memory as they are read in.
|    Training proceeds as the character is read in.  Then the
|    guessing occurs.
|
\***************************************************************/

#define FILE_SIGNATURE 0x2121

#define MAX_CHARS 256
private lpList  CharData[MAX_CHARS];
private double  ElapsedTime;
private double  TotalCharsProcessed;
private double  NumRight;

#ifdef USE_INCREMENTAL_IO
typedef unsigned char	    BYTE;
typedef unsigned short	    WORD;
private jmp_buf             main_loop;
private long                num_bytes = 0;

private INT16   GetWord(FILE *data_file);
private void    ParseDataFormat(FILE *data_file);
#else
private void    ParseDataFormat(LPINT16 p,long size);
#endif

private void    ReadDataIntoMemory(LPSTR name);
private void    FreeMemStructures(void);
private void    LoadAndTrain(LPSTR name,INT16 begin,INT16 end);
private void    LoadAndTest(LPSTR name,INT16 begin,INT16 end);
private void    FatalError(char* msg);

public  void    NormalizeChar(int char_code,int num_proto,lpList gesture);


/****************************************************************/
public int main()
{
    char file_name[64];
    INT16 begin,end;
    FILE *control_file;
    
#ifdef REDUCED_DATA_SET
    printf("DDJ Test Harness -- V 1.1 (for Reduced Data Set)");
#else
    printf("DDJ Test Harness -- V 1.0 (Standard Data Set)");
#endif

    control_file = fopen("Control.txt","r");  
    if (control_file==0) 
	FatalError("***Could not open control file CONTROL.TXT!\n");
 
    rec_InitRecognizer();

    while (fscanf(control_file,
	"Train on columns %d - %d in %s\n",&begin,&end,file_name) == 3)
    {
        LoadAndTrain(file_name, begin-1, end-1);
    }
    rec_EndTraining();

    ElapsedTime = 0;
    TotalCharsProcessed = NumRight = 0;

    while (fscanf(control_file,
	"Guess columns %d - %d in %s\n",&begin,&end,file_name)
	== 3)
    {
        LoadAndTest(file_name, begin-1, end-1);
    }

    printf("\nScore: "
	"%.0lf/%.0lf = %.1lf%% in %.0lf seconds (%.0lf chars/sec)\n",
        NumRight,
	TotalCharsProcessed,
	(TotalCharsProcessed==0)?0:100*NumRight/TotalCharsProcessed,
	ElapsedTime,
	(ElapsedTime==0)?TotalCharsProcessed:TotalCharsProcessed/ElapsedTime
	);
    return 1;
}
/****************************************************************/
private void LoadAndTrain(LPSTR file_name,INT16 begin,INT16 end)
{
    INT16 i,j;

    ReadDataIntoMemory(file_name);
    printf("\nTraining...");

#ifdef REDUCED_DATA_SET
    for (i = 47; i < 111; i++)
    {
	if(i>57 && i<64) continue;
	if(i>90 && i<94) continue;
#else
    for (i = 0; i < MAX_CHARS; i++) 
    {
#endif
	if (CharData[i])
	{
	    /* call Train() once per prototype */
            for (j = begin; j <= end && j < CharData[i]->num_items; j++)
	    {
#ifdef REDUCED_DATA_SET
		if(!CharData[i]->items[j])
		{
		    printf("\nSkipped char=%c inst=%d",i,j);
		    continue;
		}
#endif
		NormalizeChar(i,j,CharData[i]->items[j]);
                rec_Train(CharData[i]->items[j], i);
	    }
	}
    }
}
/****************************************************************/
private void LoadAndTest(LPSTR file_name,INT16 begin,INT16 end)
{
    INT16 i,j,g1,g2,g3,w1,w2,w3;
    clock_t startTime;

    ReadDataIntoMemory(file_name);
    printf("\nTesting...");
    startTime = clock();

#ifdef REDUCED_DATA_SET
    for (i = 47; i < 111; i++)
    {
	if(i>57 && i<64) continue;
	if(i>90 && i<94) continue;
#else
    for (i = 0; i < MAX_CHARS; i++)
    {
#endif	
      if (CharData[i])
      {
        for (j = begin;  j <= end && j < CharData[i]->num_items; j++) 
	{
#ifdef REDUCED_DATA_SET
	    if(!CharData[i]->items[j])
		{
		    printf("\nSkipped char=%c inst=%d",i,j);
		    continue;
		}
#endif
            TotalCharsProcessed++;
            g1 = g2 = g3 = -1; w1 = w2 = w3 = 0;

            NormalizeChar(i,j,CharData[i]->items[j]);
            rec_Guess(CharData[i]->items[j], &g1, &g2, &g3, &w1, &w2, &w3);

            if      (i == g1)   NumRight += w1/(double)(w1+w2+w3);
            else if (i == g2)   NumRight += w2/(double)(w1+w2+w3);
            else if (i == g3)   NumRight += w3/(double)(w1+w2+w3);
        }
    }
    }
    ElapsedTime += (clock()-startTime)/CLOCKS_PER_SEC;
}
/****************************************************************/
private void ReadDataIntoMemory(LPSTR file_name)
{
#ifdef USE_INCREMENTAL_IO
    FILE *data_file;
#else
    LPINT16 buf;
#endif

    printf("\nReading datafile into memory...");

#ifdef USE_INCREMENTAL_IO
    data_file = fopen(file_name,"rb");
    if (!data_file) 
	FatalError("Could not open datafile for reading.");

    FreeMemStructures();
    ParseDataFormat(data_file);
    fclose(data_file);
#else
    long file_size;
    FILE *data_file = fopen(file_name,"rb");
    if (!data_file) { printf("File %s not found\n",file_name); return; }
    fseek(data_file,0L,SEEK_END);
    file_size = ftell(data_file);
    rewind(data_file);
    buf = mem_Alloc(file_size); 
    if (!buf) mem_AllocFailed(file_size);
    fread(buf,1,file_size,data_file);
    fclose(data_file);
    FreeMemStructures();
    ParseDataFormat(buf,file_size);
    mem_Free(buf);
#endif
}
/****************************************************************/
private void FreeMemStructures(void)
{
    INT16 i,j,k;
    for (i = 0; i < MAX_CHARS; i++) if (CharData[i]) 
    {
        for (j = 0; j < CharData[i]->num_items; j++) 
	{
            lpList gesture = CharData[i]->items[j];
            for (k = 0; k < gesture->num_items; k++)
                mem_Free(gesture->items[k]);
            mem_Free(gesture);
        }
        mem_Free(CharData[i]);
        CharData[i] = NULL;
    }
}
/****************************************************************/
#ifdef USE_INCREMENTAL_IO
private INT16  GetWord(FILE *data_file)
{
    char s[2];
    s[0] = s[1] = 0;
    if(fread(&s, sizeof(char), 2, data_file) != 2)
	longjmp( main_loop, -1 );
    num_bytes += 2;
    return (INT16)( ((BYTE)s[1]) | ((WORD)((BYTE)s[0])<<8) );
}
/****************************************************************/
private void ReadHeader(FILE *data_file)
{
    INT16 signature,header_size,version_major,version_minor;
    signature = GetWord(data_file);
    if(signature != FILE_SIGNATURE) FatalError("***Header format error!***");
    header_size = GetWord(data_file); //number of bytes should be even
    if(header_size & 01) FatalError("***Header format error #2!***");
    header_size >>= 1;  // number of words
    header_size--;
    version_major = GetWord(data_file); header_size--;
    version_minor = GetWord(data_file); header_size--;    
    while(header_size--) GetWord(data_file);
    //the following line serves only to get rid of a compiler warning
    version_major++; version_minor++; 
}
#else
private LPINT16 ReadHeader(LPINT16 p)
{
    INT16 signature,header_size,version_major,version_minor;
    signature = *p++; //number of bytes should be even
    if(signature != FILE_SIGNATURE) FatalError("***Header format error!***");
    header_size = *p++; //number of bytes should be even
    if(header_size & 01) FatalError("***Header format error #2!***");
    header_size >>= 1;  // number of words
    version_major = *p++; header_size--;
    version_minor = *p++; header_size--;    
    while(header_size--) p++;
    return p;
}
#endif
/****************************************************************/
#ifdef USE_INCREMENTAL_IO
private void ParseDataFormat(FILE *data_file)
{
    VHPoint Pt; 
    lpList   gesture;
    lpList   stroke;
    INT16    i,j,k,num_strokes,num_pts,which;
    INT16    char_code,num_protos;
    

    if( setjmp( main_loop )==0 ) 
    {
	ReadHeader(data_file);
	while (1)
        {
        char_code = GetWord(data_file);
        num_protos = GetWord(data_file);

	if(char_code < 127 && char_code > 32) 
	    printf(" <%c>",char_code);
	else
	    printf(" <c=%d>",char_code);
#else
private void ParseDataFormat(LPINT16 p,long size)
{
    VHPoint Pt; 
    lpList  gesture;
    lpList  stroke;
    INT16 i,j,k,num_protos,num_strokes,num_pts,char_code,which;
    LPINT16 end = p + size/2;

    p = ReadHeader(p);
    
    while (p < end) 
    {
        char_code = *p++;
        num_protos = *p++;
#endif
        which = (char_code < MAX_CHARS) ? char_code : 1;
        CharData[which] = mem_Alloc(sizeof(INT16) + num_protos * sizeof(long));
	if(!CharData[which]) 
	    mem_AllocFailed(sizeof(INT16) + num_protos * sizeof(long));
        CharData[which]->num_items = num_protos;
        for (i = 0; i < num_protos; i++) 
	{
#ifdef USE_INCREMENTAL_IO
            num_strokes = GetWord(data_file);
#else
            num_strokes = *p++;
#endif	
#ifdef REDUCED_DATA_SET
	    if(num_strokes==0)
	    {
		--(CharData[which]->num_items);
		continue;
	    }
#endif
            gesture = mem_Alloc(sizeof(INT16) + num_strokes * sizeof(long));
	    if(!gesture) 
		mem_AllocFailed(sizeof(INT16) + num_strokes * sizeof(long));
            gesture->num_items = num_strokes;
            for (j = 0; j < num_strokes; j++) 
	    {
#ifdef USE_INCREMENTAL_IO
                num_pts = GetWord(data_file);
		Pt.h = GetWord(data_file);
		Pt.v = GetWord(data_file);
#else	    
                num_pts = *p++; Pt.h = *p++; Pt.v = *p++;
#endif	    
                stroke = mem_Alloc(sizeof(INT16) + num_pts * sizeof(long));
		if(!stroke) 
		    mem_AllocFailed(sizeof(INT16) + num_pts * sizeof(long));
                stroke->num_items = num_pts;
                stroke->items[0] = *(void**)&Pt;
                for (k = 1; k < num_pts; k++) 
		{
#ifdef USE_INCREMENTAL_IO
		    INT16 s = GetWord(data_file);
		    LPSTR ps = (LPSTR )&s;
                    Pt.h += ps[1]; Pt.v += ps[0];
#else
                    Pt.h += *(char*)p; Pt.v += *(((char*)p)+1); p++;
#endif		
                    stroke->items[k] = *(void**)&Pt;
                } /* for loop (k) */
                gesture->items[j] = stroke;

            } /* for loop (j) */
            CharData[which]->items[i] = gesture;
           }  /* for loop (i) */                  
#ifdef USE_INCREMENTAL_IO
         } /*while loop */
       } /*if*/
       else if (!feof(data_file))
       {
	   FatalError("Error reading data file!");
       }
#endif    
}
/****************************************************************/
private void FatalError(char* msg)
{
    fprintf(stderr,"\n****Fatal Error: ");
    fprintf(stderr,msg);
    exit(0);
}

/****************************************************************
NormalizeChar -- 
    determines bounding box, 
    then normalizes coordinates so that (0,0) is the base.
    Also displays the character.
****************************************************************/
#ifdef TURBO_C
#pragma argsused
#endif
#define MIN_VALUE (-0x7FFF)
#define MAX_VALUE ( 0x7FFF)

public void NormalizeChar(int char_code,int num_proto,lpList gesture)
{
    INT16 i,j;
    INT16 x_min,x_max,y_min,y_max;
    INT16 height, width;

#ifdef MSOFT_C7
    /*this is in place of Borland's #pragma argused  */
    num_proto; char_code; height; width;  
#endif
     x_min = y_min = MAX_VALUE;
     x_max = y_max = MIN_VALUE;

    /* first find the maximum and minimum */
    for (i = 0; i < gesture->num_items; i++) 
    {
        lpList    stroke =   gesture->items[i];

        for (j = 0; j < stroke->num_items; j++) 
	{
            VHPoint p = *(lpVHPoint)&stroke->items[j];
            if (p.h > x_max) x_max = p.h;
            if (p.h < x_min) x_min = p.h;
            if (p.v > y_max) y_max = p.v;
            if (p.v < y_min) y_min = p.v;
        }
    }
	/* then subtract minimum from the character */
    for (i = 0; i < gesture->num_items; i++) 
    {
        lpList    stroke =   gesture->items[i];
	
        for (j = 0; j < stroke->num_items; j++) 
	{
            VHPoint p = *(lpVHPoint)&stroke->items[j];
	    p.h -= (x_min - 100);
	    p.v -= (y_min - 100);	    
            *(lpVHPoint)&stroke->items[j] = p;
        }
    }
}
/****************************************************END*********/


