#define RV
/*****************************************************************
 * db_recog.c Copyright 1992 by Don Branson.
 *	Initial coding: July 23, 1992 - September 07, 1992
 *
 * Replacement recognizer for Ron Avitzur's handwriting recognition
 * code.
 *
 *
 * Input:
 *	Incoming data is a gesture (character image) which consists of a
 *	list of strokes.  Each stroke is pen movement from pen-down to
 *	pen-up.  Each stroke consists of a list of points.
 *
 * Output:
 *      rec_Train(): Saves a character code in a list of recognizable
 *	characters.  Each character includes identifying attributes.  These
 *	attributes include a list of each component in the character image,
 *	and component attributes.  A component is a line segment,
 *	or an enclosed region.  Each component includes attributes which
 *	describe its orientation and position.  Each component also includes
 *	attributes which quantify its (rough) spatial relationship to other
 *	components in the character.
 *
 *	rec_Guess(): Calculates components and component attributes for
 *	a character, then scans the list for the closest match or matches.
 *
 *
 * Algorithm:
 *      rec_train() and rec_Guess() take the character image in the form
 *	of a list of strokes.  Each stroke is smoothed. After smoothing,
 *	the character is examined for recognizable features. (Regions and
 *	line segments.) The components are then saved with attributes
 *	that roughly identify the component's orientation, position, and
 *	relationships to other components of the character.
 *
 *
 */


#include "h_config.h"

#ifdef TURBO_C
#include <stdlib.h>
#include <math.h>
#include <graphics.h>
#endif
#ifdef MSOFT_C7
#include <search.h>
#include <math.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>



/****************************************************************/

#include "h_stddef.h"
#include "h_mem.h"
#include "h_list.h"
#include "h_recog.h"

/****************************************************************/
#ifndef NIL
#define TRUE               1
#define FALSE              0
#ifdef MSOFT_C7
#define NIL               ((void FAR*)0)
#else
#define NIL                0L
#endif
#define ABS(x)           ((x)>0?(x):(-(x)))
#define MAX(a,b)         ((a)>(b)?(a):(b))
#define MIN(a,b)         ((a)<(b)?(a):(b))
#endif

#define SIGN(x) ((x) < 0 ? -1 : 1)
#define PI (3.14159265)
#define CORNER_THRESHOLD   		(PI/2)	/* RAD's */
#define COMBINE_THRESHOLD		(0)		/* RAD's */
#define ATTACH_THRESHOLD		(0)		/* Distance in points */
#define SHORT_LINE_THRESHOLD 	(25)	/* Percentage of max character dimension */
#define SMALL_REGION_THRESHOLD	(05)	/* Percentage of character bounding region */
#define NEAR_CLOSURE_THRESHOLD	(PI/10)	/* RAD's */

/* Used by GetScore(): */
/* For line segments */
#define LINE_LENGTH_MATCH		(10)
#define LINE_ARC_SIG			(15)	/* Lines with a greater arc should be
										 * checked for an arc match.
										 */
#define LINE_ARC_MATCH			(10)	/* Lines with significant arc should
										 * not differ more than this in arc
										 * for a match.
										 */
#define LINE_NOARC_MATCH		(10)	/* Lines without significant arc
										 * should not differ more than this
										 * in arc for a match.
										 */
#define LINE_SLOPE_MATCH		(PI/8)
#define LINE_FACE_MATCH			(PI/8)
#define LINE_LENGTH_2DIFF		(30)	/* If the lengths of 2 line segs
										 * differ by more that more than this
										 * percentage, the lines don't match,
										 * even if other attributes match.
										 */
/* For regions: */
#define BOUNDING_BOX_MATCH		(10)
#define REGION_FACE_MATCH		(PI/4)
#define BOX_PERCENT_2DIFF		(50)	/* If the bounding box sizes differ by
										 * more than this percentage, the
										 * regions don't match, even if other
										 * attributes match.
										 */
/* For both: */
#define CENTROID_DIST_MATCH		(15)	/* Percentage of max dimension */
#define CENTROID_ANGLE_MATCH	(PI/4)
/* For the character: */
#define CHAR_MAX_2DIFF			(60)	/* If the character size varies by
										 * more than this percentage of
										 * of the larger, the character
										 * doesn't match.
										 */


#define TWO_GUESSES 0

/*---------------------------------------------------------------------------
 * Function prototypes
 * ------------------------------------------------------------------------*/
/* Level-two */
private void ConvertStrokesToComponents(lpList);
private void RecordCharacter(lpList, INT16);
private void ScanForMatch(lpList, LPINT16, LPINT16, LPINT16, LPINT16);

/* Level-three */
private void ExtractDirections(lpList, char *, char *, char *);
private void ExtractRegions(lpList, lpLpList, lpLpList);
private void ExtractNearClosures(lpLpList, lpLpList);
private void SeparateAtCorner(lpList, lpLpList);
private void DiscardShortLineSegments(lpLpList, INT16);
private void DiscardSmallRegions(lpLpList, long);

#if 0
private int  GetScore(struct CharDef *, struct CharDef *);
private void DefineGesture(lpList, lpList, lpList, struct CharDef *);
private void DefineRegions(lpList, struct CharDef *);
private void DefineLineSegments(lpList, struct CharDef *);
#endif
/* Level-four */

/* Defining functions */


/* Gesture functions */
private void GetGestureHeightWidth(lpList, LPINT16, LPINT16);

/* Stroke functions */
private void CleanStroke(lpList);
private double GetArcPercentage(lpList);
private VHPoint GetCentroid(lpList, int, int);

/* Line functions */
private double GetDeltaTheta(VHPoint, VHPoint, VHPoint, VHPoint);
private double GetAverageTheta(VHPoint, VHPoint, VHPoint, VHPoint);
private double GetTheta(INT16, INT16);
private lpVHPoint TheyIntersect(VHPoint, VHPoint, VHPoint, VHPoint, lpVHPoint);
private double DistanceToLine(VHPoint, VHPoint, VHPoint);
private double GetDistance(VHPoint, VHPoint);

/* Subroutines */
private void AppendToList(lpLpList, long);
private int  MatchDirectionStrings(char *, char *);
private void RemoveFromList(lpLpList, INT16);
private void ListPoints(lpList, char *);
private void putCross(int, int, int);
private void TabletToScreen(INT16 x,INT16 y, int* xx, int* yy);
private void CharacterDump(struct CharDef *, char *);


/*---------------------------------------------------------------------------
 * Global data
 * ------------------------------------------------------------------------*/

private char DirectNSEW[9];
private char DirectABCD[9];
private char Direct0123[9];
private lpList Regions;
private lpList LineSegments;
private INT16 gesture_height, gesture_width;

struct RegionDef{
	VHPoint region_centroid;
	VHPoint region_centroid_offset;
	int bounding_box_percent;
	double region_face_angle;
} ;

struct LineSegDef{
	VHPoint line_centroid;
	VHPoint line_centroid_offset;
	int length_percent;
	double slope;
	double line_face_angle;
	double arc_percent;
} ;

struct CharDef{
	INT16 char_code;
	lpList Regions;
	lpList LineSegments;
	VHPoint upper_left;
	VHPoint lower_right;
	VHPoint centroid;
	INT16 initial_strokes;
	char DirectNSEW[9];
	char DirectABCD[9];
	char Direct0123[9];
	char StartZone;
	char EndZone;
} ;

lpList CharDefs;


/****************************************************************/

#ifdef RV
private int GetScore(struct CharDef *testchar, struct CharDef *listchar);
private void DefineGesture(lpList gesture, lpList regions, lpList linesegs,
			  struct CharDef *chardef);
private void DefineRegions(lpList Regions, struct CharDef *chardef);
private void DefineLineSegments(lpList LineSegments, struct CharDef *chardef);
#endif

/* ==========================================================================
 * Level-one routines
 * ========================================================================*/


/*--------------------------------------------------------------------------
 * rec_InitRecognizer()
 *
 *
 *
 *
 */

public void
rec_InitRecognizer(void)
{
	CharDefs = NULL;
}

/*--------------------------------------------------------------------------
 * rec_Train()
 *
 *
 * Input:
 *	- A gesture (character image in the form of a list of strokes.)
 *	- The character code represented by the gesture.
 *
 * Output:
 *	- The character added to the list of known characters, along with
 *        its attributes.
 *
 * Algorithm:
 *	- Convert strokes to components
 *	- Identify component attributes
 *	- Add character to list of known characters
 *
 */

public void rec_Train(lpList gesture, INT16 char_code)
{
    printf("\nTrain: code=%d",char_code);
	ConvertStrokesToComponents(gesture);
	RecordCharacter(gesture, char_code);
	return;
}

/*--------------------------------------------------------------------------
 * rec_Guess()
 *
 * Input:
 *	- A gesture (character image in the form of a list of strokes.)
 *	- Pointers to three possible guesses and guess weights.
 *
 * Output:
 *	- The best guess(es) stored at the addresses passed.
 *
 * Algorithm:
 *	- Convert strokes to components
 *	- Identify component attributes
 *	- Scan for closest matches
 *
 */

#ifdef TURBO_C
#pragma argsused
#endif
public void rec_Guess(lpList gesture,
		LPINT16 g1,LPINT16 g2,LPINT16 g3,
		LPINT16 w1,LPINT16 w2,LPINT16 w3)
{
	int c;
printf("\nGuess: start");
	ConvertStrokesToComponents(gesture);
	ScanForMatch(gesture, g1, g2, w1, w2);
printf("\nGuess: end %d %d %d",*g1,*g2,*g3);
	return;
}


/*--------------------------------------------------------------------------
 * rec_EndTraining()
 *
 *
 *
 *
 */

public void rec_EndTraining(void)
{
}

/* ==========================================================================
 * Level-two routines
 * ========================================================================*/

/*--------------------------------------------------------------------------
 * ConvertStrokesToComponents()
 *
 * Input:
 *	- A list of strokes
 *
 * Output:
 *	- A list of components
 *
 * Algorithm:
 *      - Initialize list of components
 *	- For each stroke
 *		+ Smooth stroke and separate components
 *
 *
 */

#define MAGNIFY 5
private void
ConvertStrokesToComponents(lpList gesture)
{
	INT16 i;
	long gesture_area;
	lpList Components;
	int j, c;

	Components = NULL;
	Regions = NULL;
	LineSegments = NULL;

	GetGestureHeightWidth(gesture, &gesture_height, &gesture_width);
	/* Smooth all strokes in character */
	for(i = 0; i < gesture->num_items; i++)
		CleanStroke(gesture->items[i]);

	/* Get bounding region of character */
	GetGestureHeightWidth(gesture, &gesture_height, &gesture_width);
	gesture_area = (long)gesture_height * (long)gesture_width;

	strcpy(DirectNSEW, "");
	strcpy(DirectABCD, "");
	strcpy(Direct0123, "");
	ExtractDirections(gesture, DirectNSEW, DirectABCD, Direct0123);

	/* Copy all line segments which are part of regions
	 * from gesture into Regions. All other segments go into Components.
	 */
	ExtractRegions(gesture, &Components, &Regions);
	if(Components)
		ExtractNearClosures(&Components, &Regions);
	if(Components)
		SeparateAtCorner(Components, &LineSegments);

	if(Components){
		for(i = 0; i < Components->num_items; i++)
			mem_Free(Components->items[i]);
		mem_Free(Components);
	}

	if(LineSegments)
		DiscardShortLineSegments(&LineSegments, MAX(gesture_height, gesture_width));

	if(Regions)
		DiscardSmallRegions(&Regions, gesture_area);
}

/*--------------------------------------------------------------------------
 * RecordCharacter()
 *
 * Input:
 *	- A character code, and the global list Components and Regions
 *
 * Output:
 *	- The character and components added to the list
 *
 * Algorithm:
 *
 *
 */

private void
RecordCharacter(lpList gesture, INT16 char_code)
{
	int i, j;
	struct CharDef *chardef;
	int c;
	struct RegionDef *rd;
	struct LineSegDef *ld;

	chardef = mem_Alloc(sizeof(struct CharDef));
	if (!chardef) mem_AllocFailed(sizeof(struct CharDef));

	chardef->char_code = char_code;
	DefineGesture(gesture, Regions, LineSegments, chardef);
	DefineRegions(Regions, chardef);
	DefineLineSegments(LineSegments, chardef);

	if(Regions){
		for(i = 0; i < Regions->num_items; i++)
			mem_Free(Regions->items[i]);
		mem_Free(Regions);
	}

	if(LineSegments){
		for(i = 0; i < LineSegments->num_items; i++)
			mem_Free(LineSegments->items[i]);
		mem_Free(LineSegments);
	}

	AppendToList(&CharDefs, ((long)chardef));

	return;
}

/*--------------------------------------------------------------------------
 * ScanForMatch()
 *
 * Input:
 *	- The analysis of the character, and the list of possible characters
 *
 * Output:
 *	- The closest character match
 *
 * Algorithm:
 *
 *
 */

#ifdef TURBO_C
#pragma argsused
#endif
private void
ScanForMatch(lpList gesture, LPINT16 g1, LPINT16 g2, LPINT16 w1, LPINT16 w2)
{
	struct CharDef *chardef;
	struct RegionDef *rd;
	struct LineSegDef *ld;
	int i, j, score, equal_scores;
	int best_match, best_score, c1;
#if TWO_GUESSES
	int second_match, second_score, c2;
#endif
	char buffer[100];
	struct CharDef *cd;

	FILE *out;

	chardef = mem_Alloc(sizeof(struct CharDef));
	if (!chardef) mem_AllocFailed(sizeof(struct CharDef));

	DefineGesture(gesture, Regions, LineSegments, chardef);
	DefineRegions(Regions, chardef);
	DefineLineSegments(LineSegments, chardef);

	best_score = 0;
	best_match = 0;

#if TWO_GUESSES
	second_score = 0;
	second_match = 0;
#endif

	equal_scores = 0;
	for(i=0; i<CharDefs->num_items; i++){
		score = GetScore(chardef, (struct CharDef *)(CharDefs->items[i]));
		if(score == best_score)
			++equal_scores;
		if(score > best_score){
			best_score = score;
			best_match = i;
			equal_scores = 0;
		}
#if TWO_GUESSES
		else
			if(score > second_score){
				second_score = score;
				second_match = i;
			}
#endif
	}

	c1 = ((struct CharDef *)(CharDefs->items[best_match]))->char_code;
#if TWO_GUESSES
	c2 = ((struct CharDef *)(CharDefs->items[second_match]))->char_code;
#endif

	if(Regions){
		for(i = 0; i < Regions->num_items; i++)
			mem_Free(Regions->items[i]);
		mem_Free(Regions);
	}
	if(LineSegments){
		for(i = 0; i < LineSegments->num_items; i++)
			mem_Free(LineSegments->items[i]);
		mem_Free(LineSegments);
	}

	if(chardef->Regions){
		for(i = 0; i < (chardef->Regions)->num_items; i++)
			mem_Free((chardef->Regions)->items[i]);
		mem_Free(chardef->Regions);
	}
	if(chardef->LineSegments){
		for(i = 0; i < (chardef->LineSegments)->num_items; i++)
			mem_Free((chardef->LineSegments)->items[i]);
		mem_Free(chardef->LineSegments);
	}

	mem_Free(chardef);

	*g1 = c1;
	*w1 = best_score;
#if TWO_GUESSES
	if((second_score*100)/best_score > 98){
		*g2 = c2;
		*w2 = second_score;
	}
#endif
	return;
}

/* ==========================================================================
 * Level-three routines
 * ========================================================================*/

/*--------------------------------------------------------------------------
 * ExtractDirections()
 *
 * Input:
 *	- A gesture
 *
 * Output:
 *	- A list of compass directions
 *
 * Algorithm:
 *
 */

private void
ExtractDirections(lpList gesture, char *DirectNSEW, char *DirectABCD,
					char *Direct0123)
{
	int i, j;
	char DNSEW[2], NewDNSEW[2];
	char DABCD[2], NewDABCD[2];
	char D0123[2], NewD0123[2];
	lpList stroke;
	VHPoint p1, p2;
	double theta;
	INT16 MAX, end;

	MAX = MAX(gesture_width, gesture_height);
	for(i=0; i<gesture->num_items; i++){
		stroke = gesture->items[i];
		*(DNSEW+0) = 0;
		*(DNSEW+1) = 0;
		*(DABCD+0) = 0;
		*(DABCD+1) = 0;
		*(D0123+0) = 0;
		*(D0123+1) = 0;
		p1 = *(lpVHPoint)&stroke->items[0];
		j = 1;
		end = stroke->num_items;
		if(stroke->num_items > 3){
			p1 = *(lpVHPoint)&stroke->items[1];
			j = 2;
		}
		if(stroke->num_items > 3)
			end = stroke->num_items-1;
		while(j < end){
			p2 = *(lpVHPoint)&stroke->items[j];
			theta = GetTheta((p2.h-p1.h), (p1.v-p2.v));

			++j;
			if(GetDistance(p1, p2) < MAX/12)
				continue;
			p1 = p2;


			/* Determine NESW */
			if((theta > (7*PI/4)) || (theta <= (PI/4)))
				strcpy(NewDNSEW, "E");
			else
			if((theta > (PI/4)) && (theta <= (3*PI/4)))
				strcpy(NewDNSEW, "N");
			else
			if((theta > (3*PI/4)) && (theta <= (5*PI/4)))
				strcpy(NewDNSEW, "W");
			else
				strcpy(NewDNSEW, "S");

			if((*NewDNSEW != *DNSEW) && strlen(DirectNSEW) < 8)
				strcat(DirectNSEW, NewDNSEW);
			*DNSEW = *NewDNSEW;


			/* Determine ABCD */
			if(theta <= (PI/2))
				strcpy(NewDABCD, "A");
			else
			if((theta > (PI/2)) && (theta <= (PI)))
				strcpy(NewDABCD, "B");
			else
			if((theta > (PI)) && (theta <= (3*PI/2)))
				strcpy(NewDABCD, "C");
			else
				strcpy(NewDABCD, "D");

			if((*NewDABCD != *DABCD) && strlen(DirectABCD) < 8)
				strcat(DirectABCD, NewDABCD);
			*DABCD = *NewDABCD;


			/* Determine 012345678 */
			if(theta <= (PI/4))
				strcpy(NewD0123, "0");
			else
			if((theta > (PI/4)) && (theta <= (2*PI/4)))
				strcpy(NewD0123, "1");
			else
			if((theta > (2*PI/4)) && (theta <= (3*PI/4)))
				strcpy(NewD0123, "2");
			else
			if((theta > (3*PI/4)) && (theta <= (4*PI/4)))
				strcpy(NewD0123, "3");
			else
			if((theta > (4*PI/4)) && (theta <= (5*PI/4)))
				strcpy(NewD0123, "4");
			else
			if((theta > (5*PI/4)) && (theta <= (6*PI/4)))
				strcpy(NewD0123, "5");
			else
			if((theta > (6*PI/4)) && (theta <= (7*PI/4)))
				strcpy(NewD0123, "6");
			else
			if(theta > (7*PI/4))
				strcpy(NewD0123, "7");

			if((*NewD0123 != *D0123) && strlen(Direct0123) < 8)
				strcat(Direct0123, NewD0123);
			*D0123 = *NewD0123;

		}
	}
}

/*--------------------------------------------------------------------------
 * ExtractRegions()
 *
 * Input:
 *	- A list of strokes
 *
 * Output:
 *	- A list of components, with self-intersecting strokes isolated
 *	  at the intersection
 *  When a stroke is separated at an intersection, the result should be
 *  a single region and a single line segment. Don't separate the two
 *	halves of the line segment.
 *
 * Algorithm:
 *	- Scan strokes, looking for intersections that are within a single
 *	  stroke, and separate into components at the intersection.
 *
 */

private void
ExtractRegions(lpList gesture, lpLpList Components, lpLpList Regions)
{
	int i, j, k, l;
	VHPoint A1, A2, B1, B2, intersect, p1;
	lpList Component, Region;
	int region_flag;

	for(i=0; i<gesture->num_items; i++){

		/* Transfer all points to Component */
		lpList stroke = gesture->items[i];
		Component = NULL;
		for(j=0; j<stroke->num_items; j++)
			AppendToList(&Component, *(long *)&stroke->items[j]);

		/* Scan for regions */
		region_flag = 0;
		j=2;
		while(j < Component->num_items-1){

			A1 = *(lpVHPoint)&Component->items[j+0];
			A2 = *(lpVHPoint)&Component->items[j+1];
			k=0;
			while(k < j-1){
				B1 = *(lpVHPoint)&Component->items[k+0];
				B2 = *(lpVHPoint)&Component->items[k+1];
				if(TheyIntersect(A1, A2, B1, B2, &intersect)){
					region_flag = 1;

					/* Take all points from region and save */
					Region = NULL;
					AppendToList(&Region, (*(long *)&intersect));
					for(l = k+1; l <= j; l++){
						p1 = *(lpVHPoint)&Component->items[l];
						AppendToList(&Region, (*(long *)&p1));
					}
					AppendToList(&Region, (*(long *)&intersect));
					if(Region)
						AppendToList(Regions, (long)Region);

					/* Remove Region points from stroke, leaving
					 * intersection point on list.
					 */
					Component->items[k+1] = *(GenericT *)&intersect;
					for(l=k+2; l<=j; l++)
						RemoveFromList(&Component, k+2);

					/* Restart scan of stroke from beginning */
					j = 1; /* Will be incremented to 2 */
					break;
				}
				++k;
			}
			++j;
		}
		/* At this point, only non-region points are left in Component.
		 *
		 * If we joined to line segments on either side of a region,
		 * smooth the result before it goes on the line segment list so
		 * we don't make a corner.
		 */
		if(region_flag && Component)
			CleanStroke(Component);
		if(Component)
			AppendToList(Components, (long)Component);
	}
}

/*--------------------------------------------------------------------------
 * ExtractNearClosures()
 *
 * Input:
 *	- A pointer a list of line segments, and a pointer to a list of
 *	  regions
 *
 * Output:
 *	- All near-closures from the line segment list added to the region
 *	  list
 *
 * Algorithm:
 *	- Examine the angle of the lines passing from the centroid through
 *	  each end point, and the distance of each endpoint from the centroid.
 *
 */

private void
ExtractNearClosures(lpLpList LineSegments, lpLpList Regions)
{
	int i, st, en, j;
	VHPoint p1, p2, centroid;
	double theta, d1, d2;
	lpList stroke;
	lpList Component;


	/* Scan forward checking for near-closures  by checking angle */
	for(i=0; i<(*LineSegments)->num_items; i++)	{
		stroke = (*LineSegments)->items[i];
		en = stroke->num_items-1;
		p1 = *(lpVHPoint)&stroke->items[en];
		Component = NULL;

		if(en > 0)
			for(st=0; st<en; st++){
				p2 = *(lpVHPoint)&stroke->items[st+0];
				centroid = GetCentroid(stroke, st, en);
				theta = GetDeltaTheta(centroid, p1, centroid, p2);
				if((theta < NEAR_CLOSURE_THRESHOLD)
				&& (GetDistance(centroid, p1) < 2*GetDistance(centroid, p2))
				&& (GetDistance(centroid, p2) < 2*GetDistance(centroid, p1))){
					/* Transfer line segment to region,
					 * Continue with another stroke.
					 */
					for(j=st; j<en; j++){
						p2 = *(lpVHPoint)&stroke->items[j];
						AppendToList(&Component, (*(long *)&p2));
					}
					if(Component)
						AppendToList(Regions, (long)Component);
					Component = NULL;
					/* Remove region points from stroke */
					for(j=st; j<=en; j++)
						RemoveFromList(&stroke, st);

					/* Create a new line segment with all of the points
					 * after the region.
					 */
					en = stroke->num_items-1;
					if(en > 1){
						for(j=st; j<en; j++){
							p2 = *(lpVHPoint)&stroke->items[j];
							AppendToList(&Component, (*(long *)&p2));
						}
						if(Component)
							AppendToList(LineSegments, (long)Component);
						Component = NULL;
						/* Remove trailing segment points from stroke */
						for(j=st; j<en; j++)
							RemoveFromList(&stroke, st);
					}
					else{
						mem_Free((*LineSegments)->items[i]);
						RemoveFromList(LineSegments, i);
						--i;
					}

					/* Continue at outermost for loop */
					break;
				}
			}
	}

	/* Scan backward looking for near-closures by checking angle */
	for(i=0; i<(*LineSegments)->num_items; i++){
		stroke = (*LineSegments)->items[i];
		st = 0;
		p1 = *(lpVHPoint)&stroke->items[0];
		Component = NULL;

		en = stroke->num_items-1;
		if(en > 0)
			for(; en>st; en--){
				p2 = *(lpVHPoint)&stroke->items[en];
				centroid = GetCentroid(stroke, st, en);
				theta = GetDeltaTheta(centroid, p1, centroid, p2);
				if((theta < NEAR_CLOSURE_THRESHOLD)
				&& (GetDistance(centroid, p1) < 2*GetDistance(centroid, p2))
				&& (GetDistance(centroid, p2) < 2*GetDistance(centroid, p1))){
					/* Transfer line segment to region,
					 * Continue with another stroke.
					 */
					for(j=st; j<en; j++){
						p2 = *(lpVHPoint)&stroke->items[j];
						AppendToList(&Component, (*(long *)&p2));
					}
					if(Component)
						AppendToList(Regions, (long)Component);
					Component = NULL;
					/* Remove region points from stroke */
					for(j=st; j<=en; j++)
						RemoveFromList(&stroke, st);

					/* Create a new line segment with all of the points
					 * after the region.
					 */
					en = stroke->num_items-1;
					if(en > 1){
						for(j=st; j<en; j++){
							p2 = *(lpVHPoint)&stroke->items[j];
							AppendToList(&Component, (*(long *)&p2));
						}
						if(Component)
							AppendToList(LineSegments, (long)Component);
						Component = NULL;
						/* Remove trailing segment points from stroke */
						for(j=st; j<en; j++)
							RemoveFromList(&stroke, st);
					}
					else{
						mem_Free((*LineSegments)->items[i]);
						RemoveFromList(LineSegments, i);
						--i;
					}

					/* Continue at outermost for loop */
					break;
				}
			}
	}
}

/*--------------------------------------------------------------------------
 * SeparateAtCorner()
 *
 * Input:
 *	- A list of strokes
 *
 * Output:
 *	- A list of components, without attributes
 *
 * Algorithm:
 *	- Scan strokes (or components), looking for corners as specified
 *	  by CORNER_THRESHOLD, and separate at the corner.
 *
 */

private void
SeparateAtCorner(lpList gesture, lpLpList LineSegments)
{
	INT16 i,j, k;
	double delta_theta;
	VHPoint p1, p2, p3, p4;
	lpList Component;

	/* Identify component breaks */
	Component = NULL;	/* List of points */
	*LineSegments = NULL;	/* List of line segments */
	for(i = 0; i < gesture->num_items; i++){
		lpList stroke = gesture->items[i];
		delta_theta = 0;
		for(j = 0; j < stroke->num_items-3; j++){
			p1 = *(lpVHPoint)&stroke->items[j+0];
			p2 = *(lpVHPoint)&stroke->items[j+1];
			p3 = *(lpVHPoint)&stroke->items[j+2];
			if(j+4 < stroke->num_items)
				p4 = *(lpVHPoint)&stroke->items[j+4];
			else
				p4 = *(lpVHPoint)&stroke->items[j+3];

			AppendToList(&Component, (*(long *)&p1));

			delta_theta = GetDeltaTheta(p1, p2, p3, p4);
			if(delta_theta > CORNER_THRESHOLD){

				AppendToList(&Component, (*(long *)&p2));
				AppendToList(&Component, (*(long *)&p3));

				if(Component)
					AppendToList(LineSegments, (long)Component);
				Component = NULL;

				AppendToList(&Component, (*(long *)&p3));
				j+=2;
			}
		}
		if(j >= stroke->num_items-4){
			for(k=j; k<stroke->num_items; k++){
				p1 = *(lpVHPoint)&stroke->items[k];
				AppendToList(&Component, (*(long *)&p1));
			}
		}
		if(Component)
			AppendToList(LineSegments, (long)Component);
		Component = NULL;
	}
}

/*--------------------------------------------------------------------------
 * DiscardShortLineSegments()
 *
 * Input:
 *	- A list of line-segment strokes
 *	  The maximum dimension of the character
 *
 * Output:
 *	- The list with short line segments gone
 *
 * Algorithm:
 *	If the length of the segment is less than SHORT_LINE_THRESHOLD percent
 *	of the length of the longest dimension of the character, throw it away
 *
 *
 */

private void
DiscardShortLineSegments(lpLpList gesture, INT16 max_dimension)
{
	INT16 i, j;
	lpList stroke;
	VHPoint p;
	double len, dX, dY;
	VHPoint p1, p2;

	for(i = 0; i < (*gesture)->num_items; i++){
		stroke = (*gesture)->items[i];
		/* Get length of stroke */
		len = 0;
		for(j=0; j<stroke->num_items-1; j++){
			p1 = *(lpVHPoint)&stroke->items[j+0];
			p2 = *(lpVHPoint)&stroke->items[j+1];
			dX = p2.h-p1.h;
			dY = p2.v-p1.v;
			len += sqrt(dX*dX + dY*dY);
		}

		if(len < ((double)max_dimension *
		(double)SHORT_LINE_THRESHOLD)/(double)100){
			mem_Free((*gesture)->items[i]);
			RemoveFromList(gesture, i);
			--i;
		}
	}
}

/*--------------------------------------------------------------------------
 * DiscardSmallRegions()
 *
 * Input:
 *	- A list of line-segment strokes
 *	  The maximum dimension of the character
 *
 * Output:
 *	- The list with short line segments gone
 *
 * Algorithm:
 *	If the area of the bounding box of the region is less than
 *	SMALL_REGION_THRESHOLD percent of the area of the bounding box of the
 *	character, throw it away
 *
 *
 */

private void
DiscardSmallRegions(lpLpList gesture, long charArea)
{
	INT16 i, j;
	lpList stroke;
	VHPoint p;
	INT16 YMIN, YMAX, XMIN, XMAX;
	long region_area;

	for(i = 0; i < (*gesture)->num_items; i++){
		stroke = (*gesture)->items[i];
		p = *(lpVHPoint)&stroke->items[0];
		YMIN = p.v;
		YMAX = 0;
		XMIN = p.h;
		XMAX = 0;
		/* Get bounding region of stroke */
		for(j=0; j<stroke->num_items; j++){
			p = *(lpVHPoint)&stroke->items[j];
			YMIN = MIN(p.v, YMIN);
			YMAX = MAX(p.v, YMAX);
			XMIN = MIN(p.h, XMIN);
			XMAX = MAX(p.h, XMAX);
		}

		region_area = (long)(YMAX - YMIN) * (long)(XMAX - XMIN);
		if(region_area < (charArea *
		(long)SMALL_REGION_THRESHOLD)/(long)100){
			mem_Free((*gesture)->items[i]);
			RemoveFromList(gesture, i);
			--i;
		}
	}
}

/*--------------------------------------------------------------------------
 * GetScore()
 *
 * Input:
 *	-
 *
 * Output:
 *	-
 *
 * Algorithm:
 *
 */

private int
GetScore(struct CharDef *testchar, struct CharDef *listchar)
{
	int score;
	double dX1, dX2, dY1, dY2;
	int i, j;
	double MAXX, MAXY;
	struct RegionDef *rt, *rl;
	struct LineSegDef *lt, *ll;
	double theta;
	int region_score, best_region_score;
	int line_score, best_line_score;
	int regions_matched, lines_matched;
	int centroid_dist;

	score = 0;

	if(testchar->initial_strokes != listchar->initial_strokes)
		return 0;

	/* Examine direction strings */
	score += 2*MatchDirectionStrings(testchar->DirectNSEW, listchar->DirectNSEW);
	score += 2*MatchDirectionStrings(testchar->DirectABCD, listchar->DirectABCD);
	score += MatchDirectionStrings(testchar->Direct0123, listchar->Direct0123);

	if(strcmp(testchar->DirectNSEW, listchar->DirectNSEW) == 0)
		score += 10;
	if(strcmp(testchar->DirectABCD, listchar->DirectABCD) == 0)
		score += 10;

	if(testchar->StartZone == listchar->StartZone)
		score += 7;
	if(testchar->EndZone == listchar->EndZone)
		score += 7;


	/* Look for character size match */
	dX1 = ((listchar->lower_right).h - (listchar->upper_left).h);
	dY1 = ((listchar->lower_right).v - (listchar->upper_left).v);
	dX2 = ((testchar->lower_right).h - (testchar->upper_left).h);
	dY2 = ((testchar->lower_right).v - (testchar->upper_left).v);
	MAXX = MAX(dX1, dX2);
	MAXY = MAX(dY1, dY2);
	if(MAXY >= MAXX){
		if(ABS(dY1 - dY2) < MAXY/5)
			score += 1;
		if(ABS(dY1 - dY2) > MAXY*CHAR_MAX_2DIFF/100)
			score = -1;
	}
	else{
		if(ABS(dX1 - dX2) < MAXX/5)
			score += 1;
		if(ABS(dX1 - dX2) > MAXX*CHAR_MAX_2DIFF/100)
			score = -1;
	}

	/* Check for a match in the number of regions and line Segments */
	if((testchar->Regions)->num_items
	== (listchar->Regions)->num_items)
		score += 2;
	if((testchar->LineSegments)->num_items
	== (listchar->LineSegments)->num_items)
		score += 1;

	regions_matched = 0;
	lines_matched = 0;

	MAXX = ABS(dX1);
	MAXY = ABS(dY1);

	/* Scan for region matches */
	if((testchar->Regions) && (listchar->Regions)){

		i = 0;
		j = 0;
		do{
			if(i >= (testchar->Regions)->num_items)
				break;
			if(j >= (listchar->Regions)->num_items)
				break;

			rt = (struct RegionDef *)(testchar->Regions)->items[i];
			rl = (struct RegionDef *)(listchar->Regions)->items[j];

			region_score = 0;

			/* Look for bounding box area % match */
			if(ABS(rt->bounding_box_percent - rl->bounding_box_percent)
			< BOUNDING_BOX_MATCH)
				region_score += 1;

			/* Check region face angle */
			if(ABS(rt->region_face_angle - rl->region_face_angle)
			< REGION_FACE_MATCH){
				region_score += 2;
			}
			else{
				if((rt->region_face_angle > 7*PI/4)
				&& (rl->region_face_angle < PI/4)){
					if(ABS(((2*PI)-rt->region_face_angle)
					- rl->region_face_angle) < REGION_FACE_MATCH)
						region_score += 2;
				}
				else{
					if((rl->region_face_angle > 7*PI/4)
					&& (rt->region_face_angle < PI/4)){
						if(ABS(rt->region_face_angle
						- ((2*PI)-rl->region_face_angle))
						< REGION_FACE_MATCH)
							region_score += 2;
					}
					else
						region_score = 0;
				}
			}

			/* The centroids of the regions can match on distance, since
			 * they might be close to the character centroid, and have
			 * greatly different angles, but be close, or the centroids
			 * can match on the angle.
			 */
			centroid_dist = GetDistance(rt->region_centroid_offset, rl->region_centroid_offset);
			if(centroid_dist < MAX(MAXX, MAXY)*CENTROID_DIST_MATCH/100)
				region_score += 2;
			else{
				/* Check for region centroid angle match
				 */
				theta = GetDeltaTheta(
					testchar->centroid, rt->region_centroid,
					listchar->centroid, rl->region_centroid);
				if((theta < CENTROID_ANGLE_MATCH)
				&& (centroid_dist < 2*MAX(MAXX, MAXY)*CENTROID_DIST_MATCH/100))
					region_score += 2;
				else
					region_score = 0;
			}

			/* If the bounding box percent is too different, subtract points
			 */
			if(ABS(rt->bounding_box_percent - rl->bounding_box_percent)
			> BOX_PERCENT_2DIFF)
				region_score -= 1;

			if(region_score > 4)
				++regions_matched;
			if(region_score > 2){
				++regions_matched;
				++i;
			}

			++j;
		}while(1);

		score += ((5*regions_matched / (testchar->Regions)->num_items))
			  +  ((5*regions_matched / (listchar->Regions)->num_items));
	}

	/* Scan for line segment matches */
	if((testchar->LineSegments) && (listchar->LineSegments)){

		i = 0;
		j = 0;

		do{
			if(i >= (testchar->LineSegments)->num_items)
				break;
			if(j >= (listchar->LineSegments)->num_items)
				break;

			lt = (struct LineSegDef *)(testchar->LineSegments)->items[i];
			ll = (struct LineSegDef *)(listchar->LineSegments)->items[j];

			line_score = 0;

			/* Look for length % match */
			if(ABS(lt->length_percent - ll->length_percent) < LINE_LENGTH_MATCH){
				line_score += 1;
				if(ABS(lt->length_percent - ll->length_percent) < LINE_LENGTH_MATCH/2)
					line_score += 1;
			}

			/* Check for arc percentage match */
			if(lt->arc_percent > LINE_ARC_SIG){
				if(ABS(lt->arc_percent - ll->arc_percent) < LINE_ARC_MATCH){
					/* Arc percent is significant, check for open face match.
					 */
					if(ABS(lt->line_face_angle - ll->line_face_angle)
					< LINE_FACE_MATCH)
						line_score += 2;
					if((lt->line_face_angle > 7*PI/4)
					&& (ll->line_face_angle < PI/4))
						if(ABS(((2*PI)-lt->line_face_angle) - ll->line_face_angle)
						< LINE_FACE_MATCH)
							line_score += 2;
					if((ll->line_face_angle > 7*PI/4)
					&& (lt->line_face_angle < PI/4))
						if(ABS(lt->line_face_angle - ((2*PI)-ll->line_face_angle))
						< LINE_FACE_MATCH)
							line_score += 2;
				}
			}
			else{
				/* Line does not have significant curvature, check slope
				 * instead.
				 */
				if(ABS(lt->arc_percent - ll->arc_percent) < LINE_NOARC_MATCH){
					if(ABS(lt->slope - ll->slope) < LINE_SLOPE_MATCH)
						line_score += 2;
					else{
						if((lt->slope > 7*PI/4) && (ll->slope < PI/4)){
							if(ABS(((2*PI)-lt->slope) - ll->slope)
							< LINE_SLOPE_MATCH)
								line_score += 2;
						}
						else{
							if((ll->slope > 7*PI/4) && (lt->slope < PI/4)){
								if(ABS(lt->slope - ((2*PI)-ll->slope))
								< LINE_SLOPE_MATCH)
									line_score += 2;
							}
						}
					}
				}
			}

			/* The centroids of the lines can match on distance, since
			 * they might be close to the character centroid, and have
			 * greatly different angles, but be close, or the centroids
			 * can match on the angle.
			 */
			centroid_dist = GetDistance(lt->line_centroid_offset, ll->line_centroid_offset);
			if(centroid_dist < MAX(MAXX, MAXY)*CENTROID_DIST_MATCH/100)
				line_score += 2;
			else{
				/* Check for line centroid angle match
				 */
				theta = GetDeltaTheta(
					testchar->centroid, lt->line_centroid,
					listchar->centroid, ll->line_centroid);
				if((theta < CENTROID_ANGLE_MATCH)
				&& (centroid_dist < 2*MAX(MAXX, MAXY)*CENTROID_DIST_MATCH/100))
					line_score += 2;
				else
					line_score = 0;
			}

			/* If the line length percent is too different, subtract points.
			 */
			if(ABS(lt->length_percent - ll->length_percent) > LINE_LENGTH_2DIFF)
				line_score -= 1;

			if(line_score > 7)
				++lines_matched;
			if(line_score > 3){
				++lines_matched;
				++i;
			}
			++j;


		}while(1);

		score += (5*lines_matched / (testchar->LineSegments)->num_items)
			  +  (5*lines_matched / (listchar->LineSegments)->num_items);
	}

	return score;
}

/* ==========================================================================
 * Level-four routines
 * ========================================================================*/

/*--------------------------------------------------------------------------
 * DefineGesture()
 *
 * Input:
 *	- A list of regions and line segments
 *
 * Output:
 *	- The CharDef structure upper-left corner and lower-right corner
 *	  defined, the list pointers NULL'd
 *
 * Algorithm:
 *
 *
 */

private void
DefineGesture(lpList gesture, lpList regions, lpList linesegs,
			  struct CharDef *chardef)
{
	INT16 i, j;
	INT16 YMIN, YMAX, XMIN, XMAX;
	lpList stroke;
	VHPoint p, ps, pe;
	INT16 upper_threshold, lower_threshold;

	chardef->Regions = NULL;
	chardef->LineSegments = NULL;

	if(regions)
		stroke = regions->items[0];
	else{
		if(linesegs)
			stroke = linesegs->items[0];
		else{
			(chardef->upper_left).h = NULL;
			(chardef->upper_left).v = NULL;
			(chardef->lower_right).h = NULL;
			(chardef->lower_right).v = NULL;
			(chardef->centroid).h = NULL;
			(chardef->centroid).v = NULL;

			return;
		}
	}

	p = *(lpVHPoint)&stroke->items[0];
	YMIN = p.v;
	YMAX = 0;
	XMIN = p.h;
	XMAX = 0;

	if(regions)
		for(i = 0; i < regions->num_items; i++){
			stroke = regions->items[i];
			for(j = 0; j < stroke->num_items; j++){
				p = *(lpVHPoint)&stroke->items[j];
				YMIN = MIN(p.v, YMIN);
				YMAX = MAX(p.v, YMAX);
				XMIN = MIN(p.h, XMIN);
				XMAX = MAX(p.h, XMAX);
			}
		}
	if(linesegs)
		for(i = 0; i < linesegs->num_items; i++){
			stroke = linesegs->items[i];
			for(j = 0; j < stroke->num_items; j++){
				p = *(lpVHPoint)&stroke->items[j];
				YMIN = MIN(p.v, YMIN);
				YMAX = MAX(p.v, YMAX);
				XMIN = MIN(p.h, XMIN);
				XMAX = MAX(p.h, XMAX);
			}
		}
	(chardef->upper_left).h = XMIN;
	(chardef->upper_left).v = YMIN;
	(chardef->lower_right).h = XMAX;
	(chardef->lower_right).v = YMAX;

	(chardef->centroid).h = (XMIN+XMAX)/2;
	(chardef->centroid).v = (YMIN+YMAX)/2;

	(chardef->initial_strokes) = gesture->num_items;
	strcpy((chardef->DirectNSEW), DirectNSEW);
	strcpy((chardef->DirectABCD), DirectABCD);
	strcpy((chardef->Direct0123), Direct0123);

	/* Determine starting and ending quadrant */
	stroke = gesture->items[0];
	ps = *(lpVHPoint)&stroke->items[0];
	stroke = gesture->items[gesture->num_items-1];
	pe = *(lpVHPoint)&stroke->items[stroke->num_items-1];

	upper_threshold = (YMIN + YMAX)/3;
	lower_threshold = 2*(YMIN + YMAX)/3;
	if(ps.h > (XMIN + XMAX)/2)
		if(ps.v < upper_threshold)
			chardef->StartZone = 1;
		else
			if(ps.v < lower_threshold)
				chardef->StartZone = 2;
			else
				chardef->StartZone = 3;
	else
		if(ps.v < upper_threshold)
			chardef->StartZone = 4;
		else
			if(ps.v < lower_threshold)
				chardef->StartZone = 5;
			else
				chardef->StartZone = 6;


	if(pe.h > (XMIN + XMAX)/2)
		if(pe.v < upper_threshold)
			chardef->EndZone = 1;
		else
			if(pe.v < lower_threshold)
				chardef->EndZone = 2;
			else
				chardef->EndZone = 3;
	else
		if(pe.v < upper_threshold)
			chardef->EndZone = 4;
		else
			if(pe.v < lower_threshold)
				chardef->EndZone = 5;
			else
				chardef->EndZone = 6;


	return;
}

/*--------------------------------------------------------------------------
 * DefineRegions()
 *
 * Input:
 *	- A list of regions
 *
 * Output:
 *	- Region summary information
 *
 * Algorithm:
 *
 *
 */

private void
DefineRegions(lpList Regions, struct CharDef *chardef)
{
	struct RegionDef FAR *rd;
	VHPoint char_centroid, p, ps, pe;
	long region_area, char_area;
	int i, j;
	INT16 YMIN, YMAX, XMIN, XMAX;
	lpList stroke;

	if(!Regions)
		return;

	char_centroid.h = ((chardef->upper_left).h + (chardef->lower_right).h)/2;
	char_centroid.v = ((chardef->upper_left).v + (chardef->lower_right).v)/2;
	char_area = (long)((chardef->lower_right).h - (chardef->upper_left).h) *
				(long)((chardef->lower_right).v - (chardef->upper_left).v);

	for(i=0; i<Regions->num_items; i++){
		rd = mem_Alloc(sizeof(struct RegionDef));
		if (!rd) mem_AllocFailed(sizeof(struct RegionDef));

		(rd->region_centroid_offset).h=0;
		(rd->region_centroid_offset).v=0;
		(rd->region_centroid).h=0;
		(rd->region_centroid).v=0;
		rd->bounding_box_percent = -1;
		rd->region_face_angle = -1;

		/* Get max and min points of region */
		stroke = Regions->items[i];
		p = *(lpVHPoint)&stroke->items[0];
		YMIN = p.v;
		YMAX = 0;
		XMIN = p.h;
		XMAX = 0;
		for(j = 0; j < stroke->num_items; j++){
			p = *(lpVHPoint)&stroke->items[j];
			YMIN = MIN(p.v, YMIN);
			YMAX = MAX(p.v, YMAX);
			XMIN = MIN(p.h, XMIN);
			XMAX = MAX(p.h, XMAX);
		}

		/* Calculate region centroid*/
		(rd->region_centroid_offset).h = (XMAX+XMIN)/2 - char_centroid.h;
		(rd->region_centroid_offset).v = (YMAX+YMIN)/2 - char_centroid.v;
		(rd->region_centroid).h=(XMAX+XMIN)/2;
		(rd->region_centroid).v=(YMAX+YMIN)/2;

		/* Calculate region bounding box relative size */
		region_area = (long)(XMAX-XMIN) * (long)(YMAX-YMIN);
		rd->bounding_box_percent = (region_area*100)/char_area;

		/* Calculate region opening angle */
		p.h = (XMAX+XMIN)/2;
		p.v = (YMAX+YMIN)/2;
		ps = *(lpVHPoint)&stroke->items[0];
		pe = *(lpVHPoint)&stroke->items[stroke->num_items - 1];
		rd->region_face_angle = GetAverageTheta(p, ps, p, pe);

		AppendToList(&chardef->Regions, (long)rd);
	}
}


/*--------------------------------------------------------------------------
 * DefineLineSegments()
 *
 * Input:
 *	- A list of line segments
 *
 * Output:
 *	- Line segment summary information
 *
 * Algorithm:
 *
 *
 */

private void
DefineLineSegments(lpList LineSegments, struct CharDef *chardef)
{
	struct LineSegDef FAR *ld;
	VHPoint char_centroid, p, ps, pe;
	int i, j;
	INT16 YMIN, YMAX, XMIN, XMAX;
	lpList stroke;
	INT16 dX, dY;
	int char_max;

	if(!LineSegments)
		return;

	char_centroid.h = ((chardef->upper_left).h + (chardef->lower_right).h)/2;
	char_centroid.v = ((chardef->upper_left).v + (chardef->lower_right).v)/2;
	char_max = MAX(((chardef->lower_right).h - (chardef->upper_left).h),
				   ((chardef->lower_right).v - (chardef->upper_left).v));

	for(i=0; i<LineSegments->num_items; i++){
		ld = mem_Alloc(sizeof(struct LineSegDef));
		if (!ld) mem_AllocFailed(sizeof(struct LineSegDef));

		(ld->line_centroid_offset).h = 0;
		(ld->line_centroid_offset).v = 0;
		(ld->line_centroid).h=0;
		(ld->line_centroid).v=0;
		ld->length_percent = -1;
		ld->slope = 0;
		ld->line_face_angle = -1;
		ld->arc_percent = -1;

		/* Get max and min points of line segment */
		stroke = LineSegments->items[i];
		p = *(lpVHPoint)&stroke->items[0];
		YMIN = p.v;
		YMAX = 0;
		XMIN = p.h;
		XMAX = 0;
		for(j = 0; j < stroke->num_items; j++){
			p = *(lpVHPoint)&stroke->items[j];
			YMIN = MIN(p.v, YMIN);
			YMAX = MAX(p.v, YMAX);
			XMIN = MIN(p.h, XMIN);
			XMAX = MAX(p.h, XMAX);
		}

		/* Calculate line segment centroid*/
		(ld->line_centroid_offset).h = (XMAX+XMIN)/2 - char_centroid.h;
		(ld->line_centroid_offset).v = (YMAX+YMIN)/2 - char_centroid.v;
		(ld->line_centroid).h=(XMAX+XMIN)/2;
		(ld->line_centroid).v=(YMAX+YMIN)/2;

		ps = *(lpVHPoint)&stroke->items[0];
		if(stroke->num_items > 1)
			pe = *(lpVHPoint)&stroke->items[stroke->num_items - 1];
		else
			pe = ps;

		/* calculate line curve face angle */
		p.h = (XMAX+XMIN)/2;
		p.v = (YMAX+YMIN)/2;
		ld->line_face_angle = GetAverageTheta(p, ps, p, pe);

		/* calculate length percent */
		ld->length_percent = (GetDistance(ps, pe)*100) / char_max;

		/* calculate line slope */
		dX = pe.h - ps.h;
		dY = ps.v - pe.v;	/* Reversed, since y-coord increases downward */
		ld->slope = GetTheta(dX, dY);

		ld->arc_percent = GetArcPercentage(stroke);

		AppendToList(&chardef->LineSegments, (long)ld);
	}

}

/*--------------------------------------------------------------------------
 * GetGestureHeightWidth()
 *
 * Input:
 *	- A list of line-segment strokes
 *
 * Output:
 *	- The difference between the highest and lowest point
 *
 * Algorithm:
 *
 *
 */

private void
GetGestureHeightWidth(lpList gesture, LPINT16 gesture_height,
					  LPINT16 gesture_width)
{
	INT16 i, j;
	INT16 YMIN, YMAX, XMIN, XMAX;
	lpList stroke;
	VHPoint p;

	stroke = gesture->items[0];
	p = *(lpVHPoint)&stroke->items[0];
	YMIN = p.v;
	YMAX = 0;
	XMIN = p.h;
	XMAX = 0;

	for(i = 0; i < gesture->num_items; i++){
		stroke = gesture->items[i];
		for(j = 0; j < stroke->num_items; j++){
			p = *(lpVHPoint)&stroke->items[j];
			YMIN = MIN(p.v, YMIN);
			YMAX = MAX(p.v, YMAX);
			XMIN = MIN(p.h, XMIN);
			XMAX = MAX(p.h, XMAX);
		}
	}
	*gesture_height = YMAX - YMIN;
	*gesture_width = XMAX - XMIN;

	return;
}

/*--------------------------------------------------------------------------
 * CleanStroke()
 *
 * Input:
 *	- A stroke (A list of points from pen-down to pen-up)
 *
 * Output:
 *	- A cleaned-up stroke
 *
 * Algorithm:
 *	- Scan the list of points. Each point from the second to the
 *	  next-to-last point are changed to be the average of that point
 *	  and the next. The first and last points are untouched.
 *	- Then, scan the list from the second from the last to the second
 *	  from the beginning, and average the points.
 *	- Then, combine consecutive line segments within a stroke that
 *	  vary little, this improves corner detection.
 *
 */

private void
CleanStroke(lpList stroke)
{
	int i;
	VHPoint p1, p2, p3;
	double delta_theta;
	INT16 MAX;

	/* Smooth forward: */
	for(i = 1; i < stroke->num_items-1; i++){
		p1 = *(lpVHPoint)&stroke->items[i];
		p2 = *(lpVHPoint)&stroke->items[i+1];
		p1.h = (p1.h + p2.h)/2;
		p1.v = (p1.v + p2.v)/2;

		*(lpVHPoint)&stroke->items[i] = p1;
	}

	/* Smooth backward: */
	for(i = stroke->num_items-1; i > 1; i--){
		p1 = *(lpVHPoint)&stroke->items[i];
		p2 = *(lpVHPoint)&stroke->items[i-1];
		p1.h = (p1.h + p2.h)/2;
		p1.v = (p1.v + p2.v)/2;

		*(lpVHPoint)&stroke->items[i-1] = p1;
	}

	/* Remove close points */
	MAX = MAX(gesture_width, gesture_height);
	for(i = 0; i < stroke->num_items-2; i++){
		p1 = *(lpVHPoint)&stroke->items[i+0];
		p2 = *(lpVHPoint)&stroke->items[i+1];
		if(GetDistance(p1, p2) < MAX/17){
			RemoveFromList(&stroke, i+1);
			--i;
		}
	}

	/* If angle varies by less than COMBINE_THRESHOLD RAD, combine successive
	 * line segments in the stroke by deleting the intermediate point
	 * from list.
	 */
#if COMBINE_THRESHOLD
	for(i = 0; i < stroke->num_items-2; i++){
		p1 = *(lpVHPoint)&stroke->items[i+0];
		p2 = *(lpVHPoint)&stroke->items[i+1];
		p3 = *(lpVHPoint)&stroke->items[i+2];

		delta_theta = GetDeltaTheta(p1, p2, p2, p3);
		if(delta_theta <= COMBINE_THRESHOLD)
			RemoveFromList(&stroke, i+1);
	}
#endif

	return;
}

/*--------------------------------------------------------------------------
 * GetArcPercentage()
 *
 * Input:
 *	- A stroke
 *
 * Output:
 *	- The percentage of Arc, relative to the length
 *
 * Algorithm:
 *	- Calculate the distance from the center point of the center line segment
 *	  to the line through the enpoints.
 *
 */

private double
GetArcPercentage(lpList stroke)
{
	int i;
	VHPoint p1, p2;
	double len1, len2, dist;
	double dX, dY;
	VHPoint P;

	/* Find center line point of center line segment */
	len1 = 0;
	for(i=0; i<stroke->num_items-1; i++){
		p1 = *(lpVHPoint)&stroke->items[i+0];
		p2 = *(lpVHPoint)&stroke->items[i+1];
		dX = p2.h-p1.h;
		dY = p2.v-p1.v;
		len1 += sqrt(dX*dX + dY*dY);
	}
	len2 = len1/2;
	for(i=0; i<stroke->num_items-1; i++){
		p1 = *(lpVHPoint)&stroke->items[i+0];
		p2 = *(lpVHPoint)&stroke->items[i+1];
		dX = p2.h-p1.h;
		dY = p2.v-p1.v;
		len2 -= sqrt(dX*dX + dY*dY);
		if(len2 <0)
			break;
	}
	P.h = (p2.h + p1.h)/2;
	P.v = (p2.v + p1.v)/2;

	/* Get distance from calculated center point to line drawn
	 * from starting point to ending point.
	 */
	p1 = *(lpVHPoint)&stroke->items[0];
	p2 = *(lpVHPoint)&stroke->items[stroke->num_items-1];
	dist = DistanceToLine(P, p1, p2);

	return dist*100/len1;
}

/*--------------------------------------------------------------------------
 * GetCentroid()
 *
 * Input:
 *	- A stroke and a start and end point
 *
 * Output:
 *	- The centroid of the part of the stroke given by st and en
 *
 * Algorithm:
 *
 *
 */

private VHPoint GetCentroid(lpList stroke, int st, int en)
{
	INT16 i;
	INT16 YMIN, YMAX, XMIN, XMAX;
	VHPoint p;

	p = *(lpVHPoint)&stroke->items[st];
	YMIN = p.v;
	YMAX = 0;
	XMIN = p.h;
	XMAX = 0;

	for(i = st; i < en; i++){
		p = *(lpVHPoint)&stroke->items[i];
		YMIN = MIN(p.v, YMIN);
		YMAX = MAX(p.v, YMAX);
		XMIN = MIN(p.h, XMIN);
		XMAX = MAX(p.h, XMAX);
	}
	p.h = (XMIN + XMAX)/2;
	p.v = (YMIN + YMAX)/2;

	return p;
}

/*--------------------------------------------------------------------------
 * GetDeltaTheta()
 *
 * Input:
 *	- Four points
 *
 * Output:
 *	- The difference in angle in the line segment (P1,P2) and (P3,P4)
 *	  The angle returned is always a positive angle
 *
 *
 */

private double
GetDeltaTheta(VHPoint p1, VHPoint p2, VHPoint p3, VHPoint p4)
{
	INT16 dX1, dY1, dX2, dY2;
	double theta1, theta2, delta_theta;

	dX1 = p2.h - p1.h;
	dY1 = p1.v - p2.v;	/* Reversed, since y-coord increases downward */
	dX2 = p4.h - p3.h;
	dY2 = p3.v - p4.v;	/* Reversed, since y-coord increases downward */

	theta1 = GetTheta(dX1, dY1);
	theta2 = GetTheta(dX2, dY2);

	/* If one angle is in quadrant 1, and the other in quadrant 4,
	 * then calculate the difference through the positive X. Otherwise,
	 * just calculate the difference.
	 */
	if((SIGN(dX1) == 1) && (SIGN(dX2) == 1)
	&& (SIGN(dY1) != SIGN(dY2))){
		if(SIGN(dY1) == 1)
			delta_theta = theta1 + 2*PI - theta2;
		else
			delta_theta = theta2 + 2*PI - theta1;
	}
	else
		delta_theta = ABS(theta2 - theta1);

	return delta_theta;
}

/*--------------------------------------------------------------------------
 * GetAverageTheta()
 *
 * Input:
 *	- Four points
 *
 * Output:
 *	- The angle which bisects the angle between the two line segments.
 *	  Note: The two line segments should a point in common
 *
 */

private double
GetAverageTheta(VHPoint p1, VHPoint p2, VHPoint p3, VHPoint p4)
{
	INT16 dX1, dY1, dX2, dY2;
	double theta1, theta2, avg_theta;

	dX1 = p2.h - p1.h;
	dY1 = p1.v - p2.v;	/* Reversed, since y-coord increases downward */
	dX2 = p4.h - p3.h;
	dY2 = p3.v - p4.v;	/* Reversed, since y-coord increases downward */

	theta1 = GetTheta(dX1, dY1);
	theta2 = GetTheta(dX2, dY2);

	/* If one angle is in quadrant 1, and the other in quadrant 4,
	 * then calculate the through the positive X. Otherwise, just
	 * calculate the average.
	 */
	if((SIGN(dX1) == 1) && (SIGN(dX2) == 1)
	&& (SIGN(dY1) != SIGN(dY2)))
		avg_theta = (theta1 + theta2 - 2*PI)/2;
	else
		avg_theta = (theta1 + theta2)/2;

	return avg_theta;
}

/*--------------------------------------------------------------------------
 * GetTheta()
 *
 * Input:
 *	- X and and Y values of a point
 *
 * Output:
 *	- The angle of the line segment (P1,P2) in Radians from the X-axis
 *
 *
 */

private double
GetTheta(INT16 dX, INT16 dY)
{
	double theta;

	if((dX == 0) && (dY == 0))
		theta = 0;
	else
		theta = atan2((double)dY, (double)dX);

	/* Always adjust theta to be the angle counter-clockwise. */
	if(dY < 0)
		theta = 2*PI + theta;

	return theta;
}


/*--------------------------------------------------------------------------
 * TheyIntersect()
 *
 * Input:
 *	- Four points, making up two line segments
 *
 * Output:
 *	- NULL = no points in common
 *	  Other = Address of VHPoint with point of intersection
 *
 * Algorithm:
 *	- Determine the formula for each line extension of the line segments,
 *	  then determine if they intersect. Then, determine if the point of
 *	  intersection lies within both line segments.
 *
 */

private lpVHPoint
TheyIntersect(VHPoint A1, VHPoint A2, VHPoint B1, VHPoint B2,
			  lpVHPoint intersect)
{
	int dYA, dXA, dYB, dXB;
	int X_MAX, X_MIN, Y_MAX, Y_MIN;
	double IX, IY, CA, CB;

	dXA = A2.h - A1.h;
	dXB = B2.h - B1.h;

	/* If the lines are vertical and parallel, return NULL. The lines
	 * segments are parallel and could be colinear, but even so, don't
	 * pick an arbitrary point of intersection.
	 */
	if((dXA == 0) && (dXB == 0))
		return NULL;

	dYA = A2.v - A1.v;
	dYB = B2.v - B1.v;

	/* Horizontal parallel lines, return NULL here for the same reason
	 * as above.
	 */
	if((dYA == 0) && (dYB == 0))
		return NULL;

	/* Return NULL for parallel lines, same reason. */
	if((dXA != 0) && (dXB != 0))
		if(((double)dYA/(double)dXA) == ((double)dYB/(double)dXB))
			return NULL;

	if(dXA != 0)
		CA = A1.v - ((double)dYA/(double)dXA)*(double)A1.h;
	if(dXB != 0)
		CB = B1.v - ((double)dYB/(double)dXB)*(double)B1.h;

	/* IX and IY are the Intersect X and Intersect Y */
	if(dXA == 0)
		IX = A1.h;
	else
		if(dXB == 0)
			IX = B1.h;
		else
			IX = (CB - CA)/(((double)dYA/(double)dXA)
				 - ((double)dYB/(double)dXB));
	if(dXA == 0)
		IY = ((double)dYB/(double)dXB)*IX + CB;
	else
		IY = ((double)dYA/(double)dXA)*IX + CA;

	/* Get minimum range for X */
	X_MIN = MAX(MIN(A1.h, A2.h), MIN(B1.h, B2.h));
	X_MAX = MIN(MAX(A1.h, A2.h), MAX(B1.h, B2.h));
	/* Get minimum range for Y */
	Y_MIN = MAX(MIN(A1.v, A2.v), MIN(B1.v, B2.v));
	Y_MAX = MIN(MAX(A1.v, A2.v), MAX(B1.v, B2.v));

	if((IX >= X_MIN - ATTACH_THRESHOLD)
	&& (IX <= X_MAX + ATTACH_THRESHOLD)
	&& (IY >= Y_MIN - ATTACH_THRESHOLD)
	&& (IY <= Y_MAX + ATTACH_THRESHOLD)){
		intersect->h = IX;
		intersect->v = IY;
		return intersect;
	}
	else
		return NULL;	/* No intersection */
}

/*--------------------------------------------------------------------------
 * DistanceToLine()
 *
 * Input:
 *	- A point and two line segment ends
 *
 * Output:
 *	- The shortest distance from the point to the line which is
 *	  colinear with the line segment
 *
 * Algorithm:
 *	Calculate the line which passes thru the point and crosses the given
 *	line at a right angle, then calculate the distance from the point
 *	given to the point of intersection.
 *
 *	Note: The point of intersection is not checked to see if it lies on
 *	the line segment.
 *
 */

private double
DistanceToLine(VHPoint P, VHPoint A1, VHPoint A2)
{
	double dYA, dXA;
	double IX, IY, CA, CB;

	dXA = A2.h - A1.h;
	if(dXA == 0)
		return ABS(A1.h - P.h);

	dYA = A2.v - A1.v;
	if(dYA == 0)
		return ABS(A1.v - P.v);

	CA = A1.v - (dYA/dXA)*(double)A1.h;
	CB = P.v + (dXA/dYA)*(double)P.h;

	IX = (CB - CA)/((dYA/dXA) + (dXA/dYA));
	IY = (dYA/dXA)*IX + CA;

	dXA = IX - P.h;
	dYA = IY - P.v;
	return sqrt(dXA*dXA + dYA*dYA);

}

/*--------------------------------------------------------------------------
 * GetDistance()
 *
 * Input:
 *	- Two points
 *
 * Output:
 *	- The distance
 *
 * Algorithm:
 *
 *
 */

private double GetDistance(VHPoint p1, VHPoint p2)
{
	double dX, dY;

	dX = p2.h-p1.h;
	dY = p2.v-p1.v;

	return sqrt(dX*dX + dY*dY);
}

/*--------------------------------------------------------------------------
 * AppendToList()
 *	From Ron Avitzur's recog.c
 *
 * Input:
 *	- The address of the list and the address of the item
 *
 * Output:
 *	- The item added to the list
 *
 * Algorithm:
 *
 *
 */

private void
AppendToList(lpLpList list,long item)
{
	if (*list == NULL){
		*list = mem_Alloc(sizeof(INT16) + sizeof(long));
		if (!*list) mem_AllocFailed(sizeof(INT16) + sizeof(long));
		(*list)->num_items = 1;
		(*list)->items[0] = (void*)item;
	}
	else{
		(*list)->num_items += 1;
		*list = mem_Realloc(*list,
			sizeof(INT16) + (*list)->num_items*sizeof(long));
		if (!*list) mem_AllocFailed(
			sizeof(INT16) + (*list)->num_items*sizeof(long));
		(*list)->items[(*list)->num_items-1] = (void*)item;
	}
}

/*--------------------------------------------------------------------------
 * RemoveFromList()
 *
 * Input:
 *	- The address of the list and the item number to remove, 0-indexed
 *
 * Output:
 *	- The list without the item
 *
 * Algorithm:
 *
 *
 */

private void
RemoveFromList(lpLpList list, INT16 item_number)
{
	int i;

	if (*list == NULL)
		return;

	for(i=item_number; i<(*list)->num_items-1; i++)
		(*list)->items[i] = (*list)->items[i+1];

	(*list)->num_items -= 1;
	*list = mem_Realloc(*list,
		sizeof(INT16) + (*list)->num_items*sizeof(long));
	if (!*list) mem_AllocFailed(
		sizeof(INT16) + (*list)->num_items*sizeof(long));
}

/*--------------------------------------------------------------------------
 * MatchDirectionStrings()
 *
 * Input:
 *	- Two strings
 *
 * Output:
 *	- Number of matches
 *
 * Algorithm:
 *
 *
 */

private int
MatchDirectionStrings(char *str1, char *str2)
{
	char *p1, *p2;
	int matches;

	p1 = str1;
	p2 = str2;
	matches = 0;
	while(*p1 && *p2){
		if(*p1 == *p2){
			++p1;
			++p2;
			++matches;
		}
		else{
			if(*p1 == *(p2+1)){
				++p1;
				p2 += 2;
				++matches;
			}
			else{
				if(*(p1+1) == *p2){
					p1 += 2;
					++p2;
					++matches;
				}
				else{
					++p1;
					++p2;
				}
			}
		}
	}

	return matches;
}


public void TabletToScreen(INT16 x,INT16 y, int * xx, int * yy)
{
	*xx = x >> 3;
	*yy = y >> 3;
}
