#include "h_config.h"

#ifdef TURBO_C
#include <stdlib.h>
#include <math.h>
#endif
#ifdef MSOFT_C7
#include <search.h>
#include <math.h>
#endif

#include "h_stddef.h"
#include "h_mem.h"
#include "h_list.h"
#include "h_recog.h"

/*************************************************************
***/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include <values.h>

/*************************************************************
***/

#ifndef NIL
#define TRUE               1
#define FALSE              0
#ifdef MSOFT_C7
#define NIL               ((void FAR*)0)
#else
#define NIL                0L
#endif
#define ABS(x)           ((x)>0?(x):(-(x)))
#define MAX(a,b)         ((a)>(b)?(a):(b))
#define MIN(a,b)         ((a)<(b)?(a):(b))
#endif

#define PATYPE INT16
#define MAX_CHARS 128

#define SCL 100L         /*  scaling factor for fuzzies  */

#define GWEIGHT 4
#define NUM_GEST_VARS 3
#define NUM_STROKE_VARS 55
#define FRACTION 2
#define MAX_STROKES 10
#define MAX_POINTS 200
#define MAX_N 100

#define PRE_ABS           (20)   /* was: (40) */

typedef struct
{
    INT16 Xmin, Xmax, Ymin, Ymax,
          QXmin, QXmax, QYmin, QYmax;
    INT16 Height, Width, QHeight, QWidth;
    VHPoint Simp[MAX_N];
    INT16 SimpNum;
    INT16 lensofar[MAX_N];
} stroketype;
typedef stroketype * LPstroke;

typedef struct
{
    INT16 Xmin, Xmax, Ymin, Ymax,
          QXmin, QXmax, QYmin, QYmax;
    INT16 Height, Width, QHeight, QWidth;
    INT16 totalen;
    INT16 numstrokes;
} gesturetype;
typedef gesturetype * LPgest;

typedef struct
{
    INT16 numstrokes;
    PATYPE *gestfuz;
    PATYPE **strokefuz;
} shrunktype;
typedef shrunktype * LPshrunk;

private LPshrunk    NowShrunk = NULL;
private lpList         Learned[MAX_CHARS];
private LPshrunk    Shrunk[MAX_CHARS];
private INT16         ATAN2_TABLE[21][21];

private void     BoilDownToLittleTinyMorsels(lpList origstrokes);
private PATYPE* gestanalyze(LPgest G);
private PATYPE* strokeanalyze(LPstroke S, LPgest G);
private void     Simplify(lpVHPoint Ink, INT16 NumInk, LPstroke S);

private PATYPE* meanlist(INT16 numvars, lpList L);
private void     Learn(INT16 char_name);
private void     Recognize(LPINT16 g1,LPINT16 g2,LPINT16 g3,
                          LPINT16 w1,LPINT16 w2,LPINT16 w3);
private INT16     weighting(INT16 numvars, PATYPE *nowm,
PATYPE *means);
private void     Extremes(lpVHPoint ink, INT16 num_ink,
LPstroke S);
private INT16 calclength(LPstroke S);
private void     EmptyNowShrunk();
private void     InitAtanTable(void);
private INT16   MY_ATAN2(INT16, INT16);
private void     AppendToList(lpLpList list,long item);


/****************************************************************/

public void rec_InitRecognizer(void)
{
    InitAtanTable();
}

/****************************************************************/

public void rec_Train(lpList gesture,INT16 char_code)
{
    printf("\nTrain: %d",char_code);
    BoilDownToLittleTinyMorsels(gesture);

    Learn(char_code);
}

/****************************************************************/
public void rec_Guess(lpList gesture,
        LPINT16 g1,LPINT16 g2,LPINT16 g3,
        LPINT16 w1,LPINT16 w2,LPINT16 w3)
{
    
    BoilDownToLittleTinyMorsels(gesture);

    Recognize(g1,g2,g3,w1,w2,w3);
}

/****************************************************************/

/*  NOTE:  during the following shrinking of learned characters to
           one of each, we assume that there are the same number of
           strokes in each sample of a given letter.  I hope this
           is not unreasonable.  Anyone writing a G with one stroke
           first, and then with two strokes is hereby warned that
           it will probably crash the program.  The same goes for
           any other letter you care to mention.
*/
public void rec_EndTraining(void)
{
    INT16 i, j, k;
    LPshrunk *G, S;
    lpList templist = NULL;

    
    for (i = 0; i < MAX_CHARS; i++)
    {
        if (!Learned[i]) continue;

        if (Learned[i]->num_items == 1)
        {
            Shrunk[i] = (LPshrunk)Learned[i]->items[0];
            mem_Free(Learned[i]);
            Learned[i] = NULL;

            continue;
        }

        S = Shrunk[i] = (LPshrunk)mem_Alloc(sizeof(shrunktype));
        if (!S) mem_AllocFailed(sizeof(shrunktype));

        G = (LPshrunk*)Learned[i]->items;

        for (k = 0; k < Learned[i]->num_items; k++)
            AppendToList(&templist, (long)(G[k]->gestfuz));

        S->gestfuz = meanlist(NUM_GEST_VARS, templist);

        mem_Free(templist);
        templist = NULL;

        /*  with the G[0] in the following lines we assume that all the
            rest of the samples of this letter have the same number of
            strokes as the first.  If some have more, the recognition
            is bad, and if some have less, we probably crash...
        */
        S->numstrokes = G[0]->numstrokes;
        S->strokefuz = (PATYPE**)mem_Alloc(S->numstrokes * sizeof(PATYPE*));
        if (!S->strokefuz) mem_AllocFailed(S->numstrokes * sizeof(PATYPE*));

        for (j = 0; j < G[0]->numstrokes; j++)
        {
            for (k = 0; k < Learned[i]->num_items; k++)
                AppendToList(&templist, (long)G[k]->strokefuz[j]);

            S->strokefuz[j] = meanlist(NUM_STROKE_VARS, templist);

            mem_Free(templist);
            templist = NULL;
        }

        /*  free all the unnecessary bits again  */
        for (j = 0; j < Learned[i]->num_items; j++)
        {
            mem_Free(G[j]->gestfuz);

            for (k = 0; k < G[j]->numstrokes; k++)
                mem_Free(G[j]->strokefuz[k]);

            mem_Free(G[j]->strokefuz);
        }

        mem_Free(Learned[i]);
        Learned[i] = NULL;
    }
}

/****************************************************************/

private PATYPE *meanlist(INT16 numvars, lpList L)
{
    register INT16 i, j;
    PATYPE *s, *x1;
    PATYPE *mean = (PATYPE*)mem_CAlloc(numvars, sizeof(PATYPE));
    if (!mean) mem_AllocFailed(numvars * sizeof(PATYPE));

    for (i = 0; i < L->num_items; i++)
    {
        s = (PATYPE*)L->items[i];

        for (j = 0; j < numvars; j++)
            mean[j] += s[j];
    }

    for (j = 0; j < numvars; j++)
        mean[j] /= L->num_items;

    return mean;
}

/****************************************************************/

private void Learn(INT16 char_name)
{
    AppendToList(&Learned[char_name], (long)NowShrunk);
    NowShrunk = NULL;
}

/****************************************************************/

private void BoilDownToLittleTinyMorsels(lpList origstrokes)
{
    INT16 i, snum = origstrokes->num_items;
    lpList ink;
    LPgest G = (LPgest)mem_Alloc(sizeof(gesturetype));
    LPshrunk N;
    LPstroke S;
    LPstroke *Sarray = (LPstroke*)mem_Alloc(snum * sizeof(LPstroke));
    if (!G) mem_AllocFailed(sizeof(gesturetype));
    if (!Sarray) mem_AllocFailed(snum * sizeof(LPstroke));

    EmptyNowShrunk();

    N = NowShrunk;

    N->numstrokes = snum;
    N->gestfuz = (PATYPE*)mem_CAlloc(NUM_GEST_VARS, sizeof(PATYPE));
    if (!N->gestfuz) mem_AllocFailed(sizeof(PATYPE) * NUM_GEST_VARS);
    N->strokefuz = (PATYPE**)mem_Alloc(snum * sizeof(PATYPE*));
    if (!N->gestfuz) mem_AllocFailed(snum * sizeof(PATYPE*));

    G->totalen = 0;
    G->Xmax = G->Ymax = G->QXmax = G->QYmax = -MAXINT;
    G->Xmin = G->Ymin = G->QXmin = G->QYmin = MAXINT;

    for (i = 0; i < snum; i++)
    {
        ink = (lpList)origstrokes->items[i];

        S = (LPstroke)mem_Alloc(sizeof(stroketype));

        Extremes((VHPoint*)ink->items, ink->num_items, S);

        Simplify((VHPoint*)ink->items, ink->num_items, S);

        Extremes(S->Simp, S->SimpNum, S);

        if (S->Xmax > G->Xmax)     G->Xmax = S->Xmax;
        if (S->Ymax > G->Ymax)     G->Ymax = S->Ymax;
        if (S->Xmin < G->Xmin)     G->Xmin = S->Xmin;
        if (S->Ymin < G->Ymin)     G->Ymin = S->Ymin;

        if (S->QXmax > G->QXmax)   G->QXmax = S->QXmax;
        if (S->QYmax > G->QYmax)   G->QYmax = S->QYmax;
        if (S->QXmin < G->QXmin)   G->QXmin = S->QXmin;
        if (S->QYmin < G->QYmin)   G->QYmin = S->QYmin;

        G->totalen += calclength(S);

        Sarray[i] = S;
    }

    G->Width   = G->Xmax - G->Xmin + 1;
    G->Height  = G->Ymax - G->Ymin + 1;
    G->QWidth  = G->QXmax - G->QXmin + 1;
    G->QHeight = G->QYmax - G->QYmin + 1;

    N->gestfuz = gestanalyze(G);

    for (i = 0; i < snum; i++)
    {
        N->strokefuz[i] = strokeanalyze(Sarray[i], G);
        mem_Free(Sarray[i]);
    }
    mem_Free(Sarray);
}

/****************************************************************/

private PATYPE *gestanalyze(LPgest G)
{
    INT16 j;
    PATYPE *blip = (PATYPE*)mem_CAlloc(NUM_GEST_VARS, sizeof(PATYPE));
    if (!blip)    mem_AllocFailed(NUM_GEST_VARS * sizeof(PATYPE));

    blip[0] = SCL * G->totalen / (G->totalen + hypot(G->Height, G->Width));
    blip[1] = SCL * G->Height  / (G->Height  + G->Width);
    blip[2] = SCL * G->QHeight / (G->QHeight + G->QWidth);

    for (j = 0; j < NUM_GEST_VARS; j++)
        blip[j]++;

    return blip;
}

/****************************************************************/

private PATYPE *strokeanalyze(LPstroke S, LPgest G)
{
    register INT16 i, j, k;
    INT16 hmove[2][FRACTION], vmove[2][FRACTION],
          Qhmove[2][FRACTION], Qvmove[2][FRACTION];
    INT16 hmT[2], vmT[2], QhmT[2], QvmT[2];
    INT32 Ibot[2][FRACTION], Iright[2][FRACTION],
          Itop[2][FRACTION], Ileft[2][FRACTION];
    INT32 IbotT[2], IrightT[2], ItopT[2], IleftT[2];
    INT16 fractlen, Xdiff, Ydiff, f;
    VHPoint p, pold, ps, pe, pm;
    PATYPE *blip;

    for (k = 0; k < 2; k++)
    {
        for (j = 0; j < FRACTION; j++)
        {
            Ibot[k][j] = Iright[k][j] = Itop[k][j] = Ileft[k][j] = (k ? 1 : -1);
            hmove[k][j] = vmove[k][j] = Qhmove[k][j] = Qvmove[k][j] = (k ? 1 : -1);
        }

        /*  should be safe initializing these to 0, as all the others
            have been set to at least 1       */
        IbotT[k] = IrightT[k] = ItopT[k] = IleftT[k] = 0;
        hmT[k] = vmT[k] = QhmT[k] = QvmT[k] = 0;

    }

    fractlen = S->lensofar[S->SimpNum - 1] / FRACTION + 1;
    f = 0;
    p = S->Simp[0];
    for (j = 1; j < S->SimpNum; j++)
    {
        float step;
        INT32 Xd, Yd;
        INT16 fold;

        pold = p;
        p = S->Simp[j];

        fold = f;
        f = S->lensofar[j] / fractlen;

        Xdiff = p.h - pold.h;
        Ydiff = p.v - pold.v;

        pe = pold;

        for (k = fold; k <= f; k++)
        {
            step = MIN((k + 1) * fractlen, S->lensofar[j])
                   - S->lensofar[j - 1];
            step /= S->lensofar[j] - S->lensofar[j - 1];

            ps = pe;
            pe.h = pold.h + Xdiff * step;
            pe.v = pold.v + Ydiff * step;

            Xd = pe.h - ps.h;
            Yd = pe.v - ps.v;

            /*  Do ya groovy integrals  */
            /*  NB: these are in fact double the integral values  */
            /*  NB2: contrary to my second thoughts, top-bot are not
                *always* related, ditto left-right, so I leave em  */

            Itop   [Xd > 0][k] += (pe.v + ps.v - 2 * G->Ymin) * Xd;
            Ileft  [Yd > 0][k] += (pe.h + ps.h - 2 * G->Xmin) * Yd;
            Ibot   [Xd > 0][k] += (2 * G->Ymax - (pe.v + ps.v)) * Xd;
            Iright [Yd > 0][k] += (2 * G->Xmax - (pe.h + ps.h)) * Yd;

            /*  could do the same for Q as well, I spose  */
            /*  only if this is a little bit fast  */

            hmove   [Xd > 0     ][k] += Xd;
            vmove   [Yd > 0     ][k] += Yd;
            Qhmove  [Xd - Yd > 0][k] += Xd - Yd;
            Qvmove  [Xd + Yd > 0][k] += Xd + Yd;
        }
    }

    pm = S->Simp[0];    //  in case there's only one point
    for (j = 0; j < S->SimpNum - 1; j++)
    {
        if (S->lensofar[j] / fractlen != S->lensofar[j + 1] / fractlen)
        {
            double step = fractlen - S->lensofar[j];
            step /= S->lensofar[j + 1] - S->lensofar[j];

            pm.h = S->Simp[j].h + (S->Simp[j + 1].h - S->Simp[j].h) * step;
            pm.v = S->Simp[j].v + (S->Simp[j + 1].v - S->Simp[j].v) * step;
        }
    }

    for (j = 0; j < FRACTION; j++)
    {
        Ibot[0][j] *= -1;
        Iright[0][j] *= -1;
        Itop[0][j] *= -1;
        Ileft[0][j] *= -1;

        hmove[0][j] *= -1;
        vmove[0][j] *= -1;
        Qhmove[0][j] *= -1;
        Qvmove[0][j] *= -1;

        for (k = 0; k < 2; k++)
        {
            IbotT[k] += Ibot[k][j];
            IrightT[k] += Iright[k][j];
            ItopT[k] += Itop[k][j];
            IleftT[k] += Ileft[k][j];

            hmT[k] += hmove[k][j];
            vmT[k] += vmove[k][j];
            QhmT[k] += Qhmove[k][j];
            QvmT[k] += Qvmove[k][j];
        }
    }

    ps = S->Simp[0];
    pe = S->Simp[S->SimpNum - 1];

    blip = (PATYPE*)mem_CAlloc(NUM_STROKE_VARS, sizeof(PATYPE));
    if (!blip) mem_AllocFailed(NUM_STROKE_VARS * sizeof(PATYPE));

    /*  curvinexx & aspect ratios  */
    blip[0] = SCL * hypot(S->Height, S->Width) / S->lensofar[S->SimpNum - 1];
    blip[1] = SCL * S->Height / (S->Height + S->Width);
    blip[2] = SCL * S->QHeight / (S->QHeight + S->QWidth);

    /*  relativeness to 0 (of gesture) of begin and endpoints */
    blip[3] = SCL * (ps.h - G->Xmin) / G->Width;
    blip[4] = SCL * (ps.v - G->Ymin) / G->Height;
    blip[5] = SCL * (pe.h - G->Xmin) / G->Width;
    blip[6] = SCL * (pe.v - G->Ymin) / G->Height;

    /*  ditto for rotation of 45 degrees  */
    blip[7] = SCL * (ps.h - ps.v - G->QXmin) / G->QWidth;
    blip[8] = SCL * (ps.h + ps.v - G->QYmin) / G->QHeight;
    blip[9] = SCL * (pe.h - pe.v - G->QXmin) / G->QWidth;
    blip[10] = SCL * (pe.h + pe.v - G->QYmin) / G->QHeight;

    /*  effective horizontal and vertical movement  */
    blip[11] = SCL * ABS(hmT[1] - hmT[0]) / G->Width;
    blip[12] = SCL * ABS(vmT[1] - vmT[0]) / G->Height;
    blip[13] = SCL * ABS(QhmT[1] - QhmT[0]) / G->QWidth;
    blip[14] = SCL * ABS(QvmT[1] - QvmT[0]) / G->QHeight;

    /*  relative positive to total movement  */
    blip[15] = SCL * hmT[1] / (hmT[1] + hmT[0]);
    blip[16] = SCL * vmT[1] / (vmT[1] + vmT[0]);
    blip[17] = SCL * QhmT[1] / (QhmT[1] + QhmT[0]);
    blip[18] = SCL * QvmT[1] / (QvmT[1] + QvmT[0]);

    /*  effective integral in all directions  */
    blip[19] = SCL * ABS(IbotT[1] - IbotT[0]) / G->Width / G->Height;
    blip[20] = SCL * ABS(IrightT[1] - IrightT[0]) / G->Width / G->Height;
    blip[21] = SCL * ABS(ItopT[1] - ItopT[0]) / G->Width / G->Height;
    blip[22] = SCL * ABS(IleftT[1] - IleftT[0]) / G->Width / G->Height;

    /*  relative positive to total integral  */
    blip[23] = SCL * IbotT[1] / (IbotT[1] + IbotT[0]);
    blip[24] = SCL * IrightT[1] / (IrightT[1] + IrightT[0]);
    blip[25] = SCL * ItopT[1] / (ItopT[1] + ItopT[0]);
    blip[26] = SCL * IleftT[1] / (IleftT[1] + IleftT[0]);

    blip[27] = SCL * hmove[0][0] / hmT[0];
    blip[28] = SCL * hmove[0][1] / hmT[0];
    blip[29] = SCL * vmove[0][0] / vmT[0];
    blip[30] = SCL * vmove[0][1] / vmT[0];

    blip[31] = SCL * hmove[1][0] / hmT[1];
    blip[32] = SCL * hmove[1][1] / hmT[1];
    blip[33] = SCL * vmove[1][0] / vmT[1];
    blip[34] = SCL * vmove[1][1] / vmT[1];

    blip[35] = SCL * Ibot[0][0] / IbotT[0];
    blip[36] = SCL * Ibot[0][1] / IbotT[0];
    blip[37] = SCL * Ibot[1][0] / IbotT[1];
    blip[38] = SCL * Ibot[1][1] / IbotT[1];

    blip[39] = SCL * Iright[0][0] / IrightT[0];
    blip[40] = SCL * Iright[0][1] / IrightT[0];
    blip[41] = SCL * Iright[1][0] / IrightT[1];
    blip[42] = SCL * Iright[1][1] / IrightT[1];

    blip[43] = SCL * Itop[0][0] / ItopT[0];
    blip[44] = SCL * Itop[0][1] / ItopT[0];
    blip[45] = SCL * Itop[1][0] / ItopT[1];
    blip[46] = SCL * Itop[1][1] / ItopT[1];

    blip[47] = SCL * Ileft[0][0] / IleftT[0];
    blip[48] = SCL * Ileft[0][1] / IleftT[0];
    blip[49] = SCL * Ileft[1][0] / IleftT[1];
    blip[50] = SCL * Ileft[1][1] / IleftT[1];

    blip[51] = SCL * (pm.h - G->Xmin) / G->Width;
    blip[52] = SCL * (pm.v - G->Ymin) / G->Height;
    blip[53] = SCL * (pm.h - pm.v - G->QXmin) / G->QWidth;
    blip[54] = SCL * (pm.h + pm.v - G->QYmin) / G->QHeight;

    for (j = 0; j < NUM_STROKE_VARS; j++)
        blip[j]++;

    return blip;
}

/****************************************************************/

private void Simplify(lpVHPoint Ink, INT16 NumInk, LPstroke S)
{
    INT16 dx,dy,i,n = 0;

    S->Simp[0] = Ink[0];
    for (i = 1; i < NumInk - 1; i++)
    {
        dx = Ink[i].h - S->Simp[n].h;
        dy = Ink[i].v - S->Simp[n].v;

        dx = abs(dx);
        dy = abs(dy);

        if (dx + dy < PRE_ABS)    continue;
        n++;
        S->Simp[n] = Ink[i];
    }
    dx = Ink[NumInk - 1].h - S->Simp[n].h;
    dy = Ink[NumInk - 1].v - S->Simp[n].v;
    if (ABS(dx) + ABS(dy) > PRE_ABS)
        n++;
    S->Simp[n] = Ink[NumInk - 1];

    S->SimpNum = n + 1;
}

/****************************************************************/

private void Extremes(lpVHPoint ink, INT16 num_ink, LPstroke S)
{
    register INT16 j;
    VHPoint p;

    p = ink[0];

    S->Xmin = S->Xmax = p.h;
    S->Ymin = S->Ymax = p.v;
    S->QXmin = S->QXmax = p.h - p.v;
    S->QYmin = S->QYmax = p.h + p.v;

    for (j = 1; j < num_ink; j++)
    {
        p = ink[j];
        if (p.h > S->Xmax) S->Xmax = p.h;
        if (p.h < S->Xmin) S->Xmin = p.h;
        if (p.v > S->Ymax) S->Ymax = p.v;
        if (p.v < S->Ymin) S->Ymin = p.v;

        if (p.h - p.v > S->QXmax)    S->QXmax = p.h - p.v;
        if (p.h - p.v < S->QXmin)    S->QXmin = p.h - p.v;
        if (p.h + p.v > S->QYmax)    S->QYmax = p.h + p.v;
        if (p.h + p.v < S->QYmin)    S->QYmin = p.h + p.v;

    }

    S->Width = S->Xmax - S->Xmin;
    S->Height = S->Ymax - S->Ymin;
    S->QWidth = S->QXmax - S->QXmin;
    S->QHeight = S->QYmax - S->QYmin;
}

/****************************************************************/

private INT16 calclength(LPstroke S)
{
    register INT16 j, Xdiff, Ydiff;
    VHPoint p, pold;

    p = S->Simp[0];
    S->lensofar[0] = 0;

    for (j = 1; j < S->SimpNum; j++)
    {
        pold = p;
        p = S->Simp[j];

        Xdiff = p.h - pold.h;
        Ydiff = p.v - pold.v;

        S->lensofar[j] = S->lensofar[j - 1] + hypot(ABS(Xdiff), ABS(Ydiff));
    }

    return S->lensofar[S->SimpNum - 1];
}

/****************************************************************/

private void Recognize(
        LPINT16 g1,LPINT16 g2,LPINT16 g3,
        LPINT16 w1,LPINT16 w2,LPINT16 w3)
{
    register INT16 i, j, k;
    LPshrunk G;
    INT32 weight, we1, we2, we3;
    INT16 pos1, pos2, pos3;

    we1 = we2 = we3 = MAXLONG;

    for (i = 0; i < MAX_CHARS; i++)
    {
        if (!Shrunk[i]) continue;
        if (Shrunk[i]->numstrokes != NowShrunk->numstrokes) continue;

        G = Shrunk[i];
        weight = 0;
        for (j = 0; j < G->numstrokes; j++)
        {
            weight += weighting(NUM_STROKE_VARS, NowShrunk->strokefuz[j],
                                G->strokefuz[j]);
        }

        weight += GWEIGHT * weighting(NUM_GEST_VARS, NowShrunk->gestfuz, G->gestfuz);

        if (weight < we3)
           if (weight < we2)
              if (weight < we1)
              {
                we3 = we2; we2 = we1; we1 = weight;
                pos3 = pos2; pos2 = pos1; pos1 = i;
              }
              else
              {
                we3 = we2; we2 = weight;
                pos3 = pos2; pos2 = i;
              }
           else
           {
            we3 = weight;
            pos3 = i;
           }
    }

    *w1 = 100;
    *g1 = pos1;
}

/****************************************************************/

private INT16 weighting(INT16 numvars, PATYPE *nowm, PATYPE *means)
{
    register INT16 i, total = 0;

    for (i = 0; i < numvars; i++)
        total += abs(nowm[i] - means[i]);

    return total;
}

/****************************************************************/

private void EmptyNowShrunk()
{
    INT16 i;

    if (NowShrunk)
    {
        mem_Free(NowShrunk->gestfuz);

        for (i = 0; i < NowShrunk->numstrokes; i++)
            mem_Free(NowShrunk->strokefuz[i]);
        mem_Free(NowShrunk->strokefuz);
    }

    NowShrunk = (LPshrunk)mem_Alloc(sizeof(shrunktype));
}

/****************************************************************/

private void InitAtanTable(void)
{
    INT16 i,j;
    double r2d = 180 / 3.141592654;
    for (i = 0; i <= 10; i++)
    {
        for (j = 0; j <= 10; j++)
    {
            INT16 t;
        t = (i==0 && j==0)? 0 : (INT16)(r2d * atan2(i,j));
            ATAN2_TABLE[ i + 10][ j + 10] =  t;
            ATAN2_TABLE[-i + 10][ j + 10] = -t;
            ATAN2_TABLE[ i + 10][-j + 10] = 180 - t;
            ATAN2_TABLE[-i + 10][-j + 10] = t - 180;
        }
    }
}

/****************************************************************/

private INT16 MY_ATAN2(register INT16 dy,register INT16 dx)
{
    while (dx > 10 || dx < -10 || dy > 10 || dy < -10)
        { dx >>= 1; dy >>= 1; } /*was 2*/
    return ATAN2_TABLE[dy + 10][dx + 10];
}

/****************************************************************/

private void AppendToList(lpLpList list, long item)
{
    if (*list == NULL)
    {
        *list = mem_Alloc(sizeof(INT16) + sizeof(long));
        if (!*list) mem_AllocFailed(sizeof(INT16) + sizeof(long));
        (*list)->num_items = 1;
        (*list)->items[0] = (void*)item;
    }
    else
    {
        (*list)->num_items += 1;
        *list = mem_Realloc(*list,
            sizeof(INT16) + (*list)->num_items*sizeof(long));
        if (!*list) mem_AllocFailed(
        sizeof(INT16) + (*list)->num_items*sizeof(long));
        (*list)->items[(*list)->num_items-1] = (void*)item;
    }
}
/****************************************************************/
