
/*--------------------------------------------------------------*\
|   RECOG.C   V.4.0 (LARGE MEMORY VERSION - NO MOMENTUM)        |
|   Recognizer for Handprinted Text.                            |
+---------------------------------------------------------------+
|   This is a the sample recognition engine for my entry into   |
|   the DDJ Handwriting Recognition Competition. The idea is    |
|   very simple, calculate a bunch of features, then find the   |
|   stored character that most closely matches (is the least    |
|   distance from) the character to guess. Obviously, some      |
|   features are better than other features, for determining    |
|   the difference between characters. However, what is better  |
|   for one writer may not be better for another. Therefore     |
|   the program optimizes the weights for the writer.           |
|   The optimization method uses a neural net to optimize the   |
|   weights. It is seem slow to learn, but it needs to          |
|   only be done once.                                          |
|                                                               |
|   This recognizer was written by Kipton Moravec.              |
|   (c) Copyright 1992 by Kipton Moravec                        |
|   Limited license for noncommercial use granted.              |
|                                                               |
|   The only limit to the use for this program is the           |
|   acnowledgement of its use in any future programs that may   |
|   be derived from it. It can be copied an unlimited number of |
|   times and can be used in any way the user sees fit.         |
|                                                               |
|                                                               |
|   I have just the Borland C++ 3.0 compiler. So I do not think |
|   it will work with any other compiler, without some          |
|   modifications.                                              |
|                                                               |
|   This should be optimized or it will take forever (>96 hours |
|   qualifies as forever) to learn. Also a 486 (which has a     |
|   built-in processor and a built-in cache is about 100 times  |
|   faster than a 386 with no coprocessor.                      |
|                                                               |
|   The net has 4 layers: an input layer, two hidden layers and |
|   an output layer. I have not found any good theory as to how |
|   many nodes in the hidden layers I should have. I am         |
|   currently testing the following configurations:             |
|     IN_NODES    L1_NODES    L2_NODES   OUT_NODES              |
|       148         148         148         65                  |
|       148         592         260         65                  |
|       148         300         300         65                  |
|       148         600         600         65                  |
|   I do not know which is better. All I do know is that a      |
|   148 128 128 65 net is the largest I can fit in the 640 K    |
|   PC memory.                                                  |
|                                                               |
|   People who have figured out how to use extended memory have |
|   not been a generous as I, when it comes to giving out       |
|   source code! So I write the data to disk and that slows the |
|   program down alot.                                          |
|                                                               |
\*--------------------------------------------------------------*/

/*--------------------------------------------------------------*\
 |                                                              |
 |      include files                                           |
 |                                                              |
\*--------------------------------------------------------------*/

#include "h_config.h"

#ifdef TURBO_C
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <io.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <alloc.h>
#endif

#ifdef MSOFT_C7
#include <search.h>
#include <math.h>
#endif

#include "h_stddef.h"
#include "h_mem.h"
#include "h_list.h"
#include "h_recog.h"

#include <string.h>
#include <math.h>

/*--------------------------------------------------------------*\
 |                                                              |
 |      Constants                                               |
 |                                                              |
\*--------------------------------------------------------------*/

#define MAX_FEATURES     148

#define MAX_POINTS       200

#ifndef NIL
#define TRUE               1
#define FALSE              0
#ifdef MSOFT_C7
#define NIL               ((void FAR*)0)
#else
#define NIL                0L
#endif

#define SGN(x)           ((x)>0?(1.0):(-1.0))
#define ABS(x)           ((x)>0?(x):(-(x)))
#define MAX(a,b)         ((a)>(b)?(a):(b))
#define MIN(a,b)         ((a)<(b)?(a):(b))

#endif

#define WRITE_VAR        (O_CREAT | O_TRUNC | O_WRONLY | O_BINARY)
#define READ_VAR         (O_RDONLY | O_BINARY)
#define FILE_VAR         (S_IWRITE | S_IREAD)

#define ABORT            -1
/*--------------------------------------------------------------*\
 |                                                              |
 |      Attribute/Value Pair (AVPair), and corresponding List   |
 |      this is my version of the input data list. It was       |
 |      easier for me to remember.                              |
 |                                                              |
\*--------------------------------------------------------------*/

typedef struct tagPointSet
{
    INT16         num_points;
    VHPoint       p[1];
} PointSet;

typedef PointSet FAR *  lpPointSet;

typedef struct tagStrokeSet
{
    INT16         num_strokes_in;
    lpPointSet    ptr[1];
} StrokeSet;

typedef StrokeSet FAR * lpStrokeSet;

/*--------------------------------------------------------------*\
 |                                                              |
 |      my storage structure for the characters                 |
 |                                                              |
\*--------------------------------------------------------------*/

#define FTYPE float

typedef struct f_d
{
    INT16            char_code;
    FTYPE            Features[MAX_FEATURES];
} Feature_Data;

typedef Feature_Data FAR * Feature_Data_Ptr;

typedef struct typscore
{
    INT16           char_code;
    FTYPE           score;
}  Score_Type;

/*--------------------------------------------------------------*\
 |                                                              |
 |      my storage structure for the Neural Net                 |
 |                                                              |
\*--------------------------------------------------------------*/


#define IN_NODES    (MAX_FEATURES)      /* input layer          */
#define L1_NODES    (600)               /* first hidden layer   */
#define L2_NODES    (600)               /* second hidden layer  */
#define OUT_NODES   65                  /* output layer         */

#define GAIN        0.3 /* gain, this affects how fast the net  */
                        /* converges, however too high a value  */
                        /* can cause the net to diverge, because*/
                        /* the value can get too far from the   */
                        /* "surface" it is following and trying */
                        /* to minimize. The theory says this    */
                        /* step size should approach zero. I have */
                        /* had problems on some nets, sometimes */
                        /* when this value was over 0.5. This   */
                        /* will hopefully be a stable number.   */
                        /* Usually everybody uses 0.25 or 0.3.  */

#define BIAS        1.0 /* I do not have any idea what a good   */
                        /* value should be. Too big should have */
                        /* same problem as the gain, especially */
                        /* since it is multiplied by the gain.  */

#define MAX_ITERATIONS 5000
                        /* this is the cutoff if the iterations */
                        /* fail to converge.                    */
#define TOLERANCE   0.01
                        /* when all of the answers are within   */
                        /* this tolerance we can stop learning  */

#define W1_SIZE     (IN_NODES*sizeof(float))
#define W2_SIZE     (L1_NODES*sizeof(float))
#define W3_SIZE     (L2_NODES*sizeof(float))


float   *x1;            /* node values of the first hidden layer    */
float   *w1;            /* weights bet. input and first h. layer    */
float   *theta1;        /* bias term for each node                  */
int     w1_FILE;        /* the file handle for the weights          */

float   *x2;            /* node values of the second hidden layer   */
float   *w2;            /* weights bet. first and second h. layer   */
float   *theta2;        /* bias term for each node                  */
int     w2_FILE;        /* the file handle for the weights          */


float   *x3;            /* node values of the output hidden layer   */
float   *w3;            /* weights bet. second h. and output layer  */
float   *theta3;        /* bias term for each node                  */
int     w3_FILE;        /* the file handle for the weights          */


double delta1[L1_NODES]; /* the delta terms are the difference      */
double delta2[L2_NODES]; /* between the desired node value and the  */
double delta3[OUT_NODES];/* actual node value multiplied by the     */
                        /* first derivative of the sigmoid function */
                        /* this tells how much and in which         */
                        /* direction the weights and bias feeding   */
                        /* the layer must go.                       */

double t1[L1_NODES];    /* the t1 and t2 hold temporary sums so I   */
double t2[L2_NODES];    /* only have to pass through the weight     */
                        /* array one time.                          */

/*--------------------------------------------------------------*\
 |                                                              |
 |      Files                                                   |
 |                                                              |
\*--------------------------------------------------------------*/

FILE    *out, *out1, *out2, *gues;
int     scratch;

/*--------------------------------------------------------------*\
 |                                                              |
 |      Global Data                                             |
 |                                                              |
\*--------------------------------------------------------------*/

FTYPE           intercept[MAX_FEATURES], slope[MAX_FEATURES];
                /* structure for linierly normalizing the data  */

int     count;  /* way to keep track of the count of the guesses */
int     inCount;

FTYPE   m[4][4];        /* structure for moment calculations    */

int    cTable[OUT_NODES];
                        /* table of characters to correspond    */
                        /* with the output of the Neural Net    */

int     iTableCount;    /* Count of the different characters    */

Feature_Data FD;        /* where I store the features while I   */
                        /* am working on them.                  */

/*--------------------------------------------------------------*\
 |                                                              |
 |      Prototypes of private subroutines                       |
 |                                                              |
\*--------------------------------------------------------------*/

private void    Calculate_Features (lpStrokeSet gesture,
                                    INT16 char_code,
                                    Feature_Data_Ptr FDP);

private void    moment (lpStrokeSet gesture);
private void    calculate_scores (void);
private float * vector (int n);
private int     init_net (void);
private float **matrix (int r, int c);
private long    train_net (void);
private void    use_net (Feature_Data_Ptr FDP);
private void    save_net (void);
private int     load_net (void);
private int     c_break (void);
private double  sigmoid_logistic_nonlinearity(double alpha);
private void    out_of_memory_error(void);

/*--------------------------------------------------------------*\
 |                                                              |
 |      rec_InitRecognizer                                      |
 |                                                              |
\*--------------------------------------------------------------*/

public void rec_InitRecognizer(void)
{
    int i;

    if ((gues = fopen("guess.prn","wt")) == NULL)
    {
            fprintf(stderr, "Can not open guess.prn.\n");
    }

    if ((scratch = open("SCRATCH.KIP", 
                         O_CREAT | O_WRONLY | O_TRUNC | O_BINARY,
                         S_IWRITE | S_IREAD)) == -1)
    {
        printf("Can not open SCRATCH.KIP in rec_InitRecognizer\n");
    }

    count = 0;
    iTableCount = 0;
    inCount = 0;

    for (i=0; i<128; i++) cTable[i] = 0;
    

}

/*--------------------------------------------------------------*\
 |                                                              |
 |      rec_Train                                               |
 |                                                              |
\*--------------------------------------------------------------*/

public void rec_Train(lpList gesture,INT16 char_code)
{
    int              i;
    lpStrokeSet      ss;
    Feature_Data_Ptr FDP;

    FDP = &FD;

    ss = (lpStrokeSet) gesture;  /* I like my structure better */

    printf("Moravec ch= %c  ",char_code);

    memset(&FD, 0, sizeof(Feature_Data));

    Calculate_Features(ss, char_code, FDP);

    for (i=0; (i<128) && (cTable[i] != 0) && (cTable[i] != FD.char_code); i++);    if (cTable[i] == 0)
    {
        cTable[i] = FD.char_code;
        iTableCount = i;
    }
    
    if (_write(scratch, &FD.char_code, sizeof(Feature_Data)) == -1)
    {
        printf("Error writing SCRATCH.KIP\n");
        exit(1);
    }
    inCount++;    

}

/*--------------------------------------------------------------*\
 |                                                              |
 |      rec_Guess                                               |
 |                                                              |
\*--------------------------------------------------------------*/

public void rec_Guess(lpList gesture,
        LPINT16 g1,LPINT16 g2,LPINT16 g3,
        LPINT16 w1,LPINT16 w2,LPINT16 w3)
{
    INT16 i,j;
    INT16 char_code;
    lpStrokeSet      ss;
    Feature_Data_Ptr FDP, p;
    FTYPE            t;
    Score_Type       n_score, temp;
    Score_Type       Scr[3];
                        /* structure for storing highest values */

    FDP = &FD;

    printf("\nStarting Guess %i  ", ++count);

    ss = (lpStrokeSet) gesture;  /* I like my structure better */
    char_code = 0;

    for (i=0; i<3; i++)
    {
        Scr[i].char_code = 0;
        Scr[i].score     = -1.0e300;
    }

    Calculate_Features(ss, char_code, FDP);

    use_net(FDP);

    for ( i=0; i<iTableCount; i++)
    {
        n_score.score     = x3[i];
        n_score.char_code = cTable[i];

        for (j=0; j<3; j++)
        {
            if (n_score.score > Scr[j].score)
            {
                temp    =   n_score;
                n_score =   Scr[j];
                Scr[j]  =   temp;
            }
        }
    }
    *g1 = Scr[0].char_code;
    *g2 = Scr[1].char_code;
    *g3 = Scr[2].char_code;

    *w1 = (int) (1000 * Scr[0].score);
    *w2 = (int) (1000 * Scr[1].score);
    *w3 = (int) (1000 * Scr[2].score);

    printf("Guess 1: %c",*g1);
    printf(" %4u ", *w1);
    printf("    2: %c ", *g2);
    printf(" %4u ", *w2);
    printf("    3: %c ", *g3);
    printf(" %4u\n",*w3);

    fprintf(gues, "%5% %c %4u %c %4u %c %4u \n",
        count, *g1, *w1, *g2, *w2, *g3, *w3);
}

public void rec_EndTraining(void)
{
    int i, j;
    long  lT;
    int   scratch2;
    double sum;

    printf("EndTraining -- Start\n");

    close(scratch);
    
    init_net();
    lT  = train_net();
    printf("net training count: %lu\n",lT);

    printf("EndTraining -- End\n");

    /* feature statistics */
    
    if ((out2 = fopen("feature.sta","wt")) == NULL)
    {
            fprintf(stderr, "Can not open FEATURE.STA\n");
    }


    for (i=0; i<MAX_FEATURES; i++)
    {
        sum = 0.0;
        for (j=0; j<L1_NODES; j++) sum += ABS(w1[i]);
        fprintf(out2, " %4d  %13.6E\n", i, sum);
    }

    fclose(out2);

}
/*--------------------------------------------------------------*\
 |                                                              |
 |      Calculate_Features                                      |
 |      This routine fills the FD structure with the character  |
 |      features. New features can be added or removed easily.  |
 |      Just make sure the MAX_FEATURES define matches the      |
 |      number printed at the end of the routine.               |
 |      To be very safe, try to keep the value of the feature   |
 |      between -1 and 1 or 0 and 1.                            |
 |                                                              |
\*--------------------------------------------------------------*/


#define BOXROW 3
#define BOXCOL 3
#define SGN(x) ((x)>0?(1.0):(-1.0))

private void Calculate_Features(lpStrokeSet gesture,
                                INT16 char_code,
                                Feature_Data_Ptr FDP)
{
    int i, j, k, ii;
    double cm[4][4];        /* central moment of order [p][q] */
    double ncm[4][4];       /* normalized central moment of order [p][q] */
    double im[8];           /* invariant moments */
    double x_mean, y_mean;
    double x_mean_2_inv;
    double y_mean_2_inv;
    double x_center, y_center;
    double norm_2;
    double norm_2_5;
    double hypinv,r2;
    double  height, width;
    int  n;
    int  indx = 0;
    int  cnt;
    int  x, xx, x_min, x_max;
    int  y, yy, y_min, y_max;
    int  strk, pts;
    int  bx[BOXCOL][BOXROW];
    double a, b, c, d;
    int  x_threshold_1, y_threshold_1, x_threshold_2, y_threshold_2;
    int  dx, dy, adx, ady;

    /* if we treat the points as an image and the points have value 1 and
       non points have a value 0, then the moments can be calculated
       by method in Gonzalez & Wintz, Page 355, Eqn (7.2-3). */

    moment(gesture);

    /* this calculates the central moments from the moments */

    x_mean = m[1][0] / m[0][0];
    y_mean = m[0][1] / m[0][0];
    x_mean_2_inv = 1.0 / (x_mean * x_mean);
    y_mean_2_inv = 1.0 / (y_mean * y_mean);

    cm[0][0] = m[0][0];                     /* Eqn (7.2.12) */
    cm[1][0] = 0;                           /* Eqn (7.2-4)  */
    cm[0][1] = 0;                           /* Eqn (7.2-4)  */
    cm[2][0] = m[2][0] - x_mean * m[1][0];  /* Eqn (7.2-6)  */
    cm[0][2] = m[0][2] - y_mean * m[0][1];  /* Eqn (7.2-7)  */
    cm[1][1] = m[1][1] - y_mean * m[1][0];  /* Eqn (7.2-5)  */
    cm[3][0] = m[3][0] - 3 * x_mean * m[2][0] + 2 * x_mean_2_inv * m[1][0];
                                            /* Eqn (7.2-8)  */
    cm[0][3] = m[0][3] - 3 * y_mean * m[0][2] + 2 * y_mean_2_inv * m[0][1];
                                            /* Eqn (7.2-11) */
    cm[1][2] = m[1][2] - 2 * y_mean * m[1][1] - x_mean * m[0][2] +
                2 * y_mean_2_inv * m[1][0]; /* Eqn (7.2-9)  */
    cm[2][1] = m[2][1] - 2 * x_mean * m[1][1] - y_mean * m[2][0] +
                2 * x_mean_2_inv * m[0][1]; /* Eqn (7.2-10) */

    /* the rest are not of interest */

    /* now to calculate the normalized central moments from the
       central moments as given in Gonzales & Wintz, Pages 356-357. */

    norm_2   = 1.0 / (cm[0][0] * cm[0][0]);
    norm_2_5 = pow(cm[0][0], -2.5);

    ncm[1][1] = cm[1][1] * norm_2;
    ncm[2][0] = cm[2][0] * norm_2;
    ncm[0][2] = cm[0][2] * norm_2;
    ncm[3][0] = cm[3][0] * norm_2_5;
    ncm[0][3] = cm[0][3] * norm_2_5;
    ncm[1][2] = cm[1][2] * norm_2_5;
    ncm[2][1] = cm[2][1] * norm_2_5;

    /* now we calculate the set of seven invariant moments which have
       been shown to be invariant to translation, rotation, and scale
       change. It may help on recognition. Gonzales & Wintz, Page 358 */

    im[1] = ncm[2][0] + ncm[0][2];          /* Eqn (7.2-14) */

    im[2] = (ncm[2][0] - ncm[0][2]) * (ncm[2][0] - ncm[0][2]) +
            4.0 * ncm[1][1] * ncm[1][1];    /* Eqn (7.2-15) */

    im[3] = (ncm[3][0] - 3 * ncm[1][2]) * (ncm[3][0] - 3 * ncm[1][2]) +
            (ncm[0][3] - 3 * ncm[2][1]) * (ncm[0][3] - 3 * ncm[2][1]);
                                            /* Eqn (7.2-16) */

    im[4] = (ncm[3][0] + ncm[1][2]) * (ncm[3][0] + ncm[1][2]) +
            (ncm[0][3] + ncm[2][1]) * (ncm[0][3] + ncm[2][1]);
                                            /* Eqn (7.2-17) */

    im[5] = (ncm[3][0] - 3 * ncm[1][2]) * (ncm[3][0] + ncm[1][2])
            * ( (ncm[3][0] + ncm[1][2]) * (ncm[3][0] + ncm[1][2])
            - 3 * (ncm[2][1] + ncm[0][3]) * (ncm[2][1] + ncm[0][3]) )
            + (3 * ncm[2][1] - ncm[0][3]) * (ncm[2][1] + ncm[0][3])
            * (3 * (ncm[3][0] + ncm[1][2]) * (ncm[3][0] + ncm[1][2])
            - (ncm[2][1] + ncm[0][3]) * (ncm[2][1] + ncm[0][3]) );
                                            /* Eqn (7.2-18) */

    im[6] = (ncm[2][0] - ncm[0][2]) *
            ( (ncm[3][0] + ncm[1][2]) * (ncm[3][0] + ncm[1][2])
            - (ncm[2][1] + ncm[0][3]) * (ncm[2][1] + ncm[0][3]) )
            + 4.0 * ncm[1][1] * (ncm[3][0] + ncm[1][2])
            * (ncm[2][1] +ncm[0][3]);

    im[7] = (3 * ncm[1][2] - ncm[3][0]) * (ncm[3][0] + ncm[1][2]) *
            ( (ncm[3][0] + ncm[1][2]) * (ncm[3][0] + ncm[1][2])
            - 3 * (ncm[2][1] + ncm[0][3]) * (ncm[2][1] + ncm[0][3]) )
            + (3 * ncm[1][2] - ncm[0][3]) * (ncm[2][1] + ncm[0][3]) *
            ( 3 * (ncm[3][0] + ncm[1][2]) * (ncm[3][0] + ncm[1][2])
            - (ncm[2][1] + ncm[0][3]) * (ncm[2][1] + ncm[0][3]) );

        
    /* calculate the minimum and maximum point coordinates */
    x_max = gesture->ptr[0]->p[0].h;
    x_min = gesture->ptr[0]->p[0].h;
    y_max = gesture->ptr[0]->p[0].v;
    y_min = gesture->ptr[0]->p[0].v;

    for (strk=0; strk<gesture->num_strokes_in; strk++)
    {
        for (pts=0; pts< gesture->ptr[strk]->num_points; pts++)
        {
            x = gesture->ptr[strk]->p[pts].h;
            y = gesture->ptr[strk]->p[pts].v;
            if (x > x_max) x_max = x;
            if (x < x_min) x_min = x;
            if (y > y_max) y_max = y;
            if (y < y_min) y_min = y;
        }
    }
    /* Clear Data */
    
    for (i=0; i<MAX_FEATURES; i++) FDP->Features[i] = 0; /* 0  x */

    FDP->char_code   = char_code;
    FDP->Features[indx++]  = gesture->num_strokes_in;  /* number of strokes */

    /*  try to see if the aspect ratio is a good feature */

    height = y_max - y_min;
    width  = x_max - x_min;
    
    if (width < 1.0) width = 1.0; /* divide by zero prevention */
    FDP->Features[indx] = height / width;  
    if (FDP->Features[indx] > 7.0) FDP->Features[indx] = 7.0;
    indx++;
    
    y_center = (height / 2) + y_min;
    x_center = (width / 2) + x_min;

    /* where does the first stroke start and end and 6 spaced points */
  
    ii = gesture->ptr[0]->num_points;
    FDP->Features[indx++] = ii / 100.0;
    a = (ii-1) / 7.0;

    for (i=0; i<8; i++)
    {
        j = (a * i) + 0.5;
        FDP->Features[indx++]  = (gesture->ptr[0]->p[j].h - x_center) / 300.0;
        FDP->Features[indx++]  = (gesture->ptr[0]->p[j].v - y_center) / 300.0;
    }


    /* where does the second stroke start and end */

    if (gesture->num_strokes_in > 1)
    {
        ii = gesture->ptr[1]->num_points;
        FDP->Features[indx++]  = ii / 100.0;
        a = (ii-1) / 5.0;
        for (i=0; i<6; i++)
        {
            j = (a * i) + 0.5;
            FDP->Features[indx++] = (gesture->ptr[1]->p[j].h - x_center)/300.0;
            FDP->Features[indx++] = (gesture->ptr[1]->p[j].v - y_center)/300.0;
        }
    }
    else indx += 13;

    /* where does the third stroke start and end */

    if (gesture->num_strokes_in > 2)
    {
        FDP->Features[indx++] = gesture->ptr[2]->num_points / 100.0;
        FDP->Features[indx++] = (gesture->ptr[2]->p[0].h - x_center) / 300.0;
        FDP->Features[indx++] = (gesture->ptr[2]->p[0].v - y_center) / 300.0;
        
        FDP->Features[indx++]  =
            (gesture->ptr[2]->p[gesture->ptr[2]->num_points-1].h
                - x_center) / 300.0; 
        FDP->Features[indx++]  =
            (gesture->ptr[2]->p[gesture->ptr[2]->num_points-1].v
                - y_center) / 300.0;
    }
    else indx += 5;

    if (gesture->num_strokes_in > 3)
        FDP->Features[indx++] = gesture->ptr[3]->num_points / 100.0;
    else indx++;

    if (gesture->num_strokes_in > 4)
        FDP->Features[indx++] = gesture->ptr[4]->num_points / 100.0;
    else indx++;


    a = 3.0 / (x_max - x_min);
    b = 3.0 * x_min / (x_max - x_min);
    c = 3.0 / (y_max - y_min);
    d = 3.0 * y_min / (y_max - y_min);
    cnt = 0;

    for (i=0; i<3; i++)
        for (j=0; j<3; j++)
            bx[j][i] = 0;

    for (strk=0; strk<gesture->num_strokes_in; strk++)
    {
        for (pts=0; pts< gesture->ptr[strk]->num_points; pts++)
        {
            x = gesture->ptr[strk]->p[pts].h;
            y = gesture->ptr[strk]->p[pts].v;
            i = x * a - b;
            j = y * c - d;
            bx[i][j]++;
            cnt++;
        }
    }

    for (j=0; j<3; j++)
        for (i=0; i<3; i++)
            FDP->Features[indx++]  = 1.0 * bx[i][j] / cnt;

                                                    /* 69-77 */


    for (j=0; j<3; j++)
    {
        FDP->Features[indx]  = bx[0][j] + bx[1][j] + bx[2][j];
        FDP->Features[indx++] /= cnt;                   
    }

    for (i=0; i<3; i++)
    {
        FDP->Features[indx]  = bx[i][0] + bx[i][1] + bx[i][2];
        FDP->Features[indx++] /= cnt;                  
    }

    /* this is the second try at a feature in the demo code,
       but modified for my structures                          */

#define RANGE(x1,x2) ((x1>x2/0.196349541)&&(x1<(((x2+4)%16)/0.196349541)))
#define PRE_ABS 90

    ii = indx;

    x_threshold_1 = width  / 8;
    y_threshold_1 = height / 8;
    x_threshold_2 = width  / 4;
    y_threshold_2 = height / 4;

    for (j=0; j<MIN(2,gesture->num_strokes_in); j++)
    {
        n  = 0;

        for (i=1; i<gesture->ptr[j]->num_points-1; i++)
        {
            x = gesture->ptr[j]->p[i].h;
            y = gesture->ptr[j]->p[i].v;

            dx  = xx - x;
            dy  = yy - x;
            adx = ABS(dx);
            ady = ABS(dy);

            if ((adx + ady) < PRE_ABS) continue;
            if ((adx < x_threshold_1) && (ady < y_threshold_1)) continue;
            if ((n == 0) &&
                (adx < x_threshold_2) && (ady < y_threshold_2)) continue;

            n++;
            a = atan2((double)dx,(double)dy);
            for (k=0; k<15; k++)
                if RANGE(a,k) FDP->Features[ii+k] += 1.0;
            xx = x;
            yy = y;
        }

        dx = xx - gesture->ptr[j]->p[i].h;
        dy = yy - gesture->ptr[j]->p[i].v;
        if ((ABS(dx) + ABS(dy)) > PRE_ABS)
        {
            n++;
            a = atan2((double)dx,(double)dy);
            for (k=0; k<15; k++)
                if RANGE(a,k) FDP->Features[ii+k] += 1.0;
        }
        ii += 16;
    }
    indx += 32;


/*
       This is inspired by the demo code. However with my scoring and
       optimization technique, there is a problem with the difference
       between 359 degrees and 1 degree. I always take the real
       difference. If you think about a angle being the arctan of the
       (y / x) why can't we leave it as the normalized x and y pieces.
       Therefore y = 1 x = 0; is the same as 0 (or 360) degrees, but
       tracked as two features instead of one. That way I do not have
       to have all of the angles transforms for quadrants that the
       demo has.
*/

    hypinv = 1.0 / sqrt(height * height + width * width);

    for (j=0; j<MIN(2,gesture->num_strokes_in); j++)
    {
        x = gesture->ptr[j]->p[0].h;
        y = gesture->ptr[j]->p[0].v;
        n = gesture->ptr[j]->num_points - 1;

        for (i=1; i<9; i++)
        {
            a = gesture->ptr[j]->p[(i*n)/8].h;
            b = gesture->ptr[j]->p[(i*n)/8].v;
            c = a - x;
            d = b - y;
            r2 = (c * c) + (d * d);
            if (r2 < 0.001)
            {
                FDP->Features[indx++] = 0;
                FDP->Features[indx++] = 0;
                FDP->Features[indx++] = 0;
            }
            else
            {
                FDP->Features[indx++] = SGN(c) * sqrt(c*c / r2);
                FDP->Features[indx++] = SGN(d) * sqrt(d*d / r2);
                FDP->Features[indx++] = sqrt(r2) * hypinv;
                }
            x = gesture->ptr[j]->p[(i*n)/8].h;
            y = gesture->ptr[j]->p[(i*n)/8].v;
        }
    }

    if (gesture->num_strokes_in == 1) indx += 24; 

/* 8 central moments */
/*
    FDP->Features[indx++]  = cm[0][0];
    FDP->Features[indx++]  = cm[1][1];
    FDP->Features[indx++]  = cm[0][2];
    FDP->Features[indx++]  = cm[2][0];
    FDP->Features[indx++]  = cm[0][3];
    FDP->Features[indx++]  = cm[3][0];
    FDP->Features[indx++]  = cm[1][2];
    FDP->Features[indx++]  = cm[2][1];
*/
    /* 7 normalized central moments */

    FDP->Features[indx++]  = ncm[1][1] / 400.0;
    FDP->Features[indx++]  = ncm[0][2] / 1600.0;
    FDP->Features[indx++]  = ncm[2][0] / 1600.0;
    FDP->Features[indx++]  = ncm[0][3] / 1000000.0;
    FDP->Features[indx++]  = ncm[3][0] / 1000000.0;
    FDP->Features[indx++]  = ncm[1][2] / 640000.0;
    FDP->Features[indx++]  = ncm[2][1] / 640000.0;

    /* 7 invariant moments */
    
    FDP->Features[indx++]  = im[1] / 2000.0;
    FDP->Features[indx++]  = im[2] / 3500000.0;
    FDP->Features[indx++]  = im[3] / 5.5E12;
    FDP->Features[indx++]  = im[4] / 5.5E12;
    FDP->Features[indx++]  = im[5] / 3.0E25;
    FDP->Features[indx++]  = im[6] / 5.0E15;
    FDP->Features[indx++]  = im[7] / 1.6E25;
    
    printf(" Features : %u  %5.1f\n", indx, FDP->Features[0]);
}

/*--------------------------------------------------------------*\
 |                                                              |
 |      moment                                                  |
 |                                                              |
 |      This routine generates moments. Moments are used to     |
 |      to classify images they may help here.                  |
 |                                                              |
\*--------------------------------------------------------------*/
    
private void moment(lpStrokeSet gesture)
{
    int i, j;
    INT16 strk;
    INT16 pts;
    double x, y, x2, y2;

    for (i=0; i<4; i++)
        for (j=0; j<4; j++)
            m[i][j] = 0;

    for (strk=0; strk<gesture->num_strokes_in; strk++)
    {
        for (pts=0; pts< gesture->ptr[strk]->num_points; pts++)
        {
            x = gesture->ptr[strk]->p[pts].h;
            y = gesture->ptr[strk]->p[pts].v;

            x2 = x * x;
            y2 = y * y;

            m[0][0] += 1;
            m[0][1] += y;
            m[1][0] += x;
            m[1][1] += x * y;
            m[0][2] += y2;
            m[2][0] += x2;
            m[0][3] += y2 * y;
            m[3][0] += x2 * x;
            m[1][2] += x * y2;
            m[2][1] += x2 * y;

        }
    }
}

/*------------------------------------------------------------------*\
 |  NEURAL NET based on the features extracted. This is an example  |
 |      based on the multi-layer (3) perceptron as described by     |
 |      Richard P. Lippmann in "An Introduction to Computing with   |
 |      with Neural Nets" IEEE ASSP Magazine, April 1987, pp. 4-22. |
\*------------------------------------------------------------------*/


/*--------------------------------------------------------------*\
 |                                                              |
 |      vector                                                  |
 |                                                              |
 |      Routine to generate a vector array.                     |
 |                                                              |
\*--------------------------------------------------------------*/

private float *vector(int n)
{
    float huge *v;  /* I use huge to make sure it is normalized */
    v = (float huge *) mem_Alloc((unsigned) n * sizeof(float));
    if (v == NULL) out_of_memory_error();
    return (float *) v;
}

/*--------------------------------------------------------------*\
 |                                                              |
 |      matrix                                                  |
 |                                                              |
 |      Routine to generate a matrix using pointers to          |
 |      pointers. The advantage of this technique is that I     |
 |      can have very large matricies. A normal float matrix    |
 |      128 x 128 takes 64K bytes. Since the number of weights  |
 |      between the layers is the product of the the nodes of   |
 |      the two layers. Since I think the input features will   |
 |      have at least 128 features and the hidden layers will   |
 |      have at least as many nodes as the input. The other     |
 |      problem is this d*** segmented architecture with        |
 |      limited memory. I long for the good old days when I had |
 |      a VAX with seemingly endless memory! In some ways we go |
 |      forward, in others we go backward.                      |
 |                                                              |
\*--------------------------------------------------------------*/

private float **matrix(int r, int c)
{
    int i;
    float huge **m;

    m=(float huge **) mem_Alloc((unsigned) r * sizeof(float *));
    if (m == NULL) out_of_memory_error();

    for (i=0; i<r; i++)
    {
        m[i] = (float *) mem_Alloc((unsigned) c * sizeof(float));
        if (m[i] == NULL) out_of_memory_error();
    }

    return (float **) m;
}

/*--------------------------------------------------------------*\
 |                                                              |
 |      out_of_memory_error                                     |
 |                                                              |
 |      This routine shuts everything down if there is not      |
 |      enough memory for the arrays. It should not be a        |
 |      problem.                                                |
 |                                                              |
\*--------------------------------------------------------------*/

private void out_of_memory_error(void)
{
    printf("Out of memory\n");
    exit(-1);
}


/*--------------------------------------------------------------*\
 |                                                              |
 |      init_net                                                |
 |                                                              |
 |      This routine initializes the Neural Net using back      |
 |      propagation.                                            |
 |                                                              |
\*--------------------------------------------------------------*/
#define RANDOM_FUNCTION ((random(1000) - 500.0) / 500.0)

private int     init_net(void)
{
    int i,j;
    
    int ret  = 0;
    int ret1 = 0;
    float a;    
    ctrlbrk (c_break);

/*    srand(31);   */
    randomize(); 
    
    x1     = vector(L1_NODES);
    theta1 = vector(L1_NODES);
    printf("Core Left after x1 : %ld\n", (unsigned long) coreleft());
    mem_CheckHeap();

    x2     = vector(L2_NODES);
    theta2 = vector(L2_NODES);
    printf("Core Left after x2 : %ld\n", (unsigned long) coreleft());
    mem_CheckHeap();

    x3     = vector(OUT_NODES);
    theta3 = vector(OUT_NODES);
    printf("Core Left after x3 : %ld\n", (unsigned long) coreleft());
    mem_CheckHeap();


    for (i=0; i<(L1_NODES); i++)
    {
        theta1[i] = i * 4.0 / (L1_NODES) - 2.0;
    }

    for (i=0; i<(L2_NODES); i++)
    {
        theta2[i] = i * 4.0 / (L2_NODES) - 2.0;
    }

    for (i=0; i<(OUT_NODES); i++)
    {
        theta3[i] = i * 4.0 / (OUT_NODES) - 2.0;
    }

    /* initialize the extra large weight arrays */

    w1     = vector(IN_NODES);
    w2     = vector(L1_NODES);
    w3     = vector(L2_NODES);
    printf("Core Left after weights : %ld\n", (unsigned long) coreleft());
    mem_CheckHeap();

    
    if (load_net() == 1) return 0; 


    
    /*------------------------------------------------*/
    /* w1         create and init with random numbers */
    /*------------------------------------------------*/

    if  ((w1_FILE = open("w1.dat", WRITE_VAR, FILE_VAR)) == -1)
        printf("Can not open w1.dat in INIT_NET\n");
    
    for ( i=0; ret >= 0 && ret1 >= 0 && i<L1_NODES; i++ )
    {
        for (j=0; j<IN_NODES; j++)
        {
            w1[j] = RANDOM_FUNCTION;
        }
        ret = _write(w1_FILE, w1, W1_SIZE );
        }

    if ( ret < 0 || ret1 < 0 ) /* arggh! something failed in the create */
    {
        printf ( "\n\nwrite failed, w1, with return code %d %d\n\n",
            ret, ret1 );
        exit ( -1 );
    }

    printf ( "w1 -- Records are now on disk \n" );

    close(w1_FILE);
    w1_FILE = 0;

    /*------------------------------------------------*/
    /* w2         create and init with random numbers */
    /*------------------------------------------------*/
    if  ((w2_FILE = open("w2.dat", WRITE_VAR, FILE_VAR)) == -1)
        printf("Can not open w2.dat in INIT_NET\n");

    for ( i=0; ret >= 0 && ret1 >= 0 && i<L2_NODES; i++ )
    {
        for (j=0; j<L1_NODES; j++)
        {
            w2[j] = RANDOM_FUNCTION;
        }
        ret = _write ( w2_FILE, w2, W2_SIZE );
    }

    if ( ret < 0 || ret1 < 0 ) /* arggh! something failed in the _write */
    {
        printf ( "\n\n_write failed, w2, with return code %d %d\n\n",
            ret, ret1 );
        exit ( -1 );
    }

    printf ( "w2 -- Records are now on disk \n" );

    close(w2_FILE);
    w2_FILE = 0;
    
    /*------------------------------------------------*/
    /* w3         create and init with random numbers */
    /*------------------------------------------------*/
    if  ((w3_FILE = open("w3.dat", WRITE_VAR, FILE_VAR)) == -1)
        printf("Can not open w3.dat in INIT_NET\n");

    for ( i=0; ret >= 0 && ret1 >= 0 && i<OUT_NODES; i++ )
    {
        for (j=0; j<L2_NODES; j++)
        {
            w3[j] = RANDOM_FUNCTION;
        }
        ret = _write ( w3_FILE, w3, W3_SIZE );
    }

    if ( ret < 0 || ret1 < 0 ) /* arggh! something failed in the _write */
    {
        printf ( "\n\n_write failed, w3, with return code %d %d\n\n",
            ret, ret1 );
        exit ( -1 );
    }

    printf ( "w3 -- Records are now on disk \n" );

    close(w3_FILE);
    w3_FILE = 0;

    return 0;
}

/*------------------------------------------------------------------*\
 | train_net: this is the routine necessary to train the net. In a  |
 | "real machine" the feature vectors would be in an array and I    |
 | would not have to read them from a disk. Because I have so       |
 | little RAM I have to put it out to disk. On the other hand I     |
 | have 2 MB of disk cache, so it should still work fairly well.    |
 | It would also help if SCRATCH.KIP was on a RAM Disk!             |
\*------------------------------------------------------------------*/

private long     train_net(void)
{
    int i, j, k, l;
    double alpha, tmp;
    int iter;
    long over = 1000;
    int handle, bytes;
    int cnt;
    int temp_1;
    double sumsqrs;


    for (iter=0; (iter<MAX_ITERATIONS) && (over > 0); iter++)
    {
        over = 0;
        cnt  = 0;
        sumsqrs = 0.0;

        if ((handle = open("SCRATCH.KIP", O_RDONLY | O_BINARY,
                                           S_IWRITE | S_IREAD)) == -1)
        {
            printf("error opening SCRATCH.KIP in train_net\n");
            exit(1);
        }

        while (!eof(handle)) 
        {
            bytes = _read(handle, &FD.char_code,sizeof(Feature_Data));

            if (bytes != sizeof(Feature_Data)) break;
            if (FD.char_code == 0) break;

            /* Calculate forward */

            use_net(&FD);

            /* Adapt Weights output layer (Back Propagation) */

            if  ((w3_FILE = open("w3.dat", READ_VAR, FILE_VAR)) == -1)
                printf("Can not open w3.dat in TRAIN_NET\n");
            if  ((temp_1 = open("temp_1.dat", WRITE_VAR, FILE_VAR)) == -1)
                printf("Can not open temp_1.dat in TRAIN_NET\n");

            for(k=0; k<L2_NODES; k++) t2[k] = 0.0;
            for (l=0; l<OUT_NODES; l++)
            {
                if (cTable[l] == FD.char_code)
                {
                    delta3[l] = x3[l] * (1.0 - x3[l]) * (0.99999 - x3[l]);
                    if (ABS(1.0-x3[l]) > TOLERANCE) over++;
                    sumsqrs += (1.0 - x3[l]) * (1.0 - x3[l]);
                }
                else
                {
                    delta3[l] = x3[l] * (1.0 - x3[l]) * (0.00001 - x3[l]);
                    if (ABS(x3[l]) > TOLERANCE) over++;
                    sumsqrs += x3[l] * x3[l];
                }
                
                theta3[l] += GAIN * delta3[l] * BIAS;
                
                _read( w3_FILE, w3, W3_SIZE);
                for (k=0; k<L2_NODES; k++)
                {
                    t2[k] += delta3[l] * w3[k];
                    w3[k] += GAIN * delta3[l] * x2[k];
                }

                _write( temp_1 , w3, W3_SIZE);
            }

            close(w3_FILE);
            close(temp_1);
            w3_FILE = 0;
            temp_1 = 0;
            
            remove("w3.dat");
            rename("temp_1.dat","w3.dat");

            if (((cnt % 3) == 0) && (cnt < 30))
            printf(
                "%2i %c | %7.5f %7.5f %7.5f %7.5f %7.5f %7.5f %7.5f%11.5lf\n",
                cnt, FD.char_code,
                x3[0], x3[1], x3[2], x3[3], x3[4], x3[5], x3[6], sumsqrs);
            cnt++;


            /* Adapt Weights of second hidden layer */

            if  ((w2_FILE = open("w2.dat", READ_VAR, FILE_VAR)) == -1)
                printf("Can not open w2.dat in TRAIN_NET\n");
            if  ((temp_1 = open("temp_1.dat", WRITE_VAR, FILE_VAR)) == -1)
                printf("Can not open temp_1.dat in TRAIN_NET\n");

            for (j=0; j<L1_NODES; j++) t1[j] = 0.0;
            for (k=0; k<L2_NODES; k++)
            {
                delta2[k] = t2[k] * x2[k] * (1.0 - x2[k]);

                theta2[k] += GAIN * delta2[k] * BIAS;
                
                _read( w2_FILE, w2, W2_SIZE);

                for (j=0; j<L1_NODES; j++)
                {
                    t1[j] += delta2[k] * w2[j]; 
                    w2[j] += GAIN * delta2[k] * x1[j];
                }

                _write( temp_1 , w2, W2_SIZE);
            }
            
            close(w2_FILE);
            close(temp_1);
            w2_FILE = 0;
            temp_1 = 0;

            remove("w2.dat");
            rename("temp_1.dat","w2.dat");
            
            /* Adapt Weights of first hidden layer */
            if  ((w1_FILE = open("w1.dat", READ_VAR, FILE_VAR)) == -1)
                printf("Can not open w1.dat in TRAIN_NET\n");
            if  ((temp_1 = open("temp_1.dat", WRITE_VAR, FILE_VAR)) == -1)
                printf("Can not open temp_1.dat in TRAIN_NET\n");

            for (j=0; j<L1_NODES; j++)
            {
                delta1[j] = t1[j] * x1[j] * (1.0 - x1[j]);

                theta1[j] += GAIN * delta1[j] * BIAS;

                _read( w1_FILE, w1, W1_SIZE);

                for (i=0; i<IN_NODES; i++)
                {
                    w1[i] += GAIN * delta1[j] * FD.Features[i];
                }

                _write( temp_1 , w1, W1_SIZE);
            }
            
            close(w1_FILE);
            close(temp_1);
            w1_FILE = 0;
            temp_1 = 0;
            remove("w1.dat");
            rename("temp_1.dat","w1.dat");
            
        }
        printf(
    "\nIter %d In %d L1 %d L2 %d Out %d Tol%8.4f Over %ld SumSqrs%10.4lf\n",
            iter, IN_NODES, L1_NODES, L2_NODES, OUT_NODES, 
                TOLERANCE, over, sumsqrs );

        close(handle);

        if ((iter % 25) == 0) save_net();
    }

    save_net();
    printf("Ended Neural Net Training\n");
    return over;
}

/*--------------------------------------------------------------*\
 |                                                              |
 |      use_net                                                 |
 |                                                              |
 |      Use the Feature Data in the net and the x3 array will   |
 |      hold the data. The correct answer will be close to one  |
 |      and the incorrect answers will be very close to zero.   |
 |                                                              |
\*--------------------------------------------------------------*/

private void     use_net(Feature_Data_Ptr FDP)
{
    int i, j, k, l;
    double alpha;


    /* Calculate first hidden layer */
    if  ((w1_FILE = open("w1.dat", READ_VAR, FILE_VAR)) == -1)
        printf("Can not open w1.dat in use_net\n");

    for (j=0; j<L1_NODES; j++)
    {
        alpha = theta1[j];
        _read( w1_FILE, w1, W1_SIZE);
        for (i=0; i<IN_NODES; i++)
            alpha += w1[i] * FDP->Features[i];
        x1[j] = sigmoid_logistic_nonlinearity(alpha);
    }
    close(w1_FILE);
    w1_FILE = 0;
    
    /* Calculate second hidden layer */
    
    if  ((w2_FILE = open("w2.dat", READ_VAR, FILE_VAR)) == -1)
        printf("Can not open w2.dat in use_net\n");
    
    for (k=0; k<L2_NODES; k++)
    {
        alpha = theta2[k];
        _read(w2_FILE, w2,  W2_SIZE);
        for (j=0; j<L1_NODES; j++)
            alpha += w2[j] * x1[j];
        x2[k] = sigmoid_logistic_nonlinearity(alpha);
    }
    close(w2_FILE);
    w2_FILE = 0;
    
    /* Calculate third (output) layer */
    if  ((w3_FILE = open("w3.dat", READ_VAR, FILE_VAR)) == -1)
        printf("Can not open w3.dat in use_net\n");
    
    for (l=0; l<OUT_NODES; l++)
    {
        alpha = theta3[l];
        _read( w3_FILE, w3, W3_SIZE);
        for (k=0; k<L2_NODES; k++)
            alpha += w3[k] * x2[k];
        x3[l] = sigmoid_logistic_nonlinearity(alpha);
    }
    close(w3_FILE);
    w3_FILE = 0;
}

/*--------------------------------------------------------------*\
 |                                                              |
 |      save_net                                                |
 |                                                              |
 |      This function saves the net out to disk. It can be      |
 |      reloaded when the net is started and hopefully will     |
 |      cause the net to converge immediately if the same data  |
 |      is presented, or maybe faster for new data depending    |
 |      on how close the handwritings are.                      |
 |                                                              |
\*--------------------------------------------------------------*/

private void save_net(void)
{
    int i;
    unsigned parms[4];
    int savefile;
    
    if ((savefile = open("c:\\SAVENET.KIP", 
                          O_CREAT | O_TRUNC | O_WRONLY | O_BINARY,
                          S_IWRITE | S_IREAD)) == -1)
    {
        printf("Can not open file to save net\n");
    }
    
    parms[0] = IN_NODES;
    parms[1] = L1_NODES;
    parms[2] = L2_NODES;
    parms[3] = OUT_NODES;

    if (sizeof(parms) != _write(savefile, parms, sizeof(parms)))
    {
        printf("Can not write parms to save net\n");
    }

    if  ((w1_FILE = open("w1.dat", READ_VAR, FILE_VAR)) == -1)
        printf("Can not open w1.dat in save_net\n");

    for (i=0; i<L1_NODES; i++)
    {
        _read(w1_FILE, w1, W1_SIZE);
        _write(savefile, w1, W1_SIZE);
    }
    close(w1_FILE);
    w1_FILE = 0;
    
    if  ((w2_FILE = open("w2.dat", READ_VAR, FILE_VAR)) == -1)
        printf("Can not open w2.dat in save_net\n");

    for (i=0; i<L2_NODES; i++)
    {
        _read( w2_FILE, w2, W2_SIZE);
        _write(savefile, w2, W2_SIZE);
    }
    close(w2_FILE);
    w2_FILE = 0;
    
    if  ((w3_FILE = open("w3.dat", READ_VAR, FILE_VAR)) == -1)
        printf("Can not open w3.dat in save_net\n");
    for (i=0; i<OUT_NODES; i++)
    {
        _read( w3_FILE, w3, W3_SIZE);
        _write(savefile, w3, W3_SIZE);
    }
    close(w3_FILE);
    w3_FILE = 0;
    
    _write(savefile, theta1, L1_NODES * sizeof(float));
    _write(savefile, theta2, L2_NODES * sizeof(float));
    _write(savefile, theta3, OUT_NODES * sizeof(float));

    close(savefile);
}

/*--------------------------------------------------------------*\
 |                                                              |
 |      load_net                                                |
 |                                                              |
 |      Load the weights and thresholds for the net. This will  |
 |      really help debugging. Since generating the output      |
 |      from scratch takes about 24 hours!                      |
 |                                                              |
\*--------------------------------------------------------------*/

private int load_net(void)
{
    int i;
    unsigned parms[4];
    int savefile;

    if ((savefile = open("c:\\SAVENET.KIP", O_RDONLY | O_BINARY,
                            S_IWRITE | S_IREAD)) == -1)
    {
        printf("Can not open file to load net\n");
        return -1;
    }

    if (sizeof(parms) != _read(savefile, parms, sizeof(parms)))
    {
        printf("Can not write parms to save net\n");
    }

    if (!(  (parms[0] == IN_NODES) &&
            (parms[1] == L1_NODES) &&
            (parms[2] == L2_NODES) &&
            (parms[3] == OUT_NODES) ))
    {
        printf("Incompatible SAVENET file\n");
        close(savefile);
        return 0;
    }


    if  ((w1_FILE = open("w1.dat", WRITE_VAR, FILE_VAR)) == -1)
        printf("Can not open w1.dat in load_net\n");

    for (i=0; i<L1_NODES; i++)
    {
        _read(savefile, w1, W1_SIZE);
        _write( w1_FILE, w1, W1_SIZE);
    }
    close(w1_FILE);
    w1_FILE = 0;
    
    if  ((w2_FILE = open("w2.dat", WRITE_VAR, FILE_VAR)) == -1)
        printf("Can not open w2.dat in load_net\n");

    for (i=0; i<L2_NODES; i++)
    {
        _read(savefile, w2, W2_SIZE);
        _write( w2_FILE, w2, W2_SIZE);
    }
    close(w2_FILE);
    w2_FILE = 0;
    
    if  ((w3_FILE = open("w3.dat", WRITE_VAR, FILE_VAR)) == -1)
        printf("Can not open w3.dat in load_net\n");

    for (i=0; i<OUT_NODES; i++)
    {
        _read(savefile, w3, W3_SIZE);
        _write( w3_FILE, w3, W3_SIZE);
    }
    close(w3_FILE);
    w3_FILE = 0;

    
    _read(savefile, theta1, L1_NODES * sizeof(float));
    _read(savefile, theta2, L2_NODES * sizeof(float));
    _read(savefile, theta3, OUT_NODES * sizeof(float));


    close(savefile);
    return 1;
}

/***********************************************************************/

private int c_break(void)
{
   printf("Control-Break pressed.  Program aborting ...\n");
   save_net();
   exit(99);
   return (ABORT);
}

private double  sigmoid_logistic_nonlinearity(double x)
{
    if (x > 11.5129) return (0.99999);
    else if (x < - 11.5129) return (0.00001);
    else return (((double) 1.0) / (((double) 1.0) + exp (-x)));
}
