/****************************************************************\
|   RECOG.C
|   Recognizer for Handprinted Text.
+-----------------------------------------------------------------
|   This is a handwriting recognition engine for
|   Ray Valdes' contest.
|   This recognizer was written by David Randolph.
|   (c) Copyright 1992 by David Randolph.
|   All rights reserved.
\****************************************************************/

/****************************************************************\
This file is my submission for the handwritten character recognition
contest.  Using the DDJ_HWX.DAT file, it achieves above 90.8%
accuracy when trained on the first three samples depending
upon the setting of the rGA_BUF_SIZE constant, and much higher
accuracy when scanning the CLEAN.DAT file.  Unlike Ron Avitzur's
method, this algorithm does not benefit quite as much from training
on additional samples, but can still perform strongly (85+%) after
training on only one sample.  As one might expect, it performs much
more accurately, even when analyzing poor-quality handwriting, if the
training samples are clean.

Unfortunately, I have no samples of my own to send, but that may be
for the best.  Because of the primitive nature of my own computer, I
have not had very good luck trying to see the samples sent with the
contest materials, but that eliminates any preconceived notions about
the handwriting I am analyzing.  For all I know, I could have been
recognizing Kanji!

The algorithm:
This is an on-line algorithm which cheats by taking advantage of the
direction information inherent in tablet data.  Here is what it does:

1.  The multiple-stroke information is concatenated into a single
    variable-length stroke.

2.  The variable-length stroke is fitted into a pair of fixed-size
    arrays of size rGA_BUF_SIZE.  The left half of the pair is for
    the X coordinates, the right half for the matching Y coordinates.
    I have #defined rGA_BUF_SIZE to 50 because that is all my own
    computer will handle with 640k, but other platforms with more
    memory might be able to increase its size.  Keep in mind that
    the larger rGA_BUF_SIZE is, the longer the program takes to run.

3.  During the training phase, each sample is run through steps one
    and two and inserted into a linked list along with the character
    code and the number of strokes in the character.  The same character
    may appear multiple times in the list.  This allows the program to
    train on multiple styles simultaneously, although I would not
    recommend doing this.

4.  During the guessing phase, each sample is run through steps one and
    two and compared with each sample in the linked list which has the
    same number of strokes.

5.  The comparison:  is done using the Pearson Correlation formula.  The
    X arrays are compared using the formula, then the Y arrays, and then
    the results are averaged.  The correlation formula is a test to see
    if two distributions are similar.  I use it to see if two pen strokes
    follow a similar path by comparing pen coordinates which should appear
    in roughly the same place at the same time.  By using this method, I
    am making some assumptions: A) that the computer has been trained on
    the handwriting of the person it is analyzing, and B) that that person
    writes a given character consistently (ie. always starting at the top or
    bottom, etc.).


ADVANTAGES of the algorithm:
1.  It may be trained on the handwriting of multiple people at the same
    time.
2.  It is generic.  It is not tuned to work on a specific set of data and
    should work consistently regardless of what it is recognizing.
3.  The more complex the characters, the better it works.  If you need
    to examine especially detailed handwriting, this method will hold
    its own well.
4.  The program can return the set of characters which most closely
    match the test character along with their scores.  These characters
    may be used to guide a spelling checker layer which can pick the
    best one based on context.
5.  It is forgiving when the writer makes slips or odd movements with the
    pen.


DISADVANTAGES of the algorithm:
1.  It cannot train on one person's handwriting and then necessarily
    recognize another's.
2.  One of the odd quirks of this algorithm is that the more complex the
    character, the easier it is to recognize because it has more features.
    The reverse is true of simple characters, which give it trouble.  In
    fact, the character in the test data which stumped the program the
    most was the minus sign!  It does best with characters that have
    curves.
3.  The algorithm must work under the assumption that there is some
    foreknowledge about the orientation of a character.  But this is
    an assumption that all analyzers must make, or else one might not
    be able to distinguish between an "N" or a "Z", a "W" or an "M",
    a "p" or a "d", etc.
4.  It is an on-line algorithm, dependent upon the ordering of the coordinates
    in the sample data.  It cannot easily be translated to an off-line
    algorithm, which only uses a scanned picture of the character during the
    training and recognition phase, although it is possible.
5.  The algorithm does not always distinguish upper and lowercase characters
    (ie. it cannot always tell the difference between a "w" and a "W").
    I do not believe that a low-level recognizer should be expected to
    distinguish case without the help of a context-sensitive interpreter.
    I would like to ask that case-sensitivity not be made a requirement
    for this contest, since it cannot be practically achieved at this
    level, at least not when using real-world data.


WHY IT WORKS:
Since I am not mathematically inclined, I really have no technical
explanation for why the Pearson correlation formula works better
than other methods I have tried, other than to say it is highly tolerant
of error, so long as each character it trains on is sufficiently
different.  With a 50-element array of points used to define 65 characters,
the leeway allowed before an input character is mistaken for another is
formidable.  The larger the array and the more detailed the input,
the fewer the mistakes.


Although the contest places results over elegance, I have tried to keep the
code as simple and straightforward as possible to keep the algorithm clear.
I have made no changes to the harness code, so for testing, you only need
to plug in the recog.c file.

ONE NOTE before you compile.  If you have a floating point processor, you
may want to convert all the variables in the "correlate" function which
are of type LONG back to type DOUBLE.  I had to make them integers because
the floating point emulator software was too slow for my taste.

This code was created on a PC using Turbo C++ 1.0 running under DOS 3.2.

\****************************************************************/

/*** Include files ***/

#include "h_config.h"

#ifdef TURBO_C
#include <stdlib.h>
#include <math.h>
#endif
#ifdef MSOFT_C7
#include <search.h>
#include <math.h>
#endif

#include "h_stddef.h"
#include "h_mem.h"
#include "h_list.h"
#include "h_recog.h"

/****************************************************************/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>


/****************************************************************/


/* RECOG internal definitions */

  /** This defines the length of the "gesture_xy_array", in which a         */
  /** character is stored                                                   */

#define rGA_BUF_SIZE 50

  /** This defines the number of dimensions in the gesture_xy_array, which  */
  /** should always be 2, one for X and one for Y                           */

#define rNUM_COORDINATES 2

  /** These constants define where in the array mentioned above the X and Y */
  /** information should be stored                                          */

#define X_VALUE 0
#define Y_VALUE 1


/* The structure which contains the linked list of trained samples */

struct prototype
{
  INT16 char_num,                 /* The character code (ie "A")             */
        num_strokes,              /* The number of strokes in the character  */
        gesture_xy_array [rNUM_COORDINATES] [rGA_BUF_SIZE];  /* the character*/
  struct prototype *proto_next_ptr;     /* a pointer to the next record      */
};

typedef struct prototype PROTOTYPE, *PROTOTYPE_PTR;



private PROTOTYPE_PTR proto_top = NULL;  /* Always points to list top */

private PROTOTYPE_PTR proto_bot = NULL;  /* Always points to list bot */
                                         /* during training           */


/** Definitions of PRIVATE functions **/

private float correlate (INT16 *, INT16 *, int);
private void FatalError (char *);
private INT16 round (float);



/************************* Start of Code ************************/



/****************************************************************/
/* This is a dummy function; it is called by the harness        */
/* program, but never used                                      */

public void rec_InitRecognizer(void)
{

}


/****************************************************************/
/* Below is the TRAIN half of the program.  This function is    */
/* given a character, along with some information about the     */
/* character such as how many strokes it has.  The function     */
/* takes the given character and converts it into its own       */
/* format and adds it to a linked list of sample characters.    */

public void rec_Train(lpList gesture,INT16 char_code)
{
  INT16 i , k , total_input_length = 0 , convert_stroke_length ,
        array_index_ptr ;
  PROTOTYPE_PTR proto_current;


  /* display character values */

#ifdef DEBUG_TRACE
#ifndef USE_BGI_GRAPHICS
  printf("\n>For char %d (%c), number strokes = %d",char_code,char_code,
             gesture->num_items);
#endif
#endif

  /* Allocate memory for a new link */

  proto_current = (PROTOTYPE_PTR) malloc ( sizeof (PROTOTYPE) );
  if (proto_current == (PROTOTYPE_PTR) NULL)
    FatalError("\n\nFATAL ERROR!  Not Enough Memory to Train.\n\n");

  if (proto_top == (PROTOTYPE_PTR) NULL)
    proto_top = proto_current;
  else
    proto_bot->proto_next_ptr = proto_current;

  proto_bot = proto_current;

  /* set current link's next ptr to NULL */

  proto_current->proto_next_ptr = (PROTOTYPE_PTR) NULL;

  /* Load data into to new structure....*/

  proto_current->char_num = char_code;
  proto_current->num_strokes = gesture->num_items;

  /* Get the total length of the set of strokes because we are going to  */
  /* concatenate them into a single fixed-size array                     */

  for (i = 0; i < gesture->num_items; i++)
  {
    lpList stroke = gesture->items[i];
    total_input_length += stroke->num_items;
  }

  /* NOW, for each stroke, for the x and y values separately,            */
  /*   take the stroke and                                               */
  /*   force it to fit in an array of length rGA_BUF_SIZE, either by     */
  /*   stretching or compressing it...                                   */
  /* To stretch, each point in the stroke is copied into multiple        */
  /*   elements of the new array.                                        */
  /* To compress, only a sampling of new points in the stroke are copied */
  /*   to the new array.                                                 */

  array_index_ptr = 0;
  for (i = 0; i < gesture->num_items; i++)
  {

    /* Get next stroke */

    lpList stroke = gesture->items[i];

    /* Fit the stroke into its slot in the fixed array by converting its */
    /* size                                                              */

    convert_stroke_length = round(( (float) stroke->num_items /
                                    (float) total_input_length ) *
                                    (float) rGA_BUF_SIZE);

    /* Fill each empty slot in the gesture_xy_array */

    for (k = 0; (k < convert_stroke_length) &&
                (array_index_ptr < rGA_BUF_SIZE); k++, array_index_ptr++)
    {
      VHPoint p = *(lpVHPoint)&stroke->items[(int) ((float)
                   ((float) stroke->num_items / (float) convert_stroke_length) *
                   (float) (k))];
      proto_current->gesture_xy_array[X_VALUE][array_index_ptr] = p.h;
      proto_current->gesture_xy_array[Y_VALUE][array_index_ptr] = p.v;
    }
  }

}



/******************************************************************/
/* This is the GUESS portion of the program.  It is passed an     */
/* unknown character, which is compared with the linked template  */
/* list.  The best fit is the guess.  The program is passed the   */
/* character and pointers to three guesses are passed back.  This */
/* function only attempts to make a first guess.                  */

public void rec_Guess(lpList gesture,
            LPINT16 g1,LPINT16 g2,LPINT16 g3,
            LPINT16 w1,LPINT16 w2,LPINT16 w3)
{
  PROTOTYPE_PTR compare_ptr = (PROTOTYPE_PTR) NULL;
  float mean_compare;
  INT16 num_strokes, i, j, k, total_input_length = 0, convert_stroke_length,
        array_index_ptr,
        gesture_xy_array[rNUM_COORDINATES][rGA_BUF_SIZE];

  *g1 = *g2 = *g3 = 0;
  *w1 = *w2 = *w3 = 0;

  num_strokes = gesture->num_items;

  /* Get the total length of the set of strokes because we are going to  */
  /* concatenate them into a single fixed-size array                     */

  for (i = 0; i < gesture->num_items; i++)
  {
    lpList stroke = gesture->items[i];
    total_input_length += stroke->num_items;
  }

  /* NOW, for each stroke, for the x and y values separately,            */
  /*   take the stroke and                                               */
  /*   force it to fit in an array of length rGA_BUF_SIZE, either by     */
  /*   stretching or compressing it...                                   */
  /* To stretch, each point in the stroke is copied into multiple        */
  /*   elements of the new array.                                        */
  /* To compress, only a sampling of new points in the stroke are copied */
  /*   to the new array.                                                 */

  array_index_ptr = 0;
  for (i = 0; i < gesture->num_items; i++)
  {

    /* Get next stroke */

    lpList stroke = gesture->items[i];

    /* Fit the stroke into its slot in the fixed array by converting its */
    /* size                                                              */

    convert_stroke_length = round(( (float) stroke->num_items /
                                    (float) total_input_length ) *
                                    (float) rGA_BUF_SIZE);

    /* Fill each empty slot in the gesture_xy_array */

    for (k = 0; (k < convert_stroke_length) &&
                (array_index_ptr < rGA_BUF_SIZE); k++, array_index_ptr++)
    {
      VHPoint p = *(lpVHPoint)&stroke->items[(int) ((float)
                   ((float) stroke->num_items / (float) convert_stroke_length) *
                   (float) (k))];
      gesture_xy_array[X_VALUE][array_index_ptr] = p.h;
      gesture_xy_array[Y_VALUE][array_index_ptr] = p.v;
    }
  }

  compare_ptr = proto_top;

  /* Starting from the top compare the test character with each character */
  /*   in the template linked list                                        */

  while (compare_ptr != (PROTOTYPE_PTR) NULL)
  {
    /* Don't even bother to compare characters if they do not have */
    /*  the same number of strokes                                 */

    if (compare_ptr->num_strokes == num_strokes)
    {

      mean_compare = 0.0;

      mean_compare += correlate (gesture_xy_array[X_VALUE],
                                 compare_ptr->gesture_xy_array[X_VALUE],
                                 (int) (rGA_BUF_SIZE));
      mean_compare += correlate (gesture_xy_array[Y_VALUE],
                                 compare_ptr->gesture_xy_array[Y_VALUE],
                                 (int) (rGA_BUF_SIZE));

      mean_compare /= rNUM_COORDINATES;

      if ((mean_compare * 100) > *w1)
      {
        *w1 = mean_compare * 100;
        *g1 = compare_ptr->char_num;
      }

    } /* end if */

    compare_ptr = compare_ptr->proto_next_ptr;

  } /* end while */

#ifdef DEBUG_TRACE
#ifndef USE_BGI_GRAPHICS
  printf ("\n### CHAR finally evaluated as %c with prob %d",
        *g1, *w1);
#endif
#endif

}





/*
  The correlate function below was borrowed from the "pearsn" function
   found in the book "Numerical Recipies in C", edited by Press, Flannery,
   Teukolsky, and Vetterling (Cambridge University Press, Cambridge 1989)
   on page 506.
*/

/****************************************************************/
/* This function computes the correlation value between two distributions */

private float correlate (x,y,n)
INT16 x[], y[];
int n;
{
  int j;

  /* NOTE: You may want to change the word 'long' below to 'double' if
           you have a floating point processor.  It should speed things up. */

  long yt, xt, syy=0, sxy=0, sxx=0, ay=0, ax=0;
  float r;

  for (j=0; j < n; j++)
  {
    ax += x[j];
    ay += y[j];
  }

  ax /= n;
  ay /= n;

  for (j=0; j < n; j++)
  {
    xt = x[j] - ax;
    yt = y[j] - ay;
    sxx += xt * xt;
    syy += yt * yt;
    sxy += xt * yt;
  }
  r = (double) sxy / sqrt( (double) ((double) sxx * (double) syy) );
  return (float) r;
}



/****************************************************************/
/* This is a dummy function; it is called by the harness        */
/* program, but never used                                      */

public void rec_EndTraining(void)
{
}



/****************************************************************/
/* Fatal Error Handler.  Displays error message and terminates. */

private void FatalError (char* msg)
{
#ifdef USE_BGI_GRAPHICS
  gr_Close();
#endif

  fprintf (stderr,"\n*** Fatal Error:");
  fprintf (stderr, msg);
  exit (0);
}



/****************************************************************/
/* A small function to round a float to an int                  */

private INT16 round (float x)
{
  return ( (float) (x) - (float) ( (int) (x) ) >= 0.5 ) ? (int) (x) + 1 :
                                                          (int) (x);
}


/************************* END OF RECOG.C ***********************/
ZW2.10    @                                 8        PRN                                                                TXT     	     p                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               