/****************************************************************\
|   RECOG.C
|   Recognizer for Handprinted Text. 
+-----------------------------------------------------------------
|   This is a sample recognition engine to 
|   demonstrate the use of the test harness.
|   See the READ.ME file for more information.
|   This recognizer was written by Ron Avitzur.
|   (c) Copyright 1992 by Ron Avitzur.
|   Limited license for noncommercial use granted.
|   All other rights reserved.
+-----------------------------------------------------------------
|	Modified by Allen Stenger for DDJ HWX contest, August 1992
|	* Moved up definitions in AppendAVPair so would compile in THINK C
|	* RecognizeStroke: count all strokes with equal weight
|	* P8, P4: include last point
|	* Analyze: move bounding box calculation to ConvertGestureToStrokes
|	  (where it belongs!)
|	* Analyze: make bounding boxes for all strokes be the same as the
|	  bounding box for the gesture (rather than accumulate the boxes)
|	* Added a new position function: Centroid.
|	* Added an ad-hoc check to distinguish certain characters based on
|	  the extents of their centroids
|
\****************************************************************/

#define CENTROIDS	/* calculate the stroke centroid as a */
					/* position function */ /* als */
#define LIMIT_CENTROIDS	/* ad hoc method to distinguish characters */
					/* based on centroid limits */

#include "h_config.h"

#ifdef TURBO_C
#include <stdlib.h>
#include <math.h>
#endif
#ifdef MSOFT_C7
#include <search.h>
#include <math.h>
#endif

#ifdef LIGHTSPEED_C

#include <stdlib.h>
#endif

#include "h_stddef.h"
#include "h_mem.h"
#include "h_list.h"
#include "h_recog.h"  

/****************************************************************/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>


/****************************************************************/

#define MAX_HASHLISTS    20
#define MAX_FEATURES     20
#define HITLIST_MAX      128
#define MAX_POINTS       200
#define MAX_N             50

#ifndef NIL
#define TRUE               1
#define FALSE              0
#ifdef MSOFT_C7
#define NIL               ((void FAR*)0)
#else
#define NIL                0L
#endif
#define ABS(x)           ((x)>0?(x):(-(x)))
#define MAX(a,b)         ((a)>(b)?(a):(b))
#define MIN(a,b)         ((a)<(b)?(a):(b))
#endif



/***************************************************************
|      Attribute/Value Pair (AVPair), and corresponding List     
+***************************************************************/

typedef struct tagAVPair
{
    unsigned long index; 
    INT16         code; 
} AVPair;

typedef   AVPair FAR *   lpAVPair;

typedef struct tagAVList
{
    INT16   num_items; 
    AVPair  items[1]; 
}  AVList;

typedef   AVList FAR *   lpAVList;
typedef lpAVList FAR * lpLpAVList;


/****************************************************************/
typedef struct tagGesturePattern 
{
    INT16       code;
    lpList        strokes;
} GesturePattern;

typedef GesturePattern FAR * lpGesturePattern;

/****************************************************************/

typedef struct tagStrokePattern 
{
#ifdef MSOFT_C7
    bool    is_dot; 
#else
    char    is_dot;  /*<<<<<SENSITIVE TO PACKING<<<*/
#endif
    lpList    s[MAX_FEATURES];
} StrokePattern;

typedef StrokePattern FAR * lpStrokePattern;

/****************************************************************/
typedef struct StrokeData 
{
    VHPoint   Ink  [MAX_POINTS],
              P    [MAX_N];
    INT16     Ink_Num,
	      N,
	      T    [MAX_N],
	      U    [MAX_N];
    bool      IsDot;
    ULONG     S    [MAX_FEATURES];
    long      start_time, 
	      end_time;
#ifdef MSOFT_C7
    INT16     Xmax, Xmin, Ymax, Ymin, Height, Width;
    INT16     XmaxT, XminT, YmaxT, YminT, HeightT, WidthT;
#else
    long      Xmax, Xmin, Ymax, Ymin, Height, Width;
    long      XmaxT, XminT, YmaxT, YminT, HeightT, WidthT;
#endif

    INT16     possibles [128], 
	      weights   [128], 
	      num_matches;
}
    StrokeData;

typedef StrokeData FAR  * lpStroke;

/****************************************************************/

typedef void (*HashFunc)(LPSTR, lpStroke);

extern HashFunc HashFunctions[];

/****************************************************************/

#define DOT_THRESHHOLD    (40)
#define PRE_ABS           (90)   /* was: 90, was(40) */
#define PRE_RATIO_X         8
#define PRE_RATIO_Y         8
#define FIRST_RATIO_X       4
#define FIRST_RATIO_Y       4

/****************************************************************/

private  lpList          HashPatterns = NIL;
private  lpList		 Strokes = NIL;
private  lpAVList        HashList[MAX_HASHLISTS];
private  INT16           HitList[HITLIST_MAX];
private  lpStrokePattern theStrokePattern;
private  lpStroke        theStroke;
private  INT16           ATAN2_TABLE[21][21];
private  bool		 DontUseTable = TRUE;

/****************************************************************/
private INT16   VoteWeight[]    = { 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3
#ifdef CENTROIDS
									, 3 /* als */
#else
									, 0 /* als */
#endif								
									};
private char    Bits[]          = { 2, 2, 2, 2, 2, 2, 2, 3, 3, 2, 2, 2, 2,
									4};	/* als */
private LPSTR   Names[MAX_FEATURES] = {"WNES","ABCD"};
private INT16   NUM_HASHLISTS   = 10; /*2;*/
private INT16   NUM_FEATURES    = 14; /* als */ 
									/*2; /* trade accuracy for speed here*/
#define CENTROID_INDEX 13	/* als */ /* index of Centroid in HashFunctions */

/****************************************************************/
private INT16   MY_ATAN2(INT16,INT16);

private INT16   IsInList(lpList list,long n);
private void    AppendToList(lpLpList list,long item);
private void    AppendAVPair(lpLpAVList theList,lpAVPair item);

private void    SearchAndAddMatches(lpStroke the_stroke,
			lpAVList theList,unsigned long n,INT16 w);

private void    Recognize(
                LPINT16 g1,LPINT16 g2,LPINT16 g3,
		LPINT16 w1,LPINT16 w2,LPINT16 w3);

private void    Learn(INT16 learn);
private void    ComputeT(lpStroke the_stroke);
private INT16   Votes(void);
private INT16   PatternMatch(void);

private INT16   Process3(lpVHPoint ,lpVHPoint ,
			    INT16,INT16,INT16,INT16,INT16);

private INT16   Dt(INT16 t1,INT16 t2);

private void    RecognizeStroke(INT16 whichStroke);
private void    Simplify(lpStroke the_stroke,lpVHPoint Ink,INT16 N);
private void    Analyze(lpStroke);

private void    ConvertGestureToStrokes(lpList strokes);
private void    FreeStrokes(void);

private void    ConvertStringToLong(LPSTR s,LPUINT32 np,INT16 bits);
#ifdef MSOFT_C7
private int     CompareAV(void * a,void * b);
#endif
#ifdef TURBO_C
private int     CompareAV(void * a,void * b);
#endif
#ifdef LIGHTSPEED_C
private int     CompareAV(lpAVPair a,lpAVPair b);
#endif
private void    InitAtanTable(void);

/****************************************************************/
public void rec_InitRecognizer(void)
{
    printf("\nStenger recog.c: PRE_ABS=%d",PRE_ABS);
    InitAtanTable();
}
/****************************************************************/
public void rec_Train(lpList gesture,INT16 char_code)
{
    ConvertGestureToStrokes(gesture);
    Learn(char_code);
}
/****************************************************************/
public void rec_Guess(lpList gesture,
	    LPINT16 g1,LPINT16 g2,LPINT16 g3,
	    LPINT16 w1,LPINT16 w2,LPINT16 w3)
{
    INT16 i;

    ConvertGestureToStrokes(gesture);
    for (i = 0; i<Strokes->num_items; i++)
    {
        RecognizeStroke(i);
    }

    Recognize(g1,g2,g3,w1,w2,w3);
}

/****************************************************************/
public void rec_EndTraining(void)
{
    INT16 i,j,k;
    lpGesturePattern  gp;
    lpStrokePattern   sp;
    AVPair item;
    
    for (i=0;i<NUM_HASHLISTS;i++) 
	if (HashList[i]) { mem_Free(HashList[i]); HashList[i] = NIL; }

    for (i = 0; i < HashPatterns->num_items; i++) 
    {
        gp = HashPatterns->items[i];
        sp = gp->strokes->items[0];
        for (k=0; k<NUM_HASHLISTS; k++)
            if (sp->s[k]) for (j = 0; j < sp->s[k]->num_items; j++) 
	    {
                item.index = (long)sp->s[k]->items[j];
                item.code  = i;
                AppendAVPair(&HashList[k],&item);
            }
    }

    /* Sort Tables numerically by index values */
    for (i=0;i<NUM_HASHLISTS;i++) 
    {
#ifdef TURBO_C
        qsort(HashList[i]->items,HashList[i]->num_items,sizeof(AVPair),
		    (int (*)(const void*,const void*))CompareAV);
#endif
#ifdef MSOFT_C7
        qsort(  (void *) (HashList[i]->items),
		(size_t) (HashList[i]->num_items),
		(size_t)  sizeof(AVPair),
		(int (__cdecl *)(const void *, const void *))CompareAV);
#endif
#ifdef LIGHTSPEED_C
        qsort(HashList[i]->items,HashList[i]->num_items,sizeof(AVPair),
		    (__cmp_func)CompareAV);
#endif		
    }
    DontUseTable = FALSE;
}

/****************************************************************/
private void FreeStrokes(void) 
{
     if (Strokes) 
     {
	 INT16 i;
         for (i=0; i<Strokes->num_items; i++)
             mem_Free(Strokes->items[i]);
         mem_Free(Strokes);
     }
     Strokes = NIL;
}
/****************************************************************/
private void SearchAndAddMatches(
    lpStroke the_stroke,
    lpAVList list,
    unsigned long n,
    INT16 w)
{
    if (list) 
    {
        INT16 i, where;
        register lpAVPair items = list->items;
	register lpAVPair guess;
        AVPair key;
        key.index = n;
#ifdef TURBO_C
	guess = bsearch(&key,items,list->num_items,sizeof(AVPair),
		(int (*)(const void * ,const void * ))CompareAV);
#endif
#ifdef MSOFT_C7
        guess = bsearch(&key,
		    items,
		    list->num_items,
		    sizeof(AVPair),
		    ( int (__cdecl *)(const void *,const void *))CompareAV);
#endif
#ifdef LIGHTSPEED_C
        guess = bsearch(&key,items,list->num_items,sizeof(AVPair),
				    (__cmp_func)CompareAV);
#endif
        if (guess == NULL) return;
        where = guess - items;
        while (where > 0 && n == items[where-1].index) 
	    where--;
        while (where < list->num_items && n == items[where].index) 
	{
            register INT16 code = items[where].code;
            if (code < HITLIST_MAX && HitList[code])
            {
		the_stroke->weights[HitList[code] - 1] += w; 
		goto GO_ON; 
	    }
            else if (code >= HITLIST_MAX) 
	    {
                for (i = 0; i < the_stroke->num_matches; i++)
                    if (the_stroke->possibles[i] == code)
                        { the_stroke->weights[i] += w; goto GO_ON; }
            }
            the_stroke->possibles[the_stroke->num_matches] = code;
            the_stroke->weights[the_stroke->num_matches]   = w;
            the_stroke->num_matches++;
            HitList[code] = the_stroke->num_matches;
GO_ON:    where++;
        }
    }
}
/****************************************************************/
private void Learn(INT16 char_to_learn)
{
    INT16 i,j;
    lpGesturePattern  gp;
    lpStrokePattern   sp;
    lpStroke          the_stroke;


    DontUseTable = TRUE;

    if (Strokes == NIL) return;

    if (HashPatterns) for (i = 0; i < HashPatterns->num_items; i++) 
    {
        gp = HashPatterns->items[i];
        if (gp->code == char_to_learn 
	    && gp->strokes->num_items == Strokes->num_items) 
	{
            for (i = 0; i < Strokes->num_items; i++) 
	    {
                sp = gp->strokes->items[i];
                the_stroke = Strokes->items[i];
                for (j=0;j<NUM_FEATURES;j++)
                    if (!IsInList(sp->s[j], the_stroke->S[j]))
                        AppendToList(&(sp->s[j]),the_stroke->S[j]);
            }
            return;
        }
    }
    gp = mem_Alloc(sizeof(GesturePattern)); 
    if (!gp) mem_AllocFailed(sizeof(GesturePattern));
    gp->code = char_to_learn;
    gp->strokes = NIL;
    for (i = 0; i < Strokes->num_items; i++) 
    {
        sp = mem_Alloc(sizeof(StrokePattern)); 
	if (!sp) mem_AllocFailed(sizeof(StrokePattern)); 
        the_stroke = Strokes->items[i];
        sp->is_dot = the_stroke->IsDot;
        for (j=0;j<NUM_FEATURES;j++) 
	{
	    sp->s[j] = NIL; 
	    AppendToList(&(sp->s[j]), the_stroke->S[j]); 
	}
        AppendToList(&(gp->strokes),(long)sp);
    }
    AppendToList(&HashPatterns,(long)gp);
}
/****************************************************************/
private void ConvertStringToLong(LPSTR s,LPUINT32 np,INT16 bits)
{
    unsigned long n = 0;
    INT16 i,len = strlen(s);
    s[len] = s[len-1];
    if (len > 32/bits) len = 32/bits;
    for (i = 0; i <= len; i++)
        n = (n << bits) + s[len - i] - '0';
    *np = n;
}
/****************************************************************/
private INT16 Votes(void)
{
    INT16 i,w = 0;

    for (i=NUM_HASHLISTS; i<NUM_FEATURES; i++)
    {
        if (IsInList(theStrokePattern->s[i], theStroke->S[i]))
            w +=  VoteWeight[i];
    }
    return w;
}
/****************************************************************/
private INT16 PatternMatch(void)
{
    register INT16 i,w = 0;

    if (theStrokePattern->is_dot && theStroke->IsDot) return 1000;
    if (theStrokePattern->is_dot != theStroke->IsDot) return 0;
    for (i=0; i<NUM_HASHLISTS; i++)
        if (IsInList(theStrokePattern->s[i], theStroke->S[i]))
            w +=  VoteWeight[i];
    w += Votes();
    return w;
}
/****************************************************************/
private INT16 IsInList(lpList list, register long n)
{
    if (list) 
    {
        register INT16 num = list->num_items;
        register LPINT32 p = (LPINT32)list->items;
        while (num--)
            if (n == *p++)
                return 1;
    }
    return 0;
}
/****************************************************************/
private void RecognizeStroke(INT16 which_stroke)
{
    INT16            i, index;
    INT16             weight, old_weight;
    lpGesturePattern gp;
    lpStroke         prev_stroke;

    if (!(Strokes && Strokes->num_items > which_stroke)) return; 

    theStroke = (lpStroke)Strokes->items[which_stroke]; /*<<<<*/
    theStroke->num_matches = 0;

    if (!HashPatterns) return;
    
    if (which_stroke == 0) 
    {
        if (DontUseTable) 
	{
            for (i = 0; i < HashPatterns->num_items; i++) 
	    {
                gp = HashPatterns->items[i];
                theStrokePattern = gp->strokes->items[0];
                weight = PatternMatch();
                if (weight) 
		{
                    theStroke->possibles[theStroke->num_matches] = i;
                    theStroke->weights[theStroke->num_matches] = weight;
                    theStroke->num_matches++;
                }
            }
        }
        else 
	{
            memset(HitList, 0, sizeof(HitList));

            for (i=0;i<NUM_HASHLISTS;i++)
                SearchAndAddMatches(theStroke,
			HashList[i],theStroke->S[i],VoteWeight[i]);
            for (i = 0; i < theStroke->num_matches; i++) 
	    {
                gp = HashPatterns->items[theStroke->possibles[i]];
                theStrokePattern = gp->strokes->items[0];
                theStroke->weights[i] += Votes();
            }
        }
    }
    else 
    {
        prev_stroke = Strokes->items[which_stroke - 1];

        for (i = 0; i < prev_stroke->num_matches; i++) 
	{
            index = prev_stroke->possibles[i];
            old_weight = prev_stroke->weights[i];
            gp = HashPatterns->items[index];
            if (gp->strokes->num_items <= which_stroke)
                continue;
            theStrokePattern = gp->strokes->items[which_stroke];
            weight = PatternMatch();
            if (weight) 
	    {
                theStroke->possibles[theStroke->num_matches] = index;
                theStroke->weights[theStroke->num_matches] = 
				weight + old_weight;	/* als */
                theStroke->num_matches++;
                }
            }
        }
}
/****************************************************************/
/* als */
/* apply ad-hoc checks to rule out particular characters as as match */
/* based on out-of-range centroid values */
private bool MightBe(INT16 c);
private bool MightBe(INT16 c)
{
	INT16			which_stroke ;
	lpStroke		aStroke ;
	bool			mightbe = TRUE ;
	INT16			centroid ;
	
	for (which_stroke = 0 ; 
			which_stroke < Strokes->num_items ;
			which_stroke++)
		{ 				
		aStroke = Strokes->items[which_stroke] ;
		centroid = aStroke->S[CENTROID_INDEX] & 0x0000000F ;
		switch (c)
			{
			case 'F':	/* no centroid in last row */
				if (centroid >= 12)
					mightbe = FALSE ;
				break ;
			case 'H':	/* no centroid in last or first rows */
				if ((centroid >= 12) || (centroid <= 3))
					mightbe = FALSE ;
				break ;
			case 'I':	/* no centroid in first or last columns */
				if  (
						((centroid % 4 == 0) || (centroid % 4 == 3))
						&& (Strokes->num_items == 3)
					)
					mightbe = FALSE ;
				break ;
			case 'b':	/* no centroid in top half */
				if ((centroid <= 7) && (Strokes->num_items == 1))
					mightbe = FALSE ;
				break ;
			case 'p':	/* no centroid in bottom half */
				if ((centroid >= 8) && (Strokes->num_items == 1))
					mightbe = FALSE ;
				break ;
			}
		}
	return mightbe ;
}
/****************************************************************/
private void Recognize(
		LPINT16 g1,LPINT16 g2,LPINT16 g3,
		LPINT16 w1,LPINT16 w2,LPINT16 w3)
{
    INT16 i,n;

    *w1 = *w2 = *w3 = 0;

    if (Strokes == NIL) return;

    theStroke = Strokes->items[Strokes->num_items - 1];

    n = theStroke->num_matches;

    for (i = 0; i < n; i++) 
    {
        INT16 index  = theStroke->possibles[i];
        INT16 weight = theStroke->weights[i];
        GesturePattern *gp = HashPatterns->items[index];

        if (Strokes->num_items == gp->strokes->num_items) 
	{	
#ifdef LIMIT_CENTROIDS
			if (!MightBe(gp->code)) weight = 0 ;
#endif
            if (weight > *w1) 
	    {
                *g3 = *g2; *g2 = *g1; *g1 = gp->code;
                *w3 = *w3; *w2 = *w1; *w1 = weight;
            }
            else if (weight > *w2) 
	    {
                *g3 = *g2; *g2 = gp->code;
                *w3 = *w2; *w2 = weight;
            }
            else if (weight > *w3) 
	    {
                *g3 = gp->code;
                *w3 = weight;
            }
        }
    }
    if (*w1 != *w2) *w2 = 0;
    if (*w1 != *w3) *w3 = 0;
}
/****************************************************************/
/* als */ /* moved material from Analyze, propagated final bounding */
		  /* box of gesture to all bounding boxes of strokes */
private void ConvertGestureToStrokes(lpList strokes)
{
    INT16 i,j;
    INT16 Xmin,Xmax,Ymin,Ymax;

    FreeStrokes();

    for (i = 0; i < strokes->num_items; i++) 
    	{
        lpList    ink =   strokes->items[i];
        VHPoint    p = *(lpVHPoint)&ink->items[0];
        lpStroke stroke = mem_CAlloc(1 , sizeof(StrokeData));
        if (!stroke) mem_AllocFailed( sizeof(StrokeData));

        Xmin = Xmax = p.h;
        Ymin = Ymax = p.v;

        for (j = 0; j < ink->num_items; j++) 
			{
            VHPoint p = *(VHPoint*)&ink->items[j];
            stroke->Ink[j] = p;
            if (p.h > Xmax) Xmax = p.h;
            if (p.h < Xmin) Xmin = p.h;
            if (p.v > Ymax) Ymax = p.v;
            if (p.v < Ymin) Ymin = p.v;
        	}

        stroke->Ink_Num         = ink->num_items;
        stroke->Height          = Ymax - Ymin + 1;
        stroke->Width           = Xmax - Xmin + 1;
        stroke->Xmin            = Xmin;
        stroke->Xmax            = Xmax;
        stroke->Ymin            = Ymin;
        stroke->Ymax            = Ymax;
        AppendToList(&Strokes,(long)stroke);
    	}

							/* als */ /* moved from Analyze */
    for (i = 0; i < Strokes->num_items; i++) 
    	{
    	lpStroke stroke = Strokes->items[i] ;
    	
	    if (i>0) 
	    	{
	    	lpStroke lastStroke   = Strokes->items[i-1] ;
	    	
	        stroke->XmaxT     = MAX(stroke->Xmax,lastStroke->XmaxT);
	        stroke->XminT     = MIN(stroke->Xmin,lastStroke->XminT);
	        stroke->YmaxT     = MAX(stroke->Ymax,lastStroke->YmaxT);
	        stroke->YminT     = MIN(stroke->Ymin,lastStroke->YminT);
	        stroke->WidthT    = stroke->XmaxT - stroke->XminT + 1;
	        stroke->HeightT   = stroke->YmaxT - stroke->YminT + 1;
	    	}
	    else 
	    	{
	        stroke->XmaxT     = stroke->Xmax;
	        stroke->XminT     = stroke->Xmin;
	        stroke->YmaxT     = stroke->Ymax;
	        stroke->YminT     = stroke->Ymin;
	        stroke->WidthT    = stroke->Width;
	        stroke->HeightT   = stroke->Height;
	    	}
    }
 							/* als */ /* propagate bounding box */
       for (i = 0; i < Strokes->num_items - 1; i++) 
	    	{
	    	lpStroke stroke = Strokes->items[i] ;
	    	lpStroke finalstroke = Strokes->items[strokes->num_items-1] ;
	    	
	    	stroke->XmaxT = finalstroke->XmaxT ;
	    	stroke->XminT = finalstroke->XminT ;
	    	stroke->YmaxT = finalstroke->YmaxT ;
	    	stroke->YminT = finalstroke->YminT ;
	    	stroke->WidthT = finalstroke->WidthT ;
	    	stroke->HeightT = finalstroke->HeightT ;
	    	}
								/*als*/ /* moved down from 1st loop */
    for (i = 0; i < Strokes->num_items; i++)
    	{
    	lpStroke stroke = Strokes->items[i] ;

        Simplify(stroke,stroke->Ink,stroke->Ink_Num);
        Analyze(stroke);
    	}

}
/****************************************************************/
INT16 Process3(
    lpVHPoint P,
    lpVHPoint Q,
    INT16 num,
    INT16 xd,  INT16 yd,
    INT16 xd2, INT16 yd2)
{
    INT16 dx,dy,i,n = 0;
    P[0] = Q[0];
    for (i = 1; i < num - 1; i++) 
    {
        dx = Q[i].h - P[n].h;   dx = ABS(dx);
        dy = Q[i].v - P[n].v;   dy = ABS(dy);

        if (dx + dy < PRE_ABS)    continue;
        if (dx < xd && dy < yd)   continue;
        if (n == 0 && dx < xd2 && dy < yd2) continue;
        n++;
        P[n] = Q[i];
    }
    dx = Q[num - 1].h - P[n].h;
    dy = Q[num - 1].v - P[n].v;
    if (ABS(dx) + ABS(dy) > PRE_ABS)
        n++;
    P[n] = Q[num - 1];
    return n + 1;
}
/****************************************************************/
INT16 Dt(INT16 t1,INT16 t2)
{
    INT16 dt,a;
    dt = t2 - t1;
    a = ABS(dt);
    if         (a == 180) dt = 180;
    else if    (a >  180) 
    {
        if     (t2 > 0)   dt -= 360;
        else if(t1 > 0)   dt += 360;
    }
    return dt;
}
/****************************************************************/
private void    Simplify(lpStroke the_stroke,lpVHPoint Ink,INT16 N)
{
    INT16 w = the_stroke->Width,
	  h = the_stroke->Height;
    INT16 xd  = w / PRE_RATIO_X, xd2 = w / FIRST_RATIO_X,
          yd  = h / PRE_RATIO_Y, yd2 = h / FIRST_RATIO_Y;
    double ar = ((double)h)/w;

    if    (ar < 0.2 || ar > 5.0) 
    {
            the_stroke->P[0] = the_stroke->Ink[0];
            the_stroke->P[1] = the_stroke->Ink[the_stroke->Ink_Num - 1];
            N = 2;
    }
    else    N = Process3(the_stroke->P,Ink,N,xd,yd,xd2,yd2);
    the_stroke->N = N;
}
/****************************************************************/
private void ComputeT(lpStroke the_stroke)
{
    LPINT16   T = the_stroke->T;
    LPINT16   U = the_stroke->U;
    lpVHPoint P = the_stroke->P;
    INT16     i,N = the_stroke->N;

    for (i = 0; i < N - 1; i++)
        T[i] = MY_ATAN2(P[i+1].v-P[i].v,P[i+1].h-P[i].h);
    T[N-1] = T[N-2];
    for (i = 0; i < N - 2; i++) U[i] = Dt(T[i],T[i+1]);
}
/****************************************************************/
/* als */ /* moved material to ConvertGestureToStrokes */
private void Analyze(register lpStroke the_stroke)
{
    char  s[100];
    INT16 i;
    INT16 N      = the_stroke->N;
    INT16 Height = the_stroke->Height,
          Width  = the_stroke->Width;

    ComputeT(the_stroke);

    for (i=0;i<NUM_FEATURES;i++) 
    {
        (*HashFunctions[i])(s,the_stroke);
        ConvertStringToLong(s,&the_stroke->S[i],Bits[i]);
    }
    the_stroke->IsDot = ( (   Height < DOT_THRESHHOLD 
                           && Width < DOT_THRESHHOLD) 
                        || N == 1);
}
/****************************************************************/

#define P8m(p) \
   '0' + \
      ((3*((p).h - the_stroke->XminT) / (the_stroke->WidthT)) + \
     3*(3*((p).v - the_stroke->YminT) / (the_stroke->HeightT)))

void P8(LPSTR s,lpStroke the_stroke);
void P8(register LPSTR s,lpStroke the_stroke)
{
    register INT16 i= the_stroke->N,d,n = -1;	/* als */
    register VHPoint p,*T = the_stroke->P;
    while (i-- > 0) 
    {
        p = *T++;
        d = P8m(p);
        if (d == '4') continue;
        if (d > '4') d -= 1;
        if (n == -1 || s[n] != d)
            s[++n] = d;
    }
    s[++n] = 0;
}
/****************************************************************/
#define P4m(p) \
    '0' + ((2*((p).h - the_stroke->XminT) / (the_stroke->WidthT)) + \
         2*(2*((p).v - the_stroke->YminT) / (the_stroke->HeightT)))

#define IN_MIDDLE(p) (P8m(p) == '4')

void P4(LPSTR s,lpStroke the_stroke);
void P4(register LPSTR s,lpStroke the_stroke)
{
    register INT16 i= the_stroke->N,	/* als */
	d,
	n = -1;
    register VHPoint   p;
    register lpVHPoint T = the_stroke->P;
    while (i-- > 0) 
    {
        p = *T++;
        d = P4m(p);
        if (IN_MIDDLE(p)) continue;
        if (n == -1 || s[n] != d)
            s[++n] = d;
    }
    s[++n] = 0;
}

/****************************************************************/
/*als*/

#define P16m(p) \
   '0' + \
      ((4*((p).h - the_stroke->XminT) / (the_stroke->WidthT)) + \
     4*(4*((p).v - the_stroke->YminT) / (the_stroke->HeightT)))

void Centroid(LPSTR s,lpStroke the_stroke);
void Centroid(register LPSTR s,lpStroke the_stroke)
{
    register INT16 		N = the_stroke->N ;
    register INT16		i ;
    register VHPoint	p;
    register lpVHPoint	T = the_stroke->P;
    long 				h = 0, v = 0 ;	/* accumlator for horizontal and vertical positions */
    
    for (i = 0 ; i < N ; i++)
    {
        p = *T++;
        h += p.h; v += p.v;
    }
    p.h = h / N; p.v = v / N;
    s[0] = P16m(p);
    s[1] = 0;
}
/****************************************************************/

#define D1m(t) ('0' + ((t + 10 + 45 + 180) / 90) % 4)
#define D2m(t) ('0' + ((t + 10 + 00 + 180) / 90) % 4)
#define D3m(t) ('0' + ((t + 10 + 22 + 180) / 90) % 4)
#define D4m(t) ('0' + ((t + 10 - 22 + 180) / 90) % 4)
#define D5m(t) ('0' + ((t + 00 + 60 + 180) / 120) % 3)
#define D6m(t) ('0' + ((t + 00 + 30 + 180) / 120) % 3)
#define D7m(t) ('0' + ((t + 00 - 30 + 180) / 120) % 3)
#define D8m(t) ('0' + ((t + 10 + 22 + 180) / 45) % 8)
#define Phm(p) ('0' + (4*((p).h - the_stroke->XminT) / the_stroke->WidthT))
#define Pvm(p) ('0' + (4*((p).v - the_stroke->YminT) / the_stroke->HeightT))
#define U1m(u) ('0' + ((u + 00 + 45 + 180) / 90) % 4)

/****************************************************************/
#ifdef MSOFT_C7
#pragma warning(once:4135)
#endif

#define FOO(mNAME,mFXN,mTYPE,mARRAY,mEND,mN)    \
void mNAME(LPSTR s,lpStroke the_stroke);        \
void mNAME(LPSTR s,lpStroke the_stroke)         \
{                                               \
    register INT16 i,d,n = 0;                   \
    register mTYPE *T = the_stroke->mARRAY;     \
    s[0] = mFXN(*T++);                          \
    i = the_stroke->N - mEND;                   \
    while (i-- > 0) {                           \
        d = mFXN(*T++);                         \
        if (s[n] != d)                          \
            s[++n] = d;                         \
        }                                       \
    s[++n] = 0;                                 \
}

/****************************************************************/
FOO(Ph, Phm, VHPoint, P, 1,1)
FOO(Pv, Pvm, VHPoint, P, 1,2)
FOO(D1, D1m, INT16, T, 2,3)
FOO(D2, D2m, INT16, T, 2,4)
FOO(D3, D3m, INT16, T, 2,5)
FOO(D4, D4m, INT16, T, 2,6)
FOO(D5, D5m, INT16, T, 2,7)
FOO(D6, D6m, INT16, T, 2,8)
FOO(D7, D7m, INT16, T, 2,9)
FOO(D8, D8m, INT16, T, 2,10)
FOO(U1, U1m, INT16, U, 3,11)

/****************************************************************/

public HashFunc  HashFunctions[] = { D1,D2,D3,D4,D5,D6,D7,D8,P8,P4,Ph,Pv,U1,
										Centroid};	/* als */

/****************************************************************/
private INT16 MY_ATAN2(register INT16 dy,register INT16 dx)
{
    while (dx > 10 || dx < -10 || dy > 10 || dy < -10)
        { dx >>= 1; dy >>= 1; } /*was 2*/
    return ATAN2_TABLE[dy + 10][dx + 10];
}
/****************************************************************/
private void InitAtanTable(void)
{
    INT16 i,j;
    double r2d = 180 / 3.141592654;
    for (i = 0; i <= 10; i++)
    {
        for (j = 0; j <= 10; j++) 
	{
            INT16 t;
	    t = (i==0 && j==0)? 0 : (INT16)(r2d * atan2(i,j));
            ATAN2_TABLE[ i + 10][ j + 10] =  t;
            ATAN2_TABLE[-i + 10][ j + 10] = -t;
            ATAN2_TABLE[ i + 10][-j + 10] = 180 - t;
            ATAN2_TABLE[-i + 10][-j + 10] = t - 180;
        }
    }
}

/****************************************************************/
private void AppendToList(lpLpList list,long item)
{
    if (*list == NULL) 
    {
        *list = mem_Alloc(sizeof(INT16) + sizeof(long));
        if (!*list) mem_AllocFailed(sizeof(INT16) + sizeof(long));
        (*list)->num_items = 1;
        (*list)->items[0] = (void*)item;
    }
    else 
    {
        (*list)->num_items += 1;
        *list = mem_Realloc(*list,
		    sizeof(INT16) + (*list)->num_items*sizeof(long));
        if (!*list) mem_AllocFailed(
	    sizeof(INT16) + (*list)->num_items*sizeof(long));
        (*list)->items[(*list)->num_items-1] = (void*)item;
    }
}
/****************************************************************/
void AppendAVPair(lpLpAVList list,lpAVPair item)
{
    if (*list == NULL) 
    {
        *list = mem_Alloc(sizeof(INT16) + sizeof(AVPair));
        if (!*list) mem_AllocFailed(sizeof(INT16) + sizeof(AVPair));
        (*list)->num_items = 1;
        (*list)->items[0] = *item;
    }
    else 
    {
        (*list)->num_items += 1;
        *list = mem_Realloc(*list,
		    sizeof(INT16) + (*list)->num_items*sizeof(AVPair));
        if (!*list) mem_AllocFailed(
	        sizeof(INT16) + (*list)->num_items*sizeof(AVPair));
        (*list)->items[(*list)->num_items-1] = *item;
    }
}
/****************************************************************/
#ifdef TURBO_C
int CompareAV(void * aa,void * bb)
{
    lpAVPair a = (lpAVPair )aa;		/* als */ /*moved up */
    lpAVPair b = (lpAVPair )bb;		/* als */ /*moved up */
#endif
#ifdef MSOFT_C7
int CompareAV(void * aa,void * bb)
{
    lpAVPair a = (lpAVPair )aa;		/* als */ /*moved up */
    lpAVPair b = (lpAVPair )bb;		/* als */ /*moved up */
#endif
#ifdef LIGHTSPEED_C
int CompareAV(lpAVPair a,lpAVPair b)
{
#endif
    if (a->index > b->index) return 1;
    if (a->index < b->index) return -1;
    return 0;
}

