/*
 /  Tags file maker for GNU emacs
 /  Copyright (C) 1984, 1987, 1988
 /  Free Software Foundation, Inc. and Ken Arnold
 /
 /  adapted for djgpp/demacs by Douglas Clifton 3/12/92
 */

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <unistd.h>
#include "etags.h"

void
main(argc, argv)
    int argc;
    char **argv;
{
    char cmd[MAXCMD];
    u_int sw;
    reg int i, j;
    int fflag = 0;
    extern char *optarg;
    char **globv;
    progname = "etags";

    if (argc < 2) usage();             /* should be at least one argument */

    while ((sw = getopt(argc, argv, "BFaetuvxf:")) != EOF)  {
        switch (sw) {
            case 'B':
                searchar='?';
                eflag--;
                argc--;
                argv++;
                break;
            case 'F':
                searchar='/';
                eflag--;
                argc--;
                argv++;
                break;
            case 'a':
                aflag++;
                argc--;
                argv++;
                break;
            case 'e':
                eflag++;
                argc--;
                argv++;
                break;
            case 't':
                tflag++;
                argc--;
                argv++;
                break;
            case 'w':
                wflag++;
                argc--;
                argv++;
                break;
            case 'u':
                uflag++;
                argc--;
                argv++;
                break;
            case 'v':
                vflag++;
                xflag++;
                eflag--;
                argc--;
                argv++;
                break;
            case 'x':
                xflag++;
                eflag--;
                argc--;
                argv++;
                break;
            case 'f':
                if (fflag)  {
                    fprintf(stderr,
                            "%s: -f flag may only be given once\n",
                            progname);
                    usage();
                }
                fflag++;
                if (*optarg == '\0')    {
                    fprintf(stderr,
                            "%s: -f flag must be followed by a filename\n",
                            progname);
                    usage();
                }
                outfile = optarg;
                argc--;
                argv++;
                break;
            case '?':
                usage();
            default:
                break;
        }
    }
    if (!outfile)
        outfile = "tags";       /* msdos filenames are not case-sensitive */
    init();                             /* initialize boolean "functions" */
    initbuffer(&lb);
    initbuffer(&lb1);
    if (eflag) {             /* open alternative output file if indicated */
        outf = fopen(outfile, aflag ? "a" : "w");
        if (!outf) {
            fprintf(stderr, "%s: ", progname);
            perror(outfile);
            exit(BAD);
        }
    }
    /* glob final args and loop through files finding functions */
    for (i = 1; i < argc; i++) {
        globv = glob_filename(argv[i]);
        if (globv == NULL) {
            fprintf(stderr, "%s: %s\n", progname, "virtual memory exausted");
            exit(BAD);
        }
        else if (globv == (char **) -1)    {
            fprintf(stderr, "%s: ", progname);
            perror(argv[i]);
            exit(BAD);
        }
        else    {
            for (j = 0; globv[j] != NULL; j++)    {
                find_entries(globv[j]);
                if (eflag) {
                    fprintf(outf, "\f\n%s,%d\n", globv[j],
                            total_size_of_entries(head));
                    put_entries(head);
                    free_tree(head);
                    head = NULL;
                }
                file_num++;
                /*
                 /  NOTE: this will fail if *nix-like tools are unavailable.
                 /  ALSO NOTE: the paragraph symbols below are the command
                 /  seperators I use (with 4DOS), default is '^'. If you are
                 /  using 4DOS a simple substitution below will suffice, if
                 /  are using MKS or MS-SH a similar stratagy is in order.
                 /  If you are using COMMAND.COM (why?) you will have to
                 /  split the system command into multiple calls.
                 */
                if (uflag)  {
                    sprintf(cmd,
                        "mv %s otagsfgrep -v '\t%s\t' otags >%srm -f otags",
                        outfile, globv[j], outfile);
                    system(cmd);
                    aflag++;
                }
            }
        }
    }
    if (eflag) {
        fclose(outf);
        exit(GOOD);
    }
    if (xflag) {
        put_entries(head);
        exit(GOOD);
    }
    outf = fopen(outfile, aflag ? "a" : "w");
    if (outf == NULL) {
        fprintf(stderr, "%s: ", outfile);
        perror(outfile);
        exit(BAD);
    }
    put_entries(head);
    fclose(outf);
    if (uflag) {
        /* see NOTE above */
        sprintf(cmd, "sort %s -o %s", outfile, outfile);
        system(cmd);
    }
    exit(GOOD);
}

void
usage(void)
{
    fprintf(stderr, "\nUsage: %s [-BFaetuwvx] [-f outfile] file...\n", progname);
    _exit(BAD);
}


/*
 /  This routine sets up the boolean psuedo-functions which work by setting
 /  boolean flags dependent upon the corresponding character. Every character
 /  which is NOT in that string is not a white char.  Therefore, all of the
 /  array "_wht" is set to FALSE, and then the elements subscripted by the
 /  chars in "white" are set to TRUE.  Thus "_wht" of a char is TRUE if it
 /  is the string "white", else FALSE.
 */

void
init(void)
{
    reg char *sp;
    reg int i;

    for (i = 0; i < 0177; i++) {
        _wht[i] = _etk[i] = _itk[i] = _btk[i] = FALSE;
        _gd[i] = TRUE;
    }
    for (sp = white; *sp; sp++)
        _wht[*sp] = TRUE;
    for (sp = endtk; *sp; sp++)
        _etk[*sp] = TRUE;
    for (sp = intk; *sp; sp++)
        _itk[*sp] = TRUE;
    for (sp = begtk; *sp; sp++)
        _btk[*sp] = TRUE;
    for (sp = notgd; *sp; sp++)
        _gd[*sp] = FALSE;
    _wht[0] = _wht['\n'];
    _etk[0] = _etk['\n'];
    _btk[0] = _btk['\n'];
    _itk[0] = _itk['\n'];
    _gd[0] = _gd['\n'];
}

/*
 /  This routine opens the specified file and calls the function
 /  which finds the function and type definitions.
 */

void
find_entries(file)
    char *file;
{
    char *cp;

    if ((inf = fopen(file, "rb")) == NULL)  {
        fprintf(stderr, "%s: ", progname);
        perror(file);
        return;
    }
    curfile = savestr(file);
    cp = rindex(file, '.');
    /* .tex, .aux or .bbl implies LaTeX source */
    if (cp && (!strcmp(cp + 1, "tex")
           ||  !strcmp(cp + 1, "aux")
           ||  !strcmp(cp + 1, "bbl"))) {
        TEX_funcs(inf);
        fclose(inf);
        return;
    }
    /* .l or .el or .cl or .lsp implies lisp source */
    if (cp && (!strcmp(cp + 1, "l")
           ||  !strcmp(cp + 1, "el")
           ||  !strcmp(cp + 1, "lsp")
           ||  !strcmp(cp + 1, "cl")))  {
        L_funcs(inf);
        fclose(inf);
        return;
    }
    /* .scm or .sm or .sch implies scheme source */
    if (cp && (!strcmp(cp + 1, "sm")
           ||  !strcmp(cp + 1, "scm")
           ||  !strcmp(cp + 1, "t")
           ||  !strcmp(cp + 1, "sch")))  {
        Scheme_funcs(inf);
        fclose(inf);
        return;
    }
    /* if not a .c or .h or .y file, try fortran */
    if (cp && (cp[1] != 'c'
           ||  cp[1] != 'h'
           ||  cp[1] != 'y')
           &&  cp[2] == '\0')  {

        if (PF_funcs(inf))  {
            fclose(inf);
            return;
        }
        rewind(inf);
    }
    /* no fortran tags found, must be `C' */
    C_entries();
    fclose(inf);
}

/*
 /  Record a tag on the current line: name is the tag name, f is
 /  nonzero to use a pattern, zero to use line number instead.
 */

void
pfnote(name, f, linestart, linelen, lno, cno)
    char *name;
    char f;                         /* f == TRUE when a function */
    char *linestart;
    int  linelen;
    int  lno;
    long cno;
{
    reg char *fp;
    NODE *np;
    char *altname;
    char tem[51];

    if ((np = (NODE *) malloc(sizeof(NODE))) == NULL) {
        fprintf(stderr, "%s: too many entries to sort\n", progname);
        put_entries(head);
        free_tree(head);
        head = NULL;
        np = (NODE *) xmalloc(sizeof(NODE));
    }
    /* Change name "main" to M<thisfilename>. */
    if (!eflag && !xflag && !strcmp(name, "main")) {
        fp = rindex(curfile, '/');
        if (!fp)
            fp = curfile;
        else
            fp++;
        altname = concat("M", fp, "");
        fp = rindex(altname, '.');
        if (fp && !fp[2])
            *fp = 0;
        name = altname;
    }
    np->name = savestr(name);
    np->file = curfile;
    np->f = f;
    np->lno = lno;
    np->cno = cno;
    np->left = np->right = 0;
    if (eflag)
        linestart[linelen] = 0;
    else if (!xflag) {
        sprintf(tem, strlen(linestart) < 50 ? "%s$" : "%.50s", linestart);
        linestart = tem;
    }
    np->pat = savestr(linestart);
    if (!head)
        head = np;
    else
        add_node(np, head);
}

void
free_tree(node)
    NODE *node;
{
    while (node)    {
        free_tree(node->right);
        free(node);
        node = node->left;
    }
}

void
add_node(node, cur_node)
    NODE *node, *cur_node;
{
    reg int dif = strcmp(node->name, cur_node->name);
    /*
     /  If this tag name matches an existing one,
     /  unless -e was given, do not add the node,
     /  and possibly print a warning.
     */
    if (!eflag && !dif) {
        if (node->file == cur_node->file) {
            if (!wflag) {
                fprintf(stderr, "%s: Duplicate entry in file %s, line %d: %s\n",
                        progname, node->file, lineno, node->name);
                fprintf(stderr, "Second entry ignored\n");
            }
            return;
        }
        if (!cur_node->been_warned) {
            if (!wflag)
                fprintf(stderr,
                        "%s: Duplicate entry in files %s and %s: %s (Warning only)\n",
                        progname, node->file, cur_node->file, node->name);
        }
        cur_node->been_warned = TRUE;
        return;
    }
    /* Actually add the node */
    if (dif < 0) {
        if (cur_node->left != NULL)
            add_node(node, cur_node->left);
        else
            cur_node->left = node;
        return;
    }
    if (cur_node->right != NULL)
        add_node(node, cur_node->right);
    else
        cur_node->right = node;
}

void
put_entries(node)
    reg NODE *node;
{
    reg char *sp;

    if (!node)
        return;      /* start back */

    /* Output subentries that precede this one */
    put_entries(node->left);

    /* Output this entry */
    if (eflag)
        fprintf(outf, "%s%c%d,%d\n", node->pat, 0177, node->lno, node->cno);
    else if (!xflag) {
        fprintf(outf, "%s\t%s\t", node->name, node->file);
        if (node->f) {                                          /* a function */
            putc(searchar, outf);
            putc('^', outf);
            for (sp = node->pat; *sp; sp++) {
                if (*sp == '\\' || *sp == searchar)
                    putc('\\', outf);
                putc(*sp, outf);
            }
            putc(searchar, outf);
        }
        else                            /* a typedef; text pattern inadequate */
            fprintf(outf, "%d", node->lno);
        putc('\n', outf);
    }
    else if (vflag)
        fprintf(stdout, "%s %s %d\n",
                node->name, node->file, (node->lno + 63) / 64);
    else
        fprintf(stdout, "%-16s%4d %-16s %s\n",
            node->name, node->lno, node->file, node->pat);

    /* Output subentries that follow this one */
    put_entries(node->right);
}

/*
 /  Return total number of characters that put_entries will
 /  output for the nodes in the subtree of the specified node.
 /  Works only if eflag is set, but called only in that case.
 */

int
total_size_of_entries(node)
    reg NODE *node;
{
    reg int total = 0;
    reg long num;

    if (!node)
        return 0;               /* start back */

    /* Count subentries that precede this one */
    total = total_size_of_entries(node->left);

    /* Count subentries that follow this one */
    total += total_size_of_entries(node->right);

    /* Count this entry */
    total += strlen(node->pat) + 3;
    num = node->lno;
    while (num) {
        total++;
        num /= 10;
    }
    num = node->cno;
    if (!num) total++;
    while (num) {
        total++;
        num /= 10;
    }
    return total;
}

#define CNL_SAVE_NUMBER {              \
    linecharno = charno;               \
    lineno++;                          \
    charno += 1 + readline (&lb, inf); \
    lp = lb.buffer;                    \
}

#define CNL {       \
    CNL_SAVE_NUMBER \
    number = 0;     \
}

/*
 /  This routine finds functions and typedefs
 /  in C syntax and adds them to the list.
 */

void
C_entries(void)
{
    reg int c;
    reg char *token, *tp, *lp;
    char incomm, inquote, inchar, midtoken;
    int level;
    char tok[BUFSIZ];

    lp = lb.buffer;
    *lp = 0;
    lineno = charno = number = level = 0;
    gotone = midtoken = inquote = inchar = incomm = FALSE;
    while (!feof(inf))  {
        c = *lp++;
        if (!c) {
            CNL
            gotone = FALSE;
        }
        if (c == '\\') {
            c = *lp++;
            if (!c) CNL_SAVE_NUMBER
            c = ' ';
        }
        else if (incomm)   {
            if (c == '*')   {
                while ((c = *lp++) == '*')
                    continue;
            if (!c) CNL
            if (c == '/')
                incomm = FALSE;
            }
        }
        /*
         /  Too dumb to know about \" not being magic,
         /  but they usually occur in pairs anyway.
         */
        else if (inquote)   {
            if (c == '"')
                inquote = FALSE;
            continue;
        }
        else if (inchar)   {
            if (c == '\'')
                inchar = FALSE;
            continue;
        }
        else switch (c) {
            case '"':
                inquote = TRUE;
                continue;
            case '\'':
                inchar = TRUE;
                continue;
            case '/':
                if (*lp == '*') {
                    lp++;
                    incomm = TRUE;
                }
                continue;
            case '#':
                if (lp == lb.buffer + 1)
                    number = 1;
                continue;
            case '{':
                if (tydef == tag_ok)
                    tydef = middle;
                level++;
                continue;
            case '}':
                if (lp == lb.buffer + 1)
                    level = 0;              /* reset */
                else
                    level--;
                if (!level && tydef == middle)
                    tydef = end;
                continue;
        }
        if (!level && !inquote && !incomm && gotone == FALSE)   {
            if (midtoken)   {
                if (endtoken(c)) {
                    int  f;
                    char *buf = lb.buffer;
                    int  endpos = lp - lb.buffer;
                    char *lp1 = lp;
                    int  line = lineno;
                    long linestart = linecharno;
                    int  tem = consider_token(&lp1, token, &f, level);
                    lp = lp1;
                    if (tem) {
                        if (linestart != linecharno) {
                            getline(linestart);
                            strncpy(tok, token + (lb1.buffer - buf), tp - token + 1);
                            tok[tp - token + 1] = 0;
                            pfnote(tok, f, lb1.buffer, endpos, line, linestart);
                        }
                        else {
                            strncpy(tok, token, tp - token + 1);
                            tok[tp - token + 1] = 0;
                            pfnote(tok, f, lb.buffer, endpos, line, linestart);
                        }
                        gotone = f;     /* function */
                    }
                    midtoken = FALSE;
                    token = lp - 1;
                }
                else if (intoken(c))
                    tp++;
            }
            else if (begtoken(c)) {
                token = tp = lp - 1;
                midtoken = TRUE;
            }
        }
        if (c == ';' && tydef == end)    /* clean with typedefs */
            tydef = none;
    }
}

/*
 /  This routine  checks to see if the current token
 /  is at the start of a function, or corresponds to
 /  a typedef It updates the input line * so that
 /  the '(' will be in it when it returns.
 */

int
consider_token(lpp, token, f, level)
    char **lpp, *token;
    int *f, level;
{
    reg char *lp = *lpp;
    reg char c;
    static char next_token_is_func;
    char firsttok;                 /* T if have seen first token in ()'s */
    int bad, win;

    *f = 1;                                                /* a function */
    c = lp[-1];
    bad = FALSE;
    if (!number) {                 /* space is not allowed in macro defs */
		while (iswhite(c)) {
			c = *lp++;
            if (!c) {
                if (feof(inf))
                    break;
                CNL
			}
		}
	}
	/*
	 /	the following tries to make it so
	 /	that a #define a b(c) doesn't count
	 /	as a define of b.
	 */
    else {
        number++;
        if (number >= 4 || (number == 2 && strncmp(token, "define", 6))) {
            gotone = TRUE;
badone:
            bad = TRUE;
            goto ret;
        }
    }
	/* check for the typedef cases */
    if (tflag && istoken(token, "typedef", 7)) {
		tydef = begin;
		goto badone;
    }
    if (tydef == begin && (istoken(token, "struct", 6)
                       ||  istoken(token, "union", 5)
                       ||  istoken(token, "enum", 4)))  {
		tydef = tag_ok;
		goto badone;
    }
    if (tydef == tag_ok) {
		tydef = middle;
		goto badone;
    }
    if (tydef == begin) {               /* e.g. typedef ->int<- */
		tydef = end;
		goto badone;
    }
	if (tydef == middle && level == 0)	/* e.g. typedef struct tag ->struct_t<- */
		tydef = end;
    if (tydef == end) {
		*f = 0;
		win = 1;
		goto ret;
    }
	/* Detect GNU emacs's function-defining macros. */
    if (!number && !strncmp(token, "DEF", 3)) {
        next_token_is_func = TRUE;
		goto badone;
    }
    if (next_token_is_func) {
        next_token_is_func = FALSE;
		win = 1;
		goto ret;
    }
    if (c != '(')
		goto badone;
    firsttok = FALSE;
    while ((c = *lp++) != ')') {
        if (!c) {
			if (feof(inf))
                break;
            CNL
		}
		/*
		 /	This line used to confuse ctags: int (*oldhup)();
		 /	This fixes it. A nonwhite char before the first
		 /	token, other than a / (in case of a comment in
		 /	there) makes this not a declaration.
		 */
		if (begtoken(c) || c == '/')
			firsttok++;
		else if (!iswhite(c) && !firsttok)
			goto badone;
    }
    while (iswhite(c = *lp++)) {
        if (!c) {
            if (feof(inf))
                break;
            CNL
        }
    }
    win = isgood(c);
ret:
    *lpp = lp - 1;
    return !bad && win;
}

void
getline(atchar)
	long atchar;
{
	long saveftell = ftell(inf);

	fseek(inf, atchar, SEEK_SET);
    readline(&lb1, inf);
	fseek(inf, saveftell, SEEK_SET);
}

/* Fortran parsing */

char *dbp;
int pfcnt;

int
PF_funcs(fi)
    FILE *fi;
{
    lineno = 0;
    charno = 0;
    pfcnt = 0;

    while (!feof(fi)) {
        lineno++;
        linecharno = charno;
        charno += readline(&lb, fi) + 1;
        dbp = lb.buffer;
        if (*dbp == '%')
            dbp++;                            /* Ratfor escape to fortran */
        while (isspace(*dbp))
            dbp++;
        if (!*dbp)
            continue;
        switch (*dbp | ' ') {
            case 'i':
                if (tail("integer"))
                    takeprec();
                break;
            case 'r':
                if (tail("real"))
                    takeprec();
                break;
            case 'l':
                if (tail("logical"))
                    takeprec();
                break;
            case 'c':
                if (tail("complex") || tail("character"))
                    takeprec();
                break;
            case 'd':
                if (tail("double")) {
                    while (isspace(*dbp))
                        dbp++;
                    if (!*dbp)
                        continue;
                    if (tail("precision"))
                        break;
				continue;
			}
			break;
        }
        while (isspace(*dbp))   {
            dbp++;
            if (!*dbp)
                continue;
            switch (*dbp | ' ') {
                case 'f':
                    if (tail("function"))
                        getit();
                    continue;
                case 's':
                    if (tail("subroutine"))
                        getit();
                    continue;
                case 'p':
                    if (tail("program")) {
                        getit();
                        continue;
                    }
                    if (tail("procedure"))
                        getit();
                    continue;
            }
        }
    }
    return (pfcnt);
}

int
tail(cp)
    char *cp;
{
    reg int len = 0;

    while (*cp && (*cp & ~' ') == ((*(dbp + len)) & ~' '))
		cp++, len++;
    if (!*cp)   {
		dbp += len;
		return (1);
    }
    return (0);
}

void
takeprec(void)
{
    while (isspace(*dbp))
		dbp++;
	if (*dbp != '*')
		return;
    dbp++;
    while (isspace(*dbp))
		dbp++;
    if (!isdigit(*dbp)) {
		--dbp;				 /* force failure */
		return;
    }
	do dbp++; while (isdigit(*dbp));
}

void
getit(void)
{
    reg char *cp;
    char c;
    char nambuf[BUFSIZ];

    while (isspace(*dbp))
		dbp++;
    if (!*dbp || !isalpha(*dbp))
		return;
    for (cp = dbp + 1; *cp && (isalpha(*cp) || isdigit(*cp)); cp++)
		continue;
    c = cp[0];
    cp[0] = 0;
    strcpy(nambuf, dbp);
    cp[0] = c;
    pfnote(nambuf, TRUE, lb.buffer, cp - lb.buffer + 1, lineno, linecharno);
    pfcnt++;
}

/* lisp tag functions, just look for: (def or (DEF */

void
L_funcs(fi)
    FILE *fi;
{
	lineno = charno = pfcnt = 0;

    while (!feof(fi)) {
		lineno++;
		linecharno = charno;
		charno += readline(&lb, fi) + 1;
		dbp = lb.buffer;
		if (dbp[0] == '(' && (dbp[1] == 'D' || dbp[1] == 'd') &&
							 (dbp[2] == 'E' || dbp[2] == 'e') &&
							 (dbp[3] == 'F' || dbp[3] == 'f'))  {
            while (!isspace(*dbp)) dbp++;
			while (isspace(*dbp))  dbp++;
			L_getit();
		}
    }
}

void
L_getit(void)
{
    reg char *cp;
    char c;
    char nambuf[BUFSIZ];

    if (*dbp == 0)
		return;
    for (cp = dbp + 1; *cp && *cp != '(' && *cp != ' '; cp++)
		continue;
    c = cp[0];
    cp[0] = 0;
    strcpy(nambuf, dbp);
    cp[0] = c;
    pfnote(nambuf, TRUE, lb.buffer, cp - lb.buffer + 1, lineno, linecharno);
    pfcnt++;
}

/*
 /	Scheme tag functions, look for:
 /	(def... xyzzy or (def... (xyzzy or
 /	(def ... ((...(xyzzy ... or (set! xyzzy
 */

static void get_scheme(void);

void
Scheme_funcs(fi)
    FILE *fi;
{
    lineno = charno = pfcnt = 0;

    while (!feof(fi)) {
        lineno++;
        linecharno = charno;
        charno += readline(&lb, fi) + 1;
        dbp = lb.buffer;
        if (dbp[0] == '(' &&
           (dbp[1] == 'D' || dbp[1] == 'd') &&
           (dbp[2] == 'E' || dbp[2] == 'e') &&
           (dbp[3] == 'F' || dbp[3] == 'f'))   {
            while (!isspace(*dbp))
                dbp++;
            /* Skip over open parenthesis and white space */
            while (*dbp && (isspace(*dbp) || *dbp == '('))
                dbp++;
            get_scheme();
        }
        if (dbp[0] == '(' &&
           (dbp[1] == 'S' || dbp[1] == 's') &&
           (dbp[2] == 'E' || dbp[2] == 'e') &&
           (dbp[3] == 'T' || dbp[3] == 't') &&
           (dbp[4] == '!' || dbp[4] == '!') &&
           (isspace(dbp[5])))   {
            while (!isspace(*dbp))
                dbp++;
            /* Skip over white space */
            while (isspace(*dbp))
                dbp++;
            get_scheme();
        }
    }
}

static void
get_scheme(void)
{
    reg char *cp;
    char c;
    char nambuf[BUFSIZ];

    if (!*dbp)
        return;
    /* Go till you get to white space or a syntactic break */
    for (cp = dbp + 1; *cp && *cp != '(' && *cp != ')' && !isspace(*cp); cp++)
        continue;
    /* Null terminate the string there. */
    c = cp[0];
    cp[0] = 0;
    /* Copy the string */
    strcpy(nambuf, dbp);
    /* Unterminate the string */
    cp[0] = c;
    /* Announce the change */
    pfnote(nambuf, TRUE, lb.buffer, cp - lb.buffer + 1, lineno, linecharno);
    pfcnt++;
}

/*  Find tags in TeX and LaTeX input files.  */

/*
 /  TEX_toktab is a table of TeX control
 /  sequences that define tags. Each TEX_tabent
 /  records one such control sequence.
 */

struct TEX_tabent   {
    char *name;
    int len;
};

struct TEX_tabent *TEX_toktab = NULL;            /* Table with tag tokens */

/*
 /  Default set of control sequences to put
 /  into TEX_toktab. The value of environment
 /  variable TEXTAGS is prepended to this.
 */

static char *TEX_defenv =
":chapter:section:subsection:subsubsection:eqno:label:ref:cite:bibitem:typeout";

struct TEX_tabent *TEX_decode_env();

static char TEX_esc = '\\';
static char TEX_opgrp = '{';
static char TEX_clgrp = '}';

/*  TeX/LaTeX scanning loop. */

TEX_funcs(fi)
    FILE *fi;
{
    char *lasthit;
    lineno = charno = pfcnt = 0;

    /* Select either \ or ! as escape character.  */
    TEX_mode(fi);

    /* Initialize token table once from environment. */
    if (!TEX_toktab)
        TEX_toktab = TEX_decode_env("TEXTAGS", TEX_defenv);

    while (!feof(fi)) {
        lineno++;
        linecharno = charno;
        charno += readline(&lb, fi) + 1;
        dbp = lb.buffer;
        lasthit = dbp;
        while (!feof(fi)) {                     /* Scan each line in file */
            lineno++;
            linecharno = charno;
            charno += readline(&lb, fi) + 1;
            dbp = lb.buffer;
            lasthit = dbp;
            while (dbp = index(dbp, TEX_esc)) {
                reg int i;                 /* Look at each escape in line */
                if (!*(++dbp))
                    break;
                linecharno += dbp - lasthit;
                lasthit = dbp;
                i = TEX_Token(lasthit);
                if (0 <= i) {
                    TEX_getit(lasthit, TEX_toktab[i].len);
                    break;                       /* only save a line once */
                }
            }
        }
    }
}

#define TEX_LESC '\\'
#define TEX_SESC '!'

/*
 /  Figure out whether TeX's escape
 /  char is '\\' or '!' and set grouping
 /  chars accordingly.
 */

TEX_mode(f)
    FILE *f;
{
    reg int c;

    while ((c = getc(f)) != EOF)    {
        if (c == TEX_LESC || c == TEX_SESC)
            break;
        if (c == TEX_LESC)  {
            TEX_esc = TEX_LESC;
            TEX_opgrp = '{';
            TEX_clgrp = '}';
        }
        else {
            TEX_esc = TEX_SESC;
            TEX_opgrp = '<';
            TEX_clgrp = '>';
        }
    }
    rewind(f);
}

/*
 /  Read environment and prepend it to
 /  the default string. Build token table.
 */

struct TEX_tabent *
TEX_decode_env(evarname, defenv)
    char *evarname;
    char *defenv;
{
    reg char *env, *p;
    struct TEX_tabent *tab;
    int size, i;

    /* Append default string to environment. */
    env = (char *) getenv(evarname);
    if (!env)
        env = defenv;
    else
        env = concat(env, defenv, "");

    /* Allocate a token table */
    for (size = 1, p = env; p;)
        if ((p = index(p, ':')) && *(++p))
            size++;
    tab = (struct TEX_tabent *) xmalloc(size * sizeof(struct TEX_tabent));
    /*
     /  Unpack environment string into token table.
     /  Be careful about zero-length strings
     /  (leading ':', "::" and trailing ':')
     */
    for (i = 0; *env;) {
        p = index(env, ':');
        if (!p)              /* End of environment string. */
            p = env + strlen(env);
        if (p - env > 0) {       /* Only non-zero strings. */
            tab[i].name = savenstr(env, p - env);
            tab[i].len = strlen(tab[i].name);
            i++;
        }
        if (*p)
            env = p + 1;
        else {
            tab[i].name = NULL;      /* Mark end of table. */
            tab[i].len = 0;
            break;
        }
    }
    return tab;
}

/*
 /  Record a tag defined by a TeX command of length
 /  LEN and starting at NAME. The name being defined
 /  actually starts at (NAME + LEN + 1). But we seem
 /  to include the TeX command in the tag name.
 */

void
TEX_getit(name, len)
    char *name;
    int len;
{
    char *p = name + len;
    char nambuf[BUFSIZ];

    if (!*name)
        return;
    /* Let tag name extend to next group close (or end of line) */
    while (*p && *p != TEX_clgrp)
        p++;
    strncpy(nambuf, name, p - name);
    nambuf[p - name] = 0;
    pfnote(nambuf, TRUE, lb.buffer, strlen(lb.buffer), lineno, linecharno);
    pfcnt++;
}

/*
 /  If the text at CP matches one of the tag-defining
 /  TeX command names, return the index of that
 /  command in TEX_toktab. Otherwise return -1.
 */

/*
 /  Keep the capital `T' in `Token' for dumb
 /  truncating compilers (this distinguishes it
 /  from `TEX_toktab'.
 */

int
TEX_Token(cp)
    char *cp;
{
    reg int i;

    for (i = 0; TEX_toktab[i].len > 0; i++) {
        if (!strncmp(TEX_toktab[i].name, cp, TEX_toktab[i].len))
            return i;
    }
    return -1;
}

/*  Initialize a linebuffer for use */

#define LBSIZ 200

void
initbuffer(linebuffer)
    LB *linebuffer;
{
    linebuffer->size = LBSIZ;
    linebuffer->buffer = (char *) xmalloc(LBSIZ);
}

/*
 /  Read a line of text from `stream' into `linebuffer'.
 /  Return the length of the line.
 */

long
readline(linebuffer, stream)
    LB *linebuffer;
    reg FILE *stream;
{
    char *buffer = linebuffer->buffer;
    reg char *p = linebuffer->buffer;
    reg char *pend;
    pend = p + linebuffer->size;

    while (1) {
        int c = getc(stream);
        if (p == pend)  {
            linebuffer->size *= 2;
            buffer = (char *) xrealloc(buffer, linebuffer->size);
            p += buffer - linebuffer->buffer;
            pend = buffer + linebuffer->size;
            linebuffer->buffer = buffer;
        }
        if (c < 0 || c == '\n') {
            *p = 0;
            break;
        }
        *p++ = c;
    }
    return (p - buffer);
}

char *
savestr(cp)
    char   *cp;
{
    return savenstr(cp, strlen(cp));
}

char *
savenstr(cp, len)
    char *cp;
    int len;
{
    reg char *dp;

    dp = (char *) xmalloc(len + 1);
    strncpy(dp, cp, len);
    dp[len] = '\0';
    return dp;
}

/*  Print error message and exit. */

void
fatal(s1, s2)
    char *s1, *s2;
{
    error(s1, s2);
    exit(BAD);
}

/*
 /  Print error message. `s1' is printf
 /  control string, `s2' is arg for it.
 */

void
error(s1, s2)
    char *s1, *s2;
{
    fprintf(stderr, "%s: ", progname);
    fprintf(stderr, s1, s2);
    fprintf(stderr, "\n");
}

/*
 /  Return a newly-allocated string whose
 /  contents concatenate those of s1, s2, s3.
 */

char *
concat(s1, s2, s3)
    char *s1, *s2, *s3;
{
    int len1 = strlen(s1), len2 = strlen(s2), len3 = strlen(s3);
    char *result = (char *) xmalloc(len1 + len2 + len3 + 1);

    strcpy(result, s1);
    strcpy(result + len1, s2);
    strcpy(result + len1 + len2, s3);
    *(result + len1 + len2 + len3) = 0;
    return result;
}

/*  Like malloc but get fatal error if memory is exhausted. */

int
xmalloc(size)
    int size;
{
    int result = (int) malloc(size);

    if (!result)
        fatal("virtual memory exhausted", 0);
    return result;
}

int
xrealloc(ptr, size)
    char *ptr;
    int size;
{
    int result = (int) realloc(ptr, size);

    if (!result)
        fatal("virtual memory exhausted", 0);
    return result;
}
