(* Units and Modules Figure 1 - Rational Number Unit
   Micro Cornucopia Magazine Issue #46 *)

unit Rational;

interface

type baseType = longint;          (* shortint, integer, longint *)
     ratType = record
                 n : baseType;    (* numerator of rational number *)
                 d : baseType     (* denomenator of rational number *)
               end;

function GCD(x, y : baseType) : baseType;
function LCM(x, y : baseType) : baseType;
procedure LowestTerms(var a : ratType);
procedure ComDenom(var a, b : ratType);
procedure IncRat(var a : ratType);
procedure DecRat(var a : ratType);
procedure ZeroRat(var a : ratType);
procedure AddRat(a, b : ratType;var c : ratType);
procedure SubRat(a, b : ratType;var c : ratType);
procedure MultRat(a, b : ratType;var c : ratType);
procedure DivRat(a, b : ratType;var c : ratType);

implementation

function GCD(x, y : baseType) : baseType;

(* function GCD finds the greatest common divisor of the two numbers
   x and y by the Euclidian Algorithm. *)

var r, d : baseType;
begin
  if x < 0 then x := -1 * x;      (* set x = |x| *)
  if y < 0 then y := -1 * y;      (* set y = |y| *)
  if x < y then begin
    r := y;                  (* swap x and y if y > x    *)
    y := x;                  (* algorithm expects x >= y *)
    x := r
  end;
  if y = 0 then
    GCD := x
  else
  begin
    repeat
      d := x;             (* repeat Euclidian Algorithm until *)
      x := y;
      r := y;             (* a remainder of zero is obtained  *)
      y := d mod r;
    until y = 0;
    GCD := r              (* then set GCD to be previous remainder *)
  end
end; (* GCD *)

function LCM(x, y : baseType) : baseType;

(* function LCM finds the least common multiple of the two numbers
   x and y by determining the GCD of the two numbers and then applying
   the rule |a*b| = GCD(a,b)*LCM(a,b)    *)

var g : baseType;
begin
  LCM := abs(x*y) div GCD(x, y)
end; (* LCM *)

procedure LowestTerms(var a : ratType);
var g : baseType;
begin
  if a.n = 0 then            (* if numerator then set denominator *)
    a.d := 1                 (* to smallest positive number       *)
  else
    begin                    (* find the GCD of the numerator *)
      g := GCD(a.n, a.d);    (* and the denominator           *)
      a.n := a.n div g;      (* divide numerator by GCD       *)
      a.d := a.d div g       (* divide denominator by GCD     *)
    end
end; (* LowestTerms *)

procedure ComDenom(var a, b : ratType);     (* find a common denominator *)
var m : baseType;                           (* for a & b and adjust them *)
begin
  m := LCM(a.d, b.d);             (* find LCM of a.d and b.d *)
  a.n := a.n * (m div a.d);       (* set a.n to new value    *)
  a.d := m;                       (* set a.d to LCM of denominators *)
  b.n := b.n * (m div b.d);       (* set b.n to new value    *)
  b.d := m                        (* set b.d to LCM of denominators *)
end; (* ComDenom *)

procedure IncRat(var a : ratType);     (* increment a by 1 *)
begin
  a.n := a.n + a.d;     (* add denominator to numerator *)
  LowestTerms(a)        (* reduce a to lowest terms *)
end; (* IncRat *)

procedure DecRat(var a : ratType);     (* decrement a by 1 *)
begin
  a.n := a.n - a.d;     (* subtract numerator from denominator *)
  LowestTerms(a)        (* reduce a to lowest terms *)
end; (* DecRat *)

procedure ZeroRat(var a : ratType);    (* set a to 0 *)
begin
  a.n := 0;   (* set numerator to zero *)
  a.d := 1    (* set denominator to smallest positive number *)
end; (* ZeroRat *)

procedure AddRat(a, b : ratType;       (* add a and b and place *)
                 var c : ratType);     (* result in c, a+b = c  *)
var l, g : baseType;
begin
  LowestTerms(a);       (* reduce a to lowest terms *)
  LowestTerms(b);       (* reduce b to lowest terms *)
  ComDenom(a, b);       (* convert a & b to common denominator *)
  c.n := a.n + b.n;
  c.d := a.d;
  LowestTerms(c)        (* reduce c to lowest terms *)
end; (* AddRat *)

procedure SubRat(a, b : ratType;       (* subtract b from a and place *)
                 var c : ratType);     (* result in c, a-b = c        *)
begin
  LowestTerms(a);       (* reduce a to lowest terms *)
  LowestTerms(b);       (* reduce b to lowest terms *)
  ComDenom(a, b);       (* convert a & b to common denominator *)
  c.n := a.n - b.n;
  c.d := a.d;
  LowestTerms(c)        (* reduce c to lowest terms *)
end; (* SubRat *)

procedure MultRat(a, b : ratType;      (* multiply a and b and place *)
                  var c : ratType);    (* result in c, a*b = c       *)
begin
  LowestTerms(a);       (* reduce a to lowest terms *)
  LowestTerms(b);       (* reduce b to lowest terms *)
  c.n := a.n * b.n;     (* multiply numerators      *)
  c.d := a.d * b.d;     (* multiply denominators    *)
  LowestTerms(c)        (* reduce c to lowest terms *)
end; (* MultRat *)

procedure DivRat(a, b : ratType;       (* divide a by b and place *)
                  var c : ratType);    (* result in c, a/b = c    *)
begin
  if b.n = 0 then
    ZeroRat(c)          (* if division by 0 then c = 0 *)
  else
    begin
      LowestTerms(a);   (* reduce a to lowest terms *)
      LowestTerms(b);   (* reduce b to lowest terms *)
      c.n := a.n * b.d; (* invert b and multiply by *)
      c.d := a.d * b.n; (* a to get c               *)
      LowestTerms(c)    (* reduce c to lowest terms *)
    end
end; (* DivRat *)

begin
end.
