/*-------------------------------------------------------------*\
 |  TaskMstr.C  -- Display current Windows tasks and memory    |
 |                 usage.                                      |
\*-------------------------------------------------------------*/
#include <Windows.H>
#include <WindowsX.H>
#include <ToolHelp.H>
#include <Memory.H>
#include "TaskMstr.H"


/*-------------------------------------------------------------*\
 |                      Global Variables.                      |
\*-------------------------------------------------------------*/
APPCOLORS           ac;
APPMETRICS          am;
char                achBuffer[BUFSIZE];
char                achAppTitle[] = "Task Master";
char            *   achSegType[] =
                        {"Dyna-Segs",
                         "DGroup",
                         "Other Data",
                         "Code",
                         "Task DB",
                         "Resource",
                         "Module DB",
                         "Free",
                         "Internal",
                         "Sentinel",
                         "Burgermaster"
                        };
DWORD               dwTaskMax;
DWORD               dwSegMax;
DWORD               dwModuleTot[SEGTYPE_COUNT];
FARPROC             lpfnNotify;
GLOBALENTRY _huge * lpheList;
HBRUSH              hbr1;
HBRUSH              hbr2;
HFONT               hfont;
HICON               hIconDefault;
HINSTANCE           hInst;
HWND                hwndMain;
HWND                hwndTask;
HWND                hwndModule;
int                 cSegments;
int                 cTaskList;
int                 iCurTask = 0;
int                 nTaskFactor;
int                 nSegFactor;
LPTASKLIST          lptlList;


/*-------------------------------------------------------------*\
 |                 Main Function:  WinMain.                    |
\*-------------------------------------------------------------*/
int PASCAL WinMain (HANDLE hInstance,   HANDLE hPrevInstance,
                    LPSTR  lpszCmdLine, int    cmdShow)
    {
    HDC         hdc;
    MSG         msg;
    WNDCLASS    wc;

    if (hwndMain = FindWindow ("TaskMstr:MAIN", NULL) )
        {
        if (IsIconic (hwndMain) )
            {
            ShowWindow (hwndMain, SW_RESTORE);
            }
        BringWindowToTop (hwndMain);
        return 0;
        }

    // Save global copy of instance handle.
    hInst = hInstance;

    // Save empty icon for application's without icons.
    hIconDefault = LoadIcon (NULL, IDI_APPLICATION);

    // Initialize application metric table...
    hdc = GetDC (NULL);
    trInitMetrics(hdc);

    // ...and application color table.
    ac.cColors = GetDeviceCaps (hdc, NUMCOLORS);
    trInitColors();
    ReleaseDC (NULL, hdc);

    //  Install ToolHelp call-back notification procedure.
    //
    lpfnNotify = MakeProcInstance (trToolHelpNotify, hInst);
    NotifyRegister (NULL, lpfnNotify, NF_NORMAL);

    //  Do usual application startup.
    wc.lpszClassName = "TaskMstr:MAIN";
    wc.hInstance     = hInstance;
    wc.lpfnWndProc   = TaskMstrWndProc;
    wc.hCursor       = LoadCursor (hInstance, "hand");
    wc.hIcon         = LoadIcon (hInstance,"snapshot");
    wc.lpszMenuName  = NULL;
    wc.hbrBackground = COLOR_WINDOW+1;
    wc.style         = NULL;
    wc.cbClsExtra    = 0;
    wc.cbWndExtra    = 0;

    RegisterClass( &wc);

    hwndMain = CreateWindow("TaskMstr:MAIN",      /* Class name.   */
                            achAppTitle,          /* Title.        */
                            WS_OVERLAPPEDWINDOW,  /* Style bits.   */
                            CW_USEDEFAULT,        /* x - default.  */
                            0,                    /* y - default.  */
                            am.win_cxInit,        /* cx - default. */
                            am.win_cyInit,        /* cy - default. */
                            NULL,                 /* No parent.    */
                            NULL,                 /* Class menu.   */
                            hInstance,            /* Creator.      */
                            NULL);                /* Params.       */

    ShowWindow (hwndMain, cmdShow);

    // Create 'peek at system data' timer.
    SetTimer (hwndMain, 1, 5000, 0);

    while (GetMessage(&msg, 0, 0, 0))
        {
        TranslateMessage(&msg);       /*  Keyboard input.      */
        DispatchMessage(&msg);
        }

    //  Clean up GDI brushes.
    DeleteBrush (hbr1);
    DeleteBrush (hbr2);
    DeleteFont (hfont);

    //  Detach ToolHelp call-back notification.
    //
    NotifyUnRegister (NULL);
    FreeProcInstance (lpfnNotify);

    return 0;
    }

/*-------------------------------------------------------------*\
 |             Window Procedure:  TaskMstrWndProc.             |
\*-------------------------------------------------------------*/
long FAR PASCAL TaskMstrWndProc (HWND   hwnd,   UINT wMsg,
                                 WPARAM wParam, LPARAM lParam)
    {
    switch (wMsg)
        {
        case WM_COMMAND:
            HANDLE_WM_COMMAND(hwnd,wParam, lParam,
                              trMain_OnCommand);
            break;

        case WM_CREATE:
            return ( HANDLE_WM_CREATE(hwnd, wParam,lParam,
                                      trMain_OnCreate) );
            break;

        case WM_DESTROY:
            HANDLE_WM_DESTROY(hwnd,wParam,lParam,
                              trMain_OnDestroy);
            break;

        case WM_DRAWITEM:
            HANDLE_WM_DRAWITEM(hwnd,wParam,lParam,
                               trMain_OnDrawItem);
            break;

        case WM_GETMINMAXINFO:
            HANDLE_WM_GETMINMAXINFO(hwnd,wParam,lParam,
                                    trMain_GetMinMaxInfo);
            break;

        case WM_MEASUREITEM:
            HANDLE_WM_MEASUREITEM(hwnd,wParam,lParam,
                                  trMain_MeasureItem);
            break;

        case WM_SIZE:
            HANDLE_WM_SIZE(hwnd,wParam,lParam,trMain_Size);
            break;

        case WM_SETFOCUS:
            HANDLE_WM_SETFOCUS(hwnd,wParam,lParam,
                               trMain_SetFocus);
            break;

        case WM_TIMER:
            HANDLE_WM_TIMER(hwnd,wParam,lParam,trMain_Timer);
            break;

        case AM_BUILDTASKLIST:
            trMain_BuildTaskList();
            break;

        case AM_ERROR:
            trMain_Error(hwnd, wParam);
            break;

        default:
            return(DefWindowProc(hwnd,wMsg,wParam,lParam));
            break;
        }
    return 0L;
    }


/*-------------------------------------------------------------*\
 |                 TaskMstrWndProc  WM_COMMAND.                |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trMain_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify)

    {
    if (        id == IDL_TASK &&
        codeNotify == LBN_SELCHANGE)
        {
        iCurTask = ListBox_GetCurSel(hwndTask);
        trCalcSegmentList();
        }
    }


/*-------------------------------------------------------------*\
 |                 TaskMstrWndProc  WM_CREATE.                 |
\*-------------------------------------------------------------*/
BOOL NEAR PASCAL
trMain_OnCreate(HWND hwnd, CREATESTRUCT FAR* lpCreateStruct)

    {
    // Create a listbox to act as a task list window.
    hwndTask = CreateWindowEx(WS_EX_NOPARENTNOTIFY,
                              "listbox",            /* Class name.   */
                              "",                   /* Title.        */
                              WS_CHILD           |  /* Style bits.   */
                              WS_VISIBLE         |
                              WS_HSCROLL         |
                              WS_BORDER          |
                              LBS_USETABSTOPS    |
                              LBS_OWNERDRAWFIXED |
                              LBS_NOTIFY         |
                              LBS_NOINTEGRALHEIGHT,
                              0,                    /* x.            */
                              0,                    /* y.            */
                              0,                    /* cx.           */
                              0,                    /* cy.           */
                              hwnd,                 /* Parent window.*/
                              IDL_TASK,             /* Child Id.     */
                              hInst,                /* Creator.      */
                              NULL);                /* Params.       */

    // Create a listbox to act as a task list window.
    hwndModule = CreateWindowEx(WS_EX_NOPARENTNOTIFY,
                                "listbox",            /* Class name.   */
                                "",                   /* Title.        */
                                WS_CHILD           |  /* Style bits.   */
                                WS_VISIBLE         |
                                WS_HSCROLL         |
                                WS_BORDER          |
                                LBS_USETABSTOPS    |
                                LBS_OWNERDRAWFIXED |
                                LBS_NOINTEGRALHEIGHT,
                                0,                    /* x.            */
                                0,                    /* y.            */
                                0,                    /* cx.           */
                                0,                    /* cy.           */
                                hwnd,                 /* Parent window.*/
                                IDL_MODULE,           /* Child Id.     */
                                hInst,                /* Creator.      */
                                NULL);                /* Params.       */

    // Create empty segment list.
    //
    lpheList = (GLOBALENTRY _huge *)
               GlobalAllocPtr(GMEM_MOVEABLE,
                              SEG_INIT_COUNT * sizeof(GLOBALENTRY));

    if (!lpheList)
        {
        SendMessage (hwnd, AM_ERROR, ERR_OUTOFMEMORY, 0);
        return FALSE;
        }

    // Fill in dummy module data list.
    trFillSegmentListbox();

    // Create segment list, module list & task list.
    //
    trBuildSegmentList();
    trBuildTaskList();

    hwndMain = hwnd;
    trCalcTaskList();
    trFillTaskListbox(hwndTask);

    trCalcSegmentList();

    // Kick start initial display.
    PostMessage (hwndMain, WM_TIMER, 0, 0L);
    ListBox_SetCurSel (hwndTask, 0);
    ListBox_SetTopIndex (hwndTask, 0);

    return TRUE;
    }


/*-------------------------------------------------------------*\
 |                 TaskMstrWndProc  WM_DESTROY.                |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trMain_OnDestroy (HWND hwnd)
    {
    GlobalFreePtr(lpheList);  // Free segment list.
    GlobalFreePtr(lptlList);  // Free task list.
    PostQuitMessage(0);
    }

/*-------------------------------------------------------------*\
 |                TaskMstrWndProc  WM_DRAWITEM.                |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trMain_OnDrawItem(HWND hwnd, const DRAWITEMSTRUCT FAR* lpDrawItem)
    {

    if (lpDrawItem->CtlID == IDL_TASK)
        trDrawTaskItems (lpDrawItem);
    else
        trDrawModuleItems (lpDrawItem);
    }


/*-------------------------------------------------------------*\
 |              TaskMstrWndProc  WM_GETMINMAXINFO.             |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trMain_GetMinMaxInfo(HWND hwnd, MINMAXINFO FAR* lpMinMaxInfo)
    {
    lpMinMaxInfo->ptMinTrackSize.x = am.win_xMinTracking;
    lpMinMaxInfo->ptMinTrackSize.y = am.win_yMinTracking;
    }


/*-------------------------------------------------------------*\
 |               TaskMstrWndProc  WM_MEASUREITEM.              |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trMain_MeasureItem(HWND hwnd, MEASUREITEMSTRUCT FAR* lpMeasureItem)
    {
    if (lpMeasureItem->CtlID == IDL_TASK)
        lpMeasureItem->itemHeight = am.lbTask_cyHeight;
    else
        lpMeasureItem->itemHeight = am.lbModule_cyHeight;
    }


/*-------------------------------------------------------------*\
 |                  TaskMstrWndProc  WM_SIZE.                  |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trMain_Size(HWND hwnd, UINT state, int cx, int cy)
    {
    int xChild;
    int yChild;
    int cxChild;
    int cyChild;

    xChild  = 0 - am.sm_cxBorder;
    yChild  = 0 - am.sm_cyBorder;
    cxChild = cx / 2 + am.sm_cxBorder;
    cyChild = cy + 2 * am.sm_cyBorder;

    // Put task listbox on the left half...
    //
    MoveWindow (hwndTask,
                xChild,
                yChild,
                cxChild,
                cyChild,
                TRUE);

    // ...and module listbox on the right half.
               + (2 * am.sm_cxBorder);
               + (2 * am.sm_cyBorder);
    xChild  = cxChild - (2 * am.sm_cxBorder);
    cxChild += 2 * am.sm_cxBorder;
    MoveWindow (hwndModule,
                xChild,
                yChild,
                cxChild,
                cyChild,
                TRUE);

    InvalidateRect (hwndTask, NULL, TRUE);
    InvalidateRect (hwndModule, NULL, TRUE);
    }


/*-------------------------------------------------------------*\
 |                TaskMstrWndProc  WM_SETFOCUS.                |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trMain_SetFocus(HWND hwnd, HWND hwndOldFocus)
    {
    SetFocus (hwndTask);
    }


/*-------------------------------------------------------------*\
 |                  TaskMstrWndProc  WM_TIMER.                 |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trMain_Timer(HWND hwnd, UINT id)
    {
    trBuildSegmentList();

    trCalcSegmentList();
    trCalcTaskList();
    }


/*-------------------------------------------------------------*\
 |              TaskMstrWndProc  AM_BUILDTASKLIST.             |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trMain_BuildTaskList()
    {
    trBuildSegmentList();
    trBuildTaskList();

    trCalcTaskList();
    trFillTaskListbox(hwndTask);

    trCalcSegmentList();

    InvalidateRect (hwndTask, NULL, FALSE);
    }


/*-------------------------------------------------------------*\
 |                  TaskMstrWndProc  AM_ERROR.                 |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trMain_Error(HWND hwnd, WORD error)
    {
    int  cb;

Retry:
    cb = LoadString (hInst, STR_ERRBASE+error,
                     achBuffer, BUFSIZE);
    if (cb == 0)
        {
        if (error != ERR_MISSING)
            {
            error = ERR_MISSING;
            goto Retry;
            }
        }

    MessageBox (hwnd, achBuffer, achAppTitle, MB_OK);
    }


/*-------------------------------------------------------------*\
 |      trDrawModuleItems: Draw module info in list box.       |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trDrawModuleItems (const DRAWITEMSTRUCT FAR*  lpdraw)
    {
    DWORD        rgb;
    HDC          hdc;
    int          cbText;
    int          xText;
    int          xWidth;
    int          cxBlock;
    int          cxPixels;
    int          cxMax;
    int          yText;
    int          iSaveDC;
    int          Y1;
    RECT         rClient;
    RECT         rDraw;
    WORD         wType;

    // Make a local copy of DRAWITEMSTRUCT data.
    hdc      = lpdraw->hDC;
    rDraw    = lpdraw->rcItem;
    wType    = (WORD)lpdraw->itemData;

    if ((lpdraw->itemAction & ODA_SELECT) ||
        (lpdraw->itemAction & ODA_DRAWENTIRE))
        {
        // Push DC data on DC-stack.
        iSaveDC = SaveDC (hdc);

        // Set up DC.
        SetTextColor (hdc, ac.rgbBlack);
        SetBkColor (hdc, ac.rgbForeSelect);
        SetTextAlign (hdc, TA_RIGHT | TA_BASELINE);
        SelectObject (hdc, hfont);

        xText = rDraw.left + am.tx_xModuleText;
        yText = rDraw.top  + am.tx_yModuleText;

        cbText = wsprintf (achBuffer, "%s: ",
                           (LPSTR)achSegType[wType]);

        rDraw.right = am.tx_xModuleText;
        ExtTextOut (hdc, xText, yText, ETO_OPAQUE | ETO_CLIPPED,
                    &rDraw, achBuffer, cbText, 0);

        // Get ready to draw object size & percent bar.
        SetTextAlign (hdc, TA_LEFT | TA_BASELINE);

        // Set up buffer with object size info.
        cbText = wsprintf (achBuffer, "  %ld bytes",
                           dwModuleTot[wType]);

        // Calculate scaled segment size.
        cxBlock  = (int) (dwModuleTot[wType]/ nSegFactor);
        cxPixels = (int) lpdraw->rcItem.right -
                         am.tx_xModuleText + 1;
        cxMax    = (int) (dwSegMax/ nSegFactor);
        xWidth = MulDiv (cxBlock, cxPixels, cxMax);

        // Draw object size & percent bar.
        SetTextColor (hdc, ac.rgbForeSelect);
        rgb = ac.rgbSolidColors[wType];
        SetBkColor (hdc, rgb);
        rDraw.left  = rDraw.right;
        rDraw.right = rDraw.right + xWidth;
        ExtTextOut (hdc, xText, yText, ETO_OPAQUE | ETO_CLIPPED,
                    &rDraw, achBuffer, cbText, 0);

        // White out remaining percent bar.
        rgb = ac.rgbSolidColors[wType];
        SetTextColor (hdc, rgb);
        SetBkColor (hdc, ac.rgbForeSelect);
        rDraw.left  = rDraw.right;
        rDraw.right = lpdraw->rcItem.right;
        ExtTextOut (hdc, xText, yText, ETO_OPAQUE  | ETO_CLIPPED,
                    &rDraw, achBuffer, cbText, 0);

        // Pop DC data from DC-stack.
        RestoreDC (hdc, iSaveDC);

        // Draw bottom border of item -- only if NOT at bottom
        // of client area.
        GetClientRect (hwndModule, &rClient);
        Y1 = rDraw.bottom - am.sm_cyBorder;
        if (Y1 != rClient.bottom - am.sm_cyBorder)
            {
            MoveTo (hdc, lpdraw->rcItem.left,    Y1);
            LineTo (hdc, lpdraw->rcItem.right+1, Y1);
            }
        }
    }


/*-------------------------------------------------------------*\
 |         trDrawTaskItems: Draw task info in list box.        |
\*-------------------------------------------------------------*/
VOID NEAR PASCAL
trDrawTaskItems (const DRAWITEMSTRUCT FAR*  lpdraw)
    {
    DWORD  dw;
    HDC    hdc;
    int    cbText;
    int    xText;
    int    yText;
    int    iText;
    int    X1;
    int    Y1;
    int    X2;
    int    Y2;
    int    iSaveDC;
    POINT  pt;
    RECT   rClient;
    RECT   rDraw;

    hdc      = lpdraw->hDC;
    rDraw    = lpdraw->rcItem;
    X1       = rDraw.left + am.xIcon;
    Y1       = rDraw.top  + am.yIcon;
    xText    = rDraw.left + am.tx_xMargin;
    yText    = rDraw.top  + am.tx_yMargin;
    iText    = (int)lpdraw->itemData;

    if ((lpdraw->itemAction & ODA_SELECT) ||
        (lpdraw->itemAction & ODA_DRAWENTIRE))
        {

        // Set up DC
        iSaveDC = SaveDC (hdc);
        SelectObject (hdc, hfont);
        if (lpdraw->itemState & ODS_SELECTED)
            {
            SetTextColor (hdc, ac.rgbForeSelect);
            SetBkColor (hdc, ac.rgbBackSelect);
            }

        // Display Task name.
        cbText = wsprintf (achBuffer, "%2d. %s", iText + 1,
                           lptlList[iText].achWindowText);
        ExtTextOut (hdc, xText, yText, ETO_OPAQUE, &rDraw,
                    achBuffer, cbText, 0);

        // Draw icon associated with task's top-level window.
        DrawIcon (hdc, X1, Y1, lptlList[iText].hIcon);

        // Draw labels for segment bars.
        SetTextAlign (hdc, TA_RIGHT | TA_TOP);
        X1 = am.tx_xTaskText;
        Y1 = yText + am.tx_cyLine +
             (am.bar_cyHeight/2) - (am.tx_cyLine/2);
        cbText = wsprintf (achBuffer, "%s: ",
                           (LPSTR)achSegType[GT_UNKNOWN]);
        TextOut (hdc, X1, Y1, achBuffer, cbText);
        Y1 += am.bar_cyHeight;
        cbText = wsprintf (achBuffer, "%s: ",
                           (LPSTR)achSegType[GT_DGROUP]);
        TextOut (hdc, X1, Y1, achBuffer, cbText);

        // Draw bars for segment sizes.
        SetMapMode (hdc, MM_ANISOTROPIC);
        X1 = rDraw.left + am.tx_xTaskText;
        Y1 = rDraw.top  + am.tx_yTaskText;

        SetViewportOrg (hdc, X1, Y1);
        SetWindowExt   (hdc, (int)(dwTaskMax/ nTaskFactor), 1);
        SetViewportExt (hdc, rDraw.right - X1, 1);

        X1 = 0;
        Y1 = 0;

        // Draw Dyna-Segs Rectangle.
        dw = lptlList[iText].dwGlobal;
        dw = dw / nTaskFactor;
        X2 = (int)dw;
        Y2 = Y1 + am.bar_cyHeight + 1;
        SelectObject (hdc, hbr1);
        Rectangle (hdc, X1, Y1, X2, Y2);

        // Calculate maximum range of Rectangle
        pt.x = rDraw.right;
        pt.y = rDraw.bottom;
        DPtoLP (hdc, &pt, 1);

        // Draw DGroup Rectangle
        Y1 += am.bar_cyHeight;
        X2 = X1 + ((int)(lptlList[iText].dwDGroup / nTaskFactor) );
        Y2 = Y1 + am.bar_cyHeight + 1;
        Y2 = min (Y2, pt.y);
        SelectObject (hdc, hbr2);
        Rectangle (hdc, X1, Y1, X2, Y2);

        // Draw size of each memory area.
        SetBkMode (hdc, TRANSPARENT);
        SetTextAlign (hdc, TA_LEFT | TA_TOP);

        X1 = am.tx_xTaskText + am.tx_xMargin;
        Y1 = yText + am.tx_cyLine +
             (am.bar_cyHeight/2) - (am.tx_cyLine/2);
        SetViewportOrg (hdc, X1, Y1);

        X1 = 0;
        Y1 = 0;

        cbText = wsprintf (achBuffer, "%lu bytes",
                 lptlList[iText].dwGlobal);
        TextOut (hdc, X1, Y1, achBuffer, cbText);

        Y1 += am.bar_cyHeight;
        cbText = wsprintf (achBuffer, "%lu bytes",
                           lptlList[iText].dwDGroup);
        TextOut (hdc, X1, Y1, achBuffer, cbText);

        // Restore initial state of DC.
        RestoreDC (hdc, iSaveDC);

        // Draw bottom border of item -- only if NOT at bottom
        // of client area.
        GetClientRect (hwndTask, &rClient);
        Y1 = rDraw.bottom - am.sm_cyBorder;
        if (Y1 != rClient.bottom - am.sm_cyBorder)
            {
            MoveTo (hdc, rDraw.left,    Y1);
            LineTo (hdc, rDraw.right+1, Y1);
            }
        }


    if (lpdraw->itemAction & ODA_FOCUS)
        {
        rDraw.left   += am.sm_cxBorder;
        rDraw.top    += am.sm_cyBorder;
        rDraw.right  -= am.sm_cxBorder;
        rDraw.bottom -= am.sm_cyBorder * 2;
        DrawFocusRect (hdc, &rDraw);
        }
    }


/*-------------------------------------------------------------*\
 |      trCalcSegmentList: Calculate tasks segment sizes.      |
\*-------------------------------------------------------------*/
void NEAR PASCAL
trCalcSegmentList()
    {
    DWORD               dwModuleTotTemp[SEGTYPE_COUNT];
    DWORD               dwSegMaxTemp;
    GLOBALENTRY _huge * lpheTemp;
    GLOBALENTRY         globalentry;
    HMODULE             hModule;
    HTASK               hTask;
    int                 iSegment;
    int                 iSegType;
    RECT                rRepaint;
    WORD                wType;

    hModule = lptlList[iCurTask].taskentry.hModule;
    hTask   = lptlList[iCurTask].taskentry.hTask;

    // Init temporary total buffer to zero.
    _fmemset (&dwModuleTotTemp[0], 0, sizeof(DWORD) * SEGTYPE_COUNT);

    // Loop through entire heap, totalling for each type.
    for (iSegment = 0, lpheTemp = lpheList;
         iSegment < cSegments;
         iSegment++, lpheTemp++)
        {
        // Make a local copy of segment info values.
        _fmemcpy (&globalentry, lpheTemp, sizeof (GLOBALENTRY) );

        // Check that segment is 'one of ours.'
        if ( (globalentry.hOwner == hModule)     ||
             (globalentry.hOwner == hTask  ) )
            {

            // If segment type is in range, add it up.
            wType = globalentry.wType;
            if ((wType >= GT_UNKNOWN) &&
                (wType <= GT_BURGERMASTER) )
                {
                dwModuleTotTemp[wType] += globalentry.dwBlockSize;
                }

            } /* [if one of ours] */

        } /* [for iSegment] */


    // Check maximum value & check for changed values.
    dwSegMaxTemp = 0L;

    // Loop through type totals...
    for (iSegType = 0; iSegType < SEGTYPE_COUNT; iSegType++)
        {

        // ...if count has changed, generate a WM_PAINT.
        if (dwModuleTot[iSegType] != dwModuleTotTemp[iSegType] )
            {
            dwModuleTot[iSegType] = dwModuleTotTemp[iSegType];

            // Generate a WM_PAINT in listbox.
            ListBox_GetItemRect(hwndModule, iSegType, &rRepaint);
            rRepaint.left  += am.tx_xModuleText;
            InvalidateRect (hwndModule, &rRepaint, FALSE);
            }

        // ... also check for max.
        dwSegMaxTemp = max (dwModuleTot[iSegType], dwSegMaxTemp);

        }

    // If max has changed, repaint entire listbox.
    if (dwSegMax != dwSegMaxTemp)
        {
        dwSegMax = dwSegMaxTemp;
        GetClientRect (hwndModule, &rRepaint);
        rRepaint.left = am.tx_xModuleText;
        InvalidateRect (hwndModule, &rRepaint, FALSE);
        }

    // Calc segment scaling to fit GDI's 16-bit coordinates.
    //
    nSegFactor = 1;
    while ( (dwSegMax/ nSegFactor) > 0x8000L)
        {
        nSegFactor *= 2;
        }
    }


/*-------------------------------------------------------------*\
 |        trCalcTaskList: Calculate task segments sizes.       |
\*-------------------------------------------------------------*/
void NEAR PASCAL
trCalcTaskList()
    {
    char                achWindowText[BUFSIZE];
    DWORD               dwDGroupTemp;
    DWORD               dwGlobalTemp;
    DWORD               dwTaskMaxTemp;
    GLOBALENTRY _huge * lpheTemp;
    HICON               hIconTemp;
    HWND                hwndChild;
    HWND                hwndSearch;
    register HTASK      hTask;
    HINSTANCE           hwndInst;
    int                 iTask;
    int                 iSegment;
    LPTASKLIST          lptlTemp;
    MODULEENTRY         moduleentry;
    RECT                rRepaint;
    WORD                cbText;

    dwTaskMaxTemp = dwTaskMax;
    dwTaskMax = 0L;

    for (iTask = 0, lptlTemp = lptlList;
         iTask < cTaskList;
         iTask++, lptlTemp++)
        {
        // Get first window in top-level window list.
        //
        hwndSearch = GetWindow (hwndMain, GW_HWNDFIRST);

        cbText = 1;
        achWindowText[0] = '\0';
        hIconTemp = 0;
        while (hwndSearch)
            {
            hwndInst = GetWindowWord (hwndSearch, GWW_HINSTANCE);
            if (hwndInst == lptlTemp->taskentry.hInst)
                {
                hwndChild = hwndSearch;
                while (hwndChild = GetParent (hwndChild))
                    hwndSearch = hwndChild;

                cbText = FORWARD_WM_GETTEXT(hwndSearch, BUFSIZE,
                         achWindowText, SendMessage,) + 1;
                hIconTemp = GetClassWord (hwndSearch, GCW_HICON);
                hwndSearch = (HWND)0;
                }
            else
                {
                hwndSearch = GetWindow (hwndSearch, GW_HWNDNEXT);
                }
            }

        //  Save handle to icon.
        lptlTemp->hIcon = (hIconTemp) ? hIconTemp : hIconDefault;

        //  Copy window text into task table.
        if (achWindowText[0] != '\0')
            _fmemcpy (lptlTemp->achWindowText, achWindowText, cbText);
        else
            {
            moduleentry.dwSize = sizeof (MODULEENTRY);
            ModuleFindHandle(&moduleentry, lptlTemp->taskentry.hModule);
            cbText = lstrlen (&moduleentry.szModule[0]) + 1;
            _fmemcpy (lptlTemp->achWindowText,
                      &moduleentry.szModule[0], cbText);
            }

        // Fetch size of DGroup segment.
        //
        dwDGroupTemp = GlobalSize (lptlTemp->taskentry.hInst);

        // Loop through segment table, adding up memory
        // allocated from the global heap.
        //
        dwGlobalTemp = 0L;
        hTask = lptlTemp->taskentry.hTask;
        for (iSegment = 0, lpheTemp = lpheList;
             iSegment < cSegments;
             iSegment++, lpheTemp++)
            {
            if (hTask == lpheTemp->hOwner)
                {
                if (lpheTemp->wType == GT_UNKNOWN)
                    {
                    dwGlobalTemp += lpheTemp->dwBlockSize;
                    }
                }

            } /* [for iSegment] */

        // If DGroup size has changed or
        // if Global size has changed then generate WM_PAINT.
        //
        if (lptlTemp->dwDGroup != dwDGroupTemp ||
            lptlTemp->dwGlobal != dwGlobalTemp )
            {
            // Update values in task list.
            lptlTemp->dwDGroup = dwDGroupTemp;
            lptlTemp->dwGlobal = dwGlobalTemp;

            // Generate a WM_PAINT in listbox.
            ListBox_GetItemRect(hwndTask, iTask, &rRepaint);
            rRepaint.left  += am.tx_xTaskText;
            rRepaint.right += am.tx_yTaskText;
            InvalidateRect (hwndTask, &rRepaint, FALSE);
            }

        // Calculate maximum sizes, to allow proper scaling
        // of task segment graph.
        dwTaskMax = max (dwDGroupTemp, dwTaskMax);
        dwTaskMax = max (dwGlobalTemp, dwTaskMax);

        } /* [for iTask] */

    // If there have been a change in list box scaling factor,
    // repaint entire list box.
    if (dwTaskMaxTemp != dwTaskMax)
        {
        GetClientRect (hwndTask, &rRepaint);
        rRepaint.left = am.tx_xTaskText;
        InvalidateRect (hwndTask, &rRepaint, FALSE);
        }

    // Scale memory figures to fit in GDI's 16-bit coordinates.
    //
    nTaskFactor = 1;
    while ( (dwTaskMax/ nTaskFactor) > 0x8000L)
        {
        nTaskFactor *= 2;
        }
    }


/*-------------------------------------------------------------*\
 |            trFillTaskListbox:  Fill task lisbox.            |
\*-------------------------------------------------------------*/
void NEAR PASCAL
trFillTaskListbox(HWND hwndTask)
    {
    int  iCurSel;
    int  iTask;

    // Get index of currently selected item.
    iCurSel = ListBox_GetCurSel(hwndTask);

    // Purge listbox.
    ListBox_ResetContent(hwndTask);

    // Fill listbox with task information
    for (iTask = 0;
         iTask < cTaskList;
         iTask++)
        {
        //  Add item to listbox.
        ListBox_AddItemData (hwndTask, iTask);
        } /* [for iTask] */

    // Make sure selection stays in range.
    iCurSel = min (iCurSel, cTaskList - 1);
    iCurSel = max (iCurSel, 0);

    // Set selection to last item selected.
    ListBox_SetCurSel(hwndTask, iCurSel);

    // Reset current task index.
    iCurTask = iCurSel;
    }


/*-------------------------------------------------------------*\
 |         trFillSegmentListbox:  Fill module listbox.          |
\*-------------------------------------------------------------*/
void NEAR PASCAL
trFillSegmentListbox(void)
    {
    int iSegType;

    for (iSegType = 0; iSegType < SEGTYPE_COUNT; iSegType++)
        {
        //  Add item to listbox.
        ListBox_AddItemData (hwndModule, iSegType);
        }
    }


/*-------------------------------------------------------------*\
 |    trToolHelpNotify:  Watch for tasks starting/stopping.    |
\*-------------------------------------------------------------*/
BOOL FAR PASCAL _export
trToolHelpNotify (WORD wID, DWORD dwData)
    {
    if (wID == NFY_STARTTASK ||
        wID == NFY_EXITTASK)
        {
        PostMessage (hwndMain, AM_BUILDTASKLIST, 0, 0L);
        }

    return FALSE;
    }
