/****************************************************************************

    PROGRAM: ALLOC.C

    PURPOSE: Contains memory allocation/deallocation routines.

    NOTE:    All memory is allocated in global fixed memory.

    FUNCTIONS:
            NcbAlloc    - Returns a pointer to an NCB which resides
                          in global memory and is zero-initialized
                          as well locked in memory.

            NcbAllocBuf - Returns a pointer to a buffer which resides
                          in global memory and is zero-initialized
                          as well locked in memory.
            NcbFree     - Frees a specifies buffer from global (locked) memory

    History:
                January, 1992       Alok Sinha      Created


****************************************************************************/

// Include files
#include "windows.h"
#include "ncb.h"
#include "alloc.h"

/****************************************************************************

    STRUCTURE: BLOCK

    PURPOSE:  Any allocated memory is allocated as a block which hides
              the handle to itself within the structure. This can
              therefore be used to unlock and free the block at
              a later time.

****************************************************************************/

typedef struct _Block
{
    HANDLE hMem;
    BYTE   bMem;
} BLOCK;

typedef BLOCK FAR   *PBLOCK;
#define HANDLE_SIZE sizeof(HANDLE)

/****************************************************************************

    FUNCTION: NcbAlloc

    PURPOSE: Returns a pointer to an NCB which resides
             in global memory. It is zero-initialized as well
             as locked in memory. The number of NCB's to allocate
             is specified in 'iNosNcb'

****************************************************************************/


PNCB   NcbAlloc   ( int iNosNcb )
{
    if (iNosNcb <= 0)
        return (PNCB) NULL;

    return (PNCB) NcbAllocBuf ( (DWORD) sizeof (NCB) * iNosNcb );
}

/****************************************************************************

    FUNCTION: NcbAllocBuf

    PURPOSE: Returns a pointer to a buffer which resides
             in global memory. It is zero-initialized as well
             The size of the buffer is specified in 'wDataLen'.
****************************************************************************/

LPVOID  NcbAllocBuf( DWORD wDataLen )
{
    HANDLE hMem;
    LPSTR  lpMem;
    PBLOCK pbMem;

    // Allocate global memory
    hMem =  GlobalAlloc( GMEM_MOVEABLE | GMEM_ZEROINIT,
                         (DWORD) (wDataLen + HANDLE_SIZE)
                       );
    // Return NULL on error
    if (hMem == NULL)
        return NULL;

    // Now, lock the memory. Return NULL on error
    lpMem = GlobalLock(hMem);
    if (lpMem == NULL)
    {
        GlobalFree ( hMem);
        return NULL;
    }
    // Now, we are going to save the handle to the memory right in the
    // memory block and return a pointer to first memory byte past handle

    pbMem = (PBLOCK) lpMem;
    pbMem->hMem = hMem;
    return (LPVOID) &(pbMem->bMem);

}

/****************************************************************************

    FUNCTION: NcbFree

    PURPOSE: Frees a piece of memory which was previously locked
             and allocated in global memory. The memory to free
             is specified in 'pvBuffer'.

    CAVEAT : No check is made to see if this memory was truely
             allocated by NcbAlloc() or NcbAllocBuf().

****************************************************************************/


VOID   NcbFree(LPVOID pvBuffer)
{
    PBLOCK pbMem;

    // Check for NULL buffer.
    if (pvBuffer==NULL)
        return;

    // Get a block pointer from the passed buffer
    pbMem = (PBLOCK)  ((LPSTR) pvBuffer - HANDLE_SIZE);
    if (pbMem==NULL)
        return;

    // Now unlock and free the memory
    if ( GlobalUnlock( pbMem->hMem ) == TRUE)
    {
        GlobalFree( pbMem->hMem );
    }

    return;
}
