/****************************************************************************
    PROGRAM: WNETBIOS.C

    PURPOSE: Exports NCB submitting routines and NCB post processing
             routines
    FUNCTIONS:
               NetBiosPostMessage - Allows PostMessage Style
                                    submission of NCBs and their completion
               LibMain            - Called from Libentry() routine at
                                    DLL initialize time
               WEP                - Dll exit point called by Windows
    History:
                January, 1992       Ray Patch      Created
****************************************************************************/


// Include file
#include <windows.h>
#include "ncb.h"
#include "wnetbios.h"

//... Default number of NCB's for DOS LAN Manager.

#define DEFAULT_MAX_NCBS 12

typedef struct ENTRY
{
    PNCB     pNcb;
    HWND     hWnd;
    unsigned iMessage;
} ENTRY;

ENTRY NcbTable[DEFAULT_MAX_NCBS];
WORD  TableEntriesUsed = 0;

//... NCB table pointer used in the post routine.

ENTRY _far *NcbTablePtr = NcbTable;

//... Our private post routine.

extern DWORD PrivPostRoutine;

// The following function can be used in lieu of
//  PrivPostRoutine

extern void  cdecl interrupt far PostRoutine(
			    unsigned int es, unsigned int ds,
			    unsigned int di, unsigned int si,
			    unsigned int bp, unsigned int sp,
			    unsigned int bx, unsigned int dx,
			    unsigned int cx, unsigned int ax,
			    unsigned int ip, unsigned int cs,
			    unsigned int flags);



//===========================================================================
//  FUNCTION: LibMain(HANDLE, WORD, WORD, LPSTR)
//
//  PURPOSE : Is called by LibEntry. LibEntry is called by Windows when
//	      the DLL is loaded.
//============================================================================

int FAR PASCAL LibMain(hModule, wDataSeg, cbHeapSize, lpszCmdLine)
HANDLE	hModule;
WORD    wDataSeg;
WORD    cbHeapSize;
LPSTR   lpszCmdLine;
{
    register LPBYTE p;

    //... zero the entire NCB table.

    for(p = (LPBYTE) NcbTable; p != (LPBYTE) &NcbTable[DEFAULT_MAX_NCBS]; ++p)
    {
        *p = 0;
    }

    return 1;
}

//============================================================================
//  FUNCTION:  WEP(int)
//
//  PURPOSE:  Performs cleanup tasks when the DLL is unloaded.
//============================================================================

int FAR PASCAL WEP(int bSystemExit)
{
    return 1;
}

//============================================================================
//  FUNCTION:  NetBiosPostMessage()
//
//  PURPOSE:  Posts a message to the parent window when an NCB completes.
//============================================================================

int  _far _pascal NetBiosPostMessage(HWND hWnd, unsigned iMessage,
				     WORD wParam, PNCB pNcb)
{
    int err;

    //... if the call is asynchronous and the call turned on the NOTIFY_IF_ASYNC
    //... bit the we have to store the information in the table before calling
    //... NetBIOS.
	
    if	((pNcb->ncb_command & ASYNCH) && (wParam & NOTIFY_IF_ASYNC))
    {
        ENTRY _far *p;

        if ( TableEntriesUsed == DEFAULT_MAX_NCBS )
        {
            return -1;      //... Out of table entries (i.e. NCBs).
        }

        //... If we are here then we know we have a free
        //... table entry, all we have to do is find it.

        for(p = NcbTable; p->pNcb != (PNCB) NULL; p++) ;

        //... p = the first free entry in the table. Now we can save the
        //... information in the table and call NetBIOS.

        p->pNcb = pNcb;
        p->hWnd = hWnd;
        p->iMessage = iMessage;

        //... we need to use our post routine.

        pNcb->ncb_post  = (DWORD) PostRoutine;
        /*
         * User's can either use PrivPostRoutine (in post.asm)
         * or PostRoutine (in wnetbios.c) for post routine
         * Both perform the same function although assembly
         * routine is more efficient. In order to use
         * asm routine, use
         *       pNcb->ncb_post = PrivPostRoutine;
         *
         */

        if ((err = NetBiosRequest(pNcb)) != (int) 0xFF)
        {
            //... if the asynchronous call completed immediately the we have
            //... to free our table entry and do the PostMessage() call here
            //... since our post routine will get called by NetBIOS.

            p->pNcb = 0L;

            PostMessage(hWnd, iMessage, pNcb->ncb_command, (DWORD) pNcb);
        }
        else
        {
            ++TableEntriesUsed;
        }

        return err;
    }

    //... if we are here then we are taking care of a
    //... synchronous call or the call was asynchronous
    //... but the caller does not what a message posted.

    err = NetBiosRequest( pNcb);


    if (wParam & NOTIFY_IF_SYNC)
    {
        PostMessage(hWnd, iMessage, pNcb->ncb_command, (DWORD) pNcb);
    }

    return err;
}

//============================================================================
//  FUNCTION:  PostRoutine
//
//  PURPOSE:  Post Routine for Asynchronous NCBs. This can be used in lieu
//            of assembly routine to acheive the same result.
//
//  History:
//              January, 1992       Alok Sinha      Created
//============================================================================


void  cdecl interrupt far  PostRoutine(
				 es,	  ds,
				 di,	  si,
				 bp,	  sp,
				 bx,	  dx,
				 cx,	  ax,
				 ip,	  cs,
				 flags)
unsigned es, ds, di, si, bp, sp, bx, dx, cx, ax, ip, cs, flags;
{
    PNCB pncb;
    WORD i;

    // ES:BX contains pointer to just completed NCB
    pncb = (PNCB) MAKELONG ( bx, es);

    // Loop thru Entry table until we find an NCB pointer matching
    // NCB pointer in ES:BX.
    for ( i = 0 ; i < TableEntriesUsed; i++)
        if ( NcbTablePtr[i].pNcb == pncb)
        {
            // Found a match. Remove the entry from table and
            // post a message to the originating window
            TableEntriesUsed--;
            NcbTablePtr[i].pNcb = (PNCB) NULL;
            PostMessage ( NcbTablePtr[i].hWnd,
                          NcbTablePtr[i].iMessage,
                          pncb->ncb_command,
                          (DWORD) pncb
                        );
            break;
        }
}
