/*****************************************************************************
Module name: VerShow.C
Programmer : Jeffrey M. Richter.
Description: Version Resource Viewer.
*****************************************************************************/

#include <Windows.h>
#include <Windowsx.h>
#include <CommDlg.h>
#include <Ver.h>
#include <String.h>
#include "VerShow.h"

#define ARRAY_LEN(Array)	(sizeof(Array) / sizeof(Array[0]))

// Application global variables.
extern const HINSTANCE cdecl _hInstance;
char _szAppName[]  = "VerShow";
void VerShow_StringFileInfo (HWND hDlg, UINT uTranslationNum);

//****************************************************************************
void VerShow_ClearFields (HWND hDlg, LPCSTR szFile) {
	SetDlgItemText(hDlg, ID_PATHNAME,
		(szFile == NULL) ? "No file selected" : szFile);
	SetDlgItemText(hDlg, ID_FIXEDFILEINFODATA,
		(szFile == NULL) ? "": "No version information exists");
	(void) ComboBox_ResetContent(GetDlgItem(hDlg, ID_TRANSLATIONS));
	(void) ListBox_ResetContent(GetDlgItem(hDlg, ID_STRINGS));
}


char _szUnknown[] = "Unknown";

char *_szFileTypes[] = {
	_szUnknown,						// 0-VFT_UNKNOWN
	"Application",					// 1-VFT_APP
	"Dynamic-link library",		// 2-VFT_DLL
	"Device driver",				// 3-VFT_DRV
	"Font",							// 4-VFT_FONT
	"Virtual device",				// 5-VFT_VXD
	_szUnknown,						// 6-Unknown
	"Static library"				// 7-VFT_STATIC_LIB
};

char *_szDriverSubtypes[] = {
	_szUnknown,						// 0-VFT2_UNKNOWN
	"Printer",						// 1-VFT2_DRV_PRINTER
	"Keyboard",						// 2-VFT2_DRV_KEYBOARD
	"Language",						// 3-VFT2_DRV_LANGUAGE
	"Display",						// 4-VFT2_DRV_DISPLAY
	"Mouse",							// 5-VFT2_DRV_MOUSE
	"Network",						// 6-VFT2_DRV_NETWORK
	"System",						// 7-VFT2_DRV_SYSTEM
	"Installable",					// 8-VFT2_DRV_INSTALLABLE
	"Sound",							// 9-VFT2_DRV_SOUND
	"Comm"							// A-VFT2_DRV_COMM
};

char *_szFontSubtypes[] = {
	_szUnknown,						// 0-VFT2_UNKNOWN
	"Raster",						// 1-VFT2_FONT_RASTER
	"Vector",						// 2-VFT2_FONT_VECTOR
	"TrueType"						// 3-VFT2_FONT_TRUETYPE
};

void CreateFixedVerInfoString (LPSTR szBuf, 
	VS_FIXEDFILEINFO FAR *lpVS_FixedFileInfo) {

	DWORD dwTemp;
	LPSTR p;

	wsprintf(szBuf,
		"0x%08lX\n"			// Signature
		"0x%08lX\n"			// Structure version
		"%d.%d.%d.%d\n"	// File version
		"%d.%d.%d.%d\n"	// Product version
		"0x%08lX\n",		// File flags mask
		lpVS_FixedFileInfo->dwSignature,
		lpVS_FixedFileInfo->dwStrucVersion,

		HIWORD(lpVS_FixedFileInfo->dwFileVersionMS),
		LOWORD(lpVS_FixedFileInfo->dwFileVersionMS),
		HIWORD(lpVS_FixedFileInfo->dwFileVersionLS),
		LOWORD(lpVS_FixedFileInfo->dwFileVersionLS),
		
		HIWORD(lpVS_FixedFileInfo->dwProductVersionMS),
		LOWORD(lpVS_FixedFileInfo->dwProductVersionMS),
		HIWORD(lpVS_FixedFileInfo->dwProductVersionLS),
		LOWORD(lpVS_FixedFileInfo->dwProductVersionLS),
		lpVS_FixedFileInfo->dwFileFlagsMask);

	// Add the string equiavlent of the file flags.
	dwTemp = lpVS_FixedFileInfo->dwFileFlags;
	if (dwTemp == 0)						lstrcat(szBuf, "No file flags");
	if (dwTemp & VS_FF_DEBUG)			lstrcat(szBuf, "Debug ");
	if (dwTemp & VS_FF_PRERELEASE)	lstrcat(szBuf, "Prerelease ");
	if (dwTemp & VS_FF_PATCHED)		lstrcat(szBuf, "Patched ");
	if (dwTemp & VS_FF_PRIVATEBUILD)	lstrcat(szBuf, "PrivateBuild ");
	if (dwTemp & VS_FF_INFOINFERRED)	lstrcat(szBuf, "InfoInferred ");
	if (dwTemp & VS_FF_SPECIALBUILD)	lstrcat(szBuf, "SpecialBuild ");
	lstrcat(szBuf, "\n");

	// Add the string equiavlent of the file OS.
	dwTemp = lpVS_FixedFileInfo->dwFileOS;
	switch (LOWORD(dwTemp)) {
		case VOS__WINDOWS16:	p = "Windows (16-bit)";	break;
		case VOS__PM16:		p = "PM (16-bit)";		break;
		case VOS__PM32:		p = "PM (32-bit)";		break;
		case VOS__WINDOWS32:	p = "Windows (32-bit)";	break;
		default:					p = _szUnknown;				break;
	}
	lstrcat(szBuf, p); lstrcat(szBuf, " running under ");
	switch (HIWORD(dwTemp)) {
		case HIWORD(VOS_DOS):	p = "DOS";				break;
		case HIWORD(VOS_OS216):	p = "OS/2 (16-bit)";	break;
		case HIWORD(VOS_OS232):	p = "OS/2 (32-bit)";	break;
		case HIWORD(VOS_NT):		p = "Windows-NT";		break;
		default:						p = _szUnknown;		break;
	}
	lstrcat(szBuf, p); lstrcat(szBuf, "\n");


	// Add the string equiavlent of the file type.
	lstrcat(szBuf, _szFileTypes[(UINT) lpVS_FixedFileInfo->dwFileType]);
	lstrcat(szBuf, "\n");

	// Add the string equiavlent of the file subtype.
	switch (lpVS_FixedFileInfo->dwFileType) {
		case VFT_DRV:	
			p = _szDriverSubtypes[(UINT) lpVS_FixedFileInfo->dwFileSubtype];
			break;

		case VFT_FONT:
			p = _szFontSubtypes[(UINT) lpVS_FixedFileInfo->dwFileSubtype];
			break;

		case VFT_VXD:
			wsprintf(_fstrchr(szBuf, 0), "%ld", 
				(LPSTR) lpVS_FixedFileInfo->dwFileSubtype);
			p = NULL;
			break;

		default:
			p = NULL;
			break;
	}
	if (p != NULL) lstrcat(szBuf, p);
	lstrcat(szBuf, "\n");
}




void VerShow_FillFields (HWND hDlg, LPCSTR szFile) {
	DWORD dwVerInfoSize, dwHandle, FAR *lpdwTranslation;
	HGLOBAL hGlbl; 
	VOID FAR *lpVerInfo; 
	UINT uLen, x;
	char szBuf[500];
	VS_FIXEDFILEINFO FAR *lpVS_FixedFileInfo;

	// Clear all of the fields.
	VerShow_ClearFields(hDlg, szFile);

	// Calculate the size of the version control resource information.
	dwVerInfoSize = GetFileVersionInfoSize(szFile, &dwHandle);
	if (dwVerInfoSize == 0)	// No resource information exists.
		return;	

	// Allocate a memory block large enough to hold the version information.
	hGlbl = GlobalAlloc(GMEM_MOVEABLE, dwVerInfoSize);
	if (hGlbl == NULL) {
		MessageBox(hDlg, _szAppName, "Insufficient memory.", MB_OK);
		return;
	}

	// Load the version information into memory.
	lpVerInfo = GlobalLock(hGlbl);
	GetFileVersionInfo(szFile, dwHandle, dwVerInfoSize, lpVerInfo);

	// Get the address to the VS_FIXEDFILEINFO structure data.
	VerQueryValue(lpVerInfo, "\\",
		(VOID FAR * FAR *) &lpVS_FixedFileInfo, &uLen);

	// Create the string containing the VS_FIXEDFILEINFO data information.
	CreateFixedVerInfoString(szBuf, lpVS_FixedFileInfo);
	
	// Place the version info string in the dialog's STATIC control.
	SetDlgItemText(hDlg, ID_FIXEDFILEINFODATA, szBuf);

	// ************************************************************************
	// Load the "Translations" combo box.
	VerQueryValue(lpVerInfo, "\\VarFileInfo\\Translation",
		(VOID FAR * FAR *) &lpdwTranslation, &uLen);

	for (x = 0; x < uLen; x += 4, lpdwTranslation++) {
		// Get the string name for the language number.
		VerLanguageName(LOWORD(*lpdwTranslation), szBuf, sizeof(szBuf));
		lstrcat(szBuf, " -- ");

		// Get the string name for the code page number.
		LoadString(_hInstance, HIWORD(*lpdwTranslation), _fstrchr(szBuf, 0), 
			(UINT) (sizeof(szBuf) - (_fstrchr(szBuf, 0) - szBuf)));

		// Add the string to the combo box.
		(void) ComboBox_AddString(GetDlgItem(hDlg, ID_TRANSLATIONS), szBuf);
	}

	// Select the first translation entry as the default.
	(void) ComboBox_SetCurSel(GetDlgItem(hDlg, ID_TRANSLATIONS), 0);

	// Show the version info strings for the default translation entry.
	VerShow_StringFileInfo(hDlg, 0);

	// Free the version information because it is no longer needed.
	GlobalUnlock(hGlbl);
	GlobalFree(hGlbl);
}




char *_szStringNames[] = {
	"Comments",
	"CompanyName",
	"FileDescription",
	"FileVersion",
	"InternalName",
	"LegalCopyright",
	"LegalTrademarks",
	"OriginalFilename",
	"PrivateBuild",
	"ProductName",
	"ProductVersion",
	"SpecialBuild"
};

void VerShow_StringFileInfo (HWND hDlg, UINT uTranslationNum) {
	char szFile[128], szBuf[200]; 
	DWORD dwHandle, FAR *lpdwTranslation, dwVerInfoSize;
	HGLOBAL hGlbl;
	UINT uLen, uMaxFieldLen, x;
	VOID FAR *lpVerInfo;
	LPSTR szStringData;
	HDC hDC;
	HFONT hFont;
	RECT rc;
	HWND hWndLB = GetDlgItem(hDlg, ID_STRINGS);

	GetDlgItemText(hDlg, ID_PATHNAME, szFile, sizeof(szFile));

	dwVerInfoSize = GetFileVersionInfoSize(szFile, &dwHandle);
	if (dwVerInfoSize == 0)		// No resource information exists.
		return;

	// Allocate a memory block large enough to hold the version information.
	hGlbl = GlobalAlloc(GMEM_MOVEABLE, dwVerInfoSize);
	if (hGlbl == NULL) {
		MessageBox(hDlg, _szAppName, "Insufficient memory.", MB_OK);
		return;
	}

	// Load the version information into memory.
	lpVerInfo = GlobalLock(hGlbl);
	GetFileVersionInfo(szFile, dwHandle, dwVerInfoSize, lpVerInfo);

	VerQueryValue(lpVerInfo, "\\VarFileInfo\\Translation",
		(VOID FAR * FAR *) &lpdwTranslation, &uLen);
	lpdwTranslation += uTranslationNum;

	hDC = GetDC(hDlg);
	hFont = GetWindowFont(hDlg);
	if (hFont != NULL) SelectObject(hDC, hFont);
	uMaxFieldLen = 0;

	SetWindowRedraw(hWndLB, FALSE);
	(void) ListBox_ResetContent(hWndLB);

	for (x = 0; x < ARRAY_LEN(_szStringNames); x++) {
		uMaxFieldLen = max(uMaxFieldLen, LOWORD(
			GetTextExtent(hDC, _szStringNames[x], lstrlen(_szStringNames[x]))));

		wsprintf(szBuf,
			"\\StringFileInfo\\%04x%04x\\%s",
			LOWORD(*lpdwTranslation), HIWORD(*lpdwTranslation),
			(LPSTR) _szStringNames[x]);

		if (0 == VerQueryValue(lpVerInfo, szBuf,
			(VOID FAR * FAR *) &szStringData, &uLen)) continue;

		if (uLen == 0) continue;	// This string doesn't exist.

		wsprintf(szBuf, "%s\t%s", (LPSTR) _szStringNames[x],
			(LPSTR) szStringData);
		(void) ListBox_AddString(hWndLB, szBuf);
	}

	// Select the first string by default.
	(void) ListBox_SetCurSel(hWndLB, 0);

   // Convert pixels into dialog box units.
   SetRect(&rc, 4, 8, 4, 8);
   MapDialogRect(hDlg, &rc);
   x = 6 + ((uMaxFieldLen * 4) / rc.left);

   // Set tabstop position in listbox.  Note: listbox must
   // have LBS_USETABSTOPS style in dialog box template.
   (void) ListBox_SetTabStops(hWndLB, 1, &x);
   ListBox_SetHorizontalExtent(hWndLB, GetSystemMetrics(SM_CXSCREEN) * 2);
	ReleaseDC(hDlg, hDC);

	// Force the listbox to be repainted.
	SetWindowRedraw(hWndLB, TRUE);
	InvalidateRect(hWndLB, NULL, TRUE);

	// Free the version information because it is no longer needed.
	GlobalUnlock(hGlbl);
	GlobalFree(hGlbl);
}


#pragma argsused
BOOL VerShow_OnInitDialog (HWND hwnd, HWND hwndFocus, LPARAM lParam) {

	SetDlgItemText(hwnd, ID_FIXEDFILEINFOFIELDS,
		"Signature:\n"
		"Structure version:\n"
		"File version:\n"
		"Product version:\n"
		"File flags mask:\n"
		"File flags:\n"
		"File OS:\n"
		"File type:\n"
		"File subtype:\n"
		"File date:\n");

	VerShow_ClearFields(hwnd, NULL);
	return(TRUE);
}



void VerShow_OnCommand (HWND hwnd, int id, HWND hwndCtl, UINT codeNotify) {
	OPENFILENAME ofn;
	char szPathname[128];

	switch (id) {

      case ID_SELECTFILE:
			_fmemset(&ofn, 0, sizeof(ofn));
			ofn.lStructSize = sizeof(ofn);
			ofn.hwndOwner = hwnd;
			ofn.lpstrFilter =
				"Executables\0*.EXE\0"
				"DLLs\0*.DLL\0"
				"Device drivers\0*.DRV\0"
				"Fonts\0*.FON\0"
				"Virtual devices\0*.386\0"
				"Static libraries\0*.LIB\0";
			ofn.lpstrFile = szPathname; ofn.lpstrFile[0] = 0;
			ofn.nMaxFile = sizeof(szPathname);
			ofn.Flags = OFN_HIDEREADONLY | OFN_FILEMUSTEXIST;
			if (GetOpenFileName(&ofn)) {
				VerShow_FillFields(hwnd, ofn.lpstrFile);
			} else {
				VerShow_ClearFields(hwnd, NULL);
			}
			break;

		case ID_TRANSLATIONS:
			if (codeNotify != CBN_SELCHANGE) 
				break;
			VerShow_StringFileInfo(hwnd, ComboBox_GetCurSel(hwndCtl));
			break;

      case IDCANCEL:
         EndDialog(hwnd, 0);
         break;
	}
}


#pragma argsused
BOOL VerShow_OnEraseBkgnd (HWND hwnd, HDC hdc) {
	return(IsMinimized(hwnd) ? 1 : 0);
}


void VerShow_OnPaint (HWND hwnd) {
	PAINTSTRUCT ps;
	BeginPaint(hwnd, &ps);
	if (IsMinimized(hwnd))
		DrawIcon(ps.hdc, 0, 0,
			LoadIcon(_hInstance, MAKEINTRESOURCE(ICON_VerShow)));
	EndPaint(hwnd, &ps);
}


BOOL CALLBACK VerShowDlgProc (HWND hDlg, UINT uMsg, WPARAM wParam, 
	LPARAM lParam) {
	LRESULT lResult = 0;

   switch (uMsg) {

		case WM_INITDIALOG:
			lResult = HANDLE_WM_INITDIALOG(hDlg, wParam, lParam,
				VerShow_OnInitDialog);
			break;

		case WM_COMMAND:
			lResult = HANDLE_WM_COMMAND(hDlg, wParam, lParam,
				VerShow_OnCommand);
			break;

		case WM_ERASEBKGND:
			lResult = HANDLE_WM_ERASEBKGND(hDlg, wParam, lParam,
				VerShow_OnEraseBkgnd);
			break;

		case WM_PAINT:
			(void) HANDLE_WM_PAINT(hDlg, wParam, lParam, VerShow_OnPaint);
			break;
   }
   return((BOOL) lResult);
}

#pragma argsused
int WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
	LPSTR lpszCmdLine, int nCmdShow) {

	DialogBox(hInstance, MAKEINTRESOURCE(DLG_VERSHOW),
		NULL, (DLGPROC) VerShowDlgProc);
	return(0);
}
