/*
   Module: comio.h
   Date:   3/9/92
   Version: 1.0b
   Author:  Dave Lutz
   Email:   lutz@psych.rochester.edu
   Copyright: 1992 University of Rochester, Psychology Dept.

   Disclaimer:  This software is distributed free of charge.  As such, it
                comes with ABSOLUTELY NO WARRANTY.  The user of the software
                assumes ALL RISKS associated with its use.

                Your rights to modify and/or distribute this software are
                outlined in the file SEND7221.DOC.

   Purpose: This module provides the codes and function prototypes needed to
            use the comio input/output functions.  The comio functions use
            dos interrupts to initiate i/o operations on the com port of an
            IBM compatible PC.

            The COMPORT structure is also defined here.  This structure
            is used by all of the comio functions.
*/

#include <time.h>
#include <dos.h>

#define COMPORT struct serial_port

COMPORT {
   union REGS inregs,
              outregs;
   time_t timeout;
};

/* pass these codes to any function that requires a com port designator */
#define COM1      0
#define COM2      1
#define COM3      2
#define COM4      3

/*  The following codes are used to set: number of data bits, number of stop
    bits, parity, and baud rate.  You should select one of each, combine them
    with a bitwise or (|), and pass them to any function that requires a
    configuration paramter.

    For example, the configuration for a serial port with 7 data bits, 1 stop
    bit, even parity, and 2400 baud would be specified as follows:

       (DATA_7 | STOP_1 | PAR_E | BAUD_2400)
*/

/* number of data bits: 7 bits, 8 bits, don't care */
#define DATA_7     0x02
#define DATA_8     0x03
#define DATA_D     0x00

/* number of stop bits */
#define STOP_1     0x00
#define STOP_2     0x04

/* parity: none, odd, even, don't care */
#define PAR_N      0x00
#define PAR_O      0x08
#define PAR_E      0x18
#define PAR_D      0x10

/* baud rate */
#define BAUD_110   0x00
#define BAUD_150   0x20
#define BAUD_300   0x40
#define BAUD_600   0x60
#define BAUD_1200  0x80
#define BAUD_2400  0xA0
#define BAUD_4800  0xC0
#define BAUD_9600  0xE0



/*
   The following codes can be used to determine the Line Control Status and
   Modem Control Status of the COMPORT.  Combine them with the returned
   status value with a bitwise and (&), and if the result is non-zero that 
   flag has been set.

   Line Control Statys flags begin with LCS.
   Modem Control Status flags begin with MCS.
*/

#define LCS_RDR  0x0100    /* received data ready */
#define LCS_OVRN 0x0200    /* data overrun */
#define LCS_PAR  0x0400    /* parity error */
#define LCS_FRM  0x0800    /* framing error - incorrect start/stop bit */
#define LCS_BRK  0x1000    /* break detect */
#define LCS_TRH  0x2000    /* Transmitter hold empty - ready for new send */
#define LCS_TRS  0x4000    /* Transmitter shift empty - not transmitting */
#define LCS_TIM  0x8000    /* Timeout error */

#define MCS_CSCHNG 0x0001  /* Clear To Send (CTS) changed state */
#define MCS_DRCHNG 0x0002  /* Data Set Ready (DSR) changed state */
#define MCS_ENDRNG 0x0004  /* End of ringing pulse detector */
#define MCS_CDCHNG 0x0008  /* Carrier Detect (CD) changed state */
#define MCS_CTS    0x0010  /* Clear To Send status */
#define MCS_DSR    0x0020  /* Data Set Ready status */
#define MCS_RING   0x0040  /* Ringing indicator */
#define MCS_CD     0x0080  /* Carrier Detect Status */



/*
   Function: cominit
   Purpose:  set up a COMPORT for subsequent io operations

   Pre: cpp is a pointer to a COMPORT data structure.
        portdes is one of the port designators defined in comio.h
        config is the desired configuration for the com port, and was
        obtained by combining the configuration constants defined in
        comio.h with a bitwise or.
        timeout is the number of seconds to wait for the first character
        during an input operation.

   Post: The portdes, config, and timeout paramaters are transferred to the 
         cpp structure.
         A dos interrupt is called to open and configure the desired
         serial port.
         The bitmapped status of the port is returned.

         The status of the port can be determined by comparing it to the
         LCS_* and MCS_* values defined in comio.h.  For example:

                (MCS_DSRSTAT & retval) 
         
         will be nonzero if Modem Control Status indicates Data Set Ready.
*/
unsigned cominit (COMPORT *cpp, int portdes, unsigned config, time_t timeout);



/*
   Function: comin
   Purpose:  read a block of characters from the serial port.

   Pre: cpp is a pointer to a COMPORT structure.
        cpp has been prepared via a call to cominit.
        Nobody has been mucking around with the internals of the
        cpp structure...
        buff is a pointer to storage for the bytes read from the
        port.  There is enough storage for max characters.
        max is the maximum number of chars to read.
        term is a character which, when read, signals the end of
        the input stream.
        If there is no real term character, term = -1.

   Post:  Characters are read from the serial port and stored in buff
          until: max chars have been read, the term char is read, or an
          I/O error occurs.
          The function will wait the predefined (by cominit) number of 
          seconds for the first character to appear.
          After the first character appears, the remaining chars are
          expected to follow immediately.  If they don't, the function
          will exit.
          The number of characters actually read is returned.
          The serial port should be checked with comlstat to determine
          if an error has occurred.
*/
int comin (COMPORT *cpp, char *buff, int max, int term);



/*
   Function: comout
   Purpose:  Send a block of characters through the serial port.

   Pre: cpp is a pointer to a COMPORT structure.
        cpp has been prepared via a call to cominit.
        buff is a pointer to the block of chars to be transmitted.
        buffcnt is the number of chars to send.

   Post: An attempt is made to send the block of chars through the
         serial port.
         The number of chars actually sent is returned.
         If all bytes were not sent, it should be possible to determine
         the reason by making a call to comlstat.
*/
int comout (COMPORT *cpp, char *buff, int buffcnt);



/*
   Function: comstat
   Purpose:  Report the current status of the com port.

   Pre: cpp is a pointer to a COMPORT structure.
        cpp has been set up via a call to comopen.
        Nobody has been mucking around with the internals of the
        cpp structure...

   Post:  A DOS interrupt is called to check the status of the port
          specified in *cpp.
          The status of the port is returned as a bitmapped value.
          This value can be interpreted by comparing it to the LCS_*
          and MCS_* masks defined in comio.h
*/
unsigned comstat (COMPORT *cpp);


/*
   Function: comlstat
   Purpose:  Report the status of the com port as it was after the last
             call to one of the other comio functions.

   Pre: cpp is a pointer to a COMPORT stucture.
        cpp has been set up via a call to comopen.
        Nobody has been mucking around with the internals of the
        cpp structure...

   Post: The last status of the com port is returned as a bitmapped
         value.
         This value can be interpreted by comparing it to the LCS_*
         and MCS_* masks defined in comio.h
*/
unsigned comlstat (COMPORT *cpp);
