/*  Copyright (C) 1993   Marc Stern  (internet: stern@mble.philips.be)  */

/*
 * File         : file.c
 *
 * Functions    : filencopy
 *		  Filencopy
 *                filetrunc
 *		  Filetrunc
 *		  Filecat
 *
 * Description  : File management functions.
 *
 * Decisions    :
 *
 */


#include "strings.h"
#include "tools.h"
#include <malloc.h>
#include <io.h>
#include <fcntl.h>
#include <sys/types.h>  /* because used by stat.h in Microsoft */
#include <sys/stat.h>
#include <stdlib.h>
#include <string.h>


#define BUFSIZE         1024L 	/*  Buffer size : to be optimized        */
				/*  Very context dependant !!            */
				/*  Under Sun 3.0 : 1 page = 8192 bytes  */
				/*  Under MS-DOS  : 1 page =  512 bytes  */

#define PERMISSION	0666	/*  RW for owner, R for everybody else  */



/***
 *  Function    :  filencopy
 *
 *  Description :  Copy n characters from a file on another.
 *
 *  Parameters  :  in   int            target    target file descriptor
 *                 in   int            source    source file descriptor
 *                 in   unsigned long  length    number of bytes to copy
 *
 *  Precond     :  The two files must be opened.
 *
 *  Decisions   :  The copy can stop when eof is reached.
 *
 *  Warning     :  no rewind is performed on either file!!!
 *
 *  Return      :   0 if OK
 *                 errno if set
 *                 -1 otherwise
 *
 *  OS/Compiler :  All
 ***/

int filencopy( int source, int target, unsigned long length )

{ int nread, bufsize;
  char *buffer;

  /* Build buffer as big as possible */
  for ( bufsize = min(0x7FFF, length); bufsize >= 128; bufsize /= 2 )
      if ( buffer = (char *) malloc(bufsize) ) break;
  if ( bufsize < 128 ) return -1;

  while ( nread = read(source, buffer, min(length, bufsize)) )
        {
          if ( nread == -1 || nread != write(target, buffer, nread) )
             {
               free( buffer );
               if ( errno ) return errno;
                       else return -1;
             }

          if ( (nread == length) || eof(source) ) break;

          length -= nread;
        }

  free( buffer );
  return 0;
}

/***
 *  Function    :  Filencopy
 *
 *  Description :  Copy n characters from a file on another.
 *
 *  Parameters  :  in   char *         source    source filename
 *                 in   char *         target    target filename
 *                 in   unsigned long  length    number of bytes to copy
 *
 *  Decisions   :  The copy can stop when eof is reached.
 *
 *  Return      :  0 if OK
 *                 errno if set
 *                 -1 otherwise
 *
 *  OS/Compiler :  All
 ***/

int Filencopy( const char *source, const char *target, unsigned long length )

{ int targetfile, sourcefile;
  int error;
  char *ptr1, targetname[_MAX_PATH];

  if ( strcmp(target, source) == 0 ) return Filetrunc(source, length);

  strcpy( targetname, target );
  if ( *(ptr1 = strend(targetname) - 1) == '\\' )
     {
       const char *ptr2 = max( source, strrchr(source, '\\') + 1 );
       strcpy( ++ptr1, ptr2 );
     }

  /* change the default file mode from text to binary */
  _fmode = O_BINARY;
  if ( (sourcefile = open(source, O_RDONLY | O_BINARY)) == -1 ||
       (targetfile = creat(targetname, S_IWRITE)) == -1
     ) return errno;

  if ( error = filencopy(sourcefile, targetfile, length) ) return error;

  if ( (close(targetfile) == -1) || (close(sourcefile) == -1) ) return errno;

  return 0;
}



/***
 *  Function    :  Filecat
 *
 *  Description :  Concatenate two files on another.
 *
 *  Parameters  :  in   char *     newpath     result filename
 *                 in   char *     path1       path1 filename
 *                 in   char *     path2       path2 filename
 *
 *  Return      :  0 if OK
 *                 errno if set
 *                 -1 otherwise
 *
 *  OS/Compiler :  All
 ***/

int Filecat( const char *newpath, const char *path1, const char *path2 )

{ int fd1, fd2, newfd;
  int error;

  if ( strcmp(path2, newpath) ) return -1; /*  new path equal to second one  */

  if ( (fd2 = open(path2, O_RDONLY | O_BINARY)) == -1 ) return errno;

  /* change the default file mode from text to binary */
  _fmode = O_BINARY;

  if ( strcmp(path1, newpath) )
     {				/*  new path different than first one  */
       if ( (fd1 = open(path1, O_RDONLY | O_BINARY)) == -1 ||
            (newfd = creat(newpath, S_IWRITE)) == -1
          ) return errno;
       if ( error = filecopy(fd1, newfd) ) return error;
       if ( close(fd1) == -1) return errno;
     }
  else if ( (newfd = open(path1, O_APPEND | O_BINARY)) == -1 ) return errno;

  if ( error = filecopy(newfd, fd2) ) return error;

  if ( (close(fd2) == -1) || (close(newfd) == -1) ) return errno;

  return 0;
}

/***
 *  Function    :  filetrunc
 *
 *  Description :  Truncate a file to a specified length.
 *
 *  Parameters  :  in   int            fd        file descriptor
 *                 in   unsigned long  length    number of bytes to copy
 *
 *  Precond     :  The file must be opened with writing permission
 *
 *  Warning     :  no rewind is performed on the file!!!
 *
 *  Return      :  0 if OK
 *                 errno if error
 *
 *  OS/Compiler :  All
 ***/

int filetrunc( int fd, unsigned long length )

{
/*
  int nread;
  char buffer[BUFSIZE + 1];

  while ( nread = read(fd, buffer, min(BUFSIZE,length)) )
        {
          if ( nread == -1 ) return errno;
          length -= nread;
        }

  if ( write(fd, buffer, 0) == -1 ) return errno;
*/
  if ( chsize(fd, length) ) return errno;
  return 0;
}

/***
 *  Function    :  Filetrunc
 *
 *  Description :  Truncate a file to a specified length.
 *
 *  Parameters  :  in   char *         path      file descriptor
 *                 in   unsigned long  length    number of bytes to copy
 *
 *  Return      :  0 if OK
 *                 errno if error
 *
 *  OS/Compiler :  All
 ***/

int Filetrunc( const char *path, int length )

{ int fd, error;

  /* change the default file mode from text to binary */
  _fmode = O_BINARY;

  if ( (fd = creat(path, S_IWRITE)) == -1 ) return errno;

  if ( error = filetrunc( fd, length ) ) return error;

  if ( close(fd) == -1 ) return errno;

  return 0;
}
