/**********************************************************/
/* File Id.                  Keybd.C                      */
/* Author.                   Stan Milam.                  */
/* Date Written.             03/08/89.                    */
/*                                                        */
/*          (c) Copyright 1989-90 by Stan Milam           */
/*                                                        */
/* Comments:  This file contains functions to querry the  */
/* keyboard to see if characters are waiting and to read  */
/* the characters from the keyboard.                      */
/* Updated 12 Aug. 91 to simplify the keyboard code and   */
/* and give new abilities.  keyin() and readkey() now make*/
/* calls to an internal routine, _getkey().               */
/*                                                        */
/**********************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dos.h>
#include <io.h>
#include "pcw.i"
#include "pcwproto.h"
#include "keys.h"
#ifdef MSC
#   include <stdlib.h>
#   include <bios.h>
#endif

static union REGS regs;

/***************************************************************/
/*                    stdin_is_redirected()                    */
/*                                                             */
/* ADDED 15 DEC. 91.                                           */
/* This function is used to determine if stdin is redirected or*/
/* not.  It looks at the first file handle entry in the PSP to */
/* see if it pointing to the console device (value 1).  If is  */
/* not standard input is being directed.  We do this this way  */
/* because calling isatty() will cause Cntl-C/Break to be a    */
/* problem when stdin is not redirected.                       */
/*                                                             */
/***************************************************************/

static int stdin_is_redirected ( void ) {

    char far *fht;

    fht = MK_FP( _psp, 0x18 );
    return ( fht[0] != 1 );
}

/***************************************************************/
/*                      __bios_keyboard()                      */
/*                                                             */
/* This function is used internally to generate interrupt calls*/
/* to the BIOS keyboard code.  By doing things this way we can */
/* get out of the way of Power C Trace when it is active and   */
/* not lock up the keyboard. Added this routine 12 Aug. 91.    */
/*                                                             */
/***************************************************************/

static void __bios_keyboard( void ) {

#ifdef __POWERC
    crt_lock();
#endif

    int86(0x16, &regs, &regs);

#ifdef __POWERC
    crt_unlock();
#endif
}

/**********************************************************/
/*                         Isxkeybd                       */
/*                                                        */
/* This function will tell us if the keyboard is an en-   */
/* hanced keyboard or not.                                */
/**********************************************************/

int isxkeybd(void) {

   memset(&regs, '\0', sizeof(union REGS));
   regs.h.ah = 0x11;
   __bios_keyboard();
   if (regs.h.ah == 0x11) return(0);
   return(1);
}

/**********************************************************/
/*                        Keypressed                      */
/*                                                        */
/* This function will determine if a character is waiting */
/* in the keyboard buffer. Returns 1 if there is and 0 if */
/* one is not.                                            */
/*                                                        */
/* Rewrote function 15 Dec 91.  Now reports if input is   *
/* from stdin rather than keyboard if stdin is redirected.*/
/* Also reports if a MOUSE key has been pressed.  The ret-*/
/* urn values are as follows.                             */
/*   STDIN redirected = 1                                 */
/*   Normal Keyboard  = 2                                 */
/*   Mouse Key Pressed= 3                                 */
/**********************************************************/

int keypressed(void) {

    if ( stdin_is_redirected() )       /* Is stdin redirected ?   */
         if ( !feof( stdin ) )         /* If it is, is it at EOF? */
              return 1;                /* If Not EOF return input ready */

#ifdef MSC
    regs.x.ax = _bios_keybrd(_KEYBRD_READY);
#else
    memset(&regs, '\0', sizeof(union REGS));
    regs.h.ah = 1;
    __bios_keyboard();
    if ((regs.x.flags & 0x0040) == 0x40) regs.x.ax = 0;
#endif
    if ( regs.x.ax ) return 2;

    if ( mpresent ) {
         if ( get_mpressed( RITEM ) || get_mpressed( LEFTM ) ) {
              return 3;
         }
    }

   /* If nothing from stdin, keyboard or mouse return zero */

   return 0;
}

/***************************************************************/
/*                         getkey()                            */
/*                                                             */
/* This function added 12 Aug. 91 as a replacement for read-   */
/* key() and keyin().                                          */
/*                                                             */
/* Rewrote 15 Dec. 91 to call keypressed(). Uses return value  */
/* from keypressed to (1) read from STDIN, (2) read from key-  */
/* board using BIOS call, and (3) read from MOUSE.             */
/*                                                             */
/***************************************************************/

static int _getkey( void ) {

    int ch;
    int mrow, mcol, mstat;

    while ( 1 ) {
        switch ( keypressed() ) {
            case 0 : continue;
            case 1 :
               ch = getc( stdin );
               if ( ch == 0 ) {
                    ch = ( getc( stdin ) ) << 8;
               }
               return ch;
            case 2 :
               memset(&regs, 0, sizeof(regs));
               __bios_keyboard();
                 if (regs.h.al == 0) ch = regs.x.ax;
               else ch = (regs.x.ax & 0x00ff);
               if (ch == CR) ch = ENTER;
               return ch;
            case 3 :
               get_mpos( &mrow, &mcol, &mstat );
               switch ( mstat ) {
                   case 1 : return LEFT_MOUSE_KEY;
                   case 2 : return RITE_MOUSE_KEY;
                   case 3 : return BOTH_MOUSE_KEY;
               }
               break;
        }
    }
    return 0;                          /* Should never reach here */
}

/**********************************************************/
/*                         ReadKey                        */
/*                                                        */
/* This function will read a character from the keyboard  */
/* buffer.  If the key is a special key (function or      */
/* arrow key) we will return 0 and not read the character.*/
/* We will also set a static flag indicating the character*/
/* is still waiting to be read and when read we will      */
/* the scan code for the special key.                     */
/**********************************************************/

int readkey(void) {

    return ( _getkey() );
}

/**********************************************************/
/*                           Keyin.C                      */
/*                                                        */
/* Comments: Used to read keyboard and handle special keys*/
/**********************************************************/

int keyin(void) {

    return ( _getkey() );
}

/**********************************************************/
/*                       Keybrd_Flush                     */
/*                                                        */
/* Routine to flush the keyboard.                         */
/**********************************************************/

void keybrd_flush(void) {

   while (keypressed()) _getkey();
}

/**********************************************************/
/*                          KeyWait                       */
/*                                                        */
/* Keywait will wait for specified number of seconds      */
/* before returning control to caller.  If a key has been */
/* pressed then return immediately.                       */
/**********************************************************/

int keywait(int seconds) {

   int ticks, rc, ch;
   int mrow, mcol, mstat;

   rc = -1;                            /* Defualt return value */
   keybrd_flush();                     /* Flush out the keyboard */
   ticks = seconds * 9;                /* Approximate number of clock ticks */
   if (ticks < 0) ticks *= -1;         /* Make sure its positive */
   while(ticks) {                      /* While count not zero */
      if ((ch = keypressed())) {       /* Was a key pressed ? */
         if ( ch == 3 ) {
            get_mpos(&mrow,&mcol,&mstat);
            switch ( mstat ) {
               case 1 : return LEFT_MOUSE_KEY;
               case 2 : return RITE_MOUSE_KEY;
               case 3 : return BOTH_MOUSE_KEY;
            }
         }
         rc = _getkey();               /* Return the key that was pressed */
         keybrd_flush();               /* Flush the keyboard */
         return(rc);                   /* Return immediately */
      }
      rest(1);                         /* Wait 1 clock tick */
      ticks--;                         /* Decrement the count */
   }
   return(rc);
}

/*
int main ( void ) {

    int rv;

    for ( rv = _getkey(); rv != 27; rv = getkey() )
        printf("%04x\n", rv);

    return ( 0 );
}
*/
