*-------------------------------------------------------------------------------
*-- Program...: CONVERT.PRG
*-- Programmer: Ken Mayer (KENMAYER)
*-- Date......: 06/25/1992
*-- Notes.....: This is the numeric conversion/calculation library file. See
*--             the file README.TXT for details on the use of this file.
*-------------------------------------------------------------------------------

FUNCTION Roman
*-------------------------------------------------------------------------------
*-- Programmer..: Nick Carlin
*-- Date........: 04/26/1992
*-- Notes.......: A function designed to return a Roman Numeral based on
*--               an Arabic Numeral input ...
*-- Written for.: dBASE III+
*-- Rev. History: 04/13/1988 - original function.
*--               07/25/1991 - Ken Mayer - 1) modified for dBASE IV, 1.1,
*--                             2) updated to a function, and 3) the procedure
*--                             GetRoman was done away with (combined into the
*--                             function).
*--               04/26/1992 - Jay Parsons - shortened (seriously ...)
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Roman(<nArabic>)
*-- Example.....: ? Roman(32)
*-- Returns.....: Roman Numeral (character string) equivalent of Arabic numeral
*--               passed to it. In example:  XXXII
*-- Parameters..: nArabic = Arabic number to be converted to Roman
*-------------------------------------------------------------------------------

   parameters nArabic
   private cLetrs,nCount,nValue,cRoman,cGroup,nMod
	
   cLetrs ="MWYCDMXLCIVX"      && Roman digits
   cRoman = ""                 && this is the returned value
   nCount = 0                  && init counter
   do while nCount < 4         && loop four times, once for thousands, once
                               && for each of hundreds, tens and singles
      nValue = mod( int( nArabic /  10 ^ ( 3 - nCount ) ), 10 )
      cGroup = substr( cLetrs, nCount * 3 + 1, 3 )
      nMod = mod( nValue, 5 )
      if nMod = 4
         if nValue = 9                 && 9
            cRoman = cRoman + left( cGroup, 1 ) + right( cGroup, 1 )
         else                          && 4
            cRoman = cRoman + left( cGroup, 2 )
         endif
      else
         if nValue > 4                 && 5 - 8
            cRoman = cRoman + substr( cGroup, 2, 1 )
         endif
         if nMod > 0                   && 1 - 3 and 6 - 8
            cRoman = cRoman + replicate( left( cGroup, 1 ), nMod )
         endif
      endif
      nCount = nCount + 1
   enddo  && while nCounter < 4
	
RETURN cRoman
*-- EoF: Roman()

FUNCTION Arabic
*-------------------------------------------------------------------------------
*-- Programmer..: Ken Mayer (KENMAYER)
*-- Date........: 04/26/1992
*-- Notes.......: This function converts a Roman Numeral to an arabic one.
*--               It parses the roman numeral into an array, and checks each 
*--               character ... if the previous character causes the value to 
*--               subtract (for example, IX = 9, not 10) we subtract that value, 
*--               and then set the previous value to 0, otherwise we would get 
*--               some odd values in return.
*--               So far, it works fine.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 07/15/1991 - original function.
*--               04/26/1992 - Jay Parsons - shortened.
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Arabic(<cRoman>)
*-- Example.....: ?Arabic("XXIV")
*-- Returns.....: Arabic number (from example, 24)
*-- Parameters..: cRoman = character string containing roman numeral to be
*--               converted.
*-------------------------------------------------------------------------------

  parameters cRoman
  private cRom,cLetrs,nLast,nAt,nVal,cChar,nArabic
	
   cRom = ltrim(trim(upper(cRoman))) && convert to all caps in case ...
   cLetrs = "IVXLCDMWY"
   nArabic = 0
   nLast = 0
   do while len( cRom ) > 0
      cChar = right( cRom, 1 )
      nAt = at( cChar, cLetrs )
      nVal= 10 ^ int( nAt/2 ) / iif(nAt/2 = int(nAt/2),2,1)
      do case
         case nAt = 0
            nArabic = 0
            exit
         case nAt >= nLast
            nArabic = nArabic + nVal
            nLast = nAt
         otherwise
            if nAt/2 = int( nAt / 2 )
               nArabic = 0
               exit
            else
               nArabic = nArabic - nVal
            endif
      endcase
      cRom = left( cRom, len( cRom ) - 1 )
   enddo
	
RETURN nArabic
*-- EoF: Arabic()

FUNCTION Factorial
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Factorial of a number; returns -1 if number is not a
*--               positive integer.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Factorial(<nNumber>)
*-- Example.....: ? Factorial( 6 )
*-- Returns.....: Numeric = number factorial <in example, 6! or 720>
*-- Parameters..: nNumber = number for which factorial is to be determined
*-------------------------------------------------------------------------------

	parameters nNumber
	private nNext, nProduct
	if nNumber # int( nNumber ) .or. nNumber < 1
	  RETURN -1
	endif
	nProduct = 1
	nNext = nNumber
	do while nNext > 1
	  nProduct = nProduct * nNext
	  nNext = nNext - 1
	enddo
	
RETURN nProduct
*-- Eof: Factorial()
                                 
FUNCTION IsPrime
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 08/11/1992
*-- Notes.......: Returns .t. if argument is prime positive integer, or .f.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 03/11/92 - original function.
*--             : 08/11/92 - revised to return .T. for 2. ( Tea for two? )
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: IsPrime(<nNumber>)
*-- Example.....: ? IsPrime( 628321 )
*-- Returns.....: Logical = .t. if prime
*-- Parameters..: nNumber = positive integer to test for being prime
*-------------------------------------------------------------------------------

   parameters nNumber
   private nFactor, nLimit, lResult
   if nNumber < 1 .or. nNumber # int( nNumber ) ;
      .or. ( nNumber > 2 .AND. mod( nNumber, 2 ) = 0 )
      RETURN .f.
   endif
   nFactor = 3
   nLimit = sqrt( nNumber )
   lResult = .t.
   do while nFactor <= nLimit
      if mod( nNumber, nFactor ) = 0
         lResult = .f.
         exit
      endif
      nFactor = nFactor + 2
   enddo

RETURN lResult
*-- Eof: IsPrime()

FUNCTION BankRound
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Rounds numeric argument to given number of places,
*--               which if positive are decimal places, otherwise
*--               trailing zeroes before the decimal, in accordance
*--               with the special banker's rule that if the value
*--               lost by rounding is exactly halfway between two
*--               possible digits, the final digit expressed will be even.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: BankRound(<nNumber>,<nPlaces>)
*-- Example.....: ? BankRound( 357.725, 2 )
*-- Returns.....: Numeric = rounded value ( 357.72 in example )
*-- Parameters..: nNumber = numeric value to round
*--               nPlaces = decimal places, negative being powers of 10
*-------------------------------------------------------------------------------

	parameters nNumber, nPlaces
	private nTemp
	nTemp = nNumber * 10 ^ nPlaces +.5
	if nTemp = int( nTemp ) .and. nTemp / 2 # int( nTemp / 2 )
	  nTemp = nTemp - 1
	endif
	
RETURN int( nTemp ) / 10 ^ nPlaces
*-- Eof: BankRound()

FUNCTION Num2Str
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 06/09/1992
*-- Notes.......: Converts a number to a string like str(), but uses
*--               the ASCII 1/2 and 1/4 characters instead of decimals
*--               where appropriate. Does not require knowing the number of
*--               decimal places first.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: 06/09/1992 -- Angus Scott-Fleming (ANGUSSF) to handle
*--               not knowing the # of decimal places, and small fractions,
*--               such as: 1.000000000000009. Also returns character 
*--               representation not using ASCII characters like the 1/4
*--               symbol ...
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Num2Str(<nNumber>)
*-- Example.....: ? Num2Str( 415.25 )
*-- Returns.....: Character = representation of number ( "415.25" in example )
*-- Parameters..: nNumber = number to represent
*-------------------------------------------------------------------------------

	parameters nNumber
	private nInteger, nFraction, cFracstr, nDec
	nInteger = int( nNumber )
	nFraction = abs( nNumber - nInteger )
	if nFraction = 0
		cFracStr = ""
	else
		*-- note that the maximum # of decimals is 18
		cFracStr = ltrim(str(nFraction,19,18))
		do while right(cFracStr,1) = "0"
	      cFracstr = left(cFracStr,len(cFracStr)-1)
	   enddo
	endif
	
RETURN ltrim( str( nInteger ) ) + cFracstr
*-- Eof: Num2Str()

FUNCTION Dec2Hex
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts an integral number ( in decimal notation)
*--               to a hexadecimal string
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Dec2Hex(<nDecimal>)
*-- Example.....: ? Dec2Hex( 118 )
*-- Returns.....: Character = Hexadecimal equivalent ( "F6" in example )
*-- Parameters..: nDecimal = number to convert
*-------------------------------------------------------------------------------
	
	parameters nDecimal
	private nD, cH
	nD = int( nDecimal )
	cH= ""
	do while nD > 0
	  cH = substr( "0123456789ABCDEF", mod( nD, 16 ) + 1 , 1 ) + cH
	  nD = int( nD / 16 )
	enddo
	
RETURN iif( "" = cH, "0", cH )
*-- Eof: Dec2Hex()

FUNCTION Hex2Dec
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts a hexadecimal character string representing
*--               a positive integer to its numeric (decimal) equivalent
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Hex2Dec(<cHex>)
*-- Example.....: ? Hex2Dec( "F6" )
*-- Returns.....: Numeric = equivalent ( 118 in example )
*-- Parameters..: cHex = character string to convert
*-------------------------------------------------------------------------------
	
	parameters cHex
	private nD, cH
	cH = upper( cHex )
	nD = 0
	do while "" # cH
	  nD = nD * 16 + at( left( cH, 1 ), "123456789ABCDEF" )
	  cH = substr( cH, 2 )
	enddo
	
RETURN nD
*-- Eof: Hex2Dec()

FUNCTION Hex2Bin
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts a hexadecimal character string representing
*--               a positive integer to its binary string equivalent
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Hex2Bin(<cHex>)
*-- Example.....: ? Hex2Bin( "F6" )
*-- Returns.....: Character = binary string ( "1111 0110" in example )
*-- Parameters..: cHex = character string to convert
*-------------------------------------------------------------------------------
	
	parameters cHex
	private cH, cBits, cNybbles, cVal
	cH = upper( cHex )
	cBits = ""
	cNybbles = "00000001001000110100010101100111" ;
	          +"10001001101010111100110111101111"
	do while "" # cH
	  cVal = left( cH, 1 )
	  if cVal # " "
	    cBits = cBits + " " + substr( cNybbles, ;
	      at ( cVal, "123456789ABCDEF" ) * 4 + 1, 4 )
	  endif
	  cH = substr( cH, 2 )
	enddo
	
RETURN iif( "" = cBits, "0", ltrim( cBits ) )
*-- Eof: Hex2Bin()

FUNCTION Bin2Hex
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts a binary character string representing
*--               a positive integer to its hexadecimal string equivalent
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Bin2Hex(<cBin>)
*-- Example.....: ? Bin2Hex( "1111 0110" )
*-- Returns.....: Character = hexadecimal string ( "F6" in example )
*-- Parameters..: cBin = character string to convert
*-------------------------------------------------------------------------------
	
	parameters cBin
	private cH, cBits, nBits, nBval, cNext
	cBits = trim( ltrim( cBin ) )
	nBits = len( cBits ) - 1
	do while nBits > 0
	  if substr( cBits, nBits, 1 ) $ ", "
	    nBval = mod( 4 - mod( len( cBits ) - nBits, 4 ), 4 )
	    cBits = stuff( cBits, nBits, 1, replicate( "0", nBval ) )
	  endif
	  nBits = nBits - 1
	enddo
	cH = ""
	do while "" # cBits
	  store 0 to nBits, nBval
	  do while nBits < 4
	    cNext = right( cBits, 1 )
	    nBval = nBval + iif( cNext = "1", 2 ^ nBits, 0 )
	    cBits = left( cBits, len( cBits ) - 1 )
	    if "" = cBits
	      exit
	    endif
	    nBits = nBits + 1
	  enddo
	  cH = substr( "0123456789ABCDEF", nBval + 1, 1 ) + cH
	enddo
	
RETURN iif( "" = cH, "0", cH )
*-- Eof: Bin2Hex()

FUNCTION Dec2Oct
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts a numeric integer to its octal string equivalent
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Dec2Oct(<nDec>)
*-- Example.....: ? Dec2Oct( 118 )
*-- Returns.....: Character = octal string ( "166" in example )
*-- Parameters..: nDec = number to convert
*-------------------------------------------------------------------------------
	
	parameters nDec
	private nD, cO
	nD = int( nDec )
	cO = ""
	do while nD > 0
	  cO = substr( "01234567", mod( nD, 8 ) + 1 , 1 ) + cO
	  nD = int( nD / 8 )
	enddo

RETURN iif( "" = cO, "0", cO )
*-- Eof: Dec2Oct()

FUNCTION Oct2Dec
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts an integral number in octal, or its string
*--               representation, to a numeric (decimal) value
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Oct2Dect(<xOct>)
*-- Example.....: ? Oct2Dec( 166 )
*-- Returns.....: Numeric = decimal equivalent ( 118 in example )
*-- Parameters..: xOct = octal character string or number to convert
*-------------------------------------------------------------------------------
	
	parameters xOct
	private nD, cO, cVal
	if type( "xOct" ) $ "NF"
	  cO = str( xOct )
	else
	  cO = xOct
	endif
	cO = trim( upper( cO ) )
	nD = 0
	do while "" # cO
	  cVal = left( cO, 1 )
	  if cVal # " "
	    nD = nD * 8 + at( cVal, "1234567" )
	  endif
	  cO = substr( cO, 2 )
	enddo
	
RETURN nD
*-- Eof: Oct2Dec()

FUNCTION Cash2Check
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts a number of dollars and cents to a string of words
*--               appropriate for writing checks.
*--               To correctly evaluate values over 16 decimal places,
*--               SET PRECISION TO a value larger than the default of 16.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: NUM2WORDS()          Function in CONVERT.PRG
*--               THOU2WORDS()         Function in CONVERT.PRG
*-- Called by...: Any
*-- Usage.......: Cash2Check(<nCash>)
*-- Example.....: ? Cash2Check( 348.27 )
*-- Returns.....: Character string equivalent
*-- Parameters..: nCash = money value to convert
*-------------------------------------------------------------------------------

	parameters nCash
	private nDollars, nCents, cResult
	nDollars = int( nCash )
	nCents = 100 * round( nCash - nDollars, 2 )
	cResult = trim( Num2Words( nDollars ) )
	if left( cResult, 1 ) = "C"               && deals with oversize number
	  RETURN cResult
	endif
	cResult = cResult + " dollar" + iif( nDollars # 1, "s", "" ) + " and "
	if nCents # 0
	  RETURN cResult + Thou2Words( nCents )  + " cent" + iif( nCents # 1, "s", "" )
	else
	  RETURN cResult + "no cents"
	endif
	
*-- Eof: Cash2Check()

FUNCTION Num2Words
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts an integer to a string of words.  Limited, due to
*--               254-character limit of dBASE strings, to numbers less than
*--               10 ^ 15
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: THOU2WORDS()         Function in CONVERT.PRG
*-- Called by...: Any
*-- Usage.......: Num2Words(<nNum>)
*-- Example.....: ? Num2Words( 4321568357 )
*-- Returns.....: Character string equivalent
*-- Parameters..: nNum = numeric integer to convert
*-------------------------------------------------------------------------------
	
	parameters nNum
	private nNumleft, nScale, nGroup, cResult
	nNumleft = int( nNum )
	do case
	  case abs( nNumleft ) >= 10 ^ 15
	    RETURN "Cannot convert a number in or above the quadrillions."    
	  case nNumleft = 0
	    RETURN "zero"
	  case nNumleft < 0
	    cResult = "minus "
	    nNumleft = -nNumleft
	  otherwise 
	    cResult = ""
	endcase
	do while nNumleft > 0
	  nScale = int( log10( nNumleft ) / 3 )
	  nGroup = int( nNumleft / 10 ^ ( 3 * nScale ) )
	  nNumleft = mod( nNumleft, 10 ^ ( 3 * nScale ) )
	  cResult = cResult + Thou2Words( nGroup )
	  if nScale > 0
	    cResult = cResult + " " ;
	      + trim( substr( "thousandmillion billion trillion", nScale * 8 - 7, 8 ) )
	    if nNumleft > 0
	      cResult = cResult + ", "
	    endif
	  endif
	enddo           
	
RETURN cResult
*-- Eof: Num2Words()

FUNCTION Thou2Words
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts a positive integer less than 1000 to a string
*--               of characters.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Thou2Words(<nNum>)
*-- Example.....: ? Thou2Words( 834 )
*-- Returns.....: Character string equivalent
*-- Parameters..: nNum = numeric integer to convert
*-------------------------------------------------------------------------------
	
	parameters nNum
	private cUnits, cTens, nN, cResult
	cUnits = "one      two      " ;
	       + "three    four     " ;
	       + "five     six      " ;
	       + "seven    eight    " ;
	       + "nine     ten      " ;
	       + "eleven   twelve   " ;
	       + "thirteen fourteen " ;
	       + "fifteen  sixteen  " ;
	       + "seventeeneighteen " ;
	       + "nineteen "
	cTens = "twen thir for  fif  six  seveneigh nine "
	nN = int( nNum )
	if nN = 0
	  RETURN "zero"
	endif
	cResult = ""
	if nNum > 99
	  cResult = trim( substr(cUnits, int(nNum / 100 ) * 9 - 8, 9 ) ) + " hundred"
	  nN = mod( nN, 100 )
	  if nN = 0
	    RETURN cResult
	  else
	    cResult = cResult + " "
	  endif
	endif
	if nN > 19
	  cResult = cResult + trim( substr( cTens, int( nN / 10 ) * 5 - 9, 5 ) ) + "ty"
	  nN = mod( nN, 10 )
	  if nN = 0
	    RETURN cResult
	  else
	    cResult = cResult + "-"
	  endif
	endif
	
RETURN cResult + trim( substr( cUnits, nN * 9 - 8, 9 ) )
*-- Eof: Thou2Words()

FUNCTION ord
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts an integer to ordinal representation by adding
*--               "st", "nd", "rd" or "th" after its digit(s)
*--               of characters.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: ord(<nNum>)
*-- Example.....: ? ord( 11 )
*-- Returns.....: Character ordinal string equivalent ( "11th" in example )
*-- Parameters..: nNum = numeric integer to convert
*-------------------------------------------------------------------------------
	
	parameters nNum
	private nD
	nD = mod( nNum, 100 ) - 1     && the -1 just happens to simplify what follows
	
RETURN str( nNum ) + iif( mod( nD, 10 ) > 2 .or. abs( nD - 11 ) < 2, ;
   "th", substr( "stndrd", mod( nD, 10 ) * 2 + 1, 2 ) )
*-- Eof: ord()

FUNCTION Num2Real
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts a number to the ASCII representation of
*--               its storage in IEEE 8-byte real format, with least
*--               significant byte (lowest in memory) first.  Provided
*--               for checking the values in .MEM files, or in memory
*--               float-type variables if peeking.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: DEC2BIN()            Function in CONVERT.PRG
*--               FRAC2BIN()           Function in CONVERT.PRG
*--               BIN2HEX()            Function in CONVERT.PRG
*-- Called by...: Any
*-- Usage.......: Num2Real(<nNum>,<nExp>)
*-- Example.....: ? Num2Real( 10, 100 )
*-- Returns.....: Character string equivalent ( of a blank date, in example )
*-- Parameters..: nNum = number to convert, or mantissa if exponent given
*--               nExp = decimal exponent of nNum, 0 if not needed
*-------------------------------------------------------------------------------
	
	parameters nNum, nExp
	private nN, nSign, nExpo, nMantissa, cBitmap, nByte, cBytemap
	nN = nNum * 10 ^ nExp
	do case
	  case nN = 0
	    RETURN ltrim( replicate( " 00", 8 ) )
	  case nN < 0
	    nSign = 1
	    nN = -nN
	  otherwise
	    nSign = 0
	endcase  
	nExpo = int( log( nN ) / log( 2 ) )
	nMantissa = nN / 2 ^ nExpo - 1
	cBitmap = str( nSign, 1 ) + Dec2Bin( nExpo + 1023, 11 ) ;
	  + Frac2Bin( nMantissa, 52 )
	cBytemap = ""
	nByte = 0
	do while nByte < 8
	  cBytemap = cBytemap + " " + Bin2Hex( substr( cBitmap, 57 - 8 * nByte, 8 ) )
	  nByte = nByte + 1
	enddo
	
RETURN trim( cBytemap )
*-- Eof: Num2Real()

FUNCTION Dec2Bin
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts an integral decimal number to a character
*--               string giving its ASCII binary representation.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Dec2Bin(<nNum>,<nPlaces>)
*-- Example.....: ? Dec2Bin( 35, 8 )
*-- Returns.....: Character binary equivalent ( "0010 0011", in example )
*-- Parameters..: nNum = number to convert
*--               nPlaces = number of binary places number is to occupy
*-------------------------------------------------------------------------------
	
	parameters nNum, nPlaces
	private cBits, nN
	cBits= ""
	nN = nNum
	do while len(cBits) < nPlaces
	  if nN > 0
	    cBits = str( mod( nN, 2 ), 1 ) + cBits
	    nN = int( nN / 2 )
	  else
	    cBits = "0" +cBits
	  endif
	enddo
	
RETURN cBits
*-- Eof: Dec2Bin()

FUNCTION Frac2Bin
*-------------------------------------------------------------------------------
*-- Programmer..: Jay Parsons (JPARSONS)
*-- Date........: 03/01/1992
*-- Notes.......: Converts the fractional part of a decimal number
*--               to a character string giving its ASCII binary representation.
*-- Written for.: dBASE IV, 1.1
*-- Rev. History: None
*-- Calls.......: None
*-- Called by...: Any
*-- Usage.......: Frac2Bin(<nNum>,<nPlaces>)
*-- Example.....: ? Frac2Bin( .35, 8 )
*-- Returns.....: Character binary equivalent
*-- Parameters..: nNum = number to convert
*--               nPlaces = number of binary places number is to occupy
*-------------------------------------------------------------------------------

	parameters nNum, nPlaces
	private cBits, nN
	cBits = ""
	nN = nNum
	do while len( cBits ) < nPlaces
	  if nN > 0
	    nN = 2 * nN
	    cBits = cBits + str( int( nN ), 1 )
	    nN = nN - int( nN )
	  else
	    cBits = cBits + "0"
	  endif
	enddo
	
RETURN cBits
*-- Eof: Frac2Bin()

*-------------------------------------------------------------------------------
*-- EoP: CONVERT.PRG
*-------------------------------------------------------------------------------
