/****************************************************************************

 BBE.EXE : BRIEF Binary Editing Executable

           This programs main purpose and intent is to assist in editing
           non-standard DOS files in BRIEF.  The converter is called
           by the macros provided.  The BRIEF macros will attempt to
           second guess the files origin and call the converter accordingly.

 AUTHOR  : Timothy Hanna

 DATE    : April 15, 1991


Commands : ------------------------------------------------------------------

           BBE BH inFile outFile outFile
           BBE HB inFile outFile 
           BBE UD inFile outFile 
           BBE DU inFile outFile 
           BBE RC inFile outFile recordlength 
           BBE RD inFile outFile recordlength 

           BH Convert binary (or text) to hex formatted file.           
           HB Convert hex formatted file to binary (or text) file.      
           UD Convert Unix style LF delimited lines to DOS CR LF lines. 
           DU Convert DOS style CR LF delimited lines to Unix LF lines. 
           RC Add CR LF to a data file for record delimiting at recordLength.           
           RD Delete CR LF from a data file for data file at recordLength.

         : ------------------------------------------------------------------

 This program is hereby placed into the public domain.  No fee's other
 than nominal copying fees may be charged.  

****************************************************************************/


#include <dos.h>
#include <mem.h>
#include <stdio.h>
#include <string.h>

#define ERROR_WNOP -1 /* ERROR_WrongNumberOfParameters */
#define ERROR_IP   -2 /* ERROR_InvalidParameters       */
#define ERROR_FNF  -3 /* ERROR_FileNotFound            */

/* ------------------------------------------------------------------------ 

 main()

 This function will parse out the command line and according to the values
 call the appropriate function.  If it is determined that inappropriate
 or insufficent values were passed then the function will return the
 error value.

 ------------------------------------------------------------------------ */

int main(int argc, char *argv[])
{
 int result;

 FILE *inFile, *outFile1, *outFile2;

 if (argc<4)
   return ERROR_WNOP;

 /* ------------------------------------------------------------------------

    Identify if selected conversion is BH, RC or RD.  These options
    require four parameters.  Otherwise the check for HB, DU and UD
    options.  Then open needed files and check that the files
    have been opened.  Lastly call the selected conversion routine.

 ------------------------------------------------------------------------ */

 if (!strcmp(argv[1],"BH") || !strcmp(argv[1],"RC") || !strcmp(argv[1],"RD"))
    {
     if (argc<5)
        return ERROR_WNOP;

     if (!strcmp(argv[1],"BH"))
        {
         inFile  = fopen(argv[2],"rb");
         outFile1 = fopen(argv[3],"wb");
         outFile2 = fopen(argv[4],"wb");
        
         if ((inFile==NULL) && (outFile1==NULL) && (outFile2==NULL))
            {
             if (inFile!=NULL)
                fclose(inFile);
        
             if (outFile1!=NULL)
                fclose(outFile1);
        
             if (outFile2!=NULL)
                fclose(outFile2);
         
             return ERROR_FNF;
            }
         
         result = ConvertBinaryToHex(inFile,outFile1,outFile2);
        
         fclose(inFile);
         fclose(outFile1);
         fclose(outFile2);
        }
     else
        {
         inFile  = fopen(argv[2],"rb");
         outFile1 = fopen(argv[3],"wb");

         if ((inFile==NULL) && (outFile1==NULL))
            {
             if (inFile!=NULL)
                fclose(inFile);
        
             if (outFile1!=NULL)
                fclose(outFile1);

             return ERROR_FNF;
            }

         if (!strcmp(argv[1],"RC"))
            result = RecordsCreate(inFile,outFile1,atoi(argv[4]));

         if (!strcmp(argv[1],"RD"))
            result = RecordsDelete(inFile,outFile1,atoi(argv[4]));
        }
    }
 else if (!strcmp(argv[1],"HB")  || !strcmp(argv[1],"UD") || !strcmp(argv[1],"DU"))
    {
     inFile  = fopen(argv[2],"rb");
     outFile1 = fopen(argv[3],"wb");

     if ((inFile==NULL) && (outFile1==NULL))
        {
         if (inFile!=NULL)
            fclose(inFile);
    
         if (outFile1!=NULL)
            fclose(outFile1);
    
         return ERROR_FNF;
        }

     if (!strcmp(argv[1],"HB"))
        result = ConvertHexToBinary(inFile,outFile1);

     if (!strcmp(argv[1],"UD"))
        result = ConvertUnixToDOS(inFile,outFile1);

     if (!strcmp(argv[1],"DU"))
        result = ConvertDOSToUnix(inFile,outFile1);

     fclose(inFile);
     fclose(outFile1);
    }
 else
    return ERROR_IP;

 return result;
}

/* ------------------------------------------------------------------------ 

 ConvertBinaryToHex()
   
   This function converts the infile character by character to destination
   files of hexadecimal and ascii values.  Since BRIEF is incapable of
   displaying the ASCII values of 0, 9 and 13 they are left as dots in the
   buffer.

 ------------------------------------------------------------------------ */

int ConvertBinaryToHex(FILE *inFile, FILE *outFile1, FILE *outFile2)
{
 const valuesPerLine = 25;
 const hexLineIncrement = 2;

 unsigned char buffer[1200];

 unsigned char hexBuffer[52];
 unsigned char charBuffer[27];

 unsigned int numberOfBytes=1,index1,index2,index3,index4,charactersLeft;

 unsigned char decimalToHex[] = {'0','1','2','3','4','5','6','7','8','9','A','B','C','D','E','F'};

 hexBuffer[50] = '\r';
 hexBuffer[51] = '\n';

 charBuffer[25] = '\r';
 charBuffer[26] = '\n';

 while (numberOfBytes)
    {
     numberOfBytes = fread(buffer,1,1200,inFile);

     if (numberOfBytes)
        {
         for (index1=0;index1<numberOfBytes;index1+=valuesPerLine)
            {
             charactersLeft = (numberOfBytes-index1 < valuesPerLine) ? (numberOfBytes-index1) : valuesPerLine;

             memset(hexBuffer,' ',50); 
             memset(charBuffer,'.',25);

             for (index2=0;index2<charactersLeft;index2++)
               {
                index3 = index1+index2;
                index4 = index2*hexLineIncrement;

                hexBuffer[index4]   = decimalToHex[buffer[index3]/16];
                hexBuffer[index4+1] = decimalToHex[buffer[index3]%16];

                switch (buffer[index3])
                  {
                   case 0:
                   case 9:
                   case 13:
                        break;

                   default:
                        charBuffer[index2] = buffer[index3];

                  };
               }

             if (charactersLeft<valuesPerLine)
                {
                 numberOfBytes=0;
                 for (index2=charactersLeft;index2<valuesPerLine;index2++)
                    {
                     hexBuffer[index2*hexLineIncrement]   = 'X';
                     hexBuffer[index2*hexLineIncrement+1] = 'X';
                    }
                }

             fwrite(hexBuffer,1,52,outFile1);

             fwrite(charBuffer,1,27,outFile2);

            }
        }
    }

   return 0;
}

/* ------------------------------------------------------------------------ 

 ConvertHexToBinary()
   
   This function converts the infile based on expected two character
   hexadecimal values.  Each record is expected to be 52 chracters
   in length with the last two chraters per record being thrown away.
   Every two chracters are converted to a single binary value and stored
   into a binary buffer.  If the chracter 'X' is encountered as the
   first of the two characters then that and the next character are ignored.

 ------------------------------------------------------------------------ */

int ConvertHexToBinary(FILE *inFile, FILE *outFile)
{
 unsigned char hexToDecimal[] = { 0,1,2,3,4,5,6,7,8,9,0,0,0,0,0,0,0,10,11,12,13,14,15}; 

 unsigned char buffer[52];

 unsigned char byteBuffer[2];
 unsigned char binaryBuffer[25];

 unsigned int numberOfBytes=1,index1,index2,writeAmount;

 while (numberOfBytes)
    {
     numberOfBytes = fread(buffer,1,52,inFile);

     if (numberOfBytes==52)
        {
         writeAmount=25;

         for (index1=0;index1<25;index1++)
            {
             index2 = index1*2;

             if ((byteBuffer[0] = buffer[index2]) != 'X')
               {
                byteBuffer[1] = buffer[index2+1];
                binaryBuffer[index1] = (hexToDecimal[byteBuffer[0]-48] * 16) + (hexToDecimal[byteBuffer[1]-48]);
               }
             else
                writeAmount--;
            }

         fwrite(binaryBuffer,1,writeAmount,outFile);
        }
    }

 return 0;
}

/* ------------------------------------------------------------------------ 

   ConvertUnixToDOS()

   This function will search the file for the occurance of the unix line
   delimiter 0x0A (a line feed chracter).  Since BRIEF needs to have a
   carraige return, one is inserted into the destination buffer prior to
   the line feed chracter.

 ------------------------------------------------------------------------ */

int ConvertUnixToDOS(FILE *inFile, FILE *outFile)
{
 unsigned char inBuffer[4000];
 unsigned char outBuffer[8000];

 unsigned int numberOfBytes=1,index1,index2;

 while (numberOfBytes)
    {
     numberOfBytes = fread(inBuffer,1,4000,inFile);

     if (numberOfBytes)
        {
         for (index1=0,index2=0;index1<numberOfBytes;index1++)
            {
             if (inBuffer[index1]==0x0A)
                outBuffer[index2++]=0x0D;

             outBuffer[index2++]=inBuffer[index1];
            }

         fwrite(outBuffer,1,index2,outFile);
        }
    }
 return 0;
}

/* ------------------------------------------------------------------------ 

   ConvertDosToUnix()

   This function will search the file for the occurance of a carridge return
   line feed character sequence.  If identified it will only write the
   line feed character, and skip the carraidge return.

 ------------------------------------------------------------------------ */

int ConvertDOSToUnix(FILE *inFile, FILE *outFile)
{
 unsigned char inBuffer[4000];
 unsigned char outBuffer[4000];

 unsigned int numberOfBytes=1,index1,index2;

 while (numberOfBytes)
    {
     numberOfBytes = fread(inBuffer,1,4000,inFile);

     if (numberOfBytes)
        {
         for (index1=0,index2=0;index1<numberOfBytes;index1++)
            {
             if (inBuffer[index1]==0x0D && inBuffer[index1+1]==0x0A)
                 index1++;

             outBuffer[index2++]=inBuffer[index1];
            }

         fwrite(outBuffer,1,index2,outFile);
        }
    }
 return 0;
}

/* ------------------------------------------------------------------------ 

   RecordsCreate()

   This function will, based on a selected value, add the carridge return
   line feed sequence at every occurance of that number of characters.

 ------------------------------------------------------------------------ */

int RecordsCreate(FILE *inFile, FILE *outFile, int recordSize)
{
 unsigned char buffer[512];

 unsigned int numberOfBytes=1;

 if (recordSize>510)
    return ERROR_IP;

 buffer[recordSize]   = '\r';
 buffer[recordSize+1] = '\n';

 while (numberOfBytes)
    {
     numberOfBytes = fread(buffer,1,recordSize,inFile);

     if (numberOfBytes==recordSize)
        fwrite(buffer,1,recordSize+2,outFile);
     else
        {
         buffer[numberOfBytes] = '\r';
         buffer[numberOfBytes+1] = '\n';

         fwrite(buffer,1,numberOfBytes+1,outFile);
        }
    }
 return 0;
}

/* ------------------------------------------------------------------------ 

   RecordsDelete()

   This function will, based on a selected value, removes two chracters
   at every occurance of that number of characters.

 ------------------------------------------------------------------------ */

int RecordsDelete(FILE *inFile, FILE *outFile, int recordSize)
{
 unsigned char buffer[512];
 unsigned int numberOfBytes=1;

 if (recordSize>510)
    return ERROR_IP;

 while (numberOfBytes)
    {
     numberOfBytes = fread(buffer,1,recordSize+2,inFile);

     if (numberOfBytes==recordSize)
        fwrite(buffer,1,recordSize,outFile);
     else
        fwrite(buffer,1,numberOfBytes-2,outFile);
    }
 return 0;
}







