//-------------------------------------------------------------------------
//			ActionMedia II Programmer's Toolkit
//			
//			Windows Sample Code Shared Functions
//
// Module Name:	memalloc.c
//
// description:	Functions to allocate and free NEAR, FAR and HUGE blocks
//				of memory.  These functions act as generic shells around
//				GlobalAlloc/GlobalLock/GlobalFree and LocalAlloc/LocalLock/
//				LocalFree functions.
//
//	Copyright Intel Corp. 1991, 1992
//	All Rights Reserved.
//
//-------------------------------------------------------------------------
//
// Exported functions from this module:
//
//		MemAlloc
//		MemAllocNear
//		MemAllocHuge
//		MemFree
//		MemFreeNear
//		MemFreeHuge
//
//
//-------------------------------------------------------------------------
//	NOTES ON MemAlloc
//
//	The allocation functions take care of allocating NEAR, FAR and HUGE 
//	blocks of memory, locking the blocks and returning pointers dereferenced
//	from the Windows handles returned by the Windows allocation functions.
//	The free functions extract the handles from the pointers and free 
//	the blocks.
//-------------------------------------------------------------------------


#include <windows.h>

#define	L_FLAGS		LMEM_FIXED | LMEM_NODISCARD | LMEM_ZEROINIT
#define	G_FLAGS		GMEM_MOVEABLE | GMEM_NODISCARD | GMEM_ZEROINIT

char far	*MemAlloc(WORD);
char huge 	*MemAllocHuge(unsigned long);

//-------------------------------------------------------------------------	 
//FUNCTION:																	 
//	
//	char far *MemAlloc(Size)
//
//PARMS IN:
//
//	WORD	Size		the size of the block to be allocated.
//	
//DESCRIPTION:
//
//	Uses Windows Global allocation functions to allocate Size bytes (or
//	more) of memory.  If the allocation is successful, the allocated
//	block is locked and the resulting pointer is returned to the caller.
//	
//RETURN:
//
//	32-bit far pointer to an allocated and locked block of at least Size 
//	bytes. NULL if unsuccessful.
//	
//-------------------------------------------------------------------------

char far *
MemAlloc(WORD Size)
{
	HANDLE		 h;
	char far 	*p = NULL;

	if ((h = GlobalAlloc(G_FLAGS, (unsigned long)Size)) != (HANDLE)NULL)
	{
		if ((p = (char far *)GlobalLock(h)) == (char far *)NULL)
			GlobalFree(h);
	}
	return p;
}

//-------------------------------------------------------------------------	 
//FUNCTION:																	 
//	
//	char huge *MemAllocHuge(Size)
//
//PARMS IN:
//
//	unsigned long	Size		the size of the block to be allocated.
//	
//DESCRIPTION:
//
//	Uses Windows Global allocation functions to allocate Size bytes (or
//	more) of memory.  If the allocation is successful, the allocated
//	block is locked and the resulting pointer is returned to the caller.
//	
//RETURN:
//
//	32-bit huge pointer to an allocated and locked block of at least Size 
//	bytes. NULL if unsuccessful.
//	
//-------------------------------------------------------------------------


char huge *
MemAllocHuge(unsigned long Size)
{
	HANDLE		 h;
	char huge	*p = NULL;

	if ((h = GlobalAlloc(G_FLAGS, (unsigned long)Size)) 
	  != (HANDLE)NULL)
	{
		if ((p = (char huge *)GlobalLock(h)) == (char huge *)NULL)
			GlobalFree(h);
	}
	return p;
}

//-------------------------------------------------------------------------	 
//FUNCTION:																	 
//	
//	char near *MemAllocNear(Size)
//
//PARMS IN:
//
//	WORD	Size		the size of the block to be allocated.
//	
//DESCRIPTION:
//
//	Uses Windows Local allocation functions to allocate Size bytes (or
//	more) of memory.  If the allocation is successful, the allocated
//	block is locked and the resulting pointer is returned to the caller.
//	
//RETURN:
//
//	16-bit near pointer to an allocated and locked block of at least Size 
//	bytes. NULL if unsuccessful.
//	
//-------------------------------------------------------------------------


char near *
MemAllocNear(WORD Size)
{
	HANDLE		 h;
	char near 	*p = NULL;

	if ((h = LocalAlloc(L_FLAGS, Size)) != (HANDLE)NULL)
	{
		if ((p = (char near *)LocalLock(h)) == (char near *)NULL)
			LocalFree(h);
	}
	return p;
}


//-------------------------------------------------------------------------	 
//FUNCTION:																	 
//	
//	void MemFree(pBlock)
//
//PARMS IN:
//
//	char far *pBlock	pointer to block of memory to be freed.
//	
//DESCRIPTION:
//
//	Uses Windows Global allocation functions to unlock and free an 
//	allocated block of memory.  
//	
//-------------------------------------------------------------------------

void
MemFree(char far *p)
{
	HANDLE		h;

	h = LOWORD(GlobalHandle(HIWORD((unsigned long)p)));
	GlobalUnlock(h);
	GlobalFree(h);
}


//-------------------------------------------------------------------------	 
//FUNCTION:																	 
//	
//	void MemFreeHuge(pBlock)
//
//PARMS IN:
//
//	char huge *pBlock	pointer to block of memory to be freed.
//	
//DESCRIPTION:
//
//	Uses Windows Global allocation functions to unlock and free an 
//	allocated block of memory.  
//	
//-------------------------------------------------------------------------

void
MemFreeHuge(char huge *p)
{
	HANDLE		h;

	h = LOWORD(GlobalHandle(HIWORD((unsigned long)p)));
	GlobalUnlock(h);
	GlobalFree(h);
}

//-------------------------------------------------------------------------	 
//FUNCTION:																	 
//	
//	void MemFreeNear(pBlock)
//
//PARMS IN:
//
//	char near *pBlock	pointer to block of memory to be freed.
//	
//DESCRIPTION:
//
//	Uses Windows Local allocation functions to unlock and free an 
//	allocated block of memory.  
//	
//-------------------------------------------------------------------------


void
MemFreeNear(char near *p)
{
	HANDLE 		h;

	h = LocalHandle((WORD)p);
	LocalUnlock(h);
	LocalFree(h);
}

