/* BarChart widget source file. This widget is used to display items in a bar 
chart. The user is expected to set at least the following values for this to
work:
          Resource       Value
          ---------      -----
          itemCount      0 if nothing to be shown or # of items
          items          pointer to integer array of ItemCount
          maxItemCount   the #of items to show, default 10
          auxItemss      pointer to integer array of ItemCount 
                         with contents of secondary data. This may be empty. 
          The other resources will be listed in the man pages. 
 Author: Kamran Husain        Mentor Programming Services Inc. 
 Debug Flags
 #define DEBUG_BARCHART 2
 #define DEBUG_BARCHART_DISPLAY
 */

#include <stdio.h> 
#include <math.h>
#include <malloc.h>
#include <X11/X.h>
#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include "Xv.h"
#include "BarChartP.h"
#include "BarChart.h"

#define  MIN(a,b)   (((a)<(b))?(a):(b))
#define  MAX(a,b)   (((a)>(b))?(a):(b))
#define  ABS(x)     (((x)< 0) ? (-1 * (x)) : (x))
#define  MID(a,b)   (ABS((((a)-(b))/2)))

static void select_barchart();
static void Initialize();
static void Redisplay();
static void Resize();
static void Destroy();
static void CopyItems();
static void MakeGC();
static Boolean SetValues();

static char defaultTranslations[] = "<Btn1Down>: select()";
static XtActionsRec actionsList[] = {
     { "select", (XtActionProc)select_barchart }
};
/*  Define resources for this widget.  */
static XtResource resources[] = {
     { 
     XtNselectCallback, XtCCallback, XtRCallback, sizeof(caddr_t),
     XtOffset(XvBarChartWidget,barchart.select ), XtRCallback, NULL 
     }, 
     { 
     XtNname, XtCName, XtRString, sizeof(caddr_t),
     XtOffset(XvBarChartWidget,barchart.name ), XtRPointer, NULL 
     },
     { 
     XtNshowValue, XtCShowValue, XtRInt, sizeof(int),
     XtOffset(XvBarChartWidget,barchart.showValue), XtRString, "1"
     }, 
     { 
     XtNshowChartLabel, XtCShowChartLabel, XtRInt, sizeof(int),
     XtOffset(XvBarChartWidget,barchart.showChartLabel), XtRString, "1"
     }, 
     { 
     XtNchartLabel, XtCChartLabel, XtRString, sizeof(caddr_t),
     XtOffset(XvBarChartWidget,barchart.chartLabel), XtRPointer, NULL 
     }, 
     { 
     XtNnamelen, XtCNamelen, XtRInt, sizeof(int),
     XtOffset(XvBarChartWidget,barchart.namelen ), XtRInt, 0 
     }, 
     { 
     XtNminShowValue, XtCMinShowValue, XtRInt, sizeof(int),
     XtOffset(XvBarChartWidget,barchart.minShowValue), XtRInt, 0 
     }, 
     { 
     XtNmaxShowValue, XtCMaxShowValue, XtRInt, sizeof(int),
     XtOffset(XvBarChartWidget,barchart.maxShowValue), XtRInt, 0 
     }, 
     { 
     XtNzeroColor, XtCZeroColor, XtRPixel, sizeof(Pixel),
     XtOffset(XvBarChartWidget,barchart.zeroColor), XtRString, "Green"
     },
     { 
     XtNpositiveColor, XtCPositiveColor, XtRPixel, sizeof(Pixel),
     XtOffset(XvBarChartWidget,barchart.positiveColor), XtRString, "Yellow"
     },
     { 
     XtNnegativeColor, XtCNegativeColor, XtRPixel, sizeof(Pixel),
     XtOffset(XvBarChartWidget,barchart.negativeColor), XtRString, "Red"
     },
     { 
     XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int), 
     XtOffset(XvBarChartWidget,barchart.maxValue), XtRString, "50"
     },
     { 
     XtNminValue, XtCMinValue, XtRInt, sizeof(int),
     XtOffset(XvBarChartWidget,barchart.minValue), XtRString, "-50"
     },
     { 
     XtNmidValue, XtCMidValue, XtRInt, sizeof(int),
     XtOffset(XvBarChartWidget,barchart.midValue), XtRInt, 0
     },
     { 
     XtNmaxItemCount, XtCMaxItemCount, XtRInt, sizeof(int),
     XtOffset(XvBarChartWidget,barchart.maxItemCount), XtRString, "10"
     }, 
     { 
     XtNredraw, XtCRedraw, XtRBoolean, sizeof(Boolean),
     XtOffset(XvBarChartWidget,barchart.redraw), XtRBoolean, False  
     }, 
     { 
     XtNitemCount, XtCItemCount, XtRInt, sizeof(int),
     XtOffset(XvBarChartWidget,barchart.itemCount), XtRInt, 0 
     }, 
     { 
     XtNauxItems, XtCAuxItems, XtRPointer, sizeof(caddr_t),
     XtOffset(XvBarChartWidget,barchart.auxItems), XtRPointer, NULL  
     }, 
     { 
     XtNitems, XtCItems, XtRPointer, sizeof(caddr_t),
     XtOffset(XvBarChartWidget,barchart.items), XtRPointer, NULL  
     }
};  
/*  Define the Class record and the pointer to it. */
XvBarChartClassRec  xvbarchartClassRec = {
     { 
     (WidgetClass) &widgetClassRec, "BarChart", sizeof(XvBarChartRec), 
     NULL, NULL, FALSE, Initialize, NULL, XtInheritRealize, NULL, 0,
     resources, XtNumber(resources), NULLQUARK, TRUE, TRUE, TRUE, TRUE,
     Destroy, Resize, Redisplay, SetValues, NULL, XtInheritSetValuesAlmost,
     NULL, NULL, XtVersion, NULL, NULL, NULL, NULL, NULL
     },
     { 0 }               /*default , do NOT change */
};
WidgetClass XvbarchartWidgetClass = (WidgetClass) &xvbarchartClassRec;

/* Methods defined for this widget: Initialize, Resize, Destroy, Redisplay
 * SetValues */
/*: Initialize -- Initialize the functions in a basic barchart widget. */
static void Initialize( request, new)
XvBarChartWidget     request, new;
{
     /*  Set inherited resources first. Must have non-zero dimensions. */
     if (request->core.width == 0 ) new->core.width = 50;
     if (request->core.height == 0 ) new->core.height = 50;
     new->core.border_width = 0;
#ifdef DEBUG_BARCHART
     printf("\n BarChart at %d,%d with %d items", 
               new->core.x, new->core.y, new->barchart.itemCount);
#endif
     /*  Now set your own resources.  */
     MakeGC(new);
     if ((request->barchart.items == NULL) 
          && (request->barchart.itemCount))
          XtWarning("ItemCount nonzero for empty items list");
     if ((request->barchart.items != NULL) 
          && (request->barchart.itemCount == 0))
          XtWarning("ItemCount zero for non empty items list");
     if (request->barchart.maxValue == request->barchart.minValue)
     {
          new->barchart.maxValue = 500;
          new->barchart.midValue = 0;
          new->barchart.minValue = -500;
          XtWarning("The maximum value was equal to the minimum!");
     }
     if (request->barchart.maxValue < request->barchart.minValue)
     {
          new->barchart.maxValue = request->barchart.minValue;
          new->barchart.minValue = request->barchart.maxValue;
          XtWarning("The maximum value is greater that the minimum!");
     }
     if (request->barchart.midValue < request->barchart.minValue)
     {
          new->barchart.midValue = 
          MID(request->barchart.maxValue,request->barchart.minValue);
          XtWarning("The midrange value is less that the minimum!");
     }
     if (request->barchart.midValue == request->barchart.minValue)
     {
          new->barchart.midValue = request->barchart.minValue + 1;
          XtWarning("The midvalue is equal to the minimum!");
     }
     new->barchart.itemCount = ABS(request->barchart.itemCount);
     if (new->barchart.maxItemCount > new->barchart.itemCount)
     {
          new->barchart.maxItemCount = new->barchart.itemCount;
          XtWarning("The maximum count is greater than itemCount!");
     }
     new->barchart.itemHt = NULL;
     new->barchart.auxItemHt = NULL;
     if (request->barchart.itemCount)
     {
          new->barchart.itemHt = (XRectangle *)
              XtMalloc(sizeof(XRectangle) * (new->barchart.itemCount));
          new->barchart.auxItemHt = (XRectangle *)
              XtMalloc(sizeof(XRectangle) * (new->barchart.itemCount));
     /*  Now create an items area for this widget and copy the users 
     ** data into your own array. **/
     CopyItems(new);
     }
Resize(new);
}
/* Internal function to create GCs for the widget. */
static void MakeGC(new)
XvBarChartWidget new;
{
     XGCValues           values;
     XtGCMask         valueMask;
     valueMask = GCForeground | GCBackground | GCLineWidth;
     values.line_width = 1;
     values.foreground = new->barchart.positiveColor;
     values.background = new->core.background_pixel;
     new->barchart.positiveGC = XtGetGC(new,valueMask,&values);
     values.background = new->core.background_pixel;
     values.foreground = new->core.background_pixel;
     new->barchart.inverseGC = XtGetGC(new,valueMask,&values);
     values.foreground = new->barchart.negativeColor;
     values.background = new->core.background_pixel;
     new->barchart.negativeGC = XtGetGC(new,valueMask,&values);
     values.foreground = new->barchart.zeroColor;
     values.background = new->core.background_pixel;
     new->barchart.zeroGC = XtGetGC(new,valueMask,&values);
#ifdef DEBUG_BARCHART
     printf("\n Max, Mid, Min = %d,%d,%d", 
         new->barchart.maxValue, new->barchart.midValue,
         new->barchart.minValue);
     printf("\n fmax, fmid, fmin = %d,%d,%d", 
         new->barchart.fmax, new->barchart.fmid, new->barchart.fmin);
#endif 
}
/* This function copies the items from the user passed values into its own
** array. If you choose to not copy these items, do not use this function. */
static void CopyItems(w)
XvBarChartWidget w;
{
     register int i;
     int  *il, *ial, *iptr;
     if (w->barchart.auxItems == NULL)
     {
          printf("\n AUXITEMS IS NULL ! \n");
          iptr = w->barchart.items;
     }
     else 
          iptr = w->barchart.auxItems;
     if ((w->barchart.itemCount) && (w->barchart.items))
     {
          il = (int *)XtMalloc(sizeof(int) * (w->barchart.itemCount));
          ial = (int *)XtMalloc(sizeof(int) * (w->barchart.itemCount));
          for(i = 0; i< w->barchart.itemCount; i++)
          {
               il[i] = w->barchart.items[i];
               ial[i] = iptr[i];
          }
          w->barchart.items = il;
          w->barchart.auxItems = ial;
     }
}
/* Destroy Method. -- Note that this function does not remove any widgets
**  themselves, it simply marks them as destroyed and removes all GCs.
**  Note also the explicit check of the NULL pointer.  */
static void Destroy(w)
XvBarChartWidget     w;
{
     XtReleaseGC(w, w->barchart.positiveGC);
     XtReleaseGC(w, w->barchart.negativeGC);
     XtReleaseGC(w, w->barchart.zeroGC);
     XtReleaseGC(w, w->barchart.inverseGC);
     if (w->barchart.itemHt != NULL) XtFree(w->barchart.itemHt);
     if (w->barchart.items != NULL) XtFree(w->barchart.items);
     if (w->barchart.auxItems != NULL) XtFree(w->barchart.auxItems);
     if (w->barchart.namelen != 0) free(w->barchart.name);
     XtRemoveAllCallbacks(w, XtNselectCallback, w->barchart.select );
}
/*: Resize Method -- Resize the widgets internal geometry. Remap X & Y 
**  coordinate systems from lower left  to top right corner of window 
** (relative to user). ** Calculate for all items in items list. */
static void Resize(w)
XvBarChartWidget w;
{
     int  ndiff, pdiff, val,  i;
     XRectangle     *rptr, *auxptr;
     w->barchart.itemWidth = 0;
     if (w->barchart.itemCount == 0) return;
     if (w->barchart.minValue == w->barchart.midValue)
          w->barchart.midValue = w->barchart.minValue + 1;
     /* Get the width of each bar. */
     w->barchart.itemWidth = 
               (int)(w->core.width)/(w->barchart.maxItemCount);
     w->barchart.fmax = w->core.height - 20;
     /* Calculate the location of the axis  */
     w->barchart.fmin = 0;
     w->barchart.fmid = (w->barchart.midValue - w->barchart.minValue)
         * w->barchart.fmax;
     w->barchart.fmid /= (w->barchart.maxValue - w->barchart.minValue);
     w->barchart.fmid = w->barchart.fmax - w->barchart.fmid;
     /* Get extents of the minimum and maximum values in screen coordinates. */
     ndiff = w->barchart.fmax - w->barchart.fmid; 
     pdiff = w->barchart.fmid - w->barchart.fmin; /* lower height */
     /* Map each item in the array to screen. You already have width, now 
     ** calculate height of each item.  */
     for (i = 0; i < w->barchart.maxItemCount; i++)
     {
          rptr = &(w->barchart.itemHt[i]);
          auxptr = &(w->barchart.auxItemHt[i]);
          rptr->x = i * w->barchart.itemWidth;
          rptr->width = w->barchart.itemWidth;
          val = w->barchart.items[i];
          if (val > w->barchart.midValue)
          {
               if (val > w->barchart.maxValue)
               {
                    rptr->height = pdiff;
                    rptr->y = 0;
               }
               else {
                    rptr->height = 
                         ((val - w->barchart.midValue)*(pdiff))/
                        (w->barchart.maxValue - w->barchart.midValue);
                    rptr->y = w->barchart.fmid - rptr->height;
               }
          }
          else
          {
               rptr->y = w->barchart.fmid;
               if (val < w->barchart.minValue)
               {
                    rptr->height = ndiff;
               }
               else {
                    rptr->height = 
                         ((w->barchart.midValue - val)*(ndiff))/       
                        (w->barchart.midValue - w->barchart.minValue);
               }
          } /* end of outer if loop for lower height */
          val = w->barchart.auxItems[i];
          if (val > w->barchart.midValue)
          {
               if (val > w->barchart.maxValue)
               {
                    auxptr->height = pdiff;
                    auxptr->y = 0;
               }
               else {
                    auxptr->height = 
                         ((val - w->barchart.midValue)*(pdiff))/
                        (w->barchart.maxValue - w->barchart.midValue);
                    auxptr->y = w->barchart.fmid - auxptr->height;
               }
          }
          else
          {
               auxptr->y = w->barchart.fmid;
               if (val < w->barchart.minValue)
               {
                    auxptr->height = ndiff;
               }
               else {
                    auxptr->height = 
                    ((w->barchart.midValue - val)*(ndiff))/       
                        (w->barchart.midValue - w->barchart.minValue);
               }
          } /* end of outer if loop for lower height */
     }  /* end of for loop */
}
/* Redisplay Method -- Called when an Expose event is received. Try not to 
** set a breakpoint in here while in dbx. */
static void Redisplay( w, event, region)
XvBarChartWidget     w;
XEvent          *event;
Region          region;
{
     register int i;
     register XRectangle *rptr;
     int len;
     char str[31];
#ifdef DEBUG_BARCHART_DISPLAY 
     printf("\n Display routine called....!!");
#endif
     if (w->core.visible)
     {
          for( i=0; i< w->barchart.maxItemCount ; i++)
          {
               rptr = &(w->barchart.itemHt[i]);
               if (rptr->y < w->barchart.fmid)
                    XFillRectangle(XtDisplay(w),XtWindow(w), 
                        w->barchart.positiveGC, 
                        (int)(rptr->x), (int)(rptr->y), 
                        rptr->width, rptr->height);
               else XFillRectangle(XtDisplay(w),XtWindow(w), 
                   w->barchart.negativeGC, 
                   (int)(rptr->x), (int)(rptr->y), 
                   rptr->width, rptr->height);
               XDrawLine(XtDisplay(w), XtWindow(w), 
                   w->barchart.zeroGC, 
                   0, w->barchart.fmid, 
                   w->core.width, w->barchart.fmid);
          }
          /* If you want to display the contents of array set it up here. */
          if (w->barchart.showValue)
          {
               i = w->barchart.maxShowValue;
               sprintf(str,"%d",i);
               len = strlen(str);
               if (i >= w->barchart.midValue)
                    XDrawString(XtDisplay(w),XtWindow(w), 
                         w->barchart.positiveGC, 
                         2, (int)(w->core.height - 5), str, len);
               else
                    XDrawString(XtDisplay(w),XtWindow(w), 
                         w->barchart.negativeGC, 
                         2, (int)(w->core.height - 5), str, len);
               i = w->barchart.minShowValue;
               sprintf(str,"%d",i);
               len = strlen(str);
               if (i >= w->barchart.midValue)
                    XDrawString(XtDisplay(w),XtWindow(w), 
                         w->barchart.positiveGC, 
                        (int)(w->core.width - 25), 
                        (int)(w->core.height - 5), str, len);
               else
                    XDrawString(XtDisplay(w),XtWindow(w), 
                         w->barchart.negativeGC, 
                         (int)(w->core.width - 25), 
                        (int)(w->core.height - 5), str, len);
          }
          /* If user requested that the label be shown.  */
          if (w->barchart.showChartLabel)
          {
               /*  This is a histogram counter display label. */
               sprintf(str,"%3d Max %3d Items",
                   w->barchart.maxValue, w->barchart.itemCount);
               len = strlen(str);
               if (len > 1)
                    XDrawString(XtDisplay(w),XtWindow(w), w->barchart.zeroGC, 
                        (int)(w->core.width/4), 
                        (int)(w->core.height - 5), 
                        str, len);
               /* w->barchart.chartLabel, len); */
          }
     }
}
/* SetValues Method.  Ptr current points to the widget before changes. Ptr 
** request points to a widget with requested changes. Ptr new points to a 
** widget after requested changes are applied to the current widget. */
static Boolean SetValues( current, request, new)
XvBarChartWidget     current, request, new;
{
     XGCValues          values;
     XtGCMask          valueMask;
     Boolean          redraw = False;
     redraw = current->barchart.redraw;
#ifdef DEBUG_BARCHART_DISPLAY
     if (redraw) printf("\nBarChart.SetValues: redraw is True");
     else printf("\nBarChart.SetValues: redraw is FALSE");
#endif
/* Do not adjust the size if requested to go to zero.  */
if (new->core.width == 0) new->core.width = current->core.width;
if (new->core.height == 0) new->core.height = current->core.height;
        /* Check if the color has changed */
if ((new->core.background_pixel != current->core.background_pixel) || 
    (new->barchart.positiveColor != new->barchart.positiveColor) || 
    (new->barchart.negativeColor != new->barchart.negativeColor) || 
    (new->barchart.zeroColor != new->barchart.zeroColor))
     {
          XtReleaseGC(new, new->barchart.negativeGC);
          XtReleaseGC(new, new->barchart.positiveGC);
          XtReleaseGC(new, new->barchart.zeroGC);
          XtReleaseGC(new, new->barchart.inverseGC);
          MakeGC( new );
          redraw = True;
     }
        /* Check if the names or the label has changed.  */
     if ((new->barchart.name != NULL)
         && (current->barchart.name != NULL))
     {
          if (strcmp(new->barchart.name,current->barchart.name) != 0)
               redraw = True;
     }
     if (new->barchart.namelen != request->barchart.namelen)
     {
          if (current->barchart.namelen) free(current->barchart.name);
          new->barchart.name = malloc(request->barchart.namelen + 1);
          new->barchart.namelen = request->barchart.namelen;
          strncpy(new->barchart.name, request->barchart.name,
              request->barchart.namelen);
          redraw = True;
     }
     if (new->barchart.chartLabel != request->barchart.chartLabel)
     {
          if (current->barchart.chartLabel) 
               free(current->barchart.chartLabel);
          new->barchart.chartLabel = 
               malloc(strlen(request->barchart.chartLabel)+ 1);
          strcpy(new->barchart.chartLabel,
                    request->barchart.chartLabel);
          redraw = True;
     }
     if ( new->barchart.maxItemCount != current->barchart.itemCount)
     {
          redraw = True;
     }
     if(( new->barchart.midValue != current->barchart.midValue)||
         ( new->barchart.midValue != current->barchart.midValue)||
         ( new->barchart.midValue != current->barchart.midValue))
     {
          if  ((new->barchart.maxValue <= new->barchart.midValue) ||
              (new->barchart.maxValue <= new->barchart.minValue))
          {
               new->barchart.midValue == current->barchart.midValue;
               new->barchart.minValue == current->barchart.minValue;
               new->barchart.maxValue == current->barchart.maxValue;
          }
          if (new->barchart.midValue == new->barchart.minValue)
          {
          new->barchart.midValue == new->barchart.minValue + 1;
          }
          redraw = True;
     }
     if (new->barchart.items != current->barchart.items)
     {
     /*  You have to change entire list structure.  Delete all other items. */
#ifdef DEBUG_BARCHART
printf("\n the current item address = %lx", current->barchart.items);
#endif
          XtFree(current->barchart.items);   /* the unaltered state */
          if (new->barchart.itemCount != current->barchart.itemCount)
          {
          XtFree(current->barchart.itemHt);
          XtFree(current->barchart.auxItemHt);
          new->barchart.itemHt = (XRectangle *)XtMalloc(sizeof(XRectangle) *
                 (new->barchart.itemCount));
          new->barchart.auxItemHt = (XRectangle *)XtMalloc(sizeof(XRectangle) *
                 (new->barchart.itemCount));
          }
          CopyItems(new);
          redraw = True;
     }
#ifdef DEBUG_BARCHART_DISPLAY
     if (XtIsRealized(new)) printf("\n new is Realized");
     else printf("\n new is NOT Realized");
     if (new->core.visible) printf("\n new is VISIBLE");
     else printf("\n new is INVISIO!!!!");
#endif
     if ((redraw) && XtIsRealized(new) && (new->core.visible))
     {
          XFillRectangle(XtDisplay(current),XtWindow(current), 
              current->barchart.negativeGC, 0, 0, 
              current->core.width, current->core.height);
          Resize(new);
     }
     /* Return value to indicate if we have to redraw */
     return(redraw);
}
/* Callback function hook. */
static void select_barchart(w, event, args, n_args)
XvBarChartWidget    w;
XEvent              *event;
char                *args[];
int                 n_args;
{
     xvbarchartCallbackStruct     cb;
     cb.reason = 1;
     cb.event  = event;
     cb.name = malloc(w->barchart.namelen + 1);
     strcpy(cb.name, w->barchart.name);
     XtCallCallbacks(w, XtNselectCallback, &cb);
     free(cb.name);
}
/* Convenience Function to add a new item to the Bar Chart. This adds to items 
** array at end of array and adjusts using a rather inefficient algorithm. */
int XvBarChartAddItem( w, val)
XvBarChartWidget w;
int  val;
{
     int t, i;
     Arg wars[2];
     if ( w->barchart.itemCount > w->barchart.maxItemCount )
          w->barchart.itemCount = w->barchart.maxItemCount ;
     if ( w->barchart.itemCount == w->barchart.maxItemCount )
     {
          for (i = 0; i < w->barchart.maxItemCount - 1; i++)
               w->barchart.items[i] = w->barchart.items[i+1];
          w->barchart.items[w->barchart.itemCount - 1] = val;
#ifdef DEBUG_BARCHART
          printf("\n The list now looks like:");
          for (i = 0; i < w->barchart.maxItemCount; i++)
               printf(" [%d]", w->barchart.items[i]);
#endif
     }
     else
     {
          w->barchart.itemCount += 1;
          w->barchart.items= (int *)XtRealloc(w->barchart.items,
              w->barchart.itemCount * sizeof(int));
          w->barchart.items[w->barchart.itemCount - 1] = val;
#ifdef DEBUG_BARCHART
          printf("\n* The list now looks like:");
          for (i = 0; i < w->barchart.maxItemCount; i++)
               printf(" [%d]", w->barchart.items[i]);
#endif
     }
     Resize(w);
     w->barchart.redraw = True;
     t = 0;
     XtSetArg(wars[t], XtNredraw, True); 
     t++;
     XtSetValues(w, wars, t);
     XtManageChild(w);
}
/* Convenience Function to reset limits on minimum and maximum values 
** to display. */
int XvBarChartResetLimits(w, max, min)
XvBarChartWidget w;
int max, min;
{
     int n;
     Arg wars[5];
     if (max == min) return; /* no allowance for idiocy */
     if (max < min) { 
          n = min; 
          min = max; 
          max = n; 
     }
     n = 0;
     XtSetArg(wars[n], XtNmaxValue, max); 
     n++;
     XtSetArg(wars[n], XtNminValue, min); 
     n++;
     XtSetValues(w, wars, n);
     Resize(w);
}
