/*
** LineChart widget source file. 
** 
** This file inherits all the functionality it needs 
** from the BarChart Widget and adds a new drawing 
** function to display the auxItems and items array
** as line plots. 
**
** This is shown as an example of inheritance and can
** be used to extend the idea to add graticules, vertical
** and horizontal axes, etc. as needed.
**
** Author: Kamran Husain	Mentor Programming Services Inc. 
**
*/

#include <stdio.h> 
#include <math.h>
#include <X11/X.h>
#include <X11/IntrinsicP.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/CoreP.h>
#include "Xv.h"
#include "BarChartP.h"
#include "BarChart.h"
#include "LineChartP.h"
#include "LineChart.h"

#define  MIN(a,b)   (((a)<(b))?(a):(b))
#define  MAX(a,b)   (((a)>(b))?(a):(b))
#define  ABS(x)	(((x)< 0) ? (-1 * (x)) : (x))
#define  MID(a,b)	(ABS((((a)-(b))/2)))


/*
** These functions are redefined here. The rest are
** inherited from the BarChart widget.
*/
static void LineChartInitialize();
static void LineChartResize();
static void LineChartRedisplay();


/*
**	Set the Class record up.
*/
XvLineChartClassRec  xvlinechartClassRec = {
	{ 
	(WidgetClass) &xvbarchartClassRec, /* for super class       */
	"LineChart",                    /* class_name            */
	sizeof(XvLineChartRec),         /* widget_size           */
	NULL,                           /* class initialize      */
	NULL,                           /* class part init       */
	FALSE,                          /* class inited          */
	LineChartInitialize,            /* defined function      */
	NULL,                           /* initialize hook       */
	XtInheritRealize,               /* from the core         */
	NULL,                           /* actions               */
	0,                              /* for this widget       */
	NULL,                           /* et al.                */
	0,                              /* for this widget       */
	NULLQUARK,                      /* default initialization*/
	TRUE,                           /* compress_motion       */
	TRUE,                           /* compress_exposure     */
	TRUE,                           /* compress_enterleave   */
	TRUE,                           /* visible_interest      */
	NULL,                           /* defined function      */
	LineChartResize,                /* defined function      */
	LineChartRedisplay,             /* for expose function   */
	NULL,                           /* defined function      */
	NULL,                           /* set values hook       */
	XtInheritSetValuesAlmost,       /* set_values_almost     */
	NULL,                           /* get values hook       */
	XtInheritAcceptFocus,           /* accept focus          */
	XtVersion,                      /* default initialization*/
	NULL,                           /* callback private      */
	XtInheritTranslations,          /* default xtlations     */
	NULL,                           /* query geometry        */
	NULL,                           /* display accelerator   */
	NULL                            /* extension             */
	},
	{ 0 }, 	/* default */ 
	{ 0 }  	/* default */
};


/* 
** Declare the Class record. 
*/
WidgetClass XvlinechartWidgetClass =
		(WidgetClass) &xvlinechartClassRec;


/*
** Method. LineChartInitialize 
** Note that since we are inheriting from the BarChart 
** widget, we do not have to check the limits as 
** strictly. These are done in the BarChart's Initialize
** function for us. We test stuff that is applicable to
** this particular widget only.
**/
static void LineChartInitialize( request, new)
XvLineChartWidget     request, new;
{
	/*
	** simply resize yourself 
	*/
#ifdef DEBUG_LINECHART
	printf("\n LineChart at %d,%d with %d items", 
		new->core.x, new->core.y, new->barchart.itemCount);
#endif

	LineChartResize(new);
}

/*
** Method. LineChartResize
** This is very simiar to the BarChart's resize function.
** It does have differences, therefore cannot be inherited,
** since it uses the auxItems array where the BarChart does 
** not. 
**
** I had removed the auxItems display routine from the BarChart
** Widget to simplify that code.
**/
static void LineChartResize(w)
XvLineChartWidget w;
{
	int 	ndiff, pdiff, val,  i;
	XRectangle 	*rptr, *auxptr;

	w->barchart.itemWidth = 0;
	if (w->barchart.itemCount == 0) return;

	w->barchart.itemWidth = 
			(int)(w->core.width)/(w->barchart.maxItemCount);
	w->barchart.fmax = w->core.height - 20;
	/* m001 compensate for displayed string */
	w->barchart.fmin = 0;
	w->barchart.fmid = 
		(w->barchart.midValue - w->barchart.minValue)
	    * w->barchart.fmax;
	w->barchart.fmid /= (w->barchart.maxValue - w->barchart.minValue);
	w->barchart.fmid = w->barchart.fmax - w->barchart.fmid;

	ndiff = w->barchart.fmax - w->barchart.fmid; /* upper height */
	pdiff = w->barchart.fmid - w->barchart.fmin; /* lower height */

#define DEBUG_LINECHART 
#ifdef DEBUG_LINECHART
	printf("\n LineChart limits at %d,%d ", ndiff, pdiff);
#endif

	for (i = 0; i < w->barchart.maxItemCount; i++)
	{
		rptr = &(w->barchart.itemHt[i]);
		rptr->x = i * w->barchart.itemWidth;
		rptr->width = w->barchart.itemWidth;
		val = w->barchart.items[i];
		if (val > w->barchart.midValue)
		{	
			if (val > w->barchart.maxValue)
			{
				rptr->y = 0;
			}
			else {
				rptr->y = 
					((val - w->barchart.midValue)*(pdiff))/
				    (w->barchart.maxValue - w->barchart.midValue);
				rptr->y = w->barchart.fmid - rptr->y;
			}
		printf("\n Value %d  (%d) is greater than mid", 
				val, rptr->y); 
		}
		else
		{
			if (val < w->barchart.minValue)
			{
				/* rptr->height = ndiff; */
				rptr->y  = ndiff;
			}
			else {
				rptr->y = 
					((w->barchart.midValue - val)*(ndiff))/       
				    (w->barchart.midValue - w->barchart.minValue);
				/*
				rptr->y = ndiff - rptr->y; 
				*/
			}
		printf("\n Value %d  (%d) is less than mid", 
				val, rptr->y); 
		} /* end of outer if loop for lower height */


	/*
	** "New" code to display the auxiliary items for
	** the second data array to display.
	** This is removed from the BarChart code for clarity.
	*/
		auxptr = &(w->barchart.auxItemHt[i]);
		auxptr->x = i * w->barchart.itemWidth;
		auxptr->width = w->barchart.itemWidth;
		val = w->barchart.auxItems[i];
		if (val > w->barchart.midValue)
		{
			if (val > w->barchart.maxValue)
			{
				auxptr->height = pdiff;
				auxptr->y = 0;
			}
			else {
				auxptr->height = 
					((val - w->barchart.midValue)*(pdiff))/
				    (w->barchart.maxValue - w->barchart.midValue);
				auxptr->y = w->barchart.fmid - auxptr->height;
			}
		}
		else
		{
			auxptr->y = w->barchart.fmid;
			if (val < w->barchart.minValue)
			{
				auxptr->height = ndiff;
			}
			else {
				auxptr->height = 
					((w->barchart.midValue - val)*(ndiff))/       
				    (w->barchart.midValue - w->barchart.minValue);
			}
		} /* end of outer if loop for lower height */


#ifdef DEBUG_LINECHART
		printf("\n [%d] Value\t[%d]\tat [%d,%d]\tWd,Ht[%d,%d]", 
		    i,val, rptr->x, rptr->y, rptr->width, rptr->height);
#endif
	}  /* end of for loop */
}


/*
** Method  LineChartRedisplay 
** This is the function to redraw in the case of an 
** Expose event. 
*/
static void LineChartRedisplay( w, event, region)
XvLineChartWidget     w;
XEvent          *event;
Region          region;
{
	register int i;
	register XRectangle *rptr;
	int len;
	char str[15];
	XPoint *xpts;

	if (w->core.visible)
	{
#ifdef DEBUG_DISPLAY 
	puts("Line Chart Display routine called....!!");
	puts("and it's visible..."); 
#endif

		/*
		** Create the points to draw for items.
		*/
		xpts = (XPoint *)XtMalloc(
		    sizeof(XPoint) * w->barchart.maxItemCount);
		for( i=0; i< w->barchart.maxItemCount ; i++)
		{
			rptr = &(w->barchart.itemHt[i]);
			xpts[i].x = (int)(rptr->x) + 5;
			xpts[i].y = (int)(rptr->y);
		}
		/*
		** Draw all the lines for items now
		*/
		XDrawLines(XtDisplay(w),XtWindow(w), 
		    w->barchart.positiveGC, 
		    xpts, w->barchart.maxItemCount, 
		    CoordModeOrigin);
		/*
		** Create the points to draw for auxItems.
		*/
		for( i=0; i< w->barchart.maxItemCount ; i++)
		{
			rptr = &(w->barchart.auxItemHt[i]);
			xpts[i].x = (int)(rptr->x) + 5;
			xpts[i].y = (int)(rptr->y);
		}

		/*
		** Draw all the lines for auxItems now
		*/
		XDrawLines(XtDisplay(w),XtWindow(w), 
		    w->barchart.negativeGC, 
		    xpts, w->barchart.maxItemCount, 
		    CoordModeOrigin);
		XtFree(xpts);

		/*
		** Draw the axes.
		*/

		XDrawLine(XtDisplay(w),XtWindow(w), 
		    w->barchart.zeroGC, 
		    0, w->core.height - 5, 
		    w->core.width, 
		    w->core.height - 5);
		XDrawLine(XtDisplay(w),XtWindow(w), 
		    w->barchart.zeroGC, 
		    2, 0, 2, w->core.height);

		/*
		** Show the number of items.
		*/

		if (w->barchart.showValue)
		{
			sprintf(str,"%d",w->barchart.itemCount);
			len = strlen(str);
			XDrawString(XtDisplay(w),XtWindow(w), 
			w->barchart.positiveGC, 
		    (int)(w->core.width - 25), 
		    (int)(w->core.height - 5), str, len);

			sprintf(str,"%d",w->barchart.maxValue);
			len = strlen(str);
			XDrawString(XtDisplay(w),XtWindow(w), 
					w->barchart.positiveGC, 0, (15), str, len);

			sprintf(str,"%d",w->barchart.minValue);
			len = strlen(str);
			XDrawString(XtDisplay(w),XtWindow(w), 
				w->barchart.positiveGC, 
		    	0, (int)(w->core.height - 10), str, len);

		}

		/*
		** show the label if the resource is set
		*/
		if (w->barchart.showChartLabel)
		{
			sprintf(str,"%d,%d",
			    w->barchart.maxValue, w->barchart.minValue);
			len = strlen(str);
			if (len > 1)
				XDrawString(XtDisplay(w),XtWindow(w), 
					w->barchart.zeroGC, 
				    (int)(w->core.width/3), 
				    (int)(w->core.height - 5), 
				    str, len);
			/* w->barchart.chartLabel, len); */
		}


		/*
		** Draw the axes. 
		*/
		XDrawLine(XtDisplay(w),XtWindow(w), 
		    w->barchart.zeroGC, 
		    0, w->core.height/2,  
		    w->core.width, w->core.height/2);
		i = w->barchart.items[(w->barchart.itemCount-1)];
		sprintf(str,"%d",w->barchart.itemCount);
		len = strlen(str);
	}
}

