/****************************************************************
|
| GMHW3.C -- Module for Handwriting Data Browser
|     by Ray Valdes, DDJ
\****************************************************************/
#include <dos.h>
#include "h_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <setjmp.h>
#include <graphics.h>

/****************************************************************/

#include "h_stddef.h"
#include "h_list.h"
#include "h_mem.h"
//#include "h_fmtstr.h"
#include "h_grafic.h"
#include "h_keys.h"


/****************************************************************/

#define FILE_SIGNATURE 0x2121

#define MAX_CHARS 256
lpList  CharData[MAX_CHARS];


typedef unsigned char	    BYTE;
typedef unsigned short	    WORD;
static jmp_buf             main_loop;
static long                num_bytes = 0;
bool     graphics_enabled = FALSE;



/****************************************************************
  external prototypes
****************************************************************/
extern void gr_ShiftPoint(lpVHPoint p,INT16 x_delta,INT16 y_delta);


/****************************************************************
        These are the routines in the DDJ spec.
****************************************************************/

void     LoadHWXData(LPSTR name);
int      GetInstanceCount(int ch);
void     DisplayInstance(int char_code,int instance_num);
void     UnloadHWXData(void);

static void (*pf_MoveTo)(HDISPLAY,INT16,INT16);
static void (*pf_LineTo)(HDISPLAY,INT16,INT16);

//void RegisterCallback(
//         HDISPLAY display_entity,
//         void (*pf_move_to)(),
//         void (*pf_line_to)());

void RegisterCallback(
       HDISPLAY display_entity,
       void (*pf_move_to)(HDISPLAY,INT16,INT16),
       void (*pf_line_to)(HDISPLAY,INT16,INT16));

  /*these are the routines that get "called back". They are in this
   * module, and actually themselves call the graphics library (H_GRAFIC.C)
   */
void    myMoveTo(HDISPLAY display_entity,INT16 h,INT16 v);
void    myLineTo(HDISPLAY display_entity,INT16 h,INT16 v);

/**********These are other routines (which can also be used)**********/

static INT16   ReadWordFromFile(FILE *data_file);
static void    ParseDataFormat(FILE *data_file);

static void    FreeMemStructures(void);

//static void    ShowAllChars(INT16 begin,INT16 end);

static void    FatalError(char* msg);

//void    ShowInstanceInTable(int num_proto,lpList char_instance);
//static bool    PickSetOfInstances(void);
//void    ShowSetOfInstances(INT16 ch);
static void    CalcInstanceBoundingBox(lpList char_instance,lpRect bbox);
void    NormalizeAnInstance(lpList char_instance,lpRect bbox);
static void    NormalizeAndDisplayInstance(lpList char_instance,lpRect bbox,
      INT16 hshift,INT16 vshift,bool use_callbacks);


/****************************************************************/

/****************************************************************/
void myMoveTo(HDISPLAY display_entity,INT16 h,INT16 v)
{
    /* this implementation does not use handle to display entity */
    gr_MoveTo(h,v);
}
/****************************************************************/
void myLineTo(HDISPLAY display_entity,INT16 h,INT16 v)
{
    /* this implementation does not use handle to display entity */
    gr_LineTo(h,v);
}
/****************************************************************/
void RegisterCallback(
       HDISPLAY display_entity,
       void (*pf_move_to)(HDISPLAY,INT16,INT16),
       void (*pf_line_to)(HDISPLAY,INT16,INT16))
{
    /*stash pointers away, ignore handle to display_entity */
    pf_MoveTo = pf_move_to;
    pf_LineTo = pf_line_to;
}
/****************************************************************/
void LoadHWXData(LPSTR file_name)
{
    FILE *data_file;

    data_file = fopen(file_name,"rb");
    if (!data_file)
    {
         char msg[80];
         sprintf( msg, "\n***File %s not found!\n",file_name );
         FatalError(msg);
    }

    FreeMemStructures();
    ParseDataFormat(data_file);
    fclose(data_file);
}
/****************************************************************/
void UnloadHWXData(void)
{
    FreeMemStructures();
}
/****************************************************************/
int GetInstanceCount(int ch)
{
    if(CharData[ch]) return CharData[ch]->num_items;
    else return 0;
}
/****************************************************************/
static void FreeMemStructures(void)
{
    INT16 i,j,k;
    for (i = 0; i < MAX_CHARS; i++) if (CharData[i])
    {
        for (j = 0; j < ((struct tagList*) CharData[i])->num_items; j++)
	{
            lpList char_instance = (struct tagList*) CharData[i]->items[j];
            for (k = 0; k < char_instance->num_items; k++)
                mem_Free(char_instance->items[k]);
            mem_Free(char_instance);
        }
        mem_Free(CharData[i]);
        CharData[i] = NULL;
    }
}
/****************************************************************/
static INT16  ReadWordFromFile(FILE *data_file)
{
    char s[2];
    s[0] = s[1] = 0;
    if(fread(&s, sizeof(char), 2, data_file) != 2)
	longjmp( main_loop, -1 );
    num_bytes += 2;
    return (INT16)( ((BYTE)s[1]) | ((WORD)((BYTE)s[0])<<8) );
}
/****************************************************************/
static void ReadHeader(FILE *data_file)
{
    INT16 signature,header_size,version_major,version_minor;
    signature = ReadWordFromFile(data_file);
    if(signature != FILE_SIGNATURE) FatalError("***Header format error!***");
    header_size = ReadWordFromFile(data_file); //number of bytes should be even
    if(header_size & 01) FatalError("***Header format error #2!***");
    header_size >>= 1;  // number of words
    header_size--;
    version_major = ReadWordFromFile(data_file); header_size--;
    version_minor = ReadWordFromFile(data_file); header_size--;
    while(header_size--) ReadWordFromFile(data_file);
    //the following line serves only to get rid of a compiler warning
    version_major++; version_minor++;
}
/****************************************************************/
static void ParseDataFormat(FILE *data_file)
{
    VHPoint Pt;
    lpList   char_instance;
    lpList   stroke;
    INT16    i,j,k,num_strokes,num_pts,which;
    INT16    char_code,num_protos;


    if( setjmp( main_loop )==0 )
    {
	ReadHeader(data_file);
	while (1)
        {
        char_code = ReadWordFromFile(data_file);
        num_protos = ReadWordFromFile(data_file);

        which = (char_code < MAX_CHARS) ? char_code : 1;
        CharData[which] = (struct tagList *) mem_Alloc(sizeof(INT16) + num_protos * sizeof(long));
	if(!CharData[which])
	    mem_AllocFailed(sizeof(INT16) + num_protos * sizeof(long));
        CharData[which]->num_items = num_protos;
        for (i = 0; i < num_protos; i++)
	{
            num_strokes = ReadWordFromFile(data_file);

            char_instance = (struct tagList *)mem_Alloc(sizeof(INT16) + num_strokes * sizeof(long));
	    if(!char_instance)
		mem_AllocFailed(sizeof(INT16) + num_strokes * sizeof(long));
            char_instance->num_items = num_strokes;
            for (j = 0; j < num_strokes; j++)
	    {
                num_pts = ReadWordFromFile(data_file);
		Pt.h = ReadWordFromFile(data_file);
		Pt.v = ReadWordFromFile(data_file);

                stroke = (struct tagList *)mem_Alloc(sizeof(INT16) + num_pts * sizeof(long));
		if(!stroke)
		    mem_AllocFailed(sizeof(INT16) + num_pts * sizeof(long));
                stroke->num_items = num_pts;
                stroke->items[0] = *(void**)&Pt;
                for (k = 1; k < num_pts; k++)
		{
		    INT16 s = ReadWordFromFile(data_file);
		    LPSTR ps = (LPSTR )&s;

                    Pt.h += ps[1]; Pt.v += ps[0];
                    stroke->items[k] = *(void**)&Pt;
                } /* for loop (k) */
                char_instance->items[j] = stroke;

            } /* for loop (j) */
            CharData[which]->items[i] = char_instance;
           }  /* for loop (i) */
         } /*while loop */
       } /*if*/
       else if (!feof(data_file))
       {
         char msg[80];
         sprintf( msg, "\n***Err reading file at loc %ld!",num_bytes );
         FatalError(msg);
       }
}
/****************************************************************/
static void FatalError(char* msg)
{

    fprintf(stderr,"\n***Fatal Error: ");
    fprintf(stderr,msg);
    getch();
    exit(0);
}
/****************************************************************/

#define MIN_VALUE (-0x7FFF)
#define MAX_VALUE ( 0x7FFF)

static void CalcInstanceBoundingBox(lpList char_instance,lpRect bb)
{
    INT16 i,j;
    INT16 x_min,x_max,y_min,y_max;

     x_min = y_min = MAX_VALUE;
     x_max = y_max = MIN_VALUE;

    /*------for a given char, find the maximum and minimum -------*/
    for (i = 0; i < char_instance->num_items; i++)
    {
        lpList    stroke =   (struct tagList *)char_instance->items[i];

        if (stroke != NULL)
        {
          for (j = 0; j < stroke->num_items; j++)
          {
              VHPoint p = *(lpVHPoint)&stroke->items[j];
              if (p.h > x_max) x_max = p.h;
              if (p.h < x_min) x_min = p.h;
              if (p.v > y_max) y_max = p.v;
              if (p.v < y_min) y_min = p.v;
          }                                                 
        }
    }
    bb->left  = x_min;    bb->top   = y_min;
    bb->right = x_max;    bb->bottom = y_max;
}
/****************************************************************/
void NormalizeAnInstance(lpList char_instance,lpRect bb)
{
    /*-------for a given char instance, normalize to xmin and ymin----*/
    INT16 i,j;
    INT16 x_min,y_min;

    x_min = bb->left;
    y_min = bb->top;

    for (i = 0; i < char_instance->num_items; i++)
    {
        lpList    stroke =   (struct tagList *)char_instance->items[i];

        for (j = 0; j < stroke->num_items; j++)
	{
            VHPoint p = *(lpVHPoint)&stroke->items[j];
	    /* subtract minimum from the character */
	    p.h -= (x_min - 100);
	    p.v -= (y_min - 100);

            *(lpVHPoint)&stroke->items[j] = p;
        }
    }
}
/****************************************************************/
static void NormalizeAndDisplayInstance(lpList char_instance, lpRect bbox,
        INT16 hshift,INT16 vshift,bool use_callbacks)
{
    INT16 i,j;
    INT16 x_min,y_min;

    x_min = bbox->left;
    y_min = bbox->top;

    for (i = 0; i < char_instance->num_items; i++)
    {
        lpList    stroke =   (struct tagList *)char_instance->items[i];

        for (j = 0; j < stroke->num_items; j++)
	{
            VHPoint p = *(lpVHPoint)&stroke->items[j];
	    p.h -= x_min;
	    p.v -= y_min;
	    gr_CvtTabPtToScreen(&p);
	    gr_ShiftPoint(&p,hshift,vshift);

	    if(use_callbacks)
	    {
		if(j==0) (*pf_MoveTo)(0,p.h,p.v);
		else     (*pf_LineTo)(0,p.h,p.v);
	    }
// This isn't necessary, and wastes a lot of space in PostScript
//      gr_PutPoint(&p);
        }
    }
}

/***************************************************************/

int  omShowInstanceInTable( Rect boundR, int instance_num,
                            lpList char_instance, int shift,
                            int boxcolor, int drawcolor, int xoffset, int yoffset)
{

// RETURNS:
//   = -1;  // drawing "above" screen
//   =  0;  // Ok, drawining on screen
//   =  1;  // drawing "below" screen


// Spec'd to hardcode 5 instances, no auto-wrap.
int NUM_INST_PER_ROW = 5; // (boundR.right-boundR.left) / CELL_WIDTH;

int H_TABLE_START = xoffset;
int V_TABLE_START = yoffset;


INT16 hshift,vshift;
Rect bbox_tab;  /* in world/object/tablet coordinates */
Rect bbox_scr;  /* in screen coordinates */
Rect bbox_out;  /* shifted for output */
INT16 theCELL_WIDTH, theCELL_HEIGHT;
INT16 aCELL_WIDTH, aCELL_HEIGHT;


  aCELL_WIDTH  = CELL_WIDTH;   // defined in h_grafic.h
  aCELL_HEIGHT = CELL_HEIGHT;  // defined in h_grafic.h

//    int theCELL_WIDTH  = CELL_WIDTH  >> (3-shift);
//    int theCELL_HEIGHT = CELL_HEIGHT >> (3-shift);
  gr_SetTabToScrConversion(shift,shift);
  gr_TabletToScreen(aCELL_WIDTH<<3, aCELL_HEIGHT<<3,
      &theCELL_WIDTH, &theCELL_HEIGHT);



    hshift = (instance_num % NUM_INST_PER_ROW) * theCELL_WIDTH;
    vshift = (instance_num / NUM_INST_PER_ROW) * theCELL_HEIGHT;
    hshift += H_TABLE_START;
    vshift += V_TABLE_START;

    // Following can be used as AN EXIT CONDITION:
    if (vshift < /*0*/ - aCELL_HEIGHT)     return  -1; // drawing above boundR
    if (boundR.top+vshift>boundR.bottom)  return   1; // drawing below boundR

    CalcInstanceBoundingBox(char_instance,&bbox_tab);
    bbox_scr = bbox_tab; /*structure copy */
//  gr_SetTabToScrConversion(shift,shift);

    gr_ConvertRectTabletToScreen(&bbox_scr);
    gr_SetRect(&bbox_out,
  hshift,vshift,
  hshift+theCELL_WIDTH,
  vshift+theCELL_HEIGHT);

    setcolor(boxcolor);
    gr_DrawRect(&bbox_out);

    hshift += ((theCELL_WIDTH  - (bbox_scr.right  - bbox_scr.left)) / 2);
    vshift += ((theCELL_HEIGHT - (bbox_scr.bottom - bbox_scr.top )) / 2);

    gr_SetRect(&bbox_out,
  hshift,vshift,
  hshift+(bbox_scr.right  - bbox_scr.left),
  vshift+(bbox_scr.bottom - bbox_scr.top));
//    gr_SetColorGray();
// I'm taking out the extra box
//    setcolor(boxcolor);
//    gr_DrawRect(&bbox_out);
//    gr_SetColorYellow();
    setcolor(drawcolor);
    NormalizeAndDisplayInstance(char_instance,&bbox_tab,hshift,vshift,1);

    return 0;
}


//--------------------------------------------------------------------------
void omDisplayInstance(lpRect R, int char_code, int instance_num,
                       int shift, int boxcolor, int drawcolor)
{
    INT16 hshift,vshift, dx, dy;
    Rect bbox_tab;  /* in world/object/tablet coordinates */
    Rect bbox_scr;  /* in screen coordinates */
    Rect bbox_out;  /* shifted for output */

    lpList char_instance;

    if (CharData[char_code]==NULL) return;
    if (CharData[char_code]->num_items<=0) return;
    char_instance = (struct tagList *)CharData[char_code]->items[instance_num];
    if (char_instance == NULL || char_instance->num_items == 0) return;

    hshift = R->left;
    vshift = R->top;
    dx = R->right  - R->left;
    dy = R->bottom - R->top;

    gr_SetTabToScrConversion(shift,shift);
    CalcInstanceBoundingBox(char_instance, &bbox_tab);

    bbox_scr = bbox_tab; /*structure copy */
    gr_ConvertRectTabletToScreen(&bbox_scr);

    gr_SetRect( &bbox_out,
                hshift, vshift,
                hshift + dx,
                vshift + dy);

    setcolor(boxcolor);
    gr_DrawRect(&bbox_out);


    hshift += ((dx  - (bbox_scr.right  - bbox_scr.left)) / 2);
    vshift += ((dy  - (bbox_scr.bottom - bbox_scr.top )) / 2);

    gr_SetRect( &bbox_out,
                hshift,vshift,
                hshift+(bbox_scr.right  - bbox_scr.left),
                vshift+(bbox_scr.bottom - bbox_scr.top) );

//  setcolor(boxcolor);
//  gr_DrawRect(&bbox_out);

    setcolor(drawcolor);
    NormalizeAndDisplayInstance( char_instance, &bbox_tab,
                                 hshift, vshift, 1 );
}


//-------------------------------------------------------------------------
void omShowSetOfInstances(Rect boundR, INT16 ch, int shift,
                          int boxcolor, int drawcolor, int xoffset, int yoffset)
{
    INT16 j;

    if (CharData[ch]==NULL) return;
    for (j = 0; j < ((struct tagList *) CharData[ch])->num_items; j++)
    {
      if ( omShowInstanceInTable(  boundR, j,
                                   (struct tagList *)CharData[ch]->items[j],
                                   shift, boxcolor, drawcolor,
                                   xoffset, yoffset) == 1)  return;
    }
}

//-------------------------------------------------------------------------
void omShowAlphabet( Rect boundR, INT16 inst, int shift,
                     int boxcolor, int drawcolor, int xoffset, int yoffset)
{
    INT16 i, ctr=0;
    for (i = 0; i < MAX_CHARS; i++)
     if (CharData[i])
     {
        if (  ((struct tagList*) CharData[i])->num_items > 0)
        {
            if ( omShowInstanceInTable( boundR, ctr,
                                        (struct tagList *)CharData[i]->items[inst],
                                        shift, boxcolor, drawcolor,
                                        xoffset, yoffset) == 1)  return;
            ctr++;
         }
     }
}

//-------------------------------------------------------------------------
void omShowAll( Rect boundR, int shift,
                     int boxcolor, int drawcolor, int xoffset, int yoffset)
{
    INT16 i,j, ctr=0;
    for (i = 0; i < MAX_CHARS; i++)
    {
      if (CharData[i])
      {
        for ( j = 0;  j < ((struct tagList *) CharData[i])->num_items;  j++)
        {
          if ( omShowInstanceInTable( boundR, ctr,
                              (struct tagList *)CharData[i]->items[j],
                              shift, boxcolor, drawcolor,
                              xoffset, yoffset ) == 1 )  return;
          ctr++;
        }
      }
    }
}
